# !/usr/bin/env python
# -*- coding: utf-8 -*-
"""
********************************************************************

* QGis-Plugin Qoogle

********************************************************************

* Date                 : 2026-02-04
* Copyright            : (C) 2023 by Ludwig Kniprath
* Email                : ludwig at kni minus online dot de

********************************************************************

this program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

********************************************************************
"""

import os, qgis, webbrowser, math
from PyQt5 import QtCore, QtGui, QtWidgets


class Qoogle:
    def __init__(self, iface: qgis.gui.QgisInterface):
        """Constructor for the Plugin.
        Triggered on open QGis

        Args:
            iface (qgis.gui.QgisInterface): interface to running QGis-App
        """
        # Rev. 2026-02-04
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)

    def initGui(self):
        """"standard-to_implement-function for plugins: adapt/extend GUI triggered by plugin-activation or project-open"""
        # Rev. 2026-02-04
        # create action that will be run by the plugin
        self.action = QtWidgets.QAction(
            QtGui.QIcon(f"{self.plugin_dir}/icons/Google_Maps_icon_(2020).svg"),
            "Qoogle...",
            self.iface.mainWindow()
        )


        self.action.setCheckable(True)

        self.iface.mapToolActionGroup().addAction(self.action)

        # add plugin to menu
        self.iface.addPluginToMenu("Qoogle", self.action)

        # add icon to plugin in toolbar and menu
        self.iface.addToolBarIcon(self.action)

        # connect action
        self.action.triggered.connect(self.run)





    def unload(self):
        """Actions to run when the plugin is unloaded"""
        # Rev. 2026-02-04
        # remove menu and icon
        self.iface.removeToolBarIcon(self.action)
        self.iface.removePluginMenu("Qoogle", self.action)
        if self.iface.mapCanvas().mapTool() == self.map_tool:
            self.iface.mapCanvas().unsetMapTool(self.map_tool)

        # sys_unload dialog and temporal graphics
        self.map_tool.unload()
        del self.map_tool

    def run(self):
        """Main-Action, triggered by click on Toolbar/Menu"""
        # Rev. 2026-02-04
        self.map_tool = QoogleMapTool(self.iface)
        self.iface.mapCanvas().setMapTool(self.map_tool)


class QoogleMapTool(qgis.gui.QgsMapToolEmitPoint):
    """the MapTool-Class"""

    def __init__(self, iface: qgis.gui.QgisInterface):
        """constructor for the MapTool

        Args:
            iface (qgis.gui.QgisInterface): interface to running QGis-App
        """
        # Rev. 2026-02-04

        qgis.gui.QgsMapTool.__init__(self, iface.mapCanvas())
        self.plugin_dir = os.path.dirname(__file__)

        self.iface = iface
        self.cursor = QtCore.Qt.CrossCursor
        self.google_maps_url = "https://www.google.com/maps/@[_lon_],[_lat_],[_zoom_]z?entry=ttu"
        self.google_place_url = "https://www.google.com/maps/place/[_lon_],[_lat_]"
        # calculate route from start to end-point
        self.google_route_url = "https://www.google.com/maps/dir/[_lon_start_],[_lat_start_]/[_lon_end_],[_lat_end_]/@[_lon_end_],[_lat_end_]?entry=ttu"

        # registered Start-Point for tool_mode "routes"
        self.start_point = None

        # internal flag with currently three possible values
        # "maps" (default)
        # "places"
        # "routes"
        # "set_start_point" (special tool-mode for "routes")
        self.tool_mode = 'maps'

        # automatically open the Gogle-Browser-Windows
        self.auto_open_browser = False

        # two temporal canvas-graphics
        # Start-Point green box for tool_mode "routes"
        self.vm_start_point = qgis.gui.QgsVertexMarker(self.iface.mapCanvas())
        self.vm_start_point.setPenWidth(2)
        self.vm_start_point.setIconSize(10)
        self.vm_start_point.setIconType(3)
        self.vm_start_point.setColor(QtGui.QColor('#ff00ff00'))
        self.vm_start_point.setFillColor(QtGui.QColor('#00ffffff'))
        self.vm_start_point.hide()

        # End-Point red box for all tool_modes
        self.vm_end_point = qgis.gui.QgsVertexMarker(self.iface.mapCanvas())
        self.vm_end_point.setPenWidth(2)
        self.vm_end_point.setIconSize(10)
        self.vm_end_point.setIconType(3)
        self.vm_end_point.setColor(QtGui.QColor('#ffff0000'))
        self.vm_end_point.setFillColor(QtGui.QColor('#00ffffff'))
        self.vm_end_point.hide()

        # simple QT-Dialog for results and handling
        self.my_dialog = QoogleDialog(iface)
        self.my_dialog.dialog_close.connect(self.dlg_close)
        self.iface.addDockWidget(QtCore.Qt.RightDockWidgetArea, self.my_dialog)
        # properties for dock/undock/dockable
        self.my_dialog.setFeatures(
            QtWidgets.QDockWidget.NoDockWidgetFeatures | QtWidgets.QDockWidget.DockWidgetClosable
        )
        self.my_dialog.setFloating(True)

        # floating positioned at right edge of main-window
        start_pos_x = int(
            self.iface.mainWindow().x()
            + self.iface.mainWindow().width()
            - self.my_dialog.width()
        )
        start_pos_y = int(self.iface.mainWindow().y() + 150)
        self.my_dialog.move(start_pos_x, start_pos_y)
        self.my_dialog.hide()
        self.my_dialog.qtb_show_help.clicked.connect(self.s_show_help)
        self.my_dialog.qtb_toggle_top_level.clicked.connect(self.s_toggle_top_level)
        self.my_dialog.qpb_copy_plaintext.clicked.connect(self.s_copy_plaintext)
        self.my_dialog.qpb_clear.clicked.connect(self.s_clear)
        self.my_dialog.qcbx_map_mode_select.currentIndexChanged.connect(self.s_change_map_mode)
        self.my_dialog.qpb_set_auto_open.clicked.connect(self.s_toggle_auto_open)
        self.my_dialog.qtb_result.anchorClicked.connect(self.s_open_browser)
        self.s_show_help()

    def s_toggle_auto_open(self,checked:bool):
        """sets runtime-variable auto_open_browser
        if True, the generated google-URLs will open in browser

        Args:
            checked (bool): check-status of checkable button
        """
        # Rev. 2026-02-04
        self.auto_open_browser = checked

    def s_show_help(self):
        """shows help-text in qtb_result
        """
        # Rev. 2026-02-04
        intro = f"""
<h3>Intro</h3>
This Plugin generates URLs for Google-Maps using click-coordinates from QGis (transformed to EPSG 4326).<br/><br/>
The generated URLs are copied to clipboard and listed below<br/><br/>
Three Google-Maps-Modes are currently implemented:
<div style='margin: 20px;'>
    <b>Maps</b> => Google-Maps centered on the clicked point, zoomed to the current scale.<br /><br />
    <b>Places</b> => Google-Maps with pin-marker placed on the clicked point for querying address-infos and Plus-Codes, storing this point to address-lists or start route-calculations.<br /><br />
    <b>Routes</b> => Google-Maps with route-calculations from start- to end-point.
</div>
Function-Buttons:
<div style='margin-left: 20px;'>
    <img src='{self.plugin_dir}/icons/Google_Maps_icon_(2020).svg' style='vertical-align: top;'/>  => activate MapTool (ToolBar)<br />
    <img src='{self.plugin_dir}/icons/auto_open_browser.svg' style='vertical-align: top;'/> => toggle auto-open-browser<br />
    <img src='{self.plugin_dir}/icons/plugin-help.svg' style='vertical-align: top;'/> => show help<br />
    <img src='{self.plugin_dir}/icons/mActionEditCopy.svg' style='vertical-align: top;'/> => copy contents to clipboard<br />
    <img src='{self.plugin_dir}/icons/mActionFileNew.svg' style='vertical-align: top;'/> => clear contents, clear start-point for MapMode 'Routes'<br />
    <img src='{self.plugin_dir}/icons/mDockify.svg' style='vertical-align: top;'/> => dock/undock window
</div><br/>
"""
        self.tool_append_html(intro)


    def s_open_browser(self, url: QtCore.QUrl):
        """opens default-webbrowser with the clicked url in self.qtb_result, copies URL to clipboard"""
        # Rev. 2026-02-04
        webbrowser.open(url.toString(), new=2, autoraise=True)
        QtWidgets.QApplication.clipboard().setText(url.toString())

    def s_toggle_top_level(self,checked:bool):
        """toggle dialog between TopLevel (free undockable window) and docked (with move and docked enabled, but floatable disabled)

        Args:
            checked (bool): checked-status of self.my_dialog.qtb_toggle_top_level
        """
        # Rev. 2026-02-04
        if self.my_dialog:
            if checked:
                # DockWidgetMovable => dock-border can be changed by drag&drop
                self.my_dialog.setFeatures(QtWidgets.QDockWidget.NoDockWidgetFeatures | QtWidgets.QDockWidget.DockWidgetClosable | QtWidgets.QDockWidget.DockWidgetMovable)
                self.my_dialog.setFloating(False)

            else:
                self.my_dialog.setFeatures(QtWidgets.QDockWidget.NoDockWidgetFeatures | QtWidgets.QDockWidget.DockWidgetClosable)
                self.my_dialog.setFloating(True)

            self.my_dialog.setFocus()
            # if its tabbed:
            self.my_dialog.raise_()


    def s_change_map_mode(self,idx:int):
        """triggered from QComboBox

        Args:
            idx (int): Index of selected option
        """
        # Rev. 2026-02-04
        if idx == 0:
            self.tool_mode = 'maps'
            self.tool_append_html("Click on map to create/open Google-Maps-URL...")
        elif idx == 1:
            self.tool_mode = 'places'
            self.tool_append_html("Click on map to create/open Google-Places-URL...")
        else:
            if self.start_point is None:
                self.vm_start_point.hide()
                self.tool_mode = 'set_start_point'
                self.tool_append_html("Click on map to set Start-Point...")
            else:
                self.tool_mode = 'routes'
                self.tool_append_html("Click on map to set End-Point...")




    def tool_append_html(self,html):
        """Append html to qtb_result and scrolls to bottom"""
        # Rev. 2026-02-04
        self.my_dialog.qtb_result.append(html)
        # add linebreak
        self.my_dialog.qtb_result.append("<br/>")
        self.my_dialog.qtb_result.verticalScrollBar().setValue(self.my_dialog.qtb_result.verticalScrollBar().maximum())


    def s_clear(self):
        """clears the contents, hides the temporal canvas-graphics and resets start_point"""
        # Rev. 2026-02-04
        self.my_dialog.qtb_result.clear()
        self.vm_start_point.hide()
        self.vm_end_point.hide()
        self.start_point = None
        if self.tool_mode == "routes":
            self.tool_mode = 'set_start_point'

    def s_copy_plaintext(self):
        """copies contents to clipboard"""
        # Rev. 2026-02-04
        QtWidgets.QApplication.clipboard().setText(self.my_dialog.qtb_result.toPlainText())

        # flash-effect
        QtCore.QTimer.singleShot(10, lambda:self.my_dialog.qtb_result.setStyleSheet("color: white; background-color: blue;"))
        QtCore.QTimer.singleShot(100, lambda:self.my_dialog.qtb_result.setStyleSheet("color: black; background-color: white;"))


    def activate(self):
        """re-implemented, called when set as currently active map tool"""
        # Rev. 2026-02-04
        self.iface.mapCanvas().setCursor(self.cursor)
        self.my_dialog.show()
        super().activate()

    def canvasReleaseEvent(self, event):
        """re-implemented main-function triggered on mouse-release
        dependent on tool_mode are different actions performed"""
        # Rev. 2026-02-04
        self.my_dialog.show()

        #click-point in the current canvas-projection
        point_org = self.iface.mapCanvas().getCoordinateTransform().toMapCoordinates(event.x(), event.y())

        # Google requires WGS84-lon/lat-coordinates
        source_crs = self.iface.mapCanvas().mapSettings().destinationCrs()
        target_crs = qgis.core.QgsCoordinateReferenceSystem("EPSG:4326")
        tr = qgis.core.QgsCoordinateTransform(source_crs, target_crs, qgis.core.QgsProject.instance())
        point_wgs = tr.transform(point_org)

        if self.tool_mode == 'set_start_point':
            self.vm_start_point.setCenter(point_org)
            self.vm_start_point.show()
            self.start_point = point_wgs
            self.tool_append_html("Start-Point registered, further Click(s) will create/open Google-Maps-URLs with route from Start- to Click-Point...")
            self.tool_mode = 'routes'
        elif self.tool_mode == 'routes':
            if self.start_point is not None:
                self.vm_end_point.setCenter(point_org)
                self.vm_end_point.show()
                google_url = self.google_route_url
                google_url = google_url.replace('[_lat_start_]', str(self.start_point.x()))
                google_url = google_url.replace('[_lon_start_]', str(self.start_point.y()))
                google_url = google_url.replace('[_lat_end_]', str(point_wgs.x()))
                google_url = google_url.replace('[_lon_end_]', str(point_wgs.y()))

                # formula taken from QGis-Plugin "zoom_level"
                zoom_level = 29.1402 - math.log2(self.iface.mapCanvas().scale())
                str_zoom_level = str(round(zoom_level, 2))
                google_url = google_url.replace('[_zoom_]', str_zoom_level)
                self.tool_append_html(f"<a href='{google_url}'>{google_url}</a>")
                QtWidgets.QApplication.clipboard().setText(google_url)
                if self.auto_open_browser:
                    webbrowser.open(google_url, new=2, autoraise=True)
            else:
                self.tool_mode = 'set_start_point'
                self.tool_append_html("Click on map to set Start-Point...")
        elif self.tool_mode == 'places':
            self.vm_end_point.setCenter(point_org)
            self.vm_end_point.show()
            google_url = self.google_place_url
            google_url = google_url.replace('[_lat_]', str(point_wgs.x()))
            google_url = google_url.replace('[_lon_]', str(point_wgs.y()))
            self.tool_append_html(f"<a href='{google_url}'>{google_url}</a>")
            QtWidgets.QApplication.clipboard().setText(google_url)
            if self.auto_open_browser:
                webbrowser.open(google_url, new=2, autoraise=True)

        else:
            self.vm_end_point.setCenter(point_org)
            self.vm_end_point.show()
            google_url = self.google_maps_url
            # formula taken from QGis-Plugin "zoom_level"
            zoom_level = 29.1402 - math.log2(self.iface.mapCanvas().scale())
            str_zoom_level = str(round(zoom_level, 2))
            google_url = google_url.replace('[_zoom_]', str_zoom_level)
            google_url = google_url.replace('[_lat_]', str(point_wgs.x()))
            google_url = google_url.replace('[_lon_]', str(point_wgs.y()))
            self.tool_append_html(f"<a href='{google_url}'>{google_url}</a>")
            QtWidgets.QApplication.clipboard().setText(google_url)
            if self.auto_open_browser:
                webbrowser.open(google_url, new=2, autoraise=True)


    def dlg_close(self, visible):
        """slot for signal dialog_close, emitted on self.my_dialog closeEvent
        change MapTool, hide canvas-graphics"""
        # Rev. 2026-02-04
        try:
            self.vm_start_point.hide()
            self.vm_end_point.hide()
            self.iface.actionPan().trigger()
        except Exception as e:
            # if called on sys_unload and these Markers are already deleted
            # print(f"Expected exception in {gdp()}: \"{e}\"")
            pass

    def unload(self):
        """unloads the MapTool:
        remove canvas-graphics
        remove dialog
        triggered from plugin-sys_unload
        """
        # Rev. 2026-02-04
        try:
            self.iface.mapCanvas().scene().removeItem(self.vm_start_point)
            del self.vm_start_point
            self.iface.mapCanvas().scene().removeItem(self.vm_end_point)
            del self.vm_end_point
            self.my_dialog.close()
            del self.my_dialog


        except Exception as e:
            # AttributeError: 'PolEvt' object has no attribute 'vm_pt_sn'
            # print(f"Expected exception in {gdp()}: \"{e}\"")
            pass

class QoogleDialog(QtWidgets.QDockWidget):
    """Dockable-Dialog for QGis-Plugin Qoogle"""


    dialog_close = QtCore.pyqtSignal(bool)
    """own dialog-close-signal, emitted on closeEvent"""

    def __init__(self, iface: qgis.gui.QgisInterface, parent=None):
        """Constructor

        Args:
            iface (qgis.gui.QgisInterface):  interface to running QGis-App
            parent (_type_, optional):optional Qt-Parent-Element for Hierarchy
        """
        # Rev. 2026-02-04
        self.plugin_dir = os.path.dirname(__file__)

        QtWidgets.QDockWidget.__init__(self, parent)
        self.setWindowTitle("Qoogle: QGis -> Google-Maps")
        main_wdg = QtWidgets.QWidget()
        main_wdg.setLayout(QtWidgets.QVBoxLayout())
        qwdg_top = QtWidgets.QWidget()
        qwdg_top.setLayout(QtWidgets.QHBoxLayout())
        qwdg_top.layout().setAlignment(QtCore.Qt.AlignRight)
        qwdg_top.layout().setContentsMargins(0, 0, 0, 0)

        self.qcbx_map_mode_select = QtWidgets.QComboBox()
        self.qcbx_map_mode_select.addItems(['Maps','Places','Routes'])
        qwdg_top.layout().addWidget(self.qcbx_map_mode_select)

        qwdg_top.layout().addStretch()

        self.qpb_set_auto_open = QtWidgets.QToolButton(self)
        self.qpb_set_auto_open.setToolButtonStyle(QtCore.Qt.ToolButtonIconOnly)
        self.qpb_set_auto_open.setIcon(QtGui.QIcon(f"{self.plugin_dir}/icons/auto_open_browser.svg"))
        self.qpb_set_auto_open.setToolTip("auto-open browser-window")
        self.qpb_set_auto_open.setCheckable(True)
        self.qpb_set_auto_open.setChecked(False)
        qwdg_top.layout().addWidget(self.qpb_set_auto_open)

        self.qtb_show_help = QtWidgets.QToolButton(self)
        self.qtb_show_help.setToolButtonStyle(QtCore.Qt.ToolButtonIconOnly)
        self.qtb_show_help.setIcon(QtGui.QIcon(f"{self.plugin_dir}/icons/plugin-help.svg"))
        self.qtb_show_help.setToolTip('Show help')
        qwdg_top.layout().addWidget(self.qtb_show_help)

        self.qpb_copy_plaintext = QtWidgets.QToolButton(self)
        self.qpb_copy_plaintext.setToolButtonStyle(QtCore.Qt.ToolButtonIconOnly)
        self.qpb_copy_plaintext.setIcon(QtGui.QIcon(f"{self.plugin_dir}/icons/mActionEditCopy.svg"))
        self.qpb_copy_plaintext.setToolTip("Copy results to clipboard")
        qwdg_top.layout().addWidget(self.qpb_copy_plaintext)

        self.qpb_clear = QtWidgets.QToolButton(self)
        self.qpb_clear.setToolButtonStyle(QtCore.Qt.ToolButtonIconOnly)
        self.qpb_clear.setIcon(QtGui.QIcon(f"{self.plugin_dir}/icons/mActionFileNew.svg"))
        self.qpb_clear.setToolTip("Clear results, reset Start-Point")
        qwdg_top.layout().addWidget(self.qpb_clear)

        self.qtb_toggle_top_level = QtWidgets.QToolButton(self)
        self.qtb_toggle_top_level.setCheckable(True)
        self.qtb_toggle_top_level.setChecked(False)
        self.qtb_toggle_top_level.setToolButtonStyle(QtCore.Qt.ToolButtonIconOnly)
        self.qtb_toggle_top_level.setIcon(QtGui.QIcon(f"{self.plugin_dir}/icons/mDockify.svg"))
        self.qtb_toggle_top_level.setToolTip('Dock/Undock window')
        qwdg_top.layout().addWidget(self.qtb_toggle_top_level)

        main_wdg.layout().addWidget(qwdg_top)

        self.qtb_result = QtWidgets.QTextBrowser(self)
        self.qtb_result.setOpenExternalLinks(False)
        self.qtb_result.setOpenLinks(False)
        main_wdg.layout().addWidget(self.qtb_result)

        self.setWidget(main_wdg)


    def closeEvent(self, e: QtCore.QEvent):
        """reimplemented, emitts signal when closing this widget

        Args:
            e (QtCore.QEvent)
        """
        # Rev. 2026-02-04
        self.dialog_close.emit(False)
