# -*- coding: utf-8 -*-
"""QMapPermalink WMS Service

WMS (Web Map Service) 機能を提供する専用クラス。
QGISキャンバスから地図画像を生成し、WMSプロトコルに対応。
"""

import math
from typing import Optional, Dict, Any, Tuple
from qgis.core import (
    QgsMapSettings, QgsMapRendererParallelJob, QgsRectangle, 
    QgsCoordinateReferenceSystem, QgsCoordinateTransform, 
    QgsProject, QgsMessageLog, Qgis
)
from PyQt5.QtCore import QSize, QEventLoop, QTimer
from PyQt5.QtGui import QColor


class QMapPermalinkWMSService:
    """QMapPermalink用WMSサービスクラス

    WMS GetCapabilitiesおよびGetMapリクエストを処理し、
    QGISキャンバスから地図画像を生成します。
    """

    def __init__(self, iface, webmap_generator, server_port: int = 8089, force_epsg3857: bool = False):
        """WMSサービスを初期化

        Args:
            iface: QGISインターフェース
            webmap_generator: WebMapGeneratorインスタンス
            server_port: サーバーポート番号
            force_epsg3857: 任意のCRSを強制的にEPSG:3857として扱うかどうか
        """
        self.iface = iface
        self.webmap_generator = webmap_generator
        self.server_port = server_port
        self.force_epsg3857 = force_epsg3857

    def _safe_int(self, value, default: int) -> int:
        """文字列から安全にintに変換する。NaNや不正値は default を返す。"""
        try:
            # floatを経由して 'NaN' をはじく
            v = float(value)
            if v != v:  # NaN check
                return int(default)
            return int(v)
        except Exception:
            return int(default)

    def _get_canvas_extent_info(self) -> Dict[str, Any]:
        """QGISキャンバスから現在の範囲情報を取得"""
        try:
            canvas = self.iface.mapCanvas()
            if not canvas:
                return {}

            extent = canvas.extent()
            crs = canvas.mapSettings().destinationCrs()

            return {
                'extent': extent,
                'crs': crs.authid() if crs else 'EPSG:3857',
                'width': canvas.width(),
                'height': canvas.height()
            }
        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"❌ Failed to get canvas extent info: {e}", "QMapPermalink", Qgis.Warning)
            return {}

    def handle_wms_request(self, conn, params: Dict[str, list], host: Optional[str] = None) -> None:
        """WMSエンドポイントを処理 - パーマリンクパラメータにも対応"""
        from qgis.core import QgsMessageLog, Qgis

        QgsMessageLog.logMessage("🌐 Processing WMS endpoint", "QMapPermalink", Qgis.Info)

        # デバッグ情報
        QgsMessageLog.logMessage(f"🔍 WMS Request parameters: {list(params.keys())}", "QMapPermalink", Qgis.Info)

        # If this request contains standard WMS GetMap parameters (BBOX/CRS/WIDTH/HEIGHT/REQUEST=GetMap)
        # prefer handling it as a normal WMS GetMap. Only fall back to the permalink-style
        # (x/y/scale) processing when a standard GetMap is not present. This ensures
        # OpenLayers' ImageWMS requests (which include BBOX) are handled correctly and
        # the returned image aligns with OL's expected geographic extent.
        is_standard_getmap = False
        try:
            if 'REQUEST' in params and params.get('REQUEST'):
                if params.get('REQUEST', [''])[0].upper() == 'GETMAP':
                    is_standard_getmap = True
        except Exception:
            is_standard_getmap = False

        has_permalink_params = ('x' in params and 'y' in params and 'scale' in params)
        QgsMessageLog.logMessage(f"🔍 Has permalink params: {has_permalink_params}", "QMapPermalink", Qgis.Info)
        QgsMessageLog.logMessage(f"🔍 x in params: {'x' in params}, y in params: {'y' in params}, scale in params: {'scale' in params}", "QMapPermalink", Qgis.Info)

        if not is_standard_getmap and has_permalink_params:
            QgsMessageLog.logMessage("📍 Permalink parameters detected in WMS request (no standard GetMap)", "QMapPermalink", Qgis.Info)
            # パーマリンクパラメータをWMSパラメータに変換してGetMapとして処理
            self._handle_permalink_as_wms_getmap(conn, params)
            return

        # 通常のWMSリクエストの処理
        request = params.get('REQUEST', [''])[0].upper()
        service = params.get('SERVICE', [''])[0].upper()

        if service != 'WMS':
            from . import http_server
            http_server.send_wms_error_response(conn, "InvalidParameterValue", "SERVICE parameter must be WMS")
            return

        if request == 'GETCAPABILITIES':
            self._handle_wms_get_capabilities(conn, params, host)
        elif request == 'GETMAP':
            self._handle_wms_get_map(conn, params)
        else:
            from . import http_server
            http_server.send_wms_error_response(conn, "InvalidRequest", f"Request {request} is not supported")

    def _handle_wms_get_capabilities(self, conn, params: Dict[str, list], host: Optional[str] = None) -> None:
        """WMS GetCapabilitiesリクエストを処理 - 動的な地図範囲に対応"""
        from qgis.core import QgsMessageLog, Qgis

        # QGISキャンバスから現在の範囲情報を取得
        extent_info = self._get_canvas_extent_info()

        # determine base host for OnlineResource. Prefer Host header if provided
        try:
            if host:
                base_host = host
            else:
                base_host = f"localhost:{self.server_port}"
        except Exception:
            base_host = f"localhost:{self.server_port}"

        xml_content = f"""<?xml version="1.0" encoding="UTF-8"?>
<WMS_Capabilities version="1.3.0" xmlns="http://www.opengis.net/wms" xmlns:xlink="http://www.w3.org/1999/xlink">
  <Service>
    <Name>WMS</Name>
    <Title>QGIS Map Permalink WMS Service</Title>
    <Abstract>Dynamic WMS service for QGIS map view with zoom/pan capabilities</Abstract>
    <OnlineResource xlink:href="http://{base_host}/wms"/>
  </Service>
  <Capability>
    <Request>
      <GetCapabilities>
        <Format>text/xml</Format>
        <DCPType>
          <HTTP>
            <Get><OnlineResource xlink:href="http://{base_host}/wms"/></Get>
          </HTTP>
        </DCPType>
      </GetCapabilities>
      <GetMap>
        <Format>image/png</Format>
        <DCPType>
          <HTTP>
            <Get><OnlineResource xlink:href="http://{base_host}/wms"/></Get>
          </HTTP>
        </DCPType>
      </GetMap>
    </Request>
    <Layer>
      <Name>qgis_map</Name>
      <Title>QGIS Map</Title>
      <Abstract>Current QGIS map view</Abstract>
      <CRS>EPSG:3857</CRS>
      <CRS>EPSG:4326</CRS>
      <CRS>EPSG:32633</CRS>
      <CRS>EPSG:32634</CRS>
      <CRS>EPSG:32635</CRS>
      <CRS>EPSG:32636</CRS>
      <CRS>EPSG:32637</CRS>
      <CRS>EPSG:32638</CRS>
      <CRS>EPSG:32639</CRS>
      <CRS>EPSG:32640</CRS>
      <CRS>EPSG:32641</CRS>
      <CRS>EPSG:32642</CRS>
      <CRS>EPSG:32643</CRS>
      <CRS>EPSG:32644</CRS>
      <CRS>EPSG:32645</CRS>
      <CRS>EPSG:32646</CRS>
      <CRS>EPSG:32647</CRS>
      <CRS>EPSG:32648</CRS>
      <CRS>EPSG:32649</CRS>
      <CRS>EPSG:32650</CRS>
      <CRS>EPSG:32651</CRS>
      <CRS>EPSG:32652</CRS>
      <CRS>EPSG:32653</CRS>
      <CRS>EPSG:32654</CRS>
      <CRS>EPSG:32655</CRS>
      <CRS>EPSG:32656</CRS>
      <CRS>EPSG:32657</CRS>
      <CRS>EPSG:32658</CRS>
      <CRS>EPSG:32659</CRS>
      <CRS>EPSG:32660</CRS>
      <EX_GeographicBoundingBox>
        <westBoundLongitude>-180</westBoundLongitude>
        <eastBoundLongitude>180</eastBoundLongitude>
        <southBoundLatitude>-90</southBoundLatitude>
        <northBoundLatitude>90</northBoundLatitude>
      </EX_GeographicBoundingBox>
      <BoundingBox CRS="EPSG:3857" minx="-20037508.34" miny="-20037508.34" maxx="20037508.34" maxy="20037508.34"/>
      <BoundingBox CRS="EPSG:4326" minx="-180" miny="-90" maxx="180" maxy="90"/>
    </Layer>
  </Capability>
</WMS_Capabilities>"""

        from . import http_server
        http_server.send_http_response(conn, 200, "OK", xml_content, content_type="text/xml; charset=utf-8")

    def _handle_wms_get_map(self, conn, params: Dict[str, list]) -> None:
        """WMS GetMapリクエストを処理 - 実際のQGIS地図画像を生成"""
        from qgis.core import QgsMessageLog, Qgis
        QgsMessageLog.logMessage("🖼️ Processing WMS GetMap request", "QMapPermalink", Qgis.Info)

        try:
            # WMSパラメータを解析
            width = self._safe_int(params.get('WIDTH', ['256'])[0], 256)
            height = self._safe_int(params.get('HEIGHT', ['256'])[0], 256)
            bbox = params.get('BBOX', [''])[0]
            # WMS version and CRS/SRS handling: accept both CRS (1.3.0) and SRS (1.1.1)
            wms_version = params.get('VERSION', params.get('version', ['1.3.0']))[0]
            # Prefer CRS (WMS 1.3.0) but fall back to SRS if provided
            original_crs = None
            if 'CRS' in params and params.get('CRS'):
                original_crs = params.get('CRS', [''])[0]
            elif 'SRS' in params and params.get('SRS'):
                original_crs = params.get('SRS', [''])[0]
            if not original_crs:
                original_crs = 'EPSG:3857'

            # THEMESパラメータを取得（QGIS拡張）
            themes = params.get('THEMES', [''])[0] if 'THEMES' in params else None
            if themes:
                QgsMessageLog.logMessage(f"🎨 Using WMS theme: {themes}", "QMapPermalink", Qgis.Info)
            else:
                QgsMessageLog.logMessage("🎨 No theme specified, using current QGIS display settings", "QMapPermalink", Qgis.Info)

            # 回転パラメータを取得（WMS拡張: ANGLEパラメータ）
            rotation = 0.0
            if 'ANGLE' in params and params.get('ANGLE'):
                try:
                    rotation = float(params.get('ANGLE', ['0'])[0])
                    QgsMessageLog.logMessage(f"🔄 WMS rotation angle: {rotation}°", "QMapPermalink", Qgis.Info)
                except Exception as e:
                    rotation = 0.0
                    QgsMessageLog.logMessage(f"⚠️ Invalid ANGLE parameter: {e}, using 0°", "QMapPermalink", Qgis.Warning)

            # If WMS 1.3.0 and CRS is EPSG:4326, axis order in BBOX is lat,lon (y,x)
            # so we need to swap coordinates when parsing. For other CRSs assume BBOX
            # is minx,miny,maxx,maxy.
            try:
                bbox_coords = [float(v) for v in bbox.split(',')] if bbox else []
                if bbox_coords and wms_version and str(wms_version).startswith('1.3') and original_crs and original_crs.upper().endswith('4326') and len(bbox_coords) == 4:
                    # incoming BBOX: miny,minx,maxy,maxx -> reorder to minx,miny,maxx,maxy
                    bbox = f"{bbox_coords[1]},{bbox_coords[0]},{bbox_coords[3]},{bbox_coords[2]}"
                else:
                    # keep as-is
                    bbox = ','.join(str(v) for v in bbox_coords) if bbox_coords else ''
            except Exception:
                # if parsing fails, keep original string
                pass
            # オプションで任意CRSを強制的にEPSG:3857として扱う
            if self.force_epsg3857:
                crs = 'EPSG:3857'
                QgsMessageLog.logMessage("🔒 force_epsg3857 enabled: treating request CRS as EPSG:3857", "QMapPermalink", Qgis.Info)
            else:
                crs = original_crs

            QgsMessageLog.logMessage(f"📐 WMS GetMap parameters: {width}x{height}, requested CRS: {original_crs}, using CRS: {crs}, BBOX: {bbox}", "QMapPermalink", Qgis.Info)
            # リクエストで与えられたBBOXがある場合、必要なら元々のCRSからEPSG:3857に変換する
            # BBOXを変換するのは force_epsg3857 が無効な場合のみ
            if not self.force_epsg3857:
                try:
                    if bbox and original_crs and original_crs.upper() != 'EPSG:3857':
                        from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject, QgsRectangle
                        src_crs = QgsCoordinateReferenceSystem(original_crs)
                        tgt_crs = QgsCoordinateReferenceSystem('EPSG:3857')
                        if src_crs.isValid():
                            try:
                                coords = [float(x) for x in bbox.split(',')]
                                if len(coords) == 4:
                                            rect = QgsRectangle(coords[0], coords[1], coords[2], coords[3])
                                            transform = QgsCoordinateTransform(src_crs, tgt_crs, QgsProject.instance())
                                            rect = transform.transformBoundingBox(rect)
                                            bbox = f"{rect.xMinimum()},{rect.yMinimum()},{rect.xMaximum()},{rect.yMaximum()}"
                                            # Ensure the CRS variable matches the transformed BBOX coordinates
                                            crs = 'EPSG:3857'
                                            QgsMessageLog.logMessage(f"🔄 Transformed incoming BBOX from {original_crs} to EPSG:3857: {bbox}", "QMapPermalink", Qgis.Info)
                            except Exception as e:
                                QgsMessageLog.logMessage(f"⚠️ Failed to transform BBOX to EPSG:3857: {e}", "QMapPermalink", Qgis.Warning)
                        else:
                            QgsMessageLog.logMessage(f"⚠️ Invalid source CRS: {original_crs}", "QMapPermalink", Qgis.Warning)
                except Exception as e:
                    QgsMessageLog.logMessage(f"⚠️ BBOX transformation error: {e}", "QMapPermalink", Qgis.Warning)

            # BBOXが指定されている場合、それを直接使用して画像を生成
            if bbox:
                try:
                    coords = [float(x) for x in bbox.split(',')]
                    if len(coords) == 4:
                        self._handle_wms_get_map_with_bbox(conn, bbox, crs, width, height, themes, rotation)
                        return
                except Exception as e:
                    QgsMessageLog.logMessage(f"⚠️ Invalid BBOX format: {bbox}, error: {e}", "QMapPermalink", Qgis.Warning)

            # BBOXが指定されていない場合、エラーレスポンスを返す
            from . import http_server
            http_server.send_wms_error_response(conn, "MissingParameterValue", "BBOX parameter is required for GetMap requests")

        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            import traceback
            QgsMessageLog.logMessage(f"❌ WMS GetMap error: {e}", "QMapPermalink", Qgis.Critical)
            QgsMessageLog.logMessage(f"❌ Error traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
            from . import http_server
            http_server.send_http_response(conn, 500, "Internal Server Error", f"WMS GetMap failed: {str(e)}")

    def _handle_wms_get_map_with_bbox(self, conn, bbox: str, crs: str, width: int, height: int, themes: str = None, rotation: float = 0.0) -> None:
        """BBOX指定でWMS GetMapを処理"""
        from qgis.core import QgsMessageLog, Qgis

        try:
            # BBOXをパース
            coords = [float(x) for x in bbox.split(',')]
            if len(coords) != 4:
                raise ValueError(f"Invalid BBOX format: {bbox}")

            minx, miny, maxx, maxy = coords

            # 画像サイズの制限（最大4096x4096）
            max_dimension = 4096
            if width > max_dimension or height > max_dimension:
                from . import http_server
                http_server.send_wms_error_response(conn, "InvalidParameterValue", f"Image dimensions too large. Maximum allowed: {max_dimension}x{max_dimension}")
                return

            # 独立レンダリングで画像を生成
            try:
                image_data = self._render_map_image(width, height, bbox, crs, themes, rotation)

                if image_data:
                    from . import http_server
                    http_server.send_http_response(conn, 200, "OK", image_data, content_type="image/png")
                else:
                    from . import http_server
                    http_server.send_wms_error_response(conn, "InternalError", "Failed to generate map image")

            except Exception as e:
                from qgis.core import QgsMessageLog, Qgis
                import traceback
                QgsMessageLog.logMessage(f"❌ Map image generation error: {e}", "QMapPermalink", Qgis.Critical)
                QgsMessageLog.logMessage(f"❌ Error traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
                from . import http_server
                http_server.send_wms_error_response(conn, "InternalError", f"Map generation failed: {str(e)}")

        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            import traceback
            QgsMessageLog.logMessage(f"❌ WMS GetMap with BBOX error: {e}", "QMapPermalink", Qgis.Critical)
            QgsMessageLog.logMessage(f"❌ Error traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
            from . import http_server
            http_server.send_http_response(conn, 500, "Internal Server Error", f"WMS GetMap processing failed: {str(e)}")

    def _handle_permalink_as_wms_getmap(self, conn, params: Dict[str, list]) -> None:
        """パーマリンクパラメータをWMS GetMapパラメータに変換して処理"""
        from qgis.core import QgsMessageLog, Qgis

        try:
            # パーマリンクパラメータを取得
            x = float(params.get('x', ['0'])[0])
            y = float(params.get('y', ['0'])[0])
            scale = float(params.get('scale', ['1000'])[0])
            crs = params.get('crs', ['EPSG:3857'])[0]
            width = self._safe_int(params.get('width', ['800'])[0], 800)
            height = self._safe_int(params.get('height', ['600'])[0], 600)
            rotation = float(params.get('rotation', ['0'])[0])

            # スケールからBBOXを計算
            # スケールは地図単位/ピクセル（例: 1000 = 1000m/px）
            # 中心点から画像サイズに基づいてBBOXを計算
            half_width_meters = (width / 2) * scale
            half_height_meters = (height / 2) * scale

            minx = x - half_width_meters
            maxx = x + half_width_meters
            miny = y - half_height_meters
            maxy = y + half_height_meters

            bbox = f"{minx},{miny},{maxx},{maxy}"

            QgsMessageLog.logMessage(f"🔄 Permalink params converted to WMS: x={x}, y={y}, scale={scale}, bbox={bbox}", "QMapPermalink", Qgis.Info)

            # WMS GetMapとして処理
            self._handle_wms_get_map_with_bbox(conn, bbox, crs, width, height, None, rotation)

        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            import traceback
            QgsMessageLog.logMessage(f"❌ Permalink to WMS conversion error: {e}", "QMapPermalink", Qgis.Critical)
            QgsMessageLog.logMessage(f"❌ Error traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
            from . import http_server
            http_server.send_http_response(conn, 500, "Internal Server Error", f"Permalink processing failed: {str(e)}")

    def _render_map_image(self, width, height, bbox, crs, themes=None, rotation=0.0):
        """
        完全独立マップレンダリング

        Args:
            width (int): 画像幅
            height (int): 画像高さ
            bbox (str): "minx,miny,maxx,maxy" 形式の範囲
            crs (str): 座標系（例: "EPSG:4326"）
            rotation (float): 回転角度（度）、デフォルト0.0

        Returns:
            bytes: PNG画像データ（失敗時はNone）
        """
        try:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(
                f"🎨 WMS Independent Rendering: {width}x{height}, BBOX: {bbox}, CRS: {crs}, Themes: {themes}, Rotation: {rotation}°",
                "QMapPermalink", Qgis.Info
            )

            # 1. 現在のキャンバスのマップ設定をベースにする
            map_settings = self._create_map_settings_from_canvas(width, height, crs, themes)

            # 2. BBOXから表示範囲を設定
            extent = self._parse_bbox_to_extent(bbox, crs)
            if extent:
                map_settings.setExtent(extent)
            else:
                # フォールバック: プロジェクト全体の範囲
                map_settings.setExtent(QgsProject.instance().extent())

            # 4. 回転を設定（指定角度でレンダリング）
            #    地図（ベクタ・ラベル含む）を回転してレンダリングします。後で
            #    出力画像を逆回転して北上に補正します。
            if rotation != 0.0:
                from qgis.core import QgsMessageLog, Qgis
                QgsMessageLog.logMessage(f"🔄 Setting map rotation to {rotation}° for rendering", "QMapPermalink", Qgis.Info)
                map_settings.setRotation(rotation)

            # 5. 並列レンダリング実行（QImageを取得）
            image = self._execute_parallel_rendering(map_settings)

            png_data = None
            if image and not image.isNull():
                # map was rendered possibly with rotation; if rotation applied,
                # inverse-rotate the raster so the returned PNG is north-up while
                # labels/graphics were rendered in rotated state.
                try:
                    if rotation != 0.0:
                        image = self._inverse_rotate_image(image, rotation)
                except Exception:
                    pass

                png_data = self._save_image_as_png(image)

            if png_data:
                from qgis.core import QgsMessageLog, Qgis
                QgsMessageLog.logMessage(
                    f"✅ WMS rendering success: {len(png_data)} bytes",
                    "QMapPermalink", Qgis.Info
                )
                # ここでは回転済みマップを逆回転して北上に補正済みのPNGを返す
                return png_data
            else:
                from qgis.core import QgsMessageLog, Qgis
                QgsMessageLog.logMessage(
                    "❌ WMS rendering failed",
                    "QMapPermalink", Qgis.Warning
                )
                return None
                return None

        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            import traceback
            QgsMessageLog.logMessage(
                f"❌ WMS rendering error: {e}",
                "QMapPermalink", Qgis.Critical
            )
            QgsMessageLog.logMessage(
                f"❌ Error traceback: {traceback.format_exc()}",
                "QMapPermalink", Qgis.Critical
            )
            return None

    def _apply_label_rotation_correction(self, map_settings, rotation):
        """地図回転時の文字回転補正を適用
        
        地図が回転しても文字は上向きに保つため、
        すべてのベクタレイヤーのラベル回転を地図回転の逆方向に設定
        """
        try:
            from qgis.core import QgsMessageLog, Qgis, QgsVectorLayer
            
            # 文字の回転補正角度（地図回転の逆方向）
            label_rotation = -rotation
            
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"🔄 Applying label rotation correction: map_rotation={rotation}°, label_rotation={label_rotation}°", "QMapPermalink", Qgis.Info)
            
            # マップ設定のレイヤを取得
            layers = map_settings.layers()
            
            labeled_layer_count = 0
            for layer in layers:
                if isinstance(layer, QgsVectorLayer) and layer.labelsEnabled():
                    labeled_layer_count += 1
                    from qgis.core import QgsMessageLog, Qgis
                    QgsMessageLog.logMessage(f"🔤 Processing labeled layer: {layer.name()}", "QMapPermalink", Qgis.Info)
                    
                    # ラベル設定を取得
                    label_settings = layer.labeling()
                    if label_settings:
                        try:
                            from qgis.core import QgsPalLayerSettings, QgsVectorLayerSimpleLabeling
                            
                            # ラベル設定のタイプに応じて処理
                            if isinstance(label_settings, QgsVectorLayerSimpleLabeling):
                                # QgsVectorLayerSimpleLabelingの場合
                                pal_settings = label_settings.settings()
                                if pal_settings:
                                    # 既存の設定をコピー
                                    pal_settings = QgsPalLayerSettings(pal_settings)
                                    
                                    # 文字回転を地図回転の逆方向に設定
                                    pal_settings.rotation = label_rotation
                                    
                                    # 新しいラベル設定を作成
                                    new_labeling = QgsVectorLayerSimpleLabeling(pal_settings)
                                    layer.setLabeling(new_labeling)
                                    
                                    from qgis.core import QgsMessageLog, Qgis
                                    QgsMessageLog.logMessage(
                                        f"🔤 ✅ Applied label rotation correction: {label_rotation}° for layer '{layer.name()}' (QgsVectorLayerSimpleLabeling)", 
                                        "QMapPermalink", Qgis.Info
                                    )
                                else:
                                    from qgis.core import QgsMessageLog, Qgis
                                    QgsMessageLog.logMessage(f"⚠️ No PAL settings found in QgsVectorLayerSimpleLabeling for layer '{layer.name()}'", "QMapPermalink", Qgis.Warning)
                            
                            elif isinstance(label_settings, QgsPalLayerSettings):
                                # 直接QgsPalLayerSettingsの場合（古い形式）
                                pal_settings = QgsPalLayerSettings(label_settings)
                                
                                # 文字回転を地図回転の逆方向に設定
                                pal_settings.rotation = label_rotation
                                
                                # 新しいラベル設定をレイヤに適用
                                labeling = QgsVectorLayerSimpleLabeling(pal_settings)
                                layer.setLabeling(labeling)
                                
                                from qgis.core import QgsMessageLog, Qgis
                                QgsMessageLog.logMessage(
                                    f"🔤 ✅ Applied label rotation correction: {label_rotation}° for layer '{layer.name()}' (QgsPalLayerSettings)", 
                                    "QMapPermalink", Qgis.Info
                                )
                            else:
                                from qgis.core import QgsMessageLog, Qgis
                                QgsMessageLog.logMessage(f"⚠️ Label settings type not supported for layer '{layer.name()}': {type(label_settings)}", "QMapPermalink", Qgis.Warning)
                        
                        except Exception as e:
                            from qgis.core import QgsMessageLog, Qgis
                            import traceback
                            QgsMessageLog.logMessage(f"⚠️ Failed to apply rotation to layer '{layer.name()}': {e}", "QMapPermalink", Qgis.Warning)
                            QgsMessageLog.logMessage(f"⚠️ Traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Warning)
                    else:
                        from qgis.core import QgsMessageLog, Qgis
                        QgsMessageLog.logMessage(f"⚠️ No label settings found for layer '{layer.name()}'", "QMapPermalink", Qgis.Warning)
            
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(
                f"🔤 Label rotation correction completed: {labeled_layer_count} labeled layers processed", 
                "QMapPermalink", Qgis.Info
            )
            
        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            import traceback
            QgsMessageLog.logMessage(f"❌ Label rotation correction error: {e}", "QMapPermalink", Qgis.Critical)
            QgsMessageLog.logMessage(f"❌ Traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)

    def _apply_label_rotation_only(self, map_settings, rotation):
        """ラベルのみを指定角度で回転させる（画像は回転させない）。

        map_settings のレイヤを巡回し、ラベルが有効なベクタレイヤについて
        現在の labeling オブジェクトを保存してから、一時的に回転設定を持つ
        新しいラベリングに差し替えます。戻すために {layer: original_labeling} の辞書を返します。
        """
        original_labelings = {}
        try:
            from qgis.core import QgsMessageLog, Qgis, QgsVectorLayer, QgsVectorLayerSimpleLabeling, QgsPalLayerSettings

            layers = map_settings.layers()
            for layer in layers:
                try:
                    if isinstance(layer, QgsVectorLayer) and layer.labelsEnabled():
                        QgsMessageLog.logMessage(f"🔤 Applying label-only rotation {rotation}° to layer: {layer.name()}", "QMapPermalink", Qgis.Info)
                        orig_labeling = layer.labeling()
                        original_labelings[layer] = orig_labeling

                        # 対応するラベリング型に応じて回転を適用
                        if isinstance(orig_labeling, QgsVectorLayerSimpleLabeling):
                            pal = orig_labeling.settings()
                            if pal:
                                pal_copy = QgsPalLayerSettings(pal)
                                pal_copy.rotation = rotation
                                new_labeling = QgsVectorLayerSimpleLabeling(pal_copy)
                                layer.setLabeling(new_labeling)
                            else:
                                QgsMessageLog.logMessage(f"⚠️ No PAL settings in labeling for layer '{layer.name()}'", "QMapPermalink", Qgis.Warning)
                        else:
                            # 試しに QgsPalLayerSettings でラップして適用
                            try:
                                pal_try = QgsPalLayerSettings(orig_labeling)
                                pal_try.rotation = rotation
                                new_labeling = QgsVectorLayerSimpleLabeling(pal_try)
                                layer.setLabeling(new_labeling)
                            except Exception:
                                QgsMessageLog.logMessage(f"⚠️ Unsupported labeling type for layer '{layer.name()}': {type(orig_labeling)}", "QMapPermalink", Qgis.Warning)
                except Exception as e:
                    from qgis.core import QgsMessageLog, Qgis
                    QgsMessageLog.logMessage(f"⚠️ Failed processing layer for label-only rotation: {e}", "QMapPermalink", Qgis.Warning)

        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            import traceback
            QgsMessageLog.logMessage(f"❌ _apply_label_rotation_only error: {e}", "QMapPermalink", Qgis.Critical)
            QgsMessageLog.logMessage(f"❌ Traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)

        return original_labelings

    def _create_map_settings_from_canvas(self, width, height, crs, themes=None):
        """現在のキャンバスのマップ設定をベースにWMS設定を作成"""
        from qgis.core import QgsMapSettings, QgsCoordinateReferenceSystem

        # 現在のキャンバスのマップ設定をコピー
        canvas = self.iface.mapCanvas()
        map_settings = canvas.mapSettings()

        # WMS固有の設定を上書き
        map_settings.setOutputSize(QSize(width, height))
        map_settings.setOutputDpi(96)

        # 座標系を設定（WMSパラメータが指定されている場合）
        if crs:
            target_crs = QgsCoordinateReferenceSystem(crs)
            if target_crs.isValid():
                map_settings.setDestinationCrs(target_crs)

        # 回転をリセット（現在のキャンバス回転を無視してWMSパラメータを使用）
        map_settings.setRotation(0.0)

        # テーマが指定されている場合はテーマのレイヤ設定を適用
        if themes:
            project = QgsProject.instance()
            map_theme_collection = project.mapThemeCollection()

            if themes in map_theme_collection.mapThemes():
                from qgis.core import QgsMessageLog, Qgis
                QgsMessageLog.logMessage(f"🎨 Applying theme: {themes}", "QMapPermalink", Qgis.Info)

                # テーマを適用
                map_theme_collection.applyTheme(themes, map_settings, visibleOnly=True)
            else:
                from qgis.core import QgsMessageLog, Qgis
                QgsMessageLog.logMessage(f"⚠️ Theme '{themes}' not found, using current canvas settings", "QMapPermalink", Qgis.Warning)
        else:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage("🎨 Using current canvas settings (no theme specified)", "QMapPermalink", Qgis.Info)

        return map_settings

    def _get_visible_layers(self, themes=None):
        """現在のプロジェクトから可視レイヤを取得（テーマ対応）"""
        from qgis.core import QgsProject
        visible_layers = []

        project = QgsProject.instance()
        map_theme_collection = project.mapThemeCollection()

        if themes and themes in map_theme_collection.mapThemes():
            # 指定されたテーマのレイヤを使用
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"🎨 Applying theme: {themes}", "QMapPermalink", Qgis.Info)

            theme_record = map_theme_collection.mapThemeState(themes)
            for layer_id in theme_record.layerRecords():
                layer = project.mapLayer(layer_id)
                if layer and layer.isValid():
                    visible_layers.append(layer)
        else:
            # テーマが指定されていない場合は現在のQGIS表示状態を使用
            from qgis.core import QgsMessageLog, Qgis
            if themes:
                QgsMessageLog.logMessage(f"⚠️ Theme '{themes}' not found, using current QGIS display settings", "QMapPermalink", Qgis.Warning)
            else:
                QgsMessageLog.logMessage("🎨 Using current QGIS display settings (no theme specified)", "QMapPermalink", Qgis.Info)

            # プロジェクトのレイヤツリーを走査して可視レイヤを取得
            layer_tree_root = project.layerTreeRoot()
            for layer_tree_layer in layer_tree_root.findLayers():
                if layer_tree_layer.isVisible():
                    layer = layer_tree_layer.layer()
                    if layer and layer.isValid():
                        visible_layers.append(layer)

        return visible_layers

    def _parse_bbox_to_extent(self, bbox, crs):
        """BBOX文字列をQgsRectangleに変換"""
        from qgis.core import QgsRectangle, QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject
        try:
            coords = [float(x) for x in bbox.split(',')]
            if len(coords) == 4:
                minx, miny, maxx, maxy = coords
                return QgsRectangle(minx, miny, maxx, maxy)
        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"⚠️ Failed to parse BBOX '{bbox}': {e}", "QMapPermalink", Qgis.Warning)
        return None

    def _execute_parallel_rendering(self, map_settings):
        """並列レンダリングを実行"""
        from qgis.core import QgsMapRendererParallelJob, QgsMessageLog, Qgis
        try:
            # 並列レンダリングジョブを作成
            render_job = QgsMapRendererParallelJob(map_settings)

            # イベントループで完了を待つ
            loop = QEventLoop()
            render_job.finished.connect(loop.quit)
            render_job.start()

            # タイムアウト設定（30秒）
            timer = QTimer()
            timer.timeout.connect(loop.quit)
            timer.setSingleShot(True)
            timer.start(30000)  # 30秒

            loop.exec_()

            if render_job.isActive():
                # タイムアウトした場合
                render_job.cancel()
                QgsMessageLog.logMessage("⚠️ Rendering timeout", "QMapPermalink", Qgis.Warning)
                return None

            # レンダリング結果を取得
            image = render_job.renderedImage()
            if image and not image.isNull():
                return image
            else:
                QgsMessageLog.logMessage("⚠️ Rendered image is null", "QMapPermalink", Qgis.Warning)
                return None

        except Exception as e:
            import traceback
            QgsMessageLog.logMessage(f"❌ Parallel rendering error: {e}", "QMapPermalink", Qgis.Critical)
            QgsMessageLog.logMessage(f"❌ Traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
            return None

    def _inverse_rotate_image(self, image, rotation_degrees):
        """レンダリング済みのQImageを逆回転して北上に戻す。

        回転中心を画像中心にし、回転後は元の幅/高さで中央切り抜きして返す。
        """
        try:
            from PyQt5.QtGui import QTransform, QPainter, QImage
            from PyQt5.QtCore import QRect, Qt

            w = image.width()
            h = image.height()

            # 逆回転（ピクセル空間で）
            t = QTransform()
            cx = w / 2.0
            cy = h / 2.0
            t.translate(cx, cy)
            t.rotate(-float(rotation_degrees))
            t.translate(-cx, -cy)

            rotated = image.transformed(t, Qt.SmoothTransformation)

            # 新しいキャンバスに中央配置して必要なら切り抜く
            result = QImage(w, h, rotated.format())
            result.fill(Qt.transparent)
            painter = QPainter(result)
            # draw rotated image centered
            rx = (w - rotated.width()) // 2
            ry = (h - rotated.height()) // 2
            painter.drawImage(rx, ry, rotated)
            painter.end()
            return result
        except Exception:
            return image

    def _save_image_as_png(self, image):
        """QImageをPNGバイトデータに変換"""
        try:
            from PyQt5.QtCore import QBuffer, QIODevice
            buffer = QBuffer()
            buffer.open(QIODevice.WriteOnly)
            image.save(buffer, "PNG")
            png_data = buffer.data()
            buffer.close()
            return png_data
        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"⚠️ Failed to save image as PNG: {e}", "QMapPermalink", Qgis.Warning)
            return None