# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QMapPermalink
                                 A QGIS plugin
 Navigate QGIS map views through external permalink system
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-10-05
        git sha              : $Format:%H$
        copyright            : (C) 2025 by yamamoto-ryuzo
        email                : 
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QUrl, QThread, pyqtSignal, QObject
from qgis.PyQt.QtGui import QIcon, QDesktopServices, QClipboard
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QApplication, QDockWidget
from qgis.core import QgsProject, QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsPointXY, QgsRectangle
try:
    from qgis.core import qgsfunction
except Exception:
    qgsfunction = None

from qgis.gui import QgsMapCanvas

# ユーザー関数定義（デコレータでの自動登録は import 時に QGIS の内部で
# Cレベルの処理を行うため、環境によってはプロセスがクラッシュすることがある。
# 安全性のためここではデコレータを使わず通常の関数として定義する。必要なら
# プラグイン初期化時に明示的に登録処理を行う。
def my_custom_function(value1, value2, feature, parent):
    """単純なサンプル関数（登録は行わない）

    NOTE: QGIS の qgsfunction デコレータでの自動登録はここでは行いません。
    もしプラグイン実行環境でユーザー関数の登録が必要であれば、
    initGui() 内など安全なタイミングで登録処理を追加してください。
    """
    try:
        return value1 + value2
    except Exception:
        return None

import os.path
import urllib.parse
import json
import math

# パネルファイルのインポート（フォールバックを削除）
try:
    from .qmap_permalink_panel import QMapPermalinkPanel
    PANEL_AVAILABLE = True
except ImportError:
    # シンプル版へのフォールバックは廃止。パネル機能は利用不可とする。
    PANEL_AVAILABLE = False
    QMapPermalinkPanel = None

# WebMap生成モジュールのインポート
try:
    from .qmap_webmap_generator import QMapWebMapGenerator
    WEBMAP_AVAILABLE = True
except ImportError:
    WEBMAP_AVAILABLE = False
    QMapWebMapGenerator = None


class NavigationSignals(QObject):
    """QGIS APIへの安全なアクセスのためのシグナル"""
    navigate_requested = pyqtSignal(dict)  # 地図ナビゲーション要求
    # ブラウザから送られてきた完全なURLを通知するシグナル
    request_origin_changed = pyqtSignal(str)


class QMapPermalink:
    """QGISの地図ビューをパーマリンクで管理・ナビゲートするプラグイン"""

    def __init__(self, iface):
        """コンストラクタ

        Args:
            iface: QGISのインターフェースオブジェクト
        """
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        
        # 翻訳の初期化
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'QMapPermalink_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # プラグインの宣言
        self.actions = []
        self.menu = self.tr(u'&QMap Permalink')
        
        # パネル（ドックウィジェット）
        self.panel = None

        # ナビゲーション用シグナル
        self.navigation_signals = NavigationSignals()
        self.navigation_signals.navigate_requested.connect(self.handle_navigation_request)
        # 要求元URLを受け取ってUIを更新するハンドラを接続
        try:
            self.navigation_signals.request_origin_changed.connect(self.handle_request_origin_changed)
        except Exception:
            pass

        # WebMap生成器の初期化
        if WEBMAP_AVAILABLE and QMapWebMapGenerator:
            # pass plugin instance (self) so the generator can reuse
            # plugin utilities like _estimate_zoom_from_scale when available
            try:
                self.webmap_generator = QMapWebMapGenerator(self)
            except Exception:
                # fallback to passing iface if plugin instance cannot be used
                self.webmap_generator = QMapWebMapGenerator(self.iface)
        else:
            self.webmap_generator = None

        # HTTPサーバーマネージャー
        from .qmap_permalink_server_manager import QMapPermalinkServerManager
        
        self.server_manager = QMapPermalinkServerManager(
            self.iface, 
            self.navigation_signals, 
            self.webmap_generator,
            self  # メインプラグインインスタンスを渡す
        )
        
        # 後方互換性のための一時的な属性（削除予定）
        # server_portアトリビュートエラーを回避
        self.server_port = 8089

        # ツールバーの確認（初回実行時にツールバーが存在するかチェック）
        self.first_start = None

        # 最終的に受信した要求元のURLを保持（パネル未生成時のフォールバック用）
        self._last_request_origin = None

    def handle_request_origin_changed(self, origin):
        """サーバから受け取った完全URLをパネルのlineEdit_navigateにセットする

        origin (str): 例: 'http://localhost:8089/qgis-map?x=...'
        """
        try:
            # 最終値を保持
            self._last_request_origin = origin
            if self.panel and hasattr(self.panel, 'lineEdit_navigate'):
                try:
                    self.panel.lineEdit_navigate.setText(origin)
                except Exception:
                    pass
        except Exception:
            pass

    def tr(self, message):
        """翻訳を取得
        
        Args:
            message: 翻訳対象のメッセージ
            
        Returns:
            翻訳されたメッセージ
        """
        return QCoreApplication.translate('QMapPermalink', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """ツールバーアイコンやメニューアイテムを追加

        Args:
            icon_path: アイコンファイルのパス
            text: アクションのテキスト
            callback: アクションが実行された時のコールバック関数
            enabled_flag: アクションが有効かどうか
            add_to_menu: メニューに追加するかどうか
            add_to_toolbar: ツールバーに追加するかどうか
            status_tip: ステータスバーに表示するヒント
            whats_this: What's Thisヘルプ
            parent: 親ウィジェット

        Returns:
            作成されたQActionオブジェクト
        """
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # このプラグイン専用のツールバーに追加
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """プラグインのGUI要素を作成（プラグイン読み込み時に呼ばれる）"""
        icon_path = os.path.join(self.plugin_dir, 'icon.png')
        
        # パネル版のアクション
        if PANEL_AVAILABLE:
            self.add_action(
                icon_path,
                text=self.tr(u'QMap Permalink'),
                callback=self.toggle_panel,
                parent=self.iface.mainWindow())
        else:
            # パネルが利用できない場合は警告メッセージ
            self.iface.messageBar().pushMessage(
                self.tr("QMap Permalink"),
                self.tr("Panel functionality is not available. Please try reinstalling the plugin."),
                level=2,  # WARNING
                duration=10
            )

        # HTTPサーバーを起動
        self.server_manager.start_http_server()

        # 初回起動フラグ
        self.first_start = True

    def toggle_panel(self):
        """パネルの表示/非表示を切り替え"""
        if not PANEL_AVAILABLE:
            QMessageBox.warning(
                self.iface.mainWindow(),
                self.tr("QMap Permalink"),
                self.tr("Panel functionality is not available.")
            )
            return
            
        try:
            if self.panel is None:
                # パネル利用可能性をデバッグ
                self.iface.messageBar().pushMessage(
                    "QMap Permalink", 
                    f"PANEL_AVAILABLE: {PANEL_AVAILABLE}, QMapPermalinkPanel: {QMapPermalinkPanel}", 
                    duration=5
                )
                
                # パネルを作成
                print("パネル作成開始...")
                self.panel = QMapPermalinkPanel(self.iface.mainWindow())
                print("パネル作成完了")
                
                # パネルのボタンにイベントを接続
                print("ボタンイベント接続開始...")
                self.panel.pushButton_generate.clicked.connect(self.on_generate_clicked_panel)
                self.panel.pushButton_navigate.clicked.connect(self.on_navigate_clicked_panel)
                self.panel.pushButton_copy.clicked.connect(self.on_copy_clicked_panel)
                if hasattr(self.panel, 'pushButton_open'):
                    self.panel.pushButton_open.clicked.connect(self.on_open_clicked_panel)
                
                # Google Maps/Earthボタンのイベントを接続
                if hasattr(self.panel, 'pushButton_google_maps'):
                    self.panel.pushButton_google_maps.clicked.connect(self.on_google_maps_clicked_panel)
                if hasattr(self.panel, 'pushButton_google_earth'):
                    self.panel.pushButton_google_earth.clicked.connect(self.on_google_earth_clicked_panel)
                print("ボタンイベント接続完了")
                
                # HTTPサーバーの状態を更新
                print("サーバー状態取得開始...")
                server_running = self.server_manager.is_server_running()
                server_port = self.server_manager.get_server_port() or 8089
                print(f"サーバー状態: running={server_running}, port={server_port}")
                print("パネル更新開始...")
                self.panel.update_server_status(server_port, server_running)
                # パネルのトグルチェックボックスをサーバー制御に接続
                try:
                    def _toggle_server(checked: bool):
                        try:
                            if checked:
                                self.server_manager.start_http_server()
                            else:
                                self.server_manager.stop_http_server()
                            # 更新後の状態をラベルに反映
                            pr = self.server_manager.get_server_port() or 8089
                            self.panel.update_server_status(pr, self.server_manager.is_server_running())
                        except Exception as e:
                            print(f"サーバートグルエラー: {e}")

                    # set_server_toggle_handler はパネル側で提供
                    if hasattr(self.panel, 'set_server_toggle_handler'):
                        self.panel.set_server_toggle_handler(_toggle_server)
                        # 初期状態を反映（checkbox自体は update_server_status で同期済み）
                except Exception:
                    pass
                print("パネル更新完了")
                
                # テーマ一覧を更新
                self.update_theme_list()
                
                # QGISのメインウィンドウの左側にドッキング
                self.iface.addDockWidget(Qt.LeftDockWidgetArea, self.panel)
                
                # 既存の左側パネルがあればタブ化
                self._try_tabify_with_existing_panels()
                
                # デバッグメッセージ
                self.iface.messageBar().pushMessage(
                    self.tr("QMap Permalink"), 
                    self.tr("Panel created successfully."), 
                    duration=3
                )
            else:
                # パネルの表示/非表示を切り替え
                if self.panel.isVisible():
                    self.panel.hide()
                else:
                    self.panel.show()
                    
        except Exception as e:
            import traceback
            error_details = traceback.format_exc()
            print(f"パネル作成エラーの詳細:\n{error_details}")
            QMessageBox.critical(
                self.iface.mainWindow(),
                self.tr("QMap Permalink"),
                self.tr("Failed to create panel: {error}").format(error=str(e))
            )

    def _try_tabify_with_existing_panels(self):
        """既存の左側パネルがあればタブ化を試行"""
        try:
            # QGISメインウィンドウから左側ドックエリアのウィジェットを取得
            main_window = self.iface.mainWindow()
            
            # よく使われるパネル名のリスト（優先順位順）
            preferred_panels = [
                'Layers',           # レイヤーパネル
                'Browser',          # ブラウザパネル
                'Browser2',         # ブラウザパネル（別名）
                'LayerOrder',       # レイヤー順序パネル
                'Processing',       # プロセシングツールボックス
                'History',          # 履歴パネル
            ]
            
            target_panel = None
            
            # 優先パネルから検索
            for panel_name in preferred_panels:
                for widget in main_window.findChildren(QDockWidget):
                    if (widget != self.panel and 
                        main_window.dockWidgetArea(widget) == Qt.LeftDockWidgetArea and
                        widget.isVisible() and
                        (panel_name.lower() in widget.objectName().lower() or
                         panel_name.lower() in widget.windowTitle().lower())):
                        target_panel = widget
                        break
                if target_panel:
                    break
            
            # 優先パネルが見つからない場合は左側の最初のパネルを使用
            if not target_panel:
                for widget in main_window.findChildren(QDockWidget):
                    if (widget != self.panel and 
                        main_window.dockWidgetArea(widget) == Qt.LeftDockWidgetArea and
                        widget.isVisible()):
                        target_panel = widget
                        break
            
            # タブ化実行
            if target_panel:
                main_window.tabifyDockWidget(target_panel, self.panel)
                
                # QMapPermalinkパネルをアクティブにする
                self.panel.raise_()
                
                self.iface.messageBar().pushMessage(
                    self.tr("QMap Permalink"), 
                    self.tr("Tabified with '{panel_name}' panel.").format(panel_name=target_panel.windowTitle()), 
                    duration=3
                )
            else:
                self.iface.messageBar().pushMessage(
                    self.tr("QMap Permalink"), 
                    self.tr("Displayed as an independent panel on the left side."), 
                    duration=3
                )
                
        except Exception as e:
            # タブ化に失敗しても継続
            print(f"パネルのタブ化でエラー: {e}")
            self.iface.messageBar().pushMessage(
                self.tr("QMap Permalink"), 
                self.tr("Panel displayed on the left side."), 
                duration=3
            )

    def unload(self):
        """プラグインのアンロード時の処理"""
        # HTTPサーバーを停止
        self.server_manager.stop_http_server()
        
        # パネルを削除
        if self.panel is not None:
            self.iface.removeDockWidget(self.panel)
            self.panel = None
        
        # シグナルを切断
        if hasattr(self, 'navigation_signals'):
            self.navigation_signals.navigate_requested.disconnect()
        
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&QMap Permalink'),
                action)
            self.iface.removeToolBarIcon(action)






        
    # 純粋にHTTPサーバー関連のメソッドは qmap_permalink_server_manager.py に移動しました
    # _build_google_maps_url, _build_google_earth_url, _estimate_zoom_from_scale, _convert_to_wgs84 は
    # メインファイルでも使用されるため、こちらに残しています



    def _build_google_maps_url(self, navigation_data):
        """ナビゲーションデータからGoogle Maps用URLを生成

        Google Maps用は zoom が必要。zoom が与えられなければ scale から推定する。
        最終フォールバックは zoom=16 とする。
        """
        try:
            if navigation_data.get('type') == 'coordinates':
                lat, lon = self._resolve_coordinates(navigation_data)
                if lat is None or lon is None:
                    return None
                # zoom が無ければ scale から推定し、無ければデフォルト16を使う
                zoom_value = navigation_data.get('zoom')
                if zoom_value is None:
                    zoom_value = self._estimate_zoom_from_scale(navigation_data.get('scale'))
                if zoom_value is None:
                    zoom_value = 16.0
                # 小数点レベルのズームをサポート（最大2桁まで）
                zoom_formatted = f"{float(zoom_value):.2f}".rstrip('0').rstrip('.')
                return f"https://www.google.co.jp/maps/@{lat:.6f},{lon:.6f},{zoom_formatted}z"

            if navigation_data.get('type') == 'location':
                lat = navigation_data.get('lat')
                lon = navigation_data.get('lon')
                zoom_value = navigation_data.get('zoom')

                if lat is None or lon is None:
                    try:
                        decoded = urllib.parse.unquote(navigation_data['location'])
                        data = json.loads(decoded)
                    except Exception:
                        data = {}

                    if data:
                        center_lat = data.get('center_wgs84_lat')
                        center_lon = data.get('center_wgs84_lon')
                        if center_lat is not None and center_lon is not None:
                            lat = float(center_lat)
                            lon = float(center_lon)
                        else:
                            center_x = data.get('center_x')
                            center_y = data.get('center_y')
                            crs_authid = data.get('center_crs') or data.get('crs')
                            if center_x is not None and center_y is not None and crs_authid:
                                lat, lon = self._convert_to_wgs84(center_x, center_y, crs_authid)
                        if zoom_value is None:
                            zoom_value = self._estimate_zoom_from_scale(data.get('scale'))

                if (lat is None or lon is None) and navigation_data.get('center_x') is not None:
                    crs_authid = navigation_data.get('crs')
                    lat, lon = self._convert_to_wgs84(
                        navigation_data.get('center_x'),
                        navigation_data.get('center_y'),
                        crs_authid,
                    )

                if lat is None or lon is None:
                    return None

                if zoom_value is None:
                    zoom_value = 16.0

                # 小数点レベルのズームをサポート（最大2桁まで）
                zoom_formatted = f"{float(zoom_value):.2f}".rstrip('0').rstrip('.')
                return f"https://www.google.co.jp/maps/@{lat:.6f},{lon:.6f},{zoom_formatted}z"

        except Exception:
            return None

        return None

    def _build_google_earth_url(self, navigation_data):
        """ナビゲーションデータからGoogle Earth用URLを生成

        Google Earth Web版用のURL形式: https://earth.google.com/web/@lat,lon,altitude,heading,tilt,roll
        Google Mapsと同じスケール変換を使用してより正確な高度を計算します。
        """
        try:
            if navigation_data.get('type') == 'coordinates':
                lat, lon = self._resolve_coordinates(navigation_data)
                if lat is None or lon is None:
                    return None
                
                # Google Mapsと同じズームレベル推定を使用
                zoom_value = navigation_data.get('zoom')
                if zoom_value is None:
                    zoom_value = self._estimate_zoom_from_scale(navigation_data.get('scale'))
                if zoom_value is None:
                    zoom_value = 16.0
                
                # Google Earth用の正確なパラメータ計算（実測値に基づく）
                # 実測分析: 1:15695スケール → 距離5554m、高度22m
                if navigation_data.get('scale'):
                    scale_value = navigation_data['scale']
                else:
                    # ズームレベルからスケールを逆算
                    estimated_scale = self._estimate_scale_from_zoom(zoom_value)
                    scale_value = estimated_scale
                
                # 実測データに基づくGoogle Earth用パラメータ計算
                # 実測値: 1:15695スケール → 高度32m、距離160699m、1y角度
                if scale_value:
                    # 実測基準値
                    reference_scale = 15695.0
                    reference_altitude = 32.03670052  # 実測高度
                    reference_distance = 160699.35527964  # 実測距離
                    
                    # スケールに比例した高度計算（実測データベース）
                    altitude = reference_altitude * (scale_value / reference_scale) ** 0.5
                    altitude = max(10.0, min(2000.0, altitude))
                    
                    # スケールに比例した距離計算（実測データベース）
                    distance = reference_distance * (scale_value / reference_scale)
                    distance = max(100.0, min(500000.0, distance))
                else:
                    altitude = 100.0
                    distance = 50000.0
                
                # 実測に基づくGoogle Earth URL形式（1y角度で適切な表示）
                return f"https://earth.google.com/web/@{lat:.6f},{lon:.6f},{altitude:.8f}a,{distance:.8f}d,1y,0h,0t,0r"

            if navigation_data.get('type') == 'location':
                lat = navigation_data.get('lat')
                lon = navigation_data.get('lon')
                zoom_value = navigation_data.get('zoom')

                if lat is None or lon is None:
                    try:
                        decoded = urllib.parse.unquote(navigation_data['location'])
                        data = json.loads(decoded)
                    except Exception:
                        data = {}

                    if data:
                        center_lat = data.get('center_wgs84_lat')
                        center_lon = data.get('center_wgs84_lon')
                        if center_lat is not None and center_lon is not None:
                            lat = float(center_lat)
                            lon = float(center_lon)
                        else:
                            center_x = data.get('center_x')
                            center_y = data.get('center_y')
                            crs_authid = data.get('center_crs') or data.get('crs')
                            if center_x is not None and center_y is not None and crs_authid:
                                lat, lon = self._convert_to_wgs84(center_x, center_y, crs_authid)
                        if zoom_value is None:
                            zoom_value = self._estimate_zoom_from_scale(data.get('scale'))

                if (lat is None or lon is None) and navigation_data.get('center_x') is not None:
                    crs_authid = navigation_data.get('crs')
                    lat, lon = self._convert_to_wgs84(
                        navigation_data.get('center_x'),
                        navigation_data.get('center_y'),
                        crs_authid,
                    )

                if lat is None or lon is None:
                    return None

                if zoom_value is None:
                    zoom_value = 16.0

                # Google Earth用の正確なパラメータ計算（実測值に基づく）
                scale_value = navigation_data.get('scale')
                if not scale_value and 'location' in navigation_data:
                    try:
                        decoded = urllib.parse.unquote(navigation_data['location'])
                        data = json.loads(decoded)
                        scale_value = data.get('scale')
                    except Exception:
                        pass
                
                if scale_value:
                    # 実測データに基づくGoogle Earth用パラメータ計算
                    # 実測値: 1:15695スケール → 高度32m、距離160699m、1y角度
                    reference_scale = 15695.0
                    reference_altitude = 32.03670052  # 実測高度
                    reference_distance = 160699.35527964  # 実測距離
                    
                    # スケールに比例した高度計算（実測データベース）
                    altitude = reference_altitude * (scale_value / reference_scale) ** 0.5
                    altitude = max(10.0, min(2000.0, altitude))
                    
                    # スケールに比例した距離計算（実測データベース）
                    distance = reference_distance * (scale_value / reference_scale)
                    distance = max(100.0, min(500000.0, distance))
                else:
                    altitude = 100.0
                    distance = 50000.0
                
                # 実測に基づくGoogle Earth URL形式（1y角度で適切な表示）
                return f"https://earth.google.com/web/@{lat:.6f},{lon:.6f},{altitude:.8f}a,{distance:.8f}d,1y,0h,0t,0r"

        except Exception:
            return None

        return None

    def _resolve_coordinates(self, navigation_data):
        """ナビゲーションデータからWGS84座標を求める"""
        lat = navigation_data.get('lat')
        lon = navigation_data.get('lon')
        if lat is not None and lon is not None:
            return float(lat), float(lon)

        x = navigation_data.get('x')
        y = navigation_data.get('y')
        crs_authid = navigation_data.get('crs')
        if x is None or y is None or crs_authid is None:
            return None, None

        return self._convert_to_wgs84(x, y, crs_authid)


    def _estimate_zoom_from_scale(self, scale):
        """スケール値からGoogle Maps用ズームレベルを推定（連続値対応改良版）
        
        固定テーブル方式をベースに、テーブル間の中間値を線形補間で計算し、
        小数点レベルでの詳細なズームレベル推定を可能にします。
        """
        if not scale:
            return 16.0
        try:
            s = float(scale)
            if s <= 0:
                return 16.0

            # QGIS実スケール対応の改良版固定スケールテーブル
            # 詳細スケール（1:500〜1:25000）で+1〜2ズームレベル上げて調整
            scale_table = {
                0: 400_000_000.0, 1: 200_000_000.0, 2: 100_000_000.0, 3: 60_000_000.0, 4: 30_000_000.0,
                5: 15_000_000.0, 6: 8_000_000.0, 7: 4_000_000.0, 8: 2_000_000.0, 9: 1_000_000.0,
                # 中〜詳細スケールを高ズーム方向に調整
                10: 600_000.0,    # 元: 400_000.0 → より詳細に
                11: 300_000.0,    # 元: 200_000.0 → より詳細に
                12: 150_000.0,    # 元: 100_000.0 → より詳細に
                13: 75_000.0,     # 元: 40_000.0 → 大幅に詳細化
                14: 40_000.0,     # 元: 20_000.0 → 2倍詳細
                15: 20_000.0,     # 元: 10_000.0 → 2倍詳細
                16: 10_000.0,     # 元: 5_000.0 → 2倍詳細
                17: 5_000.0,      # 元: 2_500.0 → 2倍詳細
                18: 2_500.0,      # 元: 1_250.0 → 2倍詳細
                19: 1_250.0,      # 元: 600.0 → 大幅に詳細化
                20: 600.0,        # 元: 300.0 → 2倍詳細
                21: 300.0,        # 元: 150.0 → 2倍詳細
                22: 150.0,        # 元: 75.0 → 2倍詳細
                23: 75.0,         # 元: 40.0 → やや詳細に
            }

            # 外挿: 24-30 は 23 の値を半分ずつ外挿
            for z in range(24, 31):
                scale_table[z] = scale_table[23] / (2 ** (z - 23))

            # 対数空間での線形補間によるズームレベル推定
            target_log = math.log(s)
            
            # ソートされたズームレベルのリストを作成
            zoom_levels = sorted(scale_table.keys())
            
            # 範囲外の処理
            if s >= scale_table[zoom_levels[0]]:
                return float(zoom_levels[0])  # 最小ズームレベル
            if s <= scale_table[zoom_levels[-1]]:
                return float(zoom_levels[-1])  # 最大ズームレベル
            
            # 線形補間で中間値を計算
            for i in range(len(zoom_levels) - 1):
                z1, z2 = zoom_levels[i], zoom_levels[i + 1]
                s1, s2 = scale_table[z1], scale_table[z2]
                
                # 対象スケールが2つのテーブル値の間にある場合
                if s1 >= s >= s2:
                    # 対数空間での線形補間
                    log_s1, log_s2 = math.log(s1), math.log(s2)
                    # 補間係数を計算（0.0〜1.0）
                    t = (target_log - log_s1) / (log_s2 - log_s1) if log_s2 != log_s1 else 0.0
                    # ズームレベルを線形補間
                    interpolated_zoom = z1 + t * (z2 - z1)
                    return max(0.0, min(30.0, interpolated_zoom))
            
            # フォールバック（理論的には到達しないはず）
            return 16.0
            
        except (ValueError, TypeError, OverflowError):
            return 16.0

    def _zoom_to_earth_distance(self, zoom_level):
        """ズームレベルからGoogle Earth用の適切な距離を計算
        
        Args:
            zoom_level: ズームレベル（小数点可）
            
        Returns:
            距離（メートル）
        """
        if zoom_level is None:
            return 5000
            
        try:
            z = float(zoom_level)
            # ズームレベルに対応する距離テーブル
            zoom_distances = {
                0: 20000000, 1: 10000000, 2: 5000000, 3: 2000000, 4: 1000000,
                5: 500000, 6: 200000, 7: 100000, 8: 50000, 9: 20000,
                10: 10000, 11: 5000, 12: 2000, 13: 1000, 14: 500,
                15: 200, 16: 100, 17: 50, 18: 20, 19: 10, 20: 5
            }
            
            # 最も近い整数ズームレベルの距離を使用
            rounded_zoom = max(0, min(20, round(z)))
            return zoom_distances.get(rounded_zoom, 5000)
            
        except (ValueError, TypeError):
            return 5000

    def _estimate_scale_from_zoom(self, zoom_level):
        """ズームレベルからスケール値を逆算（小数点対応版）
        
        小数点レベルのズームレベルにも対応し、線形補間でスケール値を計算します。
        
        Args:
            zoom_level: ズームレベル（小数点可）
            
        Returns:
            推定スケール値
        """
        if zoom_level is None:
            return 20000.0  # デフォルトスケール
            
        try:
            z = float(zoom_level)
            
            # _estimate_zoom_from_scale と同じテーブル
            scale_table = {
                0: 400_000_000.0, 1: 200_000_000.0, 2: 100_000_000.0, 3: 60_000_000.0, 4: 30_000_000.0,
                5: 15_000_000.0, 6: 8_000_000.0, 7: 4_000_000.0, 8: 2_000_000.0, 9: 1_000_000.0,
                10: 600_000.0, 11: 300_000.0, 12: 150_000.0, 13: 75_000.0, 14: 40_000.0,
                15: 20_000.0, 16: 10_000.0, 17: 5_000.0, 18: 2_500.0, 19: 1_250.0,
                20: 600.0, 21: 300.0, 22: 150.0, 23: 75.0,
            }
            
            # 外挿値も計算
            for zoom in range(24, 31):
                scale_table[zoom] = scale_table[23] / (2 ** (zoom - 23))
            
            # 範囲チェック
            z = max(0.0, min(30.0, z))
            
            # 整数ズームレベルの場合はテーブルから直接取得
            if z == int(z) and int(z) in scale_table:
                return scale_table[int(z)]
            
            # 小数点ズームレベルの場合は線形補間
            z_floor = int(math.floor(z))
            z_ceil = int(math.ceil(z))
            
            # 範囲内チェック
            if z_floor < 0:
                z_floor = 0
            if z_ceil > 30:
                z_ceil = 30
            if z_floor not in scale_table:
                z_floor = max([k for k in scale_table.keys() if k <= z_floor], default=0)
            if z_ceil not in scale_table:
                z_ceil = min([k for k in scale_table.keys() if k >= z_ceil], default=30)
                
            # 同じ値の場合
            if z_floor == z_ceil:
                return scale_table.get(z_floor, 20000.0)
            
            # 線形補間（対数空間）
            s1, s2 = scale_table[z_floor], scale_table[z_ceil]
            log_s1, log_s2 = math.log(s1), math.log(s2)
            
            # 補間係数
            t = (z - z_floor) / (z_ceil - z_floor) if z_ceil != z_floor else 0.0
            
            # 対数空間で補間してから指数に戻す
            interpolated_log_scale = log_s1 + t * (log_s2 - log_s1)
            interpolated_scale = math.exp(interpolated_log_scale)
            
            return interpolated_scale
            
        except (ValueError, TypeError, OverflowError):
            return 20000.0

    def _convert_to_wgs84(self, x, y, source_crs_authid):
        """任意座標をWGS84へ変換"""
        try:
            source_crs = QgsCoordinateReferenceSystem(str(source_crs_authid))
            if not source_crs.isValid():
                return None, None
            target_crs = QgsCoordinateReferenceSystem("EPSG:4326")
            transform = QgsCoordinateTransform(source_crs, target_crs, QgsProject.instance())
            point = transform.transform(QgsPointXY(float(x), float(y)))
            return float(point.y()), float(point.x())
        except Exception:
            return None, None


    
    def generate_permalink(self, include_theme=True, specific_theme=None):
        """現在の地図ビューからパーマリンクを生成
        
        Args:
            include_theme (bool): テーマ情報を含めるかどうか
            specific_theme (str): 指定するテーマ名（None の場合は現在の状態を使用）
        
        Returns:
            パーマリンクURL文字列（HTTP形式）
        """
        # 現在のマップキャンバス情報を取得
        canvas = self.iface.mapCanvas()
        extent = canvas.extent()
        crs = canvas.mapSettings().destinationCrs()
        scale = canvas.scale()
        # 回転角度（度）
        rotation = canvas.rotation() if hasattr(canvas, 'rotation') else 0.0
        map_units_per_pixel = canvas.mapUnitsPerPixel()
        center_point = QgsPointXY(
            (extent.xMinimum() + extent.xMaximum()) / 2.0,
            (extent.yMinimum() + extent.yMaximum()) / 2.0,
        )

        # スケールからズームレベルを推定（Web標準対応でより正確）
        zoom_level = self._estimate_zoom_from_scale(scale)

        # 基本パラメータを構築
        x_val = f"{center_point.x():.6f}"
        y_val = f"{center_point.y():.6f}"
        crs_id = crs.authid()  # e.g. 'EPSG:3857' or 'EPSG:4326'
        # scale はキャンバスの scale() を使う
        scale_val = float(scale) if scale is not None else None
        if scale_val is None:
            # 万が一 scale が取得できなければ、ズームレベルから推定して scale を算出（逆算は簡易）
            # ここでは推定値として 1000 を入れておく
            scale_val = 1000.0
        
        # 基本URL構築（OpenLayersページ生成用）
        server_port = self.server_manager.get_server_port() or 8089
        
        permalink_url = (
            f"http://localhost:{server_port}/qgis-map?x={x_val}&y={y_val}"
            f"&scale={scale_val:.1f}&crs={crs_id}&rotation={rotation:.2f}"
        )
        
        # テーマ情報を追加（オプション）
        if include_theme and specific_theme:
            # シンプルなテーマ名をパラメータに追加
            theme_encoded = urllib.parse.quote(specific_theme)
            permalink_url += f"&theme={theme_encoded}"
        
        return permalink_url

    def navigate_to_permalink(self, permalink_url):
        """パーマリンクURLから地図ビューに移動
        
        Args:
            permalink_url: パーマリンクURL
        """
        try:
            # HTTP形式のURLを処理（新しいWMS形式と古い形式の両方をサポート）
            # NOTE: 以前は 'http://localhost:' で始まるURLのみ内部ナビゲーションと見なしていましたが
            # ローカルネットワークのIPやホスト名から来る場合も利用されるため、ホスト名に依存せず
            # スキームが http(s) でパスに /wms または /qgis-map を含む場合は内部ナビゲーションとして扱います。
            parsed_url = urllib.parse.urlparse(permalink_url)
            scheme = (parsed_url.scheme or '').lower()
            path = parsed_url.path or ''

            if scheme in ('http', 'https') and ('/wms' in path or '/qgis-map' in path):
                # HTTP URLから直接実行（ブラウザを経由しない）
                params = urllib.parse.parse_qs(parsed_url.query)

                # パラメータをナビゲーションデータへ変換して処理（location または coordinates をサポート）
                try:
                    navigation_data = self.server_manager._build_navigation_data_from_params(params)
                except ValueError as e:
                    raise

                if navigation_data.get('type') == 'location':
                    # エンコード済み location JSON を処理
                    self.navigate_from_http(navigation_data['location'])
                elif navigation_data.get('type') == 'coordinates':
                    x = navigation_data.get('x')
                    y = navigation_data.get('y')
                    zoom = navigation_data.get('zoom')
                    scale = navigation_data.get('scale')
                    crs = navigation_data.get('crs')
                    rotation = navigation_data.get('rotation')
                    theme_info = navigation_data.get('theme_info')
                    # 直接移動を実行（scale を優先）
                    self.navigate_to_coordinates(x, y, scale, zoom, crs, rotation, theme_info)
                else:
                    raise ValueError("HTTP URLのパラメータからナビゲーションデータを生成できませんでした。")
                    
            # 従来のカスタムプロトコル形式も維持
            elif permalink_url.startswith('qgis-permalink://'):
                encoded_data = permalink_url[17:]  # "qgis-permalink://"を除去
                json_data = urllib.parse.unquote(encoded_data)
                permalink_data = json.loads(json_data)
                
                # 座標系とextentを復元
                crs = QgsCoordinateReferenceSystem(permalink_data['crs'])
                extent = QgsRectangle(
                    permalink_data['x_min'],
                    permalink_data['y_min'],
                    permalink_data['x_max'],
                    permalink_data['y_max']
                )
                
                # マップキャンバスに適用
                canvas = self.iface.mapCanvas()
                canvas.setDestinationCrs(crs)
                canvas.setExtent(extent)
                canvas.refresh()
                
                self.iface.messageBar().pushMessage(
                    "QMap Permalink", 
                    "パーマリンクから地図ビューに移動しました。", 
                    duration=3
                )
                
            else:
                raise ValueError("サポートされていないパーマリンクURL形式です。")
                
        except Exception as e:
            QMessageBox.warning(
                self.iface.mainWindow(),
                "QMap Permalink エラー",
                f"パーマリンクの処理中にエラーが発生しました：\n{str(e)}"
            )
    
    def navigate_from_http(self, location_data):
        """HTTP経由でのナビゲーション処理
        
        Args:
            location_data: エンコードされた位置データ
        """
        try:
            # URLデコードしてJSONパース
            json_data = urllib.parse.unquote(location_data)
            permalink_data = json.loads(json_data)
            
            # 座標系とextentを復元
            crs = QgsCoordinateReferenceSystem(permalink_data['crs'])
            extent = QgsRectangle(
                permalink_data['x_min'],
                permalink_data['y_min'],
                permalink_data['x_max'],
                permalink_data['y_max']
            )
            
            # マップキャンバスに適用
            canvas = self.iface.mapCanvas()
            canvas.setDestinationCrs(crs)
            canvas.setExtent(extent)
            canvas.refresh()
            
            self.iface.messageBar().pushMessage(
                "QMap Permalink", 
                "HTTP経由で地図ビューに移動しました。", 
                duration=3
            )
            
        except Exception as e:
            raise Exception(f"HTTP地図移動の処理中にエラーが発生しました: {str(e)}")
    
    def navigate_to_coordinates(self, x, y, scale, zoom, crs_auth_id, rotation=None, theme_info=None):
        """座標指定でのナビゲーション処理
        
        Args:
            x: 経度またはX座標
            y: 緯度またはY座標  
            zoom: ズームレベル
            crs_auth_id: 座標系ID (例: "EPSG:4326")
            rotation: 回転角度（度）
            theme_info: テーマ情報（辞書）
        """
        try:
            # 座標系を設定
            crs = QgsCoordinateReferenceSystem(crs_auth_id)

            # scale があればそれを優先して、QGIS のキャンバス API に scale をそのまま適用する
            # ここでは明示的なスケール変換は行わず、canvas.zoomScale を使ってシンプルに反映する
            scale_val = None
            if scale is not None:
                try:
                    scale_val = float(scale)
                except Exception:
                    scale_val = None

            # scale が無ければ zoom から簡易推定する（従来の互換用）
            if scale_val is None and zoom is not None:
                try:
                    zoom_val = float(zoom)
                    scale_val = 1000.0 / (2 ** (zoom_val - 10))
                except Exception:
                    scale_val = None

            # 最終フォールバック
            if scale_val is None:
                scale_val = 1000.0

            # マップキャンバスに適用
            canvas = self.iface.mapCanvas()
            canvas.setDestinationCrs(crs)

            try:
                # 中心点を設定してからスケールを適用する（QGIS 側で正しい表示範囲が計算される）
                canvas.setCenter(QgsPointXY(float(x), float(y)))
                canvas.zoomScale(float(scale_val))
                # 回転が指定されていれば適用
                if rotation is not None:
                    try:
                        canvas.setRotation(float(rotation))
                    except Exception:
                        pass
            except Exception:
                # 万が一 canvas の API が使えない/失敗した場合は従来の範囲設定にフォールバック
                half_width = float(scale_val) / 2.0
                half_height = float(scale_val) / 2.0
                extent = QgsRectangle(
                    float(x) - half_width,
                    float(y) - half_height,
                    float(x) + half_width,
                    float(y) + half_height
                )
                canvas.setExtent(extent)

            # テーマ情報がある場合は適用
            theme_applied = False
            if theme_info:
                try:
                    theme_applied = self._apply_theme_from_permalink(theme_info)
                    if theme_applied:
                        print("テーマが正常に適用されました")
                except Exception as e:
                    print(f"テーマ適用エラー: {e}")
            
            canvas.refresh()
            
            # メッセージを表示
            message = f"座標 ({x:.6f}, {y:.6f}) に移動しました。"
            if theme_applied:
                message += " テーマも復元されました。"
            
            self.iface.messageBar().pushMessage(
                "QMap Permalink", 
                message, 
                duration=3
            )
            
        except Exception as e:
            raise Exception(f"座標移動の処理中にエラーが発生しました: {str(e)}")

    # パネル用のイベントハンドラ

    def handle_navigation_request(self, navigation_data):
        """HTTPリクエストからのナビゲーション要求を安全に処理（メインスレッドで実行）
        
        Args:
            navigation_data: ナビゲーション情報を含む辞書
        """
        try:
            if navigation_data['type'] == 'location':
                # JSON形式のlocationデータを処理
                location_data = navigation_data['location']
                self.navigate_from_http(location_data)
            elif navigation_data['type'] == 'coordinates':
                # 個別座標パラメータを処理
                x = navigation_data['x']
                y = navigation_data['y']
                zoom = navigation_data.get('zoom')
                scale = navigation_data.get('scale')
                crs = navigation_data['crs']
                rotation = navigation_data.get('rotation')
                theme_info = navigation_data.get('theme_info')
                # scale を優先して渡す（None の場合は zoom を使う）
                self.navigate_to_coordinates(x, y, scale, zoom, crs, rotation, theme_info)
                
            print(f"ナビゲーション完了: {navigation_data['type']}")
            
        except Exception as e:
            print(f"ナビゲーション処理エラー: {e}")
            self.iface.messageBar().pushMessage(
                "QMap Permalink エラー", 
                f"ナビゲーション処理中にエラーが発生しました: {str(e)}", 
                duration=5
            )
    


    # パネル用のイベントハンドラ
    def on_generate_clicked_panel(self):
        """パネル版：パーマリンク生成ボタンがクリックされた時の処理"""
        try:
            # ドロップダウンの選択からテーマ設定を判定
            include_theme = False
            specific_theme = None
            
            if hasattr(self.panel, 'comboBox_themes'):
                selected_option = self.panel.comboBox_themes.currentText()
                
                if selected_option == "-- No Theme (Position Only) --":
                    include_theme = False
                    specific_theme = None
                elif selected_option:  # 実際のテーマ名が選択された場合
                    include_theme = True
                    specific_theme = selected_option
            
            permalink = self.generate_permalink(include_theme=include_theme, specific_theme=specific_theme)
            self.panel.lineEdit_permalink.setText(permalink)
            
            # ナビゲート用の欄にも同じURLを自動設定
            self.panel.lineEdit_navigate.setText(permalink)
            
            # メッセージにテーマ情報の有無を含める
            if include_theme and specific_theme:
                message = self.tr("Permalink with theme '{theme}' generated successfully.").format(theme=specific_theme)
            else:
                message = self.tr("Permalink (position only) generated successfully.")
            
            self.iface.messageBar().pushMessage(
                self.tr("QMap Permalink"), 
                message, 
                duration=3
            )
        except Exception as e:
            QMessageBox.critical(
                self.iface.mainWindow(),
                self.tr("QMap Permalink"),
                self.tr("Failed to generate permalink: {error}").format(error=str(e))
            )

    def on_navigate_clicked_panel(self):
        """パネル版：ナビゲートボタンがクリックされた時の処理"""
        permalink_url = self.panel.lineEdit_navigate.text().strip()
        if not permalink_url:
            QMessageBox.warning(
                self.iface.mainWindow(),
                self.tr("QMap Permalink"),
                self.tr("Please enter a permalink URL.")
            )
            return
        # Parse URL and decide:
        # - If it's an http(s) URL pointing to localhost and contains
        #   internal endpoints (/wms or /qgis-map), route to internal navigation.
        # - Otherwise, open external http(s) URLs in the default browser.
        try:
            parsed = urllib.parse.urlparse(permalink_url)
            scheme = (parsed.scheme or '').lower()
            path = parsed.path or ''

            # Treat any http(s) URL whose path or full URL contains /wms or /qgis-map as internal navigation
            # This covers IP addresses, hostnames, and cases where path may be empty but the pattern appears elsewhere.
            lowered = permalink_url.lower()
            is_internal_http = scheme in ('http', 'https') and ('/wms' in path or '/qgis-map' in path or '/wms' in lowered or '/qgis-map' in lowered)

            if is_internal_http:
                # Internal navigation: let existing handler parse and apply
                self.navigate_to_permalink(permalink_url)
                return

            if scheme in ('http', 'https'):
                # External http(s) — but avoid opening browser if this is actually an internal navigation URL
                if is_internal_http:
                    # already handled above, but guard defensively
                    try:
                        self.navigate_to_permalink(permalink_url)
                        return
                    except Exception:
                        pass

                try:
                    QDesktopServices.openUrl(QUrl(permalink_url))
                    self.iface.messageBar().pushMessage(
                        self.tr("QMap Permalink"),
                        self.tr("URL opened in browser."),
                        duration=3
                    )
                    return
                except Exception:
                    # If opening fails, fall back to internal navigation attempt
                    pass

        except Exception:
            # Parsing failed — fall back to internal navigation
            pass

        # Fallback: perform internal navigation using the existing handler
        self.navigate_to_permalink(permalink_url)

    def on_copy_clicked_panel(self):
        """パネル版：コピーボタンがクリックされた時の処理"""
        permalink_url = self.panel.lineEdit_permalink.text().strip()
        if not permalink_url:
            QMessageBox.warning(
                self.iface.mainWindow(),
                self.tr("QMap Permalink"),
                self.tr("No permalink available to copy.")
            )
            return
            
        clipboard = QApplication.clipboard()
        success = False

        for _ in range(3):
            clipboard.setText(permalink_url, mode=QClipboard.Clipboard)
            QApplication.processEvents()
            if clipboard.text(mode=QClipboard.Clipboard) == permalink_url:
                if clipboard.supportsSelection():
                    clipboard.setText(permalink_url, mode=QClipboard.Selection)
                success = True
                break
            QThread.msleep(50)

        if success:
            self.iface.messageBar().pushMessage(
                self.tr("QMap Permalink"),
                self.tr("Permalink copied to clipboard."),
                duration=3
            )
        else:
            QMessageBox.warning(
                self.iface.mainWindow(),
                self.tr("QMap Permalink"),
                self.tr("Failed to copy to clipboard. Please try again or copy manually.")
            )

    def on_open_clicked_panel(self):
        """パネル版：ブラウザで開くボタンがクリックされた時の処理"""
        permalink_url = self.panel.lineEdit_permalink.text().strip()
        if not permalink_url:
            QMessageBox.warning(
                self.iface.mainWindow(),
                self.tr("QMap Permalink"),
                self.tr("No permalink available to open.")
            )
            return
            
        try:
            QDesktopServices.openUrl(QUrl(permalink_url))
            self.iface.messageBar().pushMessage(
                self.tr("QMap Permalink"),
                self.tr("Permalink opened in browser."),
                duration=3
            )
        except Exception as e:
            QMessageBox.critical(
                self.iface.mainWindow(),
                self.tr("QMap Permalink"),
                self.tr("Failed to open in browser: {error}").format(error=str(e))
            )

    def on_google_maps_clicked_panel(self):
        """パネル版：Google Mapsボタンがクリックされた時の処理"""
        try:
            # 現在の地図ビューから直接ナビゲーションデータを作成
            canvas = self.iface.mapCanvas()
            extent = canvas.extent()
            crs = canvas.mapSettings().destinationCrs()
            scale = canvas.scale()
            
            # 中心点を計算
            center_x = extent.center().x()
            center_y = extent.center().y()
            
            # WGS84座標に変換
            lat, lon = self._convert_to_wgs84(center_x, center_y, crs.authid())
            
            if lat is None or lon is None:
                self.iface.messageBar().pushMessage(
                    "QMap Permalink", "座標変換に失敗しました。", duration=3
                )
                return
            
            # HTTPレスポンス用と同じ形式のナビゲーションデータを作成
            navigation_data = {
                'type': 'coordinates',
                'x': center_x,
                'y': center_y,
                'lat': lat,
                'lon': lon,
                'scale': scale,
                'crs': crs.authid(),
                'zoom': self._estimate_zoom_from_scale(scale)
            }
            
            # HTTPレスポンスと同じメソッドでGoogle Maps URLを生成
            google_maps_url = self._build_google_maps_url(navigation_data)
            if google_maps_url:
                QDesktopServices.openUrl(QUrl(google_maps_url))
                self.iface.messageBar().pushMessage(
                    "QMap Permalink", "Google Mapsで開きました。", duration=3
                )
        except Exception as e:
            self.iface.messageBar().pushMessage(
                "QMap Permalink", f"Google Maps起動エラー: {str(e)}", duration=5
            )

    def on_google_earth_clicked_panel(self):
        """パネル版：Google Earthボタンがクリックされた時の処理"""
        try:
            # 現在の地図ビューから直接ナビゲーションデータを作成
            canvas = self.iface.mapCanvas()
            extent = canvas.extent()
            crs = canvas.mapSettings().destinationCrs()
            scale = canvas.scale()
            
            # 中心点を計算
            center_x = extent.center().x()
            center_y = extent.center().y()
            
            # WGS84座標に変換
            lat, lon = self._convert_to_wgs84(center_x, center_y, crs.authid())
            
            if lat is None or lon is None:
                self.iface.messageBar().pushMessage(
                    "QMap Permalink", "座標変換に失敗しました。", duration=3
                )
                return
            
            # HTTPレスポンス用と同じ形式のナビゲーションデータを作成
            navigation_data = {
                'type': 'coordinates',
                'x': center_x,
                'y': center_y,
                'lat': lat,
                'lon': lon,
                'scale': scale,
                'crs': crs.authid(),
                'zoom': self._estimate_zoom_from_scale(scale)
            }
            
            # HTTPレスポンスと同じメソッドでGoogle Earth URLを生成
            google_earth_url = self._build_google_earth_url(navigation_data)
            if google_earth_url:
                QDesktopServices.openUrl(QUrl(google_earth_url))
                self.iface.messageBar().pushMessage(
                    "QMap Permalink", "Google Earthで開きました。", duration=3
                )
        except Exception as e:
            self.iface.messageBar().pushMessage(
                "QMap Permalink", f"Google Earth起動エラー: {str(e)}", duration=5
            )

    # テーマ関連のメソッド群
    
    def _get_current_theme_info(self):
        """現在のテーマ情報を取得（無効化 - 複雑すぎるため）
        
        Returns:
            None: 常にNoneを返す（機能無効化）
        """
        # 複雑すぎて安定しないため、この機能を無効化
        return None
        try:
            from qgis.core import QgsProject, QgsMapThemeCollection
            
            project = QgsProject.instance()
            if not project:
                return None
                
            theme_collection = project.mapThemeCollection()
            if not theme_collection:
                return None
            
            # 現在のレイヤー状態を取得
            layer_states = self._get_current_layer_states()
            
            # アクティブなテーマがあるかチェック
            current_theme = self._detect_current_theme(theme_collection, layer_states)
            
            theme_info = {
                'version': '1.0',
                'current_theme': current_theme,
                'layer_states': layer_states,
                'available_themes': theme_collection.mapThemes()
            }
            
            return theme_info
            
        except ImportError:
            # QGISが利用できない環境
            return None
        except Exception as e:
            print(f"テーマ情報取得エラー: {e}")
            return None
    
    def _get_current_layer_states(self):
        """現在のレイヤー状態を取得
        
        Returns:
            dict: レイヤー状態情報
        """
        try:
            from qgis.core import QgsProject, QgsLayerTreeLayer, QgsLayerTreeGroup
            
            project = QgsProject.instance()
            root = project.layerTreeRoot()
            
            layer_states = {}
            
            def collect_layer_info(node, path=""):
                """レイヤーノードから情報を再帰的に収集"""
                if isinstance(node, QgsLayerTreeLayer):
                    layer = node.layer()
                    if layer:
                        layer_id = layer.id()
                        layer_states[layer_id] = {
                            'name': layer.name(),
                            'visible': node.isVisible(),
                            'expanded': node.isExpanded(),
                            'opacity': getattr(layer, 'opacity', lambda: 1.0)(),
                            'path': path,
                            'type': layer.type().name if hasattr(layer.type(), 'name') else str(layer.type())
                        }
                        
                        # スタイル情報も含める（可能であれば）
                        if hasattr(layer, 'styleManager'):
                            try:
                                current_style = layer.styleManager().currentStyle()
                                layer_states[layer_id]['current_style'] = current_style
                                layer_states[layer_id]['available_styles'] = layer.styleManager().styles()
                            except:
                                pass
                                
                elif isinstance(node, QgsLayerTreeGroup):
                    group_path = f"{path}/{node.name()}" if path else node.name()
                    layer_states[f"group:{node.name()}"] = {
                        'name': node.name(),
                        'type': 'group',
                        'visible': node.isVisible(),
                        'expanded': node.isExpanded(),
                        'path': path
                    }
                    
                    # 子要素を再帰的に処理
                    for child in node.children():
                        collect_layer_info(child, group_path)
            
            # ルートから開始
            for child in root.children():
                collect_layer_info(child)
            
            return layer_states
            
        except Exception as e:
            print(f"レイヤー状態取得エラー: {e}")
            return {}
    
    def _detect_current_theme(self, theme_collection, current_layer_states):
        """現在の状態に最も近いテーマを検出
        
        Args:
            theme_collection: QgsMapThemeCollection
            current_layer_states: 現在のレイヤー状態
            
        Returns:
            str or None: テーマ名、または一致するものがない場合はNone
        """
        try:
            available_themes = theme_collection.mapThemes()
            
            for theme_name in available_themes:
                # テーマの状態と現在の状態を比較
                # 実際の詳細比較は複雑になるため、簡単な一致判定を行う
                
                # ここでは簡単な実装として、テーマ名による判定のみ行う
                # 実際の実装では、レイヤーの表示状態やスタイルを詳細に比較する必要がある
                pass
            
            # 簡略化：現在は常にNoneを返す（完全一致の検出は複雑なため）
            return None
            
        except Exception as e:
            print(f"テーマ検出エラー: {e}")
            return None
    
    def _apply_theme_from_permalink(self, theme_name):
        """パーマリンクからテーマを復元・適用
        
        Args:
            theme_name (str): テーマ名
            
        Returns:
            bool: 適用成功かどうか
        """
        try:
            from qgis.core import QgsProject, QgsLayerTreeModel
            
            if not theme_name or not isinstance(theme_name, str):
                return False
            
            project = QgsProject.instance()
            if not project:
                return False
            
            # 指定されたテーマを適用
            theme_collection = project.mapThemeCollection()
            if theme_name in theme_collection.mapThemes():
                root = project.layerTreeRoot()
                model = QgsLayerTreeModel(root)
                theme_collection.applyTheme(theme_name, root, model)
                return True
            
            return False
            
        except ImportError:
            # QGISが利用できない環境
            return False
        except Exception as e:
            print(f"テーマ適用エラー: {e}")
            return False
    

    def update_theme_list(self):
        """パネルのテーマ一覧を更新"""
        if not self.panel or not hasattr(self.panel, 'comboBox_themes'):
            return
            
        try:
            from qgis.core import QgsProject, QgsMapThemeCollection
            
            project = QgsProject.instance()
            if not project:
                return
                
            theme_collection = project.mapThemeCollection()
            if not theme_collection:
                return
            
            # コンボボックスをクリア
            self.panel.comboBox_themes.clear()
            
            # システムオプションを追加
            self.panel.comboBox_themes.addItem("-- No Theme (Position Only) --")
            
            # 利用可能なテーマを追加
            available_themes = theme_collection.mapThemes()
            for theme_name in sorted(available_themes):
                self.panel.comboBox_themes.addItem(theme_name)
                
            print(f"テーマ一覧を更新: {len(available_themes)} テーマが見つかりました")
            
        except ImportError:
            # QGISが利用できない環境
            print("QGIS環境が利用できません")
        except Exception as e:
            print(f"テーマ一覧更新エラー: {e}")


