"""Minimal QMapWebMapGenerator

This module provides a small QMapWebMapGenerator class that returns a
fullscreen OpenLayers HTML page pointing to the plugin's WMS. The HTML
is constructed with simple string concatenation to avoid brace-escaping
issues when embedding JavaScript.
"""

from typing import Any, Dict


class QMapWebMapGenerator:
    """Simplified web map generator that assumes EPSG:3857 on the client.

    This generator produces a minimal OpenLayers page which ALWAYS uses
    EPSG:3857 for view/projection and expects incoming center coordinates
    (and bookmarks) to already be in EPSG:3857.
    """

    def __init__(self, owner: Any = None):
        self.owner = owner

    def generate_wms_based_html_page(self, navigation_data: Dict[str, Any], image_width: int = 800, image_height: int = 600, server_port: int = 8089) -> str:
        """Return a minimal OpenLayers HTML page assuming EPSG:3857.

        navigation_data must contain: x, y (in EPSG:3857), optional scale, rotation, bookmarks
        """
        if not navigation_data:
            navigation_data = {}

        x = navigation_data.get('x', 0)
        y = navigation_data.get('y', 0)
        rotation_deg = navigation_data.get('rotation', 0)
        scale_value = navigation_data.get('scale', None)
        bookmarks = navigation_data.get('bookmarks', [])

        try:
            import json
            bookmarks_json = json.dumps(bookmarks)
        except Exception:
            bookmarks_json = '[]'

        # Prepare a translatable prompt label for the bookmarks select
        prompt_text = 'Select bookmark'
        try:
            # prefer plugin owner's translation if available
            if hasattr(self, 'owner') and self.owner is not None and hasattr(self.owner, 'tr'):
                try:
                    prompt_text = self.owner.tr(prompt_text)
                except Exception:
                    pass
            else:
                # fallback to QCoreApplication translation if Qt is available
                try:
                    import importlib
                    qt_core = None
                    try:
                        qt_core = importlib.import_module('PyQt5.QtCore')
                    except Exception:
                        try:
                            qt_core = importlib.import_module('PySide2.QtCore')
                        except Exception:
                            qt_core = None
                    if qt_core is not None and hasattr(qt_core, 'QCoreApplication'):
                        QCoreApplication = getattr(qt_core, 'QCoreApplication')
                        try:
                            prompt_text = QCoreApplication.translate('QMapPermalink', prompt_text)
                        except Exception:
                            pass
                except Exception:
                    pass
        except Exception:
            # best effort only
            prompt_text = prompt_text

        try:
            import math
            if scale_value is None:
                zoom = 12
            else:
                sf = float(scale_value)
                baseScale = 591657527.591555
                zoom = max(1, int(math.log2(baseScale / sf))) if sf > 0 else 12
        except Exception:
            zoom = 12

        port = int(server_port)

        # If an owner (plugin) was passed in, try to reuse its URL-building
        # utilities so the OpenLayers page and the panel produce identical
        # Google Maps / Google Earth links when possible.
        google_maps_url = None
        google_earth_url = None
        try:
            if hasattr(self, 'owner') and self.owner is not None:
                nav = {
                    'type': 'coordinates',
                    'x': x,
                    'y': y,
                    'scale': scale_value,
                    'crs': 'EPSG:3857',
                    'zoom': zoom,
                }
                if hasattr(self.owner, '_build_google_maps_url'):
                    try:
                        google_maps_url = self.owner._build_google_maps_url(nav)
                    except Exception:
                        google_maps_url = None
                if hasattr(self.owner, '_build_google_earth_url'):
                    try:
                        google_earth_url = self.owner._build_google_earth_url(nav)
                    except Exception:
                        google_earth_url = None
        except Exception:
            # best-effort only; fall back to client-side calculation
            google_maps_url = None
            google_earth_url = None

        head = (
            '  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/ol@8.2.0/ol.css" type="text/css">'
            '  <script src="https://cdn.jsdelivr.net/npm/ol@8.2.0/dist/ol.js"></script>'
        )

        # Simple JS that assumes x/y are in EPSG:3857 and sets view projection accordingly
        js = (
            "const inputX = %s;\n" % (json_safe(x)) +
            "const inputY = %s;\n" % (json_safe(y)) +
            "const inputCRS = 'EPSG:3857';\n" +
            "const mapScale = %s;\n" % (json_safe(scale_value)) +
            "const rotationRad = (%s || 0) * Math.PI / 180;\n" % (rotation_deg) +
            "const bookmarks = %s;\n" % (bookmarks_json) +
            "const serverGoogleMapsUrl = %s;\n" % (json_safe(google_maps_url)) +
            "const serverGoogleEarthUrl = %s;\n" % (json_safe(google_earth_url)) +
            "const wmsBase = (typeof window !== 'undefined' && window.location && window.location.origin) ? window.location.origin : ('http://' + window.location.hostname + (window.location.port ? ':' + window.location.port : ''));\n" +
            "const wmsSource = new ol.source.ImageWMS({ url: wmsBase + '/wms', params: { 'x': inputX, 'y': inputY, 'scale': mapScale, 'crs': inputCRS }, serverType: 'qgis', crossOrigin: 'anonymous' });\n" +
            "const mapLayer = new ol.layer.Image({ source: wmsSource });\n" +
            "const map = new ol.Map({ target: 'map', layers: [ mapLayer ], view: new ol.View({ center: [inputX, inputY], projection: 'EPSG:3857', zoom: %d, rotation: rotationRad }) });\n" % (zoom) +
            "window.map = map; window.wmsSource = wmsSource;\n"
        )

    # Append bookmark population and interaction script
        js += (
            "(function(){\n"
            "  try{\n"
            "    var sel = document.getElementById('qmp-bookmarks');\n"
            "    if(sel && Array.isArray(bookmarks) && bookmarks.length){\n"
            "      // add home option already present, now add bookmarks\n"
            "      bookmarks.forEach(function(b,i){ try{ var opt = document.createElement('option'); opt.value = i; opt.text = b.name || ('Bookmark ' + (i+1)); sel.appendChild(opt);}catch(e){} });\n"
            "      sel.addEventListener('change', function(){ try{ if(this.value === '__home' || this.value === '__prompt'){ map.getView().animate({ center: [inputX, inputY], duration: 600 }); } else { var idx = parseInt(this.value); var b = bookmarks[idx]; if(b){ var bx = parseFloat(b.x || b.orig_x || b.lon || b.lng || 0); var by = parseFloat(b.y || b.orig_y || b.lat || 0); if(isFinite(bx) && isFinite(by)){ map.getView().animate({ center: [bx, by], duration: 600 }); try{ if(window.wmsSource && typeof window.wmsSource.updateParams === 'function'){ var up = { 'x': bx, 'y': by, 'crs': 'EPSG:3857' }; try{ if(typeof mapScale !== 'undefined' && mapScale !== null) up.scale = mapScale; }catch(e){} window.wmsSource.updateParams(up); window.wmsSource.refresh(); } }catch(e){} } } } }catch(e){} try{ this.value = '__prompt'; }catch(e){} });\n"
            "    }\n"
            "  }catch(e){}\n"
            "})();\n"
        )

        # Add button handlers that open external viewers (Google Maps / Google Earth)
        js += (
            "(function(){\n"
            "  try{\n"
            "    var btnMap = document.getElementById('qmp-open-googlemaps');\n"
            "    if(btnMap){\n"
            "      btnMap.addEventListener('click', function(){\n"
            "        try{\n"
            "          var view = map.getView(); var c = view.getCenter(); var x=c[0], y=c[1];\n"
            "          var lon = (x / 20037508.34) * 180.0; var lat = (y / 20037508.34) * 180.0;\n"
            "          lat = 180.0 / Math.PI * (2.0 * Math.atan(Math.exp(lat * Math.PI / 180.0)) - Math.PI / 2.0);\n"
            "          var zoom = view.getZoom ? Math.round(view.getZoom()) : 12;\n"
            "          // prefer server-computed URL if available\n"
            "          if(typeof serverGoogleMapsUrl !== 'undefined' && serverGoogleMapsUrl){ var w = window.open(serverGoogleMapsUrl, '_blank'); if(!w){ alert('ポップアップブロック'); } return; }\n"
            "          var url = 'https://www.google.com/maps/@' + encodeURIComponent(lat) + ',' + encodeURIComponent(lon) + ',' + encodeURIComponent(zoom) + 'z';\n"
            "          var w = window.open(url, '_blank'); if(!w){ alert('ポップアップブロック'); }\n"
            "        }catch(e){ console.error(e); }\n"
            "      });\n"
            "    }\n"
            "  }catch(e){}\n"
            "})();\n"
            "(function(){\n"
            "  try{\n"
            "    var btnEarth = document.getElementById('qmp-open-googleearth');\n"
            "    if(btnEarth){\n"
            "      btnEarth.addEventListener('click', function(){\n"
            "        try{\n"
            "          var view = map.getView(); var c = view.getCenter(); var x=c[0], y=c[1];\n"
            "          var lon = (x / 20037508.34) * 180.0; var lat = (y / 20037508.34) * 180.0;\n"
            "          lat = 180.0 / Math.PI * (2.0 * Math.atan(Math.exp(lat * Math.PI / 180.0)) - Math.PI / 2.0);\n"
            "          // prefer server-computed URL if available\n"
            "          if(typeof serverGoogleEarthUrl !== 'undefined' && serverGoogleEarthUrl){ var w = window.open(serverGoogleEarthUrl, '_blank'); if(!w){ alert('ポップアップブロック'); } return; }\n"
            "          // Use Google Earth Web search link (best-effort)\n"
            "          var url = 'https://earth.google.com/web/search/' + encodeURIComponent(lat) + ',' + encodeURIComponent(lon);\n"
            "          var w = window.open(url, '_blank'); if(!w){ alert('ポップアップブロック'); }\n"
            "        }catch(e){ console.error(e); }\n"
            "      });\n"
            "    }\n"
            "  }catch(e){}\n"
            "})();\n"
        )

        html = (
            '<!doctype html>\n'
            '<html lang="ja">\n'
            '<head>' + head + '\n'
            '  <style>html,body{height:100%;margin:0;padding:0}#map{width:100vw;height:100vh}.qmp-bookmarks{position:absolute;left:72px;top:10px;z-index:999;background:#fff;border-radius:4px;padding:4px 6px;box-shadow:0 2px 6px rgba(0,0,0,0.15);font-size:13px}</style>\n'
            '</head>\n'
            '<body>\n'
            '  <div id="map"></div>\n'
                '  <button id="qmp-open-googlemaps" class="qmp-bookmarks" title="Open Google Maps" style="left:auto;right:10px">Open Google Maps</button>\n'
                '  <button id="qmp-open-googleearth" class="qmp-bookmarks" title="Open Google Earth" style="left:auto;right:10px;top:44px">Open Google Earth</button>\n'
            '  <select id="qmp-bookmarks" class="qmp-bookmarks" title="Bookmarks">' +
            (('<option value="__prompt" selected>%s</option>' % ( __import__('html').escape(prompt_text) )) if True else '') +
            '<option value="__home">Home</option></select>\n'
            '  <script>\n'
            f'    const serverPort = {server_port};\n'
            f'    ' + js + '\n'
            '  </script>\n'
            '</body>\n'
            '</html>\n'
        )

        return html

    # lightweight stubs
    def get_qgis_layers_info(self):
        return {'layer_count': 0, 'visible_layers': []}

    def get_current_extent_info(self):
        return {}

    def _resolve_coordinates(self, navigation_data):
        try:
            if not navigation_data:
                return None, None
            lat = navigation_data.get('lat')
            lon = navigation_data.get('lon')
            if lat is not None and lon is not None:
                return float(lat), float(lon)
            x = navigation_data.get('x')
            y = navigation_data.get('y')
            if x is None or y is None:
                return None, None
            return float(y), float(x)
        except Exception:
            return None, None


# small helpers used only at generation time; keep them local to avoid runtime deps
def escape_js_string(s: str) -> str:
    if s is None:
        return ''
    return str(s).replace('\\', '\\\\').replace('"', '\\"').replace("\n", ' ')


def json_safe(v):
    # simple serializer for numbers/strings used in small f-strings above
    try:
        import json
        return json.dumps(v)
    except Exception:
        return 'null'