"""Minimal QMapWebMapGenerator

This module provides a small QMapWebMapGenerator class that returns a
fullscreen OpenLayers HTML page pointing to the plugin's WMS. The HTML
is constructed with simple string concatenation to avoid brace-escaping
issues when embedding JavaScript.
"""

from typing import Any, Dict


class QMapWebMapGenerator:
    """Minimal web map generator used by the plugin.

    The generator avoids importing QGIS modules at module import time.
    It exposes generate_wms_based_html_page(navigation_data, ...) which
    returns an HTML string.
    """

    def __init__(self, owner: Any = None):
        self.owner = owner
        # optional plugin-provided zoom estimator
        self._plugin_estimate_zoom = None
        try:
            if owner is not None and hasattr(owner, '_estimate_zoom_from_scale') and callable(getattr(owner, '_estimate_zoom_from_scale')):
                self._plugin_estimate_zoom = owner._estimate_zoom_from_scale
        except Exception:
            self._plugin_estimate_zoom = None

    def generate_wms_based_html_page(self, navigation_data: Dict[str, Any], image_width: int = 800, image_height: int = 600, server_port: int = 8089) -> str:
        """Return a minimal OpenLayers HTML page embedding bookmarks.

        navigation_data may contain keys: x, y, crs, rotation, scale, bookmarks
        """
        if not navigation_data:
            navigation_data = {}

        x = navigation_data.get('x', 0)
        y = navigation_data.get('y', 0)
        crs = navigation_data.get('crs', 'EPSG:3857')
        rotation_deg = navigation_data.get('rotation', 0)
        scale_value = navigation_data.get('scale')
        bookmarks = navigation_data.get('bookmarks', [])

        # serialize bookmarks safely
        try:
            import json

            bookmarks_json = json.dumps(bookmarks)
        except Exception:
            bookmarks_json = '[]'

        # estimate zoom (prefer plugin estimator if available)
        try:
            import math
            zoom = None
            if self._plugin_estimate_zoom is not None:
                try:
                    zoom = float(self._plugin_estimate_zoom(scale_value)) if scale_value is not None else float(self._plugin_estimate_zoom(None))
                except Exception:
                    zoom = None

            if zoom is None:
                if scale_value is None:
                    zoom = 12
                else:
                    sf = float(scale_value)
                    if sf > 0:
                        baseScale = 591657527.591555
                        zoom = max(1, int(math.log2(baseScale / sf)))
                    else:
                        zoom = 12
        except Exception:
            zoom = 12

        port = int(server_port)

        # head scripts (load proj4 if provided by server_manager or owner)
        # navigation_data may include an optional 'proj4' entry containing a
        # proj4js definition string for the requested CRS. If present, load
        # proj4.js before OpenLayers and register the definition so ol.proj
        # can transform custom CRS.
        proj4_def = None
        try:
            proj4_def = navigation_data.get('proj4') if isinstance(navigation_data, dict) else None
        except Exception:
            proj4_def = None

        if proj4_def:
            # embed proj4 definition and load proj4.js before OL
            # keep content minimal and escaped on server side
            head_scripts = (
                '  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/ol@8.2.0/ol.css" type="text/css">\n'
                '  <script src="https://cdnjs.cloudflare.com/ajax/libs/proj4js/2.8.0/proj4.js"></script>\n'
                '  <script src="https://cdn.jsdelivr.net/npm/ol@8.2.0/dist/ol.js"></script>\n'
                f'  <script>try{{ proj4.defs("{escape_js_string(crs)}", "{escape_js_string(proj4_def)}"); if (ol && ol.proj && ol.proj.proj4) {{ ol.proj.proj4.register(proj4); }} }}catch(e){{console.warn("proj4 registration failed", e);}}</script>'
            )
        else:
            head_scripts = (
                '  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/ol@8.2.0/ol.css" type="text/css">\n'
                '  <script src="https://cdn.jsdelivr.net/npm/ol@8.2.0/dist/ol.js"></script>'
            )

        # small JS helper that initializes map and exposes window.map
        js_init = (
            "    // register projection (if available) and initialize map\n"
            "    function tryRegisterAndInit(retries, delayMs, initCallback) {\n"
            "      retries = retries || 5; delayMs = delayMs || 200;\n"
            "      try { if (typeof proj4 !== 'undefined' && ol && ol.proj && ol.proj.proj4) { try { ol.proj.proj4.register(proj4); } catch(e){} } } catch(e){}\n"
            "      var viewCenter = null;\n"
            "      try {\n"
            "        if (inputCRS === 'EPSG:4326') { viewCenter = ol.proj.fromLonLat([inputX, inputY]); }\n"
            "        else if (inputCRS === 'EPSG:3857') { viewCenter = [inputX, inputY]; }\n"
            "        else { try { viewCenter = ol.proj.transform([inputX, inputY], inputCRS, 'EPSG:3857'); } catch(e){ viewCenter = null; } }\n"
            "      } catch(e){ viewCenter = null; }\n"
            "      if (viewCenter !== null || retries <= 0) {\n"
            f"        const rotationRad = ({rotation_deg} || 0) * Math.PI / 180;\n"
            "        const wmsBase = (typeof window !== 'undefined' && window.location && window.location.origin) ? window.location.origin : ('http://' + window.location.hostname + (window.location.port ? ':' + window.location.port : ''));\n"
            "        const map = new ol.Map({ target: 'map', layers: [ new ol.layer.Image({ source: new ol.source.ImageWMS({ url: wmsBase + '/wms', params: {LAYERS: 'project', FORMAT: 'image/png', TRANSPARENT: true, VERSION: '1.3.0'} }), opacity: 1.0 }) ], view: new ol.View({center: viewCenter || ol.proj.fromLonLat([inputX, inputY]), projection: 'EPSG:3857', zoom: " + str(zoom) + ", rotation: rotationRad}) });\n"
            "        try{ window.map = map; }catch(e){}\n"
            "        if (initCallback && typeof initCallback === 'function') { try { initCallback(map); } catch(e){} }\n"
            "        return;\n"
            "      }\n"
            "      setTimeout(function(){ tryRegisterAndInit(retries-1, delayMs, initCallback); }, delayMs);\n"
            "    }\n"
            "    tryRegisterAndInit(8,150);\n"
        )

        # Bookmark binding code: populates select and ensures re-select works
        js_bind = (
            "    (function bindBookmarks(){\n"
            "      var attempts=0;\n"
            "      function tryBind(){\n"
            "        try{\n"
            "          var sel = document.getElementById('qmp-bookmarks');\n"
            "          if(!sel) return;\n"
            "          // clear selection on pointer/mouse/touch down so choosing the same\n"
            "          // item again will still fire a change event.\n"
            "          try{\n"
            "            sel.addEventListener('pointerdown', function(){ try{ this.selectedIndex = -1; }catch(e){} });\n"
            "            sel.addEventListener('mousedown', function(){ try{ this.selectedIndex = -1; }catch(e){} });\n"
            "            sel.addEventListener('touchstart', function(){ try{ this.selectedIndex = -1; }catch(e){} });\n"
            "          }catch(e){}\n"
            "          if(window.ol && window.ol.Map && typeof map !== 'undefined' && sel){\n"
            "            try{\n"
            "              if(Array.isArray(bookmarks) && bookmarks.length){\n"
            "                bookmarks.forEach(function(b,i){ try{ var opt = document.createElement('option'); opt.value = i; opt.text = b.name || ('Bookmark ' + (i+1)); sel.appendChild(opt); }catch(e){} });\n"
            "              }\n"
            "            }catch(e){ console.warn('populate bookmarks failed', e); }\n"
            "            sel.addEventListener('change', function(){\n"
            "              try{\n"
            "                if(this.value === '__home'){\n"
            "                  var center = (inputCRS==='EPSG:4326' ? ol.proj.fromLonLat([parseFloat(inputX), parseFloat(inputY)]) : (inputCRS==='EPSG:3857' ? [parseFloat(inputX), parseFloat(inputY)] : ol.proj.transform([parseFloat(inputX), parseFloat(inputY)], inputCRS, 'EPSG:3857')) );\n"
            "                  map.getView().animate({ center: center, duration: 600 });\n"
            "                } else {\n"
            "                  var idx = parseInt(this.value); var b = bookmarks[idx]; if(b){ var lon = parseFloat(b.x || b.lon || b.lng || b.longitude); var lat = parseFloat(b.y || b.lat); if(isFinite(lon) && isFinite(lat)){ var coord = (!b.crs || b.crs === 'EPSG:4326') ? ol.proj.fromLonLat([lon, lat]) : (b.crs === 'EPSG:3857' ? [lon, lat] : ol.proj.transform([lon, lat], b.crs, 'EPSG:3857')); map.getView().animate({ center: coord, duration: 600 }); } }\n"
            "                }\n"
            "              }catch(e){ console.warn('bookmark select failed', e); }\n"
            "            });\n"
            "            return;\n"
            "          }\n"
            "        }catch(e){}\n"
            "        attempts++; if(attempts<20) setTimeout(tryBind,150);\n"
            "      }\n"
            "      tryBind();\n"
            "    })();\n"
        )

        html = (
            '<!doctype html>\n'
            '<html lang="ja">\n'
            '<head>' + head_scripts + '\n'
            '  <style>html,body{height:100%;margin:0;padding:0}#map{width:100vw;height:100vh}.qmp-bookmarks{position:absolute;left:72px;top:10px;z-index:999;background:#fff;border-radius:4px;padding:4px 6px;box-shadow:0 2px 6px rgba(0,0,0,0.15);font-size:13px}</style>\n'
            '</head>\n'
            '<body>\n'
            '  <div id="map"></div>\n'
            '  <select id="qmp-bookmarks" class="qmp-bookmarks" title="Bookmarks"><option value="__home">Home</option></select>\n'
            '  <script>\n'
            f'    const serverPort = {port};\n'
            f'    const inputX = {json_safe(x)};\n'
            f'    const inputY = {json_safe(y)};\n'
            f'    const inputCRS = "{escape_js_string(crs)}";\n'
            f'    const bookmarks = {bookmarks_json};\n'
            + js_init + '\n' + js_bind + '\n'
            '  </script>\n'
            '</body>\n'
            '</html>\n'
        )

        return html

    # lightweight stubs
    def get_qgis_layers_info(self):
        return {'layer_count': 0, 'visible_layers': []}

    def get_current_extent_info(self):
        return {}

    def _resolve_coordinates(self, navigation_data):
        try:
            if not navigation_data:
                return None, None
            lat = navigation_data.get('lat')
            lon = navigation_data.get('lon')
            if lat is not None and lon is not None:
                return float(lat), float(lon)
            x = navigation_data.get('x')
            y = navigation_data.get('y')
            if x is None or y is None:
                return None, None
            return float(y), float(x)
        except Exception:
            return None, None


# small helpers used only at generation time; keep them local to avoid runtime deps
def escape_js_string(s: str) -> str:
    if s is None:
        return ''
    return str(s).replace('\\', '\\\\').replace('"', '\\"').replace("\n", ' ')


def json_safe(v):
    # simple serializer for numbers/strings used in small f-strings above
    try:
        import json
        return json.dumps(v)
    except Exception:
        return 'null'