# -*- coding: utf-8 -*-
import socket
import threading
import urllib.parse
import json
import html
import math
import os
import re
from qgis.core import QgsProject, QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsPointXY, QgsMessageLog, Qgis

class QMapPermalinkServerManager:
    """QMapPermalink用HTTPサーバー管理クラス - WMS専用シンプル版
    
    WMSサーバーの起動・停止・リクエスト処理を担当します。
    不要なエンドポイント（画像、マップ、タイル）は全て削除されています。
    """
    
    def __init__(self, iface, navigation_signals, webmap_generator, main_plugin):
        """サーバーマネージャーを初期化
        
        Args:
            iface: QGISインターフェース
            navigation_signals: ナビゲーション用シグナル
            webmap_generator: WebMapGeneratorインスタンス
            main_plugin: メインプラグインクラスのインスタンス（共通メソッド呼び出し用）
        """
        self.iface = iface
        self.navigation_signals = navigation_signals
        self.main_plugin = main_plugin
        self.webmap_generator = webmap_generator
        
        # HTTPサーバー関連の状態
        self.http_server = None
        self.server_thread = None
        self.server_port = 8089  # デフォルトポート
        self._http_running = False
        self._last_request_text = ""
        
        # プラグインバージョン情報
        self.plugin_version = self._get_plugin_version()
        # 任意のCRSを強制的にEPSG:3857として扱うオプション（デフォルト: False）
        self.force_epsg3857 = False

    def _safe_int(self, value, default):
        """文字列から安全にintに変換する。NaNや不正値は default を返す。"""
        try:
            # floatを経由して 'NaN' をはじく
            v = float(value)
            if v != v:  # NaN check
                return int(default)
            return int(v)
        except Exception:
            return int(default)

    def _get_plugin_version(self):
        """metadata.txtからプラグインバージョンを取得"""
        try:
            plugin_dir = os.path.dirname(__file__)
            metadata_path = os.path.join(plugin_dir, 'metadata.txt')
            
            if os.path.exists(metadata_path):
                with open(metadata_path, 'r', encoding='utf-8') as f:
                    lines = f.readlines()
                    
                    # 各行をチェック（スペースを無視）
                    for line in lines:
                        # スペースを削除して "version=" で始まるかチェック
                        clean_line = line.replace(' ', '').replace('\t', '').lower()
                        if clean_line.startswith('version='):
                            # 元の行から値を抽出
                            if '=' in line:
                                version = line.split('=')[1].strip()
                                return version
                    
        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"❌ バージョン取得エラー: {e}", "QMapPermalink", Qgis.Warning)
        
        # デフォルトバージョン（metadata.txtから読み取れない場合）
        return "UNKNOWN"

    def start_http_server(self):
        """HTTPサーバーを起動"""
        try:
            if self._http_running:
                return

            # 使用可能なポートを探す
            self.server_port = self.find_available_port(8089, 8099)

            server_socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            server_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
            # bind to all interfaces so non-localhost access is possible
            server_socket.bind(('0.0.0.0', self.server_port))
            server_socket.listen(5)
            server_socket.settimeout(1.0)

            self.http_server = server_socket
            self._http_running = True

            self.server_thread = threading.Thread(
                target=self.run_server,
                name="QMapPermalinkHTTP",
                daemon=True,
            )
            self.server_thread.start()

            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"🚀 QMap Permalink v{self.plugin_version} WMS HTTPサーバーが起動しました: http://localhost:{self.server_port}/wms", "QMapPermalink", Qgis.Info)
            self.iface.messageBar().pushMessage(
                "QMap Permalink",
                f"WMS HTTPサーバーが起動しました (ポート: {self.server_port})",
                duration=3
            )

        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"HTTPサーバーの起動に失敗しました: {e}", "QMapPermalink", Qgis.Critical)
            self.iface.messageBar().pushMessage(
                "QMap Permalink エラー",
                f"HTTPサーバーの起動に失敗しました: {str(e)}",
                duration=5
            )
            self._http_running = False
            if self.http_server:
                try:
                    self.http_server.close()
                except Exception:
                    pass
                self.http_server = None
    
    def run_server(self):
        """サーバーを安全に実行"""
        try:
            while self._http_running and self.http_server:
                try:
                    conn, addr = self.http_server.accept()
                except socket.timeout:
                    continue
                except OSError:
                    break

                try:
                    self._handle_client_connection(conn, addr)
                except Exception as e:
                    from qgis.core import QgsMessageLog, Qgis
                    QgsMessageLog.logMessage(f"HTTPリクエスト処理中にエラーが発生しました: {e}", "QMapPermalink", Qgis.Critical)

        finally:
            self._http_running = False
            if self.http_server:
                try:
                    self.http_server.close()
                except Exception:
                    pass
                self.http_server = None
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage("HTTPサーバーが停止しました", "QMapPermalink", Qgis.Info)
    
    def stop_http_server(self):
        """HTTPサーバーを停止"""
        try:
            self._http_running = False

            if self.http_server:
                try:
                    self.http_server.close()
                except Exception:
                    pass
                self.http_server = None

            # スレッドの終了を待つ
            if self.server_thread and self.server_thread.is_alive():
                try:
                    self.server_thread.join(timeout=3.0)
                except Exception:
                    pass
                self.server_thread = None

            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage("QMap Permalink HTTPサーバーが停止しました", "QMapPermalink", Qgis.Info)
            
        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"HTTPサーバーの停止中にエラーが発生しました: {e}", "QMapPermalink", Qgis.Critical)

    def _handle_client_connection(self, conn, addr):
        """HTTPリクエストを解析してWMSリクエストを処理"""
        # 初期タイムアウト設定（リクエスト読み取り用）
        conn.settimeout(10.0)
        
        with conn:
            request_bytes = self._read_http_request(conn)
            if not request_bytes:
                return

            request_text = request_bytes.decode('iso-8859-1', errors='replace')
            self._last_request_text = request_text

            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"📡 QMap Permalink v{self.plugin_version} WMS HTTP request from {addr}", "QMapPermalink", Qgis.Info)
            try:
                request_line = request_text.splitlines()[0]
            except IndexError:
                self._send_http_response(conn, 400, "Bad Request", "Invalid HTTP request line.")
                return

            parts = request_line.split()
            if len(parts) < 3:
                self._send_http_response(conn, 400, "Bad Request", "Malformed HTTP request line.")
                return

            method, target, _ = parts

            if method.upper() != 'GET':
                self._send_http_response(conn, 405, "Method Not Allowed", "Only GET is supported.")
                return

            parsed_url = urllib.parse.urlparse(target)
            params = urllib.parse.parse_qs(parsed_url.query)
            # Extract Host header for use in generated URLs (used for OnlineResource)
            host = None
            for line in request_text.splitlines():
                if line.lower().startswith('host:'):
                    host = line.split(':', 1)[1].strip()
                    break

            # ブラウザで読み込まれるページURL（/qgis-map）を受け取ったときだけ
            # パネルのナビゲート欄に表示するためにemitする。
            try:
                if parsed_url.path == '/qgis-map':
                    server_port = None
                    try:
                        server_port = self.http_server.getsockname()[1] if self.http_server else self.server_port
                    except Exception:
                        server_port = self.server_port

                    if not host:
                        host = f'localhost:{server_port}'

                    # target が absolute URI の場合はそのまま使う
                    if target.startswith('http://') or target.startswith('https://'):
                        full_url = target
                    else:
                        full_url = f'http://{host}{target}'

                    # Emit the full URL for UI if signal available
                    if hasattr(self, 'navigation_signals') and self.navigation_signals:
                        try:
                            if hasattr(self.navigation_signals, 'request_origin_changed'):
                                self.navigation_signals.request_origin_changed.emit(full_url)
                        except Exception:
                            pass

                    # main_plugin にも保持（パネル未作成時のフォールバック）
                    try:
                        if hasattr(self, 'main_plugin'):
                            setattr(self.main_plugin, '_last_request_origin', full_url)
                    except Exception:
                        pass
            except Exception:
                pass

            # WMSエンドポイントの処理（直接PNG画像返却）
            if parsed_url.path == '/wms':
                QgsMessageLog.logMessage("🌐 Processing WMS request (PNG image)", "QMapPermalink", Qgis.Info)
                try:
                    self._handle_wms_endpoint(conn, params, host)
                except Exception as e:
                    QgsMessageLog.logMessage(f"❌ WMS handler error: {e}", "QMapPermalink", Qgis.Critical)
                    import traceback
                    QgsMessageLog.logMessage(f"❌ Error traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
                    self._send_http_response(conn, 500, "Internal Server Error", f"WMS processing failed: {str(e)}")
                return
            
            # OpenLayersパーマリンクエンドポイントの処理（HTMLページ生成、内部で/wmsを参照）
            if parsed_url.path == '/qgis-map':
                QgsMessageLog.logMessage("📍 Processing OpenLayers HTML page generation", "QMapPermalink", Qgis.Info)
                try:
                    self._handle_permalink_html_page(conn, params)
                except Exception as e:
                    QgsMessageLog.logMessage(f"❌ OpenLayers HTML page error: {e}", "QMapPermalink", Qgis.Critical)
                    import traceback
                    QgsMessageLog.logMessage(f"❌ Error traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
                    self._send_http_response(conn, 500, "Internal Server Error", f"OpenLayers HTML page generation failed: {str(e)}")
                return
            # Debug endpoint to inspect collected bookmarks easily
            if parsed_url.path == '/debug-bookmarks':
                try:
                    self._handle_debug_bookmarks(conn)
                except Exception as e:
                    QgsMessageLog.logMessage(f"❌ debug-bookmarks handler error: {e}", "QMapPermalink", Qgis.Critical)
                    import traceback
                    QgsMessageLog.logMessage(f"❌ Error traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
                    self._send_http_response(conn, 500, "Internal Server Error", f"debug-bookmarks failed: {str(e)}")
                return
            
            # 未対応のエンドポイントは404エラー
            QgsMessageLog.logMessage(f"❌ Unknown endpoint: {parsed_url.path}", "QMapPermalink", Qgis.Warning)
            self._send_http_response(conn, 404, "Not Found", "Available endpoints: /wms (PNG image), /qgis-map (OpenLayers HTML)")
            return

    def _handle_wms_endpoint(self, conn, params, host=None):
        """WMSエンドポイントを処理 - パーマリンクパラメータにも対応"""
        from qgis.core import QgsMessageLog, Qgis
        
        QgsMessageLog.logMessage("🌐 Processing WMS endpoint", "QMapPermalink", Qgis.Info)
        
        # パーマリンクパラメータ（x, y, scale, crs）が含まれているかチェック
        has_permalink_params = ('x' in params and 'y' in params and 'scale' in params)
        
        # デバッグ情報
        QgsMessageLog.logMessage(f"🔍 WMS Request parameters: {list(params.keys())}", "QMapPermalink", Qgis.Info)
        QgsMessageLog.logMessage(f"🔍 Has permalink params: {has_permalink_params}", "QMapPermalink", Qgis.Info)
        QgsMessageLog.logMessage(f"🔍 x in params: {'x' in params}, y in params: {'y' in params}, scale in params: {'scale' in params}", "QMapPermalink", Qgis.Info)
        
        if has_permalink_params:
            QgsMessageLog.logMessage("📍 Permalink parameters detected in WMS request", "QMapPermalink", Qgis.Info)
            # パーマリンクパラメータをWMSパラメータに変換してGetMapとして処理
            self._handle_permalink_as_wms_getmap(conn, params)
            return
        
        # 通常のWMSリクエストの処理
        request = params.get('REQUEST', [''])[0].upper()
        service = params.get('SERVICE', [''])[0].upper()
        
        if service != 'WMS':
            self._send_wms_error_response(conn, "InvalidParameterValue", "SERVICE parameter must be WMS")
            return
            
        if request == 'GETCAPABILITIES':
            self._handle_wms_get_capabilities(conn, params, host)
        elif request == 'GETMAP':
            self._handle_wms_get_map(conn, params)
        else:
            self._send_wms_error_response(conn, "InvalidRequest", f"Request {request} is not supported")

    def _handle_wms_get_capabilities(self, conn, params, host=None):
        """WMS GetCapabilitiesリクエストを処理 - 動的な地図範囲に対応"""
        from qgis.core import QgsMessageLog, Qgis
        
        # QGISキャンバスから現在の範囲情報を取得
        extent_info = self._get_canvas_extent_info()
        
        # determine base host for OnlineResource. Prefer Host header if provided
        try:
            if host:
                base_host = host
            else:
                server_port = self.http_server.getsockname()[1] if self.http_server else self.server_port
                base_host = f"localhost:{server_port}"
        except Exception:
            base_host = f"localhost:{self.server_port}"

        xml_content = f"""<?xml version="1.0" encoding="UTF-8"?>
<WMS_Capabilities version="1.3.0" xmlns="http://www.opengis.net/wms" xmlns:xlink="http://www.w3.org/1999/xlink">
  <Service>
    <Name>WMS</Name>
    <Title>QGIS Map Permalink WMS Service</Title>
    <Abstract>Dynamic WMS service for QGIS map view with zoom/pan capabilities</Abstract>
    <OnlineResource xlink:href="http://{base_host}/wms"/>
  </Service>
  <Capability>
    <Request>
      <GetCapabilities>
        <Format>text/xml</Format>
        <DCPType>
          <HTTP>
            <Get><OnlineResource xlink:href="http://{base_host}/wms"/></Get>
          </HTTP>
        </DCPType>
      </GetCapabilities>
      <GetMap>
        <Format>image/png</Format>
        <DCPType>
          <HTTP>
            <Get><OnlineResource xlink:href="http://{base_host}/wms"/></Get>
          </HTTP>
        </DCPType>
      </GetMap>
    </Request>
    <Layer>
      <Name>qgis_map</Name>
      <Title>QGIS Dynamic Map View</Title>
      <Abstract>Real-time QGIS map with dynamic zoom and pan capabilities</Abstract>
      <CRS>EPSG:4326</CRS>
      <CRS>EPSG:3857</CRS>
      <CRS>EPSG:2154</CRS>
      <CRS>EPSG:32633</CRS>
      {extent_info}
      <Style>
        <Name>default</Name>
        <Title>Default Style</Title>
      </Style>
    </Layer>
  </Capability>
</WMS_Capabilities>"""
        
        QgsMessageLog.logMessage("📄 Sending enhanced WMS GetCapabilities response", "QMapPermalink", Qgis.Info)
        self._send_xml_response(conn, xml_content)

    def _build_navigation_data_from_params(self, params):
        """クエリパラメータからナビゲーション用辞書を構築する

        params: urllib.parse.parse_qs の戻り値（辞書: key -> [values]）

        戻り値の形式:
            {'type': 'coordinates', 'x': float, 'y': float, 'scale': float, 'crs': str, ...}
            または
            {'type': 'location', 'location': '<encoded json string>'}
        """
        # location ベースのパラメータがあれば優先
        if 'location' in params:
            # location はエンコード済みJSON文字列を想定
            loc_val = params.get('location', [''])[0]
            if not loc_val:
                raise ValueError('location パラメータが空です')
            return {'type': 'location', 'location': loc_val}

        # 個別座標パラメータをチェック
        if 'x' in params and 'y' in params and ('scale' in params or 'zoom' in params or 'crs' in params):
            try:
                x = float(params.get('x', [None])[0])
                y = float(params.get('y', [None])[0])
            except Exception:
                raise ValueError('x/y パラメータが数値ではありません')

            scale = None
            if 'scale' in params:
                try:
                    scale = float(params.get('scale', [None])[0])
                except Exception:
                    scale = None

            zoom = None
            if 'zoom' in params:
                try:
                    zoom = float(params.get('zoom', [None])[0])
                except Exception:
                    zoom = None

            crs = params.get('crs', [None])[0]
            rotation = None
            if 'rotation' in params:
                try:
                    rotation = float(params.get('rotation', [None])[0])
                except Exception:
                    rotation = None

            theme_info = None
            if 'theme' in params:
                theme_info = params.get('theme', [None])[0]

            return {
                'type': 'coordinates',
                'x': x,
                'y': y,
                'scale': scale,
                'zoom': zoom,
                'crs': crs,
                'rotation': rotation,
                'theme_info': theme_info,
            }

        # 条件に合致しない場合はエラー
        raise ValueError('ナビゲーションパラメータが見つかりません')

    def _get_canvas_extent_info(self):
        """キャンバスから現在の範囲情報を取得してWMS XMLに埋め込む"""
        try:
            canvas = self.iface.mapCanvas()
            if not canvas:
                return "<EX_GeographicBoundingBox><westBoundLongitude>-180</westBoundLongitude><eastBoundLongitude>180</eastBoundLongitude><southBoundLatitude>-90</southBoundLatitude><northBoundLatitude>90</northBoundLatitude></EX_GeographicBoundingBox>"
            
            # 現在の表示範囲を取得
            extent = canvas.extent()
            crs = canvas.mapSettings().destinationCrs()
            
            # WGS84に変換
            from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject
            wgs84_crs = QgsCoordinateReferenceSystem("EPSG:4326")
            
            if crs.authid() != "EPSG:4326":
                transform = QgsCoordinateTransform(crs, wgs84_crs, QgsProject.instance())
                try:
                    extent = transform.transformBoundingBox(extent)
                except Exception:
                    pass  # 変換失敗時はそのまま使用
            
            # XML形式で範囲情報を生成
            extent_xml = f"""<EX_GeographicBoundingBox>
        <westBoundLongitude>{extent.xMinimum():.6f}</westBoundLongitude>
        <eastBoundLongitude>{extent.xMaximum():.6f}</eastBoundLongitude>
        <southBoundLatitude>{extent.yMinimum():.6f}</southBoundLatitude>
        <northBoundLatitude>{extent.yMaximum():.6f}</northBoundLatitude>
      </EX_GeographicBoundingBox>
      <BoundingBox CRS="{crs.authid()}" minx="{extent.xMinimum():.6f}" miny="{extent.yMinimum():.6f}" maxx="{extent.xMaximum():.6f}" maxy="{extent.yMaximum():.6f}"/>"""
            
            return extent_xml
            
        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"⚠️ Error getting canvas extent info: {e}", "QMapPermalink", Qgis.Warning)
            return "<EX_GeographicBoundingBox><westBoundLongitude>-180</westBoundLongitude><eastBoundLongitude>180</eastBoundLongitude><southBoundLatitude>-90</southBoundLatitude><northBoundLatitude>90</northBoundLatitude></EX_GeographicBoundingBox>"

    def _handle_wms_get_map(self, conn, params):
        """WMS GetMapリクエストを処理 - 実際のQGIS地図画像を生成"""
        from qgis.core import QgsMessageLog, Qgis
        QgsMessageLog.logMessage("🖼️ Processing WMS GetMap request", "QMapPermalink", Qgis.Info)
        
        try:
            # WMSパラメータを解析
            width = self._safe_int(params.get('WIDTH', ['256'])[0], 256)
            height = self._safe_int(params.get('HEIGHT', ['256'])[0], 256)
            bbox = params.get('BBOX', [''])[0]
            # リクエストで指定されたCRSを記録
            original_crs = params.get('CRS', ['EPSG:3857'])[0]
            # オプションで任意CRSを強制的にEPSG:3857として扱う
            if getattr(self, 'force_epsg3857', False):
                crs = 'EPSG:3857'
                QgsMessageLog.logMessage("🔒 force_epsg3857 enabled: treating request CRS as EPSG:3857", "QMapPermalink", Qgis.Info)
            else:
                crs = original_crs
            
            QgsMessageLog.logMessage(f"📐 WMS GetMap parameters: {width}x{height}, requested CRS: {original_crs}, using CRS: {crs}, BBOX: {bbox}", "QMapPermalink", Qgis.Info)
            # リクエストで与えられたBBOXがある場合、必要なら元々のCRSからEPSG:3857に変換する
            # BBOXを変換するのは force_epsg3857 が無効な場合のみ
            if not getattr(self, 'force_epsg3857', False):
                try:
                    if bbox and original_crs and original_crs.upper() != 'EPSG:3857':
                        from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject, QgsRectangle
                        src_crs = QgsCoordinateReferenceSystem(original_crs)
                        tgt_crs = QgsCoordinateReferenceSystem('EPSG:3857')
                        if src_crs.isValid():
                            try:
                                coords = [float(x) for x in bbox.split(',')]
                                if len(coords) == 4:
                                    rect = QgsRectangle(coords[0], coords[1], coords[2], coords[3])
                                    transform = QgsCoordinateTransform(src_crs, tgt_crs, QgsProject.instance())
                                    rect = transform.transformBoundingBox(rect)
                                    bbox = f"{rect.xMinimum()},{rect.yMinimum()},{rect.xMaximum()},{rect.yMaximum()}"
                                    QgsMessageLog.logMessage(f"🔄 Transformed incoming BBOX from {original_crs} to EPSG:3857: {bbox}", "QMapPermalink", Qgis.Info)
                            except Exception as e:
                                QgsMessageLog.logMessage(f"⚠️ Failed to transform BBOX to EPSG:3857: {e}", "QMapPermalink", Qgis.Warning)
                        else:
                            QgsMessageLog.logMessage(f"⚠️ Invalid source CRS '{original_crs}' - forcing to EPSG:3857", "QMapPermalink", Qgis.Warning)
                except Exception:
                    # non-fatal: continue with original crs/bbox
                    pass
            # Use the QGIS canvas-based rendering as the authoritative renderer.
            # This ensures we rely on QGIS official rendering (layer symbology, styles
            # and labeling) rather than any independent/custom renderer which may
            # produce different visuals.
            QgsMessageLog.logMessage("🎯 Using QGIS canvas rendering as authoritative method", "QMapPermalink", Qgis.Info)
            png_data = self._generate_qgis_map_png(width, height, bbox, crs)
            if png_data and len(png_data) > 1000:
                QgsMessageLog.logMessage("✅ Canvas image generated successfully", "QMapPermalink", Qgis.Info)
                self._send_binary_response(conn, 200, "OK", png_data, "image/png")
                return
            else:
                QgsMessageLog.logMessage("❌ Canvas rendering failed", "QMapPermalink", Qgis.Warning)
                # フォールバック: エラー画像を生成
                error_image = self._generate_error_image(width, height, "QGIS Map Generation Failed")
                self._send_binary_response(conn, 200, "OK", error_image, "image/png")
            
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ WMS GetMap error: {e}", "QMapPermalink", Qgis.Critical)
            import traceback
            QgsMessageLog.logMessage(f"❌ Error traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
            
            # エラー時は最小限のテスト画像を返す
            test_image = b'\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x01\x00\x00\x00\x01\x00\x08\x02\x00\x00\x00\x90wS\xde\x00\x00\x00\tpHYs\x00\x00\x0b\x13\x00\x00\x0b\x13\x01\x00\x9a\x9c\x18\x00\x00\x00\x16tEXtSoftware\x00www.inkscape.org\x9b\xee<\x1a\x00\x00\x00\x1ftEXtTitle\x00Test Image\x87\x96\xf0\x8e\x00\x00\x00\x12IDATx\x9cc\xf8\x0f\x00\x00\x01\x00\x01\x00\x18\xdd\x8d\xb4\x00\x00\x00\x00IEND\xaeB`\x82'
            self._send_binary_response(conn, 200, "OK", test_image, "image/png")

    def _handle_permalink_as_wms_getmap(self, conn, params):
        """パーマリンクパラメータをWMS GetMapリクエストとして処理"""
        from qgis.core import QgsMessageLog, Qgis
        
        try:
            QgsMessageLog.logMessage("📍 Converting permalink parameters to WMS GetMap", "QMapPermalink", Qgis.Info)
            
            # パーマリンクパラメータを取得
            x = float(params.get('x', ['0'])[0])
            y = float(params.get('y', ['0'])[0])
            scale = float(params.get('scale', ['10000'])[0])
            crs = params.get('crs', ['EPSG:3857'])[0]
            # パーマリンクでも内部処理はEPSG:3857を使用する（元CRSを保持）
            original_permalink_crs = crs
            crs = 'EPSG:3857'
            rotation = float(params.get('rotation', ['0'])[0])
            
            # デフォルトの画像サイズ（パラメータで指定可能）
            width = self._safe_int(params.get('width', ['512'])[0], 512)
            height = self._safe_int(params.get('height', ['512'])[0], 512)
            
            QgsMessageLog.logMessage(f"📍 Permalink params: x={x}, y={y}, scale={scale}, crs={crs}, rotation={rotation}", "QMapPermalink", Qgis.Info)
            
            # スケールから表示範囲（BBOX）を計算
            bbox = self._calculate_bbox_from_permalink(x, y, scale, width, height, crs)
            
            if bbox:
                QgsMessageLog.logMessage(f"📐 Calculated BBOX: {bbox}", "QMapPermalink", Qgis.Info)
                # If permalink requested CRS is not EPSG:3857, transform computed
                # BBOX into EPSG:3857 and render in EPSG:3857.
                try:
                    if original_permalink_crs and original_permalink_crs.upper() != 'EPSG:3857':
                        from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject, QgsRectangle
                        src_crs = QgsCoordinateReferenceSystem(original_permalink_crs)
                        tgt_crs = QgsCoordinateReferenceSystem('EPSG:3857')
                        if src_crs.isValid():
                            coords = [float(x) for x in bbox.split(',')]
                            if len(coords) == 4:
                                rect = QgsRectangle(coords[0], coords[1], coords[2], coords[3])
                                transform = QgsCoordinateTransform(src_crs, tgt_crs, QgsProject.instance())
                                rect = transform.transformBoundingBox(rect)
                                bbox = f"{rect.xMinimum()},{rect.yMinimum()},{rect.xMaximum()},{rect.yMaximum()}"
                                QgsMessageLog.logMessage(f"🔄 Transformed permalink BBOX from {original_permalink_crs} to EPSG:3857: {bbox}", "QMapPermalink", Qgis.Info)
                        else:
                            QgsMessageLog.logMessage(f"⚠️ Invalid permalink CRS '{original_permalink_crs}' - forcing to EPSG:3857", "QMapPermalink", Qgis.Warning)
                except Exception:
                    pass

                # Use canvas-based rendering for permalink requests. Rotation, if
                # needed, should be handled by the canvas adjustment routines.
                png_data = self._generate_qgis_map_png(width, height, bbox, crs, rotation)
                if png_data and len(png_data) > 1000:
                    QgsMessageLog.logMessage(f"✅ Permalink canvas rendering success: {len(png_data)} bytes", "QMapPermalink", Qgis.Info)
                    self._send_binary_response(conn, 200, "OK", png_data, "image/png")
                    return
                else:
                    QgsMessageLog.logMessage("❌ Permalink canvas rendering failed", "QMapPermalink", Qgis.Warning)
                    error_image = self._generate_error_image(width, height, "Permalink Rendering Failed")
                    self._send_binary_response(conn, 200, "OK", error_image, "image/png")
                    return
            else:
                QgsMessageLog.logMessage("❌ Failed to calculate BBOX from permalink parameters", "QMapPermalink", Qgis.Warning)
                error_image = self._generate_error_image(width, height, "Invalid Permalink Parameters")
                self._send_binary_response(conn, 200, "OK", error_image, "image/png")
                
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Permalink processing error: {e}", "QMapPermalink", Qgis.Critical)
            import traceback
            QgsMessageLog.logMessage(f"❌ Error traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
            
            # エラー時はエラー画像を返す
            error_image = self._generate_error_image(512, 512, f"Permalink Error: {str(e)}")
            self._send_binary_response(conn, 200, "OK", error_image, "image/png")

    def _handle_permalink_html_page(self, conn, params):
        """パーマリンクパラメータに基づいてWMSエンドポイントを使用するHTMLページを生成"""
        from qgis.core import QgsMessageLog, Qgis
        
        try:
            QgsMessageLog.logMessage("🌐 Generating permalink HTML page with WMS endpoint", "QMapPermalink", Qgis.Info)
            
            # パーマリンクパラメータを取得
            x = params.get('x', ['15560350'])[0]
            y = params.get('y', ['4274995'])[0]
            scale = params.get('scale', ['21280'])[0]
            crs = params.get('crs', ['EPSG:3857'])[0]
            rotation = params.get('rotation', ['0.0'])[0]
            width = params.get('width', ['800'])[0]
            height = params.get('height', ['600'])[0]
            
            QgsMessageLog.logMessage(f"🔍 Permalink HTML params: x={x}, y={y}, scale={scale}, crs={crs}, rotation={rotation}", "QMapPermalink", Qgis.Info)
            
            # WMSエンドポイントを使用するHTMLページを生成
            html_content = self._generate_wms_based_html_page(x, y, scale, crs, rotation, width, height)
            
            # HTMLレスポンスを送信
            self._send_http_response(conn, 200, "OK", html_content, "text/html; charset=utf-8")
            
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Permalink HTML page generation error: {e}", "QMapPermalink", Qgis.Critical)
            import traceback
            QgsMessageLog.logMessage(f"❌ Error traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
            
            # エラー時はエラーページを返す
            error_html = self._generate_error_html_page(f"Error generating permalink page: {str(e)}")
            self._send_http_response(conn, 500, "Internal Server Error", error_html, "text/html; charset=utf-8")

    def _generate_wms_based_html_page(self, x, y, scale, crs, rotation, width, height):
        """WMSエンドポイントを使用するHTMLページを生成（WebMapGeneratorを使用）"""
        
        try:
            # WebMapGeneratorを使用してWMSベースのHTMLページを生成
            navigation_data = {
                'x': float(x),
                'y': float(y),
                'scale': float(scale),
                'crs': str(crs),
                'rotation': float(rotation)
            }
            
            server_port = self.http_server.getsockname()[1] if self.http_server else 8089
            # Try to collect QGIS spatial bookmarks and inject into navigation_data
            bookmarks_list = []
            try:
                from qgis.core import QgsProject, QgsMessageLog, Qgis, QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsPointXY

                # Try common bookmark manager access patterns
                mgr = None
                try:
                    mgr = QgsProject.instance().bookmarkManager()
                except Exception:
                    try:
                        mgr = QgsProject.instance().bookmarks()
                    except Exception:
                        mgr = None

                if mgr is None:
                    QgsMessageLog.logMessage('🔍 Bookmark manager not found (mgr is None)', 'QMapPermalink', Qgis.Warning)
                else:
                    QgsMessageLog.logMessage(f'🔍 Bookmark manager found: {type(mgr)}', 'QMapPermalink', Qgis.Info)

                raw = None
                if mgr is not None:
                    try:
                        raw = mgr.bookmarks()
                    except Exception:
                        try:
                            raw = mgr.getBookmarks()
                        except Exception:
                            raw = None

                if raw is None:
                    QgsMessageLog.logMessage('🔍 Bookmark raw list is None', 'QMapPermalink', Qgis.Info)
                else:
                    try:
                        length = len(raw)
                    except Exception:
                        length = 'unknown'
                    QgsMessageLog.logMessage(f'🔍 Bookmark raw list type={type(raw)}, len={length}', 'QMapPermalink', Qgis.Info)

                if raw:
                    for b in raw:
                        # Extract name
                        name = ''
                        try:
                            if hasattr(b, 'name'):
                                name = b.name()
                            else:
                                name = str(getattr(b, 'displayName', '') or '')
                        except Exception:
                            try:
                                if isinstance(b, dict) and 'name' in b:
                                    name = str(b.get('name'))
                            except Exception:
                                name = ''

                        # Extract point-like coordinates from common accessors
                        bx = None; by = None
                        try:
                            if hasattr(b, 'point'):
                                p = b.point(); bx = p.x(); by = p.y()
                        except Exception:
                            bx = None; by = None

                        if (bx is None or by is None) and hasattr(b, 'center'):
                            try:
                                p = b.center(); bx = p.x(); by = p.y()
                            except Exception:
                                pass

                        if (bx is None or by is None) and hasattr(b, 'extent'):
                            try:
                                ext = b.extent(); bx = (ext.xMinimum() + ext.xMaximum()) / 2.0; by = (ext.yMinimum() + ext.yMaximum()) / 2.0
                            except Exception:
                                pass

                        if (bx is None or by is None) and isinstance(b, dict):
                            try:
                                if 'x' in b and 'y' in b:
                                    bx = float(b.get('x')); by = float(b.get('y'))
                                elif 'lon' in b and 'lat' in b:
                                    bx = float(b.get('lon')); by = float(b.get('lat'))
                            except Exception:
                                pass

                        if bx is None or by is None:
                            # couldn't obtain coordinates for this bookmark
                            continue

                        # Determine source CRS for bookmark if available
                        src_crs = None
                        try:
                            if hasattr(b, 'crs'):
                                src_crs = b.crs()
                        except Exception:
                            src_crs = None

                        if not src_crs:
                            try:
                                src_crs = self.iface.mapCanvas().mapSettings().destinationCrs()
                            except Exception:
                                src_crs = QgsCoordinateReferenceSystem('EPSG:3857')

                        # Transform to EPSG:4326 for client consumption
                        try:
                            tgt_crs = QgsCoordinateReferenceSystem('EPSG:4326')
                            transform = QgsCoordinateTransform(src_crs, tgt_crs, QgsProject.instance())
                            pt = transform.transform(QgsPointXY(bx, by))
                            lon = float(pt.x()); lat = float(pt.y())
                        except Exception:
                            lon = float(bx); lat = float(by)

                        bookmarks_list.append({'name': str(name), 'x': lon, 'y': lat, 'crs': 'EPSG:4326'})
            except Exception:
                # On any issue, don't block page generation; just omit bookmarks
                try:
                    from qgis.core import QgsMessageLog, Qgis
                    QgsMessageLog.logMessage('❌ Error while extracting bookmarks; skipping bookmarks', 'QMapPermalink', Qgis.Warning)
                except Exception:
                    pass

            if bookmarks_list:
                navigation_data['bookmarks'] = bookmarks_list

            # WebMapGeneratorの新しいメソッドを使用
            # Try to obtain proj4 definition for the requested CRS and pass it
            # to the webmap generator so the generated HTML can load proj4.js
            # and register the projection for OpenLayers.
            try:
                from qgis.core import QgsCoordinateReferenceSystem
                crs_obj = QgsCoordinateReferenceSystem(str(navigation_data.get('crs', 'EPSG:3857')))
                if crs_obj.isValid():
                    try:
                        proj4_def = crs_obj.toProj4()
                    except Exception:
                        proj4_def = ''
                else:
                    proj4_def = ''
            except Exception:
                proj4_def = ''

            if proj4_def:
                navigation_data['proj4'] = proj4_def

            html_content = self.webmap_generator.generate_wms_based_html_page(
                navigation_data,
                int(width),
                int(height),
                server_port
            )
            
            return html_content
            
        except Exception as e:
            # フォールバック: シンプルなWMSベースHTMLページ
            server_port = self.http_server.getsockname()[1] if self.http_server else 8089
            # Use relative WMS URL so clients use the same origin as the HTML page
            base_url = f"/"
            wms_url = f"/wms?x={x}&y={y}&scale={scale}&crs={crs}&rotation={rotation}&width={width}&height={height}"
        
        # Try to obtain proj4 definition from QGIS for the requested CRS
        proj4_registration_script = ''
        try:
            from qgis.core import QgsCoordinateReferenceSystem
            crs_obj = QgsCoordinateReferenceSystem(str(crs))
            if crs_obj.isValid():
                try:
                    proj4_def = crs_obj.toProj4()
                except Exception:
                    proj4_def = ''
                if proj4_def:
                    # Escape quotes/newlines for embedding
                    proj4_def_escaped = proj4_def.replace('"', '\\"').replace('\n', ' ')
                    # We'll build head scripts so proj4.js is loaded before OpenLayers
                    proj4_registration_script = proj4_def_escaped
        except Exception:
            proj4_registration_script = ''

        # Prepare head scripts: if we have a proj4 definition, load proj4.js
        # BEFORE OpenLayers, then register the projection. Otherwise just load OL.
        if proj4_registration_script:
            head_scripts = (
                '    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/ol@v8.2.0/ol.css" type="text/css">\n'
                '    <script src="https://cdnjs.cloudflare.com/ajax/libs/proj4js/2.8.0/proj4.js"></script>\n'
                '    <script src="https://cdn.jsdelivr.net/npm/ol@v8.2.0/dist/ol.js"></script>\n'
                f'    <script>try{{proj4.defs("{crs}", "{proj4_registration_script}"); if (ol && ol.proj && ol.proj.proj4) {{ ol.proj.proj4.register(proj4); }} else {{ console.warn("ol.proj.proj4 not available - projection registration skipped"); }} }}catch(e){{console.warn("proj4 registration failed", e);}}</script>'
            )
        else:
            head_scripts = (
                '    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/ol@v8.2.0/ol.css" type="text/css">\n'
                '    <script src="https://cdn.jsdelivr.net/npm/ol@v8.2.0/dist/ol.js"></script>\n'
            )

        # OpenLayersを使用したHTMLページテンプレート
        html_template = f"""<!DOCTYPE html>
<html lang="ja">
    <head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>QMap Permalink - 地図表示</title>
{head_scripts}
    <style>
        body {{
            margin: 0;
            padding: 20px;
            font-family: Arial, sans-serif;
            background-color: #f5f5f5;
        }}
        .container {{
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            overflow: hidden;
        }}
        .header {{
            background: #2c3e50;
            color: white;
            padding: 20px;
            text-align: center;
        }}
        .map-container {{
            position: relative;
            height: 600px;
            border: 2px solid #ddd;
        }}
        #map {{
            width: 100%;
            height: 100%;
        }}
        .info-panel {{
            padding: 20px;
            background: #ecf0f1;
            border-top: 1px solid #ddd;
        }}
        .info-grid {{
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
        }}
        .info-item {{
            background: white;
            padding: 15px;
            border-radius: 5px;
            border-left: 4px solid #3498db;
        }}
        .info-label {{
            font-weight: bold;
            color: #2c3e50;
            margin-bottom: 5px;
        }}
        .info-value {{
            color: #555;
            font-family: monospace;
        }}
        .controls {{
            padding: 20px;
            background: #f8f9fa;
            border-top: 1px solid #ddd;
            text-align: center;
        }}
        .btn {{
            padding: 10px 20px;
            margin: 5px;
            background: #3498db;
            color: white;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
        }}
        .btn:hover {{
            background: #2980b9;
        }}
        .btn-secondary {{
            background: #95a5a6;
        }}
        .btn-secondary:hover {{
            background: #7f8c8d;
        }}
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🗺️ QMap Permalink</h1>
            <p>QGISプラグインによる統合地図表示 - WMSエンドポイント使用</p>
        </div>
        
        <div class="map-container">
            <div id="map"></div>
        </div>
        
        <div class="info-panel">
            <h3>📍 地図情報</h3>
            <div class="info-grid">
                <div class="info-item">
                    <div class="info-label">中心座標 (X)</div>
                    <div class="info-value">{x}</div>
                </div>
                <div class="info-item">
                    <div class="info-label">中心座標 (Y)</div>
                    <div class="info-value">{y}</div>
                </div>
                <div class="info-item">
                    <div class="info-label">縮尺</div>
                    <div class="info-value">1:{scale}</div>
                </div>
                <div class="info-item">
                    <div class="info-label">座標系</div>
                    <div class="info-value">{crs}</div>
                </div>
                <div class="info-item">
                    <div class="info-label">回転角度</div>
                    <div class="info-value">{rotation}°</div>
                </div>
                <div class="info-item">
                    <div class="info-label">画像サイズ</div>
                    <div class="info-value">{width} × {height}</div>
                </div>
            </div>
        </div>
        
        <div class="controls">
            <h3>🔧 コントロール</h3>
            <a href="{wms_url}" target="_blank" class="btn">📷 WMS画像を表示</a>
            <a href="/wms?SERVICE=WMS&REQUEST=GetCapabilities" target="_blank" class="btn btn-secondary">📋 WMS Capabilities</a>
            <button onclick="refreshMap()" class="btn">🔄 地図を更新</button>
            <button onclick="copyPermalink()" class="btn btn-secondary">🔗 パーマリンクをコピー</button>
        </div>
    </div>

    <script>
        // OpenLayers地図の初期化
        const centerX = parseFloat('{x}');
        const centerY = parseFloat('{y}');
        const mapScale = parseFloat('{scale}');
        const mapRotation = parseFloat('{rotation}') * Math.PI / 180; // ラジアンに変換
        const mapCrs = '{crs}';
        
        // 地図の初期化
        const map = new ol.Map({{
            target: 'map',
            layers: [
                new ol.layer.Image({{
                    source: new ol.source.ImageWMS({{
                        url: '/wms',
                        params: {{
                            'x': centerX,
                            'y': centerY,
                            'scale': mapScale,
                            'crs': mapCrs,
                            'rotation': '{rotation}',
                            'width': 800,
                            'height': 600
                        }},
                        serverType: 'qgis',
                        crossOrigin: 'anonymous'
                    }})
                }})
            ],
            view: new ol.View({{
                center: [centerX, centerY],
                zoom: calculateZoomFromScale(mapScale),
                rotation: mapRotation,
                projection: mapCrs
            }})
        }});
        
        // スケールからズームレベルを計算する関数
        function calculateZoomFromScale(scale) {{
            // 概算でのズームレベル計算
            const baseScale = 591657527.591555;
            return Math.log2(baseScale / scale);
        }}
        
        // 地図を更新する関数
        function refreshMap() {{
            map.getView().setCenter([centerX, centerY]);
            map.getView().setZoom(calculateZoomFromScale(mapScale));
            map.getView().setRotation(mapRotation);
            map.getLayers().getArray()[0].getSource().refresh();
        }}
        
        // パーマリンクをコピーする関数
        function copyPermalink() {{
            const permalink = window.location.href;
            navigator.clipboard.writeText(permalink).then(() => {{
                alert('パーマリンクをクリップボードにコピーしました！');
            }}).catch(() => {{
                // フォールバック
                const textArea = document.createElement('textarea');
                textArea.value = permalink;
                document.body.appendChild(textArea);
                textArea.select();
                document.execCommand('copy');
                document.body.removeChild(textArea);
                alert('パーマリンクをクリップボードにコピーしました！');
            }});
        }}
        
        // ページ読み込み完了時の処理
        document.addEventListener('DOMContentLoaded', function() {{
            console.log('🗺️ QMap Permalink page loaded');
            console.log('📍 Center:', centerX, centerY);
            console.log('📏 Scale:', mapScale);
            console.log('🔄 Rotation:', '{rotation}°');
            console.log('🌐 CRS:', mapCrs);
        }});
    </script>
</body>
</html>"""
        
        return html_template

    def _build_navigation_data_from_params(self, params):
        """互換性用: HTTPパラメータ(dict of lists)から navigation_data を構築して返す

        params: dict where values are lists (urllib.parse.parse_qs形式)

        Returns:
            dict: navigation_data. Examples:
                {'type': 'location', 'location': '<encoded json>'}
                {'type': 'coordinates', 'x': x, 'y': y, 'scale': scale, 'crs': crs, 'zoom': zoom, 'rotation': rotation, 'theme_info': theme}

        Raises:
            ValueError: 必要なパラメータが不足している場合
        """
        from qgis.core import QgsMessageLog, Qgis

        try:
            # location パラメータがあれば location タイプ
            if 'location' in params and params['location']:
                location_val = params['location'][0]
                QgsMessageLog.logMessage("Detected 'location' param for navigation", "QMapPermalink", Qgis.Info)
                return {'type': 'location', 'location': location_val}

            # coordinates ベース: x, y が必須
            if 'x' in params and 'y' in params:
                x_val = params.get('x', [None])[0]
                y_val = params.get('y', [None])[0]
                if x_val is None or y_val is None:
                    raise ValueError('x or y parameter missing')

                # optional params
                scale_val = params.get('scale', [None])[0]
                zoom_val = params.get('zoom', [None])[0]
                crs_val = params.get('crs', [None])[0]
                rotation_val = params.get('rotation', [None])[0]
                theme_val = params.get('theme', [None])[0]

                navigation_data = {
                    'type': 'coordinates',
                    'x': float(x_val) if x_val is not None else None,
                    'y': float(y_val) if y_val is not None else None,
                    'scale': float(scale_val) if scale_val is not None else None,
                    'zoom': float(zoom_val) if zoom_val is not None else None,
                    'crs': str(crs_val) if crs_val is not None else None,
                    'rotation': float(rotation_val) if rotation_val is not None else None,
                    'theme_info': theme_val
                }

                QgsMessageLog.logMessage(f"Built navigation_data from params: {navigation_data}", "QMapPermalink", Qgis.Info)
                return navigation_data

            # フォールバック: 不明なパラメータ
            raise ValueError('No navigation parameters found')

        except Exception as e:
            QgsMessageLog.logMessage(f"Error building navigation_data from params: {e}", "QMapPermalink", Qgis.Warning)
            raise

    def _generate_error_html_page(self, error_message):
        """エラーページのHTMLを生成"""
        error_html = f"""<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>QMap Permalink - エラー</title>
    <style>
        body {{
            margin: 0;
            padding: 40px;
            font-family: Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }}
        .error-container {{
            background: white;
            padding: 40px;
            border-radius: 10px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
            text-align: center;
            max-width: 500px;
        }}
        .error-icon {{
            font-size: 4rem;
            margin-bottom: 20px;
        }}
        .error-title {{
            color: #e74c3c;
            margin-bottom: 20px;
            font-size: 1.5rem;
        }}
        .error-message {{
            color: #555;
            margin-bottom: 30px;
            line-height: 1.6;
        }}
        .btn {{
            padding: 12px 24px;
            background: #3498db;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            display: inline-block;
            margin: 5px;
        }}
        .btn:hover {{
            background: #2980b9;
        }}
    </style>
</head>
<body>
    <div class="error-container">
        <div class="error-icon">❌</div>
        <h1 class="error-title">パーマリンクページの生成に失敗</h1>
        <p class="error-message">{html.escape(error_message)}</p>
        <a href="javascript:history.back()" class="btn">🔙 戻る</a>
        <a href="/" class="btn">🏠 ホーム</a>
    </div>
</body>
</html>"""
        return error_html

    def _calculate_bbox_from_permalink(self, center_x, center_y, scale, width, height, crs):
        """パーマリンクパラメータからBBOXを計算"""
        try:
            from qgis.core import QgsMessageLog, Qgis
            
            # 画面解像度（DPI）とピクセルあたりのサイズを計算
            dpi = 96  # 標準DPI
            meters_per_inch = 0.0254
            pixels_per_meter = dpi / meters_per_inch
            
            # スケールから地図単位での表示範囲を計算
            map_width_m = (width / pixels_per_meter) * scale
            map_height_m = (height / pixels_per_meter) * scale
            
            # 中心点から範囲を計算
            half_width = map_width_m / 2
            half_height = map_height_m / 2
            
            minx = center_x - half_width
            miny = center_y - half_height
            maxx = center_x + half_width
            maxy = center_y + half_height
            
            bbox = f"{minx},{miny},{maxx},{maxy}"
            
            QgsMessageLog.logMessage(f"📊 BBOX calculation: scale={scale}, map_size=({map_width_m:.2f}m x {map_height_m:.2f}m)", "QMapPermalink", Qgis.Info)
            QgsMessageLog.logMessage(f"📊 Result BBOX: {bbox}", "QMapPermalink", Qgis.Info)
            
            return bbox
            
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ BBOX calculation error: {e}", "QMapPermalink", Qgis.Warning)
            return None

    def _handle_wms_get_map_with_bbox(self, conn, bbox, crs, width, height, rotation=0.0):
        """計算されたBBOXでWMS GetMapを処理"""
        from qgis.core import QgsMessageLog, Qgis
        
        try:
            QgsMessageLog.logMessage(f"🎨 Processing WMS GetMap with calculated BBOX: {bbox}, rotation: {rotation}°", "QMapPermalink", Qgis.Info)
            # If requested CRS is not EPSG:3857, transform bbox to EPSG:3857
            try:
                if crs and crs.upper() != 'EPSG:3857' and bbox:
                    from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject, QgsRectangle
                    src_crs = QgsCoordinateReferenceSystem(crs)
                    tgt_crs = QgsCoordinateReferenceSystem('EPSG:3857')
                    if src_crs.isValid():
                        coords = [float(x) for x in bbox.split(',')]
                        if len(coords) == 4:
                            rect = QgsRectangle(coords[0], coords[1], coords[2], coords[3])
                            transform = QgsCoordinateTransform(src_crs, tgt_crs, QgsProject.instance())
                            rect = transform.transformBoundingBox(rect)
                            bbox = f"{rect.xMinimum()},{rect.yMinimum()},{rect.xMaximum()},{rect.yMaximum()}"
                            crs = 'EPSG:3857'
                            QgsMessageLog.logMessage(f"🔄 Transformed BBOX to EPSG:3857 for GetMapWithBBox: {bbox}", "QMapPermalink", Qgis.Info)
                    else:
                        QgsMessageLog.logMessage(f"⚠️ Invalid CRS '{crs}' - forcing to EPSG:3857", "QMapPermalink", Qgis.Warning)
                        crs = 'EPSG:3857'
            except Exception:
                pass

            # Use canvas-based rendering as the authoritative method for
            # permalink BBOX requests. Rotation handling should be applied
            # via canvas extent/rotation adjustment if needed.
            png_data = self._generate_qgis_map_png(width, height, bbox, crs, rotation)
            if png_data and len(png_data) > 1000:
                QgsMessageLog.logMessage("✅ Canvas rendering success for permalink", "QMapPermalink", Qgis.Info)
                self._send_binary_response(conn, 200, "OK", png_data, "image/png")
                return
            
            # 最終フォールバック: エラー画像
            error_image = self._generate_error_image(width, height, "Permalink Rendering Failed")
            self._send_binary_response(conn, 200, "OK", error_image, "image/png")
            
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ WMS GetMap with BBOX error: {e}", "QMapPermalink", Qgis.Critical)
            error_image = self._generate_error_image(width, height, f"Error: {str(e)}")
            self._send_binary_response(conn, 200, "OK", error_image, "image/png")

    def _generate_webmap_png(self, width, height, bbox, crs):
        """WebMapGeneratorを使用してPNG画像を生成"""
        try:
            from qgis.core import QgsMessageLog, Qgis
            
            if not self.webmap_generator:
                QgsMessageLog.logMessage("❌ WebMapGenerator not available", "QMapPermalink", Qgis.Warning)
                return None
            
            # WebMapGeneratorにダミーのナビゲーションデータを渡す
            navigation_data = {
                'type': 'coordinates',
                'lat': 35.6762,  # 東京駅
                'lon': 139.6503,
                'crs': 'EPSG:4326',
                'scale': 10000,
                'zoom': 10
            }
            
            # WMSパラメータからナビゲーションデータを更新
            if bbox and crs:
                try:
                    coords = [float(x) for x in bbox.split(',')]
                    if len(coords) == 4:
                        minx, miny, maxx, maxy = coords
                        center_lon = (minx + maxx) / 2
                        center_lat = (miny + maxy) / 2
                        navigation_data.update({
                            'lon': center_lon,
                            'lat': center_lat,
                            'crs': crs
                        })
                        QgsMessageLog.logMessage(f"📍 Updated navigation data from WMS BBOX", "QMapPermalink", Qgis.Info)
                except Exception as e:
                    QgsMessageLog.logMessage(f"⚠️ Failed to parse BBOX: {e}", "QMapPermalink", Qgis.Warning)
            
            # WebMapGeneratorを使って画像を生成
            try:
                # generate_qgis_image_map メソッドを使用（HTML文字列を取得）
                html_content = self.webmap_generator.generate_qgis_image_map(navigation_data, width, height)
                
                # HTMLコンテンツからbase64画像データを抽出
                import re
                base64_match = re.search(r'data:image/png;base64,([A-Za-z0-9+/=]+)', html_content)
                if base64_match:
                    import base64
                    png_data = base64.b64decode(base64_match.group(1))
                    QgsMessageLog.logMessage(f"✅ Extracted PNG from WebMapGenerator: {len(png_data)} bytes", "QMapPermalink", Qgis.Info)
                    return png_data
                else:
                    QgsMessageLog.logMessage("❌ No base64 image found in WebMapGenerator output", "QMapPermalink", Qgis.Warning)
                    return None
                    
            except Exception as e:
                QgsMessageLog.logMessage(f"❌ WebMapGenerator generation failed: {e}", "QMapPermalink", Qgis.Warning)
                return None
                
        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"❌ Error in _generate_webmap_png: {e}", "QMapPermalink", Qgis.Critical)
            return None

    def _generate_qgis_map_png(self, width, height, bbox, crs, rotation=0.0):
        """Generate PNG using PyQGIS independent renderer only.

        This implementation avoids canvas capture and always uses the
        independent renderer (QgsMapSettings + QgsMapRendererParallelJob).
        """
        from qgis.core import QgsMessageLog, Qgis

        QgsMessageLog.logMessage("🎯 Generating map PNG via independent PyQGIS renderer (no canvas)", "QMapPermalink", Qgis.Info)
        try:
            return self._render_map_image(width, height, bbox, crs, rotation)
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Error in _generate_qgis_map_png (delegated): {e}", "QMapPermalink", Qgis.Critical)
            return None

    def _set_canvas_extent_from_bbox(self, bbox, crs):
        """WMS BBOXパラメータからキャンバスの表示範囲を設定"""
        from qgis.core import QgsMessageLog, Qgis
        
        try:
            # BBOXの解析 (minx,miny,maxx,maxy)
            coords = [float(x) for x in bbox.split(',')]
            if len(coords) != 4:
                QgsMessageLog.logMessage(f"❌ Invalid BBOX format: {bbox}", "QMapPermalink", Qgis.Warning)
                return False
            
            minx, miny, maxx, maxy = coords
            QgsMessageLog.logMessage(f"📍 BBOX coordinates: minx={minx}, miny={miny}, maxx={maxx}, maxy={maxy}", "QMapPermalink", Qgis.Info)
            
            # QGISの座標系変換を設定
            from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject, QgsRectangle
            
            # 入力CRS
            source_crs = QgsCoordinateReferenceSystem(crs)
            if not source_crs.isValid():
                QgsMessageLog.logMessage(f"❌ Invalid CRS: {crs}", "QMapPermalink", Qgis.Warning)
                return False
            
            # キャンバスのCRS取得
            canvas = self.iface.mapCanvas()
            if not canvas:
                QgsMessageLog.logMessage("❌ No map canvas available", "QMapPermalink", Qgis.Warning)
                return False
            
            dest_crs = canvas.mapSettings().destinationCrs()
            QgsMessageLog.logMessage(f"🗺️ Canvas CRS: {dest_crs.authid()}, Input CRS: {source_crs.authid()}", "QMapPermalink", Qgis.Info)
            
            # 座標変換が必要かチェック
            extent = QgsRectangle(minx, miny, maxx, maxy)
            
            if source_crs.authid() != dest_crs.authid():
                # 座標変換実行
                transform = QgsCoordinateTransform(source_crs, dest_crs, QgsProject.instance())
                try:
                    extent = transform.transformBoundingBox(extent)
                    QgsMessageLog.logMessage(f"🔄 Transformed extent: {extent.toString()}", "QMapPermalink", Qgis.Info)
                except Exception as e:
                    QgsMessageLog.logMessage(f"❌ Coordinate transformation failed: {e}", "QMapPermalink", Qgis.Warning)
                    return False
            
            # キャンバスの表示範囲を設定
            canvas.setExtent(extent)
            canvas.refresh()
            
            QgsMessageLog.logMessage(f"✅ Canvas extent set to: {extent.toString()}", "QMapPermalink", Qgis.Info)
            return True
            
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Error setting canvas extent: {e}", "QMapPermalink", Qgis.Critical)
            import traceback
            QgsMessageLog.logMessage(f"❌ Traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
            return False

    def _capture_canvas_image(self, width, height):
        """キャンバスから直接画像をキャプチャ"""
        # Qt GUI operations (like canvas.grab()) must run in the GUI/main thread.
        # The HTTP server runs in a worker thread, so calling canvas.grab() directly
        # can cause an access violation. To avoid this we request a capture on the
        # main thread via a helper QObject that emits a signal when done, and
        # we wait synchronously (with a timeout) for the result.
        try:
            from qgis.core import QgsMessageLog, Qgis
            from qgis.PyQt.QtCore import QObject, pyqtSignal, pyqtSlot, QEventLoop, QTimer, QCoreApplication

            # Define the helper only once and keep it on the instance.
            if not hasattr(self, '_capture_helper') or self._capture_helper is None:
                class _CanvasCaptureHelper(QObject):
                    request_capture = pyqtSignal(int, int)
                    finished = pyqtSignal(bytes)

                    def __init__(self, iface):
                        super().__init__()
                        self.iface = iface
                        # connect request -> internal slot which runs in helper's thread
                        self.request_capture.connect(self._do_capture)

                    @pyqtSlot(int, int)
                    def _do_capture(self, w, h):
                        try:
                            canvas = self.iface.mapCanvas()
                            if not canvas:
                                QgsMessageLog.logMessage("❌ No map canvas available (helper)", "QMapPermalink", Qgis.Warning)
                                self.finished.emit(b'')
                                return

                            # Allow pending paint events to finish so labels/symbols
                            # complete rendering before grabbing.
                            try:
                                from qgis.PyQt.QtWidgets import QApplication
                                QApplication.processEvents()
                            except Exception:
                                pass

                            pixmap = canvas.grab()
                            if pixmap.isNull():
                                QgsMessageLog.logMessage("❌ Failed to grab canvas pixmap (helper)", "QMapPermalink", Qgis.Warning)
                                self.finished.emit(b'')
                                return

                            # scale if requested size differs
                            try:
                                from qgis.PyQt.QtCore import Qt
                                # For WMS we should match the requested size exactly
                                if w and h and (w != pixmap.width() or h != pixmap.height()):
                                    pixmap = pixmap.scaled(w, h, Qt.IgnoreAspectRatio, Qt.SmoothTransformation)
                            except Exception:
                                pass

                            image = pixmap.toImage()
                            if image.isNull():
                                QgsMessageLog.logMessage("❌ Failed to convert pixmap to image (helper)", "QMapPermalink", Qgis.Warning)
                                self.finished.emit(b'')
                                return

                            from qgis.PyQt.QtCore import QByteArray, QBuffer, QIODevice
                            byte_array = QByteArray()
                            buffer = QBuffer(byte_array)
                            buffer.open(QIODevice.WriteOnly)
                            success = image.save(buffer, "PNG")
                            if not success:
                                QgsMessageLog.logMessage("❌ Failed to save image as PNG (helper)", "QMapPermalink", Qgis.Warning)
                                self.finished.emit(b'')
                                return

                            png_data = byte_array.data()
                            QgsMessageLog.logMessage(f"✅ Captured canvas image (helper): {len(png_data)} bytes", "QMapPermalink", Qgis.Info)
                            self.finished.emit(png_data)

                        except Exception as e:
                            QgsMessageLog.logMessage(f"❌ Exception in helper capture: {e}", "QMapPermalink", Qgis.Warning)
                            try:
                                self.finished.emit(b'')
                            except Exception:
                                pass

                # Create helper and move it to the main (GUI) thread so its slot runs there.
                helper = _CanvasCaptureHelper(self.iface)
                try:
                    main_thread = QCoreApplication.instance().thread()
                    helper.moveToThread(main_thread)
                except Exception:
                    # If moveToThread fails for any reason, keep helper in current thread
                    pass
                self._capture_helper = helper

            helper = self._capture_helper

            # Prepare an event loop to wait for the capture to finish (with timeout).
            loop = QEventLoop()
            captured = {'data': b''}

            def _on_finished(data):
                captured['data'] = data or b''
                try:
                    loop.quit()
                except Exception:
                    pass

            helper.finished.connect(_on_finished)

            # Emit request; because helper lives in GUI thread, the connected slot
            # will be invoked there. Emission from worker thread is safe and
            # delivery is queued to the helper's thread.
            helper.request_capture.emit(int(width), int(height))

            # Safety timeout (5 seconds) to avoid hanging the server thread forever.
            timer = QTimer()
            timer.setSingleShot(True)
            timer.timeout.connect(lambda: loop.quit())
            timer.start(5000)

            loop.exec_()
            try:
                helper.finished.disconnect(_on_finished)
            except Exception:
                pass

            png = captured.get('data')
            if png and len(png) > 0:
                return png

            # If capture failed or timed out, fallback to None so caller can try
            # other rendering approaches.
            QgsMessageLog.logMessage("⚠️ Canvas capture timed out or failed", "QMapPermalink", Qgis.Warning)
            return None

        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"❌ Error in _capture_canvas_image: {e}", "QMapPermalink", Qgis.Critical)
            return None

    def _render_map_image(self, width, height, bbox, crs, rotation=0.0):
        """独立レンダラでPNGを生成する（rotation をサポート）

        Args:
            width, height: 出力ピクセルサイズ
            bbox: 'minx,miny,maxx,maxy' 文字列または None
            crs: CRS文字列（例: 'EPSG:3857'）
            rotation: 地図回転角度（度単位）。QgsMapSettings の回転サポートがある場合に使用されます。
        """
        from qgis.core import QgsMessageLog, Qgis

        try:
            QgsMessageLog.logMessage(f"🎨 Starting professional WMS rendering: {width}x{height}, rotation={rotation}", "QMapPermalink", Qgis.Info)

            # WMS独立レンダリング設定を作成
            map_settings = self._create_wms_map_settings(width, height, bbox, crs, rotation=rotation)
            if not map_settings:
                QgsMessageLog.logMessage("❌ Failed to create WMS map settings", "QMapPermalink", Qgis.Warning)
                return None

            # 独立したマップレンダラーでPNG画像を生成
            png_data = self._execute_map_rendering(map_settings)
            if png_data:
                QgsMessageLog.logMessage(f"✅ Professional WMS rendering completed: {len(png_data)} bytes", "QMapPermalink", Qgis.Info)
                return png_data
            else:
                QgsMessageLog.logMessage("❌ Professional WMS rendering failed", "QMapPermalink", Qgis.Warning)
                return None

        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Error in professional WMS rendering: {e}", "QMapPermalink", Qgis.Critical)
            import traceback
            QgsMessageLog.logMessage(f"❌ Traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
            return None

    def _create_wms_map_settings(self, width, height, bbox, crs, rotation=0.0):
        """WMS用の独立したマップ設定を作成 - キャンバスに依存しない

        rotation: 回転角度（度） — map settings が回転をサポートする場合は適用します。
        """
        from qgis.core import QgsMapSettings, QgsRectangle, QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject, QgsMessageLog, Qgis
        
        try:
            # 新しいマップ設定オブジェクトを作成
            map_settings = QgsMapSettings()
            
            # 1. レイヤ設定 - QGISデスクトップの表示状態を踏襲
            canvas = self.iface.mapCanvas()
            if canvas:
                # アクティブなレイヤのみを取得（表示状態を踏襲）
                visible_layers = []
                layer_tree_root = QgsProject.instance().layerTreeRoot()
                
                for layer in canvas.layers():
                    layer_tree_layer = layer_tree_root.findLayer(layer.id())
                    if layer_tree_layer and layer_tree_layer.isVisible():
                        visible_layers.append(layer)
                        QgsMessageLog.logMessage(f"📊 Including visible layer: {layer.name()}", "QMapPermalink", Qgis.Info)
                
                map_settings.setLayers(visible_layers)
                map_settings.setBackgroundColor(canvas.canvasColor())
                QgsMessageLog.logMessage(f"✅ Configured {len(visible_layers)} visible layers", "QMapPermalink", Qgis.Info)
            else:
                # キャンバスが無い場合はプロジェクトの全レイヤを使用
                from qgis.core import QgsProject
                project = QgsProject.instance()
                map_settings.setLayers(project.mapLayers().values())
                QgsMessageLog.logMessage("⚠️ No canvas, using all project layers", "QMapPermalink", Qgis.Warning)
            
            # 2. 出力サイズ設定
            from PyQt5.QtCore import QSize
            map_settings.setOutputSize(QSize(width, height))
            
            # 3. 座標系と範囲設定 - WMSパラメータに基づく
            if bbox and crs:
                success = self._configure_wms_extent_and_crs(map_settings, bbox, crs)
                if not success:
                    QgsMessageLog.logMessage("❌ Failed to configure WMS extent/CRS", "QMapPermalink", Qgis.Warning)
                    return None
            else:
                # デフォルト範囲設定
                if canvas:
                    map_settings.setDestinationCrs(canvas.mapSettings().destinationCrs())
                    map_settings.setExtent(canvas.extent())
                else:
                    # 世界全体をデフォルトに
                    world_crs = QgsCoordinateReferenceSystem("EPSG:4326")
                    world_extent = QgsRectangle(-180, -90, 180, 90)
                    map_settings.setDestinationCrs(world_crs)
                    map_settings.setExtent(world_extent)
                    QgsMessageLog.logMessage("🌍 Using world extent as default", "QMapPermalink", Qgis.Info)
            
            # 4. 品質設定
            map_settings.setFlag(QgsMapSettings.Antialiasing, True)
            map_settings.setFlag(QgsMapSettings.UseAdvancedEffects, True)
            map_settings.setFlag(QgsMapSettings.ForceVectorOutput, False)
            map_settings.setFlag(QgsMapSettings.DrawEditingInfo, False)
            
            # 5. DPI設定
            map_settings.setOutputDpi(96)

            # 6. 回転（度） - QgsMapSettings には setRotation がある場合に適用
            try:
                if rotation and float(rotation) != 0.0 and hasattr(map_settings, 'setRotation'):
                    map_settings.setRotation(float(rotation))
                    QgsMessageLog.logMessage(f"🔄 Applied rotation to map settings: {rotation}°", "QMapPermalink", Qgis.Info)
            except Exception:
                pass
            
            QgsMessageLog.logMessage(f"✅ WMS map settings created successfully", "QMapPermalink", Qgis.Info)
            return map_settings
            
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Error creating WMS map settings: {e}", "QMapPermalink", Qgis.Critical)
            return None

    def _configure_wms_extent_and_crs(self, map_settings, bbox, crs):
        """WMSパラメータに基づいて範囲と座標系を設定"""
        from qgis.core import QgsRectangle, QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject, QgsMessageLog, Qgis
        
        try:
            # BBOXの解析 (minx,miny,maxx,maxy)
            coords = [float(x) for x in bbox.split(',')]
            if len(coords) != 4:
                QgsMessageLog.logMessage(f"❌ Invalid BBOX format: {bbox}", "QMapPermalink", Qgis.Warning)  
                return False
            
            minx, miny, maxx, maxy = coords
            extent = QgsRectangle(minx, miny, maxx, maxy)
            
            # CRS設定
            target_crs = QgsCoordinateReferenceSystem(crs)
            if not target_crs.isValid():
                QgsMessageLog.logMessage(f"❌ Invalid CRS: {crs}", "QMapPermalink", Qgis.Warning)
                return False
            
            map_settings.setDestinationCrs(target_crs)
            map_settings.setExtent(extent)
            
            QgsMessageLog.logMessage(f"📍 WMS extent configured: {extent.toString()} in {crs}", "QMapPermalink", Qgis.Info)
            return True
            
        except ValueError as e:
            QgsMessageLog.logMessage(f"❌ Error parsing BBOX coordinates: {e}", "QMapPermalink", Qgis.Warning)
            return False
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Error configuring WMS extent/CRS: {e}", "QMapPermalink", Qgis.Critical)
            return False

    def _execute_map_rendering(self, map_settings):
        """独立したマップレンダラーでPNG画像を生成"""
        from qgis.core import QgsMapRendererParallelJob, QgsMessageLog, Qgis
        
        try:
            QgsMessageLog.logMessage("🎨 Executing independent map rendering...", "QMapPermalink", Qgis.Info)
            
            # 並列レンダリングジョブを作成
            job = QgsMapRendererParallelJob(map_settings)
            
            # レンダリング実行
            job.start()
            job.waitForFinished()
            
            # レンダリング結果を取得
            image = job.renderedImage()
            if image.isNull():
                QgsMessageLog.logMessage("❌ Rendered image is null", "QMapPermalink", Qgis.Warning)
                return None
            
            # PNG形式でバイト配列に変換
            from PyQt5.QtCore import QByteArray, QBuffer, QIODevice
            byte_array = QByteArray()
            buffer = QBuffer(byte_array)
            buffer.open(QIODevice.WriteOnly)
            
            success = image.save(buffer, "PNG")
            if not success:
                QgsMessageLog.logMessage("❌ Failed to save rendered image as PNG", "QMapPermalink", Qgis.Warning)
                return None
            
            png_data = byte_array.data()
            QgsMessageLog.logMessage(f"✅ Map rendering completed: {len(png_data)} bytes", "QMapPermalink", Qgis.Info)
            return png_data
            
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Error executing map rendering: {e}", "QMapPermalink", Qgis.Critical)
            import traceback
            QgsMessageLog.logMessage(f"❌ Traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
            return None
            
            # マップレンダリング実行
            from qgis.core import QgsMapRendererParallelJob
            from PyQt5.QtGui import QImage
            
            job = QgsMapRendererParallelJob(map_settings)
            job.start()
            job.waitForFinished()
            
            image = job.renderedImage()
            if image.isNull():
                QgsMessageLog.logMessage("❌ Rendered image is null", "QMapPermalink", Qgis.Warning)
                return None
            
            # PNG形式でバイト配列に変換
            from PyQt5.QtCore import QByteArray, QBuffer, QIODevice
            byte_array = QByteArray()
            buffer = QBuffer(byte_array)
            buffer.open(QIODevice.WriteOnly)
            
            success = image.save(buffer, "PNG")
            if not success:
                QgsMessageLog.logMessage("❌ Failed to save image as PNG", "QMapPermalink", Qgis.Warning)
                return None
            
            png_data = byte_array.data()
            QgsMessageLog.logMessage(f"✅ Generated PNG image: {len(png_data)} bytes", "QMapPermalink", Qgis.Info)
            return png_data
            
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Error generating QGIS map PNG: {e}", "QMapPermalink", Qgis.Critical)
            import traceback
            QgsMessageLog.logMessage(f"❌ Traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
            return None

    def _generate_error_image(self, width, height, error_message):
        """エラーメッセージ付きの画像を生成"""
        try:
            from PyQt5.QtGui import QImage, QPainter, QFont, QColor
            from PyQt5.QtCore import Qt, QByteArray, QBuffer, QIODevice
            
            # 画像を作成
            image = QImage(width, height, QImage.Format_ARGB32)
            image.fill(QColor(240, 240, 240))  # 明るいグレー背景
            
            # ペインターで描画
            painter = QPainter(image)
            painter.setRenderHint(QPainter.Antialiasing)
            
            # フォントとペンを設定
            font = QFont("Arial", 12)
            painter.setFont(font)
            painter.setPen(QColor(180, 0, 0))  # 赤色のテキスト
            
            # エラーメッセージを描画
            rect = image.rect()
            painter.drawText(rect, Qt.AlignCenter | Qt.TextWordWrap, f"Error:\n{error_message}")
            
            painter.end()
            
            # PNG形式でバイト配列に変換
            byte_array = QByteArray()
            buffer = QBuffer(byte_array)
            buffer.open(QIODevice.WriteOnly)
            image.save(buffer, "PNG")
            
            return byte_array.data()
            
        except Exception as e:
            # 最小限のPNG画像を返す
            return b'\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x01\x00\x00\x00\x01\x00\x08\x02\x00\x00\x00\x90wS\xde\x00\x00\x00\tpHYs\x00\x00\x0b\x13\x00\x00\x0b\x13\x01\x00\x9a\x9c\x18\x00\x00\x00\x16tEXtSoftware\x00www.inkscape.org\x9b\xee<\x1a\x00\x00\x00\x1ftEXtTitle\x00Error Image\x87\x96\xf0\x8e\x00\x00\x00\x12IDATx\x9cc\xf8\x0f\x00\x00\x01\x00\x01\x00\x18\xdd\x8d\xb4\x00\x00\x00\x00IEND\xaeB`\x82'

    def _send_binary_response(self, conn, status_code, reason, binary_data, content_type):
        """バイナリデータのHTTPレスポンスを送信"""
        header_lines = [
            f"HTTP/1.1 {status_code} {reason}",
            f"Content-Length: {len(binary_data)}",
            f"Content-Type: {content_type}",
            "Access-Control-Allow-Origin: *",
            "Connection: close",
            "",
            "",
        ]

        header_bytes = "\r\n".join(header_lines).encode('utf-8')

        try:
            conn.sendall(header_bytes + binary_data)
        except OSError:
            pass

    def _read_http_request(self, conn):
        """HTTPリクエスト全体を読み取る"""
        data = b""
        
        while True:
            try:
                chunk = conn.recv(1024)
            except socket.timeout:
                break

            if not chunk:
                break

            data += chunk

            if b"\r\n\r\n" in data:
                break

        return data

    def _send_http_response(self, conn, status_code, reason, body, content_type="text/plain; charset=utf-8"):
        """最小限のHTTPレスポンスを送信"""
        if isinstance(body, str):
            body_bytes = body.encode('utf-8')
        else:
            body_bytes = body

        header_lines = [
            f"HTTP/1.1 {status_code} {reason}",
            f"Content-Length: {len(body_bytes)}",  
            f"Content-Type: {content_type}",
            "Connection: close",
            "",
            "",
        ]

        header_bytes = "\r\n".join(header_lines).encode('utf-8')

        try:
            conn.sendall(header_bytes + body_bytes)
        except OSError:
            pass

    def _send_xml_response(self, conn, xml_content):
        """XMLレスポンスを送信 (簡易)"""
        try:
            xml_bytes = xml_content.encode('utf-8')
            header_lines = [
                "HTTP/1.1 200 OK",
                f"Content-Length: {len(xml_bytes)}",
                "Content-Type: text/xml; charset=utf-8",
                "Connection: close",
                "",
                "",
            ]
            header_bytes = "\r\n".join(header_lines).encode('utf-8')
            conn.sendall(header_bytes + xml_bytes)
        except Exception:
            pass

    def _guess_bind_ip(self):
        """サーバの外向きIPv4アドレスを推定して返す（簡易）"""
        try:
            import socket as _socket
            # 外部に到達可能なダミー接続を作って自IPを取得
            s = _socket.socket(_socket.AF_INET, _socket.SOCK_DGRAM)
            try:
                s.connect(('8.8.8.8', 53))
                ip = s.getsockname()[0]
            except Exception:
                ip = '127.0.0.1'
            finally:
                try:
                    s.close()
                except Exception:
                    pass
            return ip
        except Exception:
            return '127.0.0.1'

    def find_available_port(self, start_port, end_port):
        """使用可能なポートを探す
        
        Args:
            start_port: 開始ポート番号
            end_port: 終了ポート番号
            
        Returns:
            使用可能なポート番号
        """
        for port in range(start_port, end_port + 1):
            try:
                # test bind on all interfaces
                with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
                    s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
                    s.bind(('0.0.0.0', port))
                    return port
            except OSError:
                continue
        raise RuntimeError(f"ポート範囲 {start_port}-{end_port} で使用可能なポートが見つかりません")

    def is_server_running(self):
        """サーバーが稼働中かどうかを確認"""
        return self._http_running and self.http_server is not None

    def get_server_port(self):
        """現在のサーバーポートを取得"""
        return self.server_port if self.is_server_running() else None

    def get_last_request(self):
        """最後のHTTPリクエストテキストを取得"""
        return self._last_request_text