# -*- coding: utf-8 -*-
"""
PyQGIS Independent Rendering Module
QGISキャンバスに依存しない完全独立レンダリング機能

このモジュールは、WMSリクエストで指定されたBBOXを使用して、
QGISキャンバスの表示状態に関係なく任意の位置・範囲の地図画像を生成します。
"""

from qgis.core import (
    QgsMapSettings, QgsMapRendererParallelJob, QgsRectangle, 
    QgsCoordinateReferenceSystem, QgsCoordinateTransform, 
    QgsProject, QgsMessageLog, Qgis
)
from PyQt5.QtCore import QSize, QEventLoop, QTimer
from PyQt5.QtGui import QColor


class PyQGISIndependentRenderer:
    """PyQGISによる完全独立レンダリングクラス"""
    
    def __init__(self, iface=None):
        self.iface = iface
        
    def render_map_image(self, width, height, bbox, crs, rotation=0.0):
        """
        完全独立マップレンダリング
        
        Args:
            width (int): 画像幅
            height (int): 画像高さ
            bbox (str): "minx,miny,maxx,maxy" 形式の範囲
            crs (str): 座標系（例: "EPSG:4326"）
            rotation (float): 回転角度（度）、デフォルト0.0
            
        Returns:
            bytes: PNG画像データ（失敗時はNone）
        """
        try:
            QgsMessageLog.logMessage(
                f"🎨 PyQGIS Independent Rendering: {width}x{height}, BBOX: {bbox}, CRS: {crs}, Rotation: {rotation}°", 
                "QMapPermalink", Qgis.Info
            )
            
            # 1. 独立したマップ設定を作成
            map_settings = self._create_map_settings(width, height, crs)
            
            # 2. 現在のプロジェクトから可視レイヤを取得
            visible_layers = self._get_visible_layers()
            map_settings.setLayers(visible_layers)
            
            # 3. BBOXから表示範囲を設定
            extent = self._parse_bbox_to_extent(bbox, crs)
            if extent:
                map_settings.setExtent(extent)
            else:
                # フォールバック: プロジェクト全体の範囲
                map_settings.setExtent(QgsProject.instance().extent())
            
            # 4. 回転を設定（0でない場合）
            if rotation != 0.0:
                QgsMessageLog.logMessage(f"🔄 Setting map rotation to {rotation}°", "QMapPermalink", Qgis.Info)
                map_settings.setRotation(rotation)
            
            # 5. 並列レンダリング実行
            png_data = self._execute_parallel_rendering(map_settings)
            
            if png_data:
                QgsMessageLog.logMessage(
                    f"✅ Independent rendering success: {len(png_data)} bytes", 
                    "QMapPermalink", Qgis.Info
                )
                return png_data
            else:
                QgsMessageLog.logMessage(
                    "❌ Independent rendering failed", 
                    "QMapPermalink", Qgis.Warning
                )
                return None
                
        except Exception as e:
            QgsMessageLog.logMessage(
                f"❌ Independent rendering error: {e}", 
                "QMapPermalink", Qgis.Critical
            )
            import traceback
            QgsMessageLog.logMessage(
                f"❌ Error traceback: {traceback.format_exc()}", 
                "QMapPermalink", Qgis.Critical
            )
            return None
    
    def _create_map_settings(self, width, height, crs):
        """独立したマップ設定を作成"""
        map_settings = QgsMapSettings()
        map_settings.setOutputSize(QSize(width, height))
        map_settings.setOutputDpi(96)
        map_settings.setBackgroundColor(QColor(255, 255, 255, 0))  # 透明背景
        
        # 座標系を設定
        if crs:
            target_crs = QgsCoordinateReferenceSystem(crs)
            if target_crs.isValid():
                map_settings.setDestinationCrs(target_crs)
                QgsMessageLog.logMessage(f"🌐 Set CRS: {crs}", "QMapPermalink", Qgis.Info)
            else:
                QgsMessageLog.logMessage(f"⚠️ Invalid CRS: {crs}, using EPSG:3857", "QMapPermalink", Qgis.Warning)
                map_settings.setDestinationCrs(QgsCoordinateReferenceSystem("EPSG:3857"))
        else:
            map_settings.setDestinationCrs(QgsCoordinateReferenceSystem("EPSG:3857"))
        
        return map_settings
    
    def _get_visible_layers(self):
        """現在のプロジェクトから可視レイヤを取得"""
        project = QgsProject.instance()
        visible_layers = []
        
        try:
            # レイヤツリーから可視レイヤのみを取得（表示順序を保持）
            root = project.layerTreeRoot()
            
            def collect_visible_layers(node):
                """再帰的に可視レイヤを収集"""
                if hasattr(node, 'layer') and node.layer():
                    # レイヤノードの場合
                    if node.isVisible():
                        layer = node.layer()
                        visible_layers.append(layer)
                        QgsMessageLog.logMessage(f"📋 Adding visible layer: {layer.name()}", "QMapPermalink", Qgis.Info)
                elif hasattr(node, 'children'):
                    # グループノードの場合、子ノードを再帰処理
                    if node.isVisible():  # グループが可視の場合のみ
                        for child in node.children():
                            collect_visible_layers(child)
            
            # ルートから再帰的に可視レイヤを収集
            for child in root.children():
                collect_visible_layers(child)
            
            if not visible_layers:
                # フォールバック: すべてのレイヤを使用
                all_layers = list(project.mapLayers().values())
                visible_layers = all_layers
                QgsMessageLog.logMessage(f"📋 No visible layers found, using all layers: {len(visible_layers)}", "QMapPermalink", Qgis.Warning)
            else:
                QgsMessageLog.logMessage(f"✅ Found {len(visible_layers)} visible layers", "QMapPermalink", Qgis.Info)
                
        except Exception as e:
            QgsMessageLog.logMessage(f"⚠️ Error getting visible layers: {e}", "QMapPermalink", Qgis.Warning)
            # 最終フォールバック
            visible_layers = list(project.mapLayers().values())
        
        return visible_layers
    
    def _parse_bbox_to_extent(self, bbox, crs):
        """BBOXを解析してQgsRectangleに変換"""
        if not bbox:
            return None
            
        try:
            coords = [float(x.strip()) for x in bbox.split(',')]
            if len(coords) != 4:
                QgsMessageLog.logMessage(f"❌ Invalid BBOX format: {bbox}", "QMapPermalink", Qgis.Warning)
                return None
            
            minx, miny, maxx, maxy = coords
            extent = QgsRectangle(minx, miny, maxx, maxy)
            
            QgsMessageLog.logMessage(
                f"📍 Parsed BBOX: minx={minx}, miny={miny}, maxx={maxx}, maxy={maxy}", 
                "QMapPermalink", Qgis.Info
            )
            
            # BBOXが有効な範囲かチェック
            if extent.isEmpty() or not extent.isFinite():
                QgsMessageLog.logMessage("❌ Invalid extent from BBOX", "QMapPermalink", Qgis.Warning)
                return None
                
            return extent
            
        except (ValueError, TypeError) as e:
            QgsMessageLog.logMessage(f"❌ BBOX parsing error: {e}", "QMapPermalink", Qgis.Warning)
            return None
    
    def _execute_parallel_rendering(self, map_settings):
        """並列レンダリングを実行してPNG画像データを返す"""
        try:
            # 並列レンダリングジョブを作成
            renderer = QgsMapRendererParallelJob(map_settings)
            
            # レンダリング完了を待機するイベントループ
            loop = QEventLoop()
            renderer.finished.connect(loop.quit)
            
            # タイムアウト設定（20秒）
            timer = QTimer()
            timer.timeout.connect(loop.quit)
            timer.start(20000)  # 20秒タイムアウト
            
            QgsMessageLog.logMessage("⚡ Starting parallel rendering job...", "QMapPermalink", Qgis.Info)
            
            # レンダリング開始
            renderer.start()
            
            # 完了またはタイムアウトまで待機
            loop.exec_()
            
            # タイマーを停止
            timer.stop()
            
            # レンダリング結果をチェック
            if renderer.isActive():
                QgsMessageLog.logMessage("❌ Rendering job timed out", "QMapPermalink", Qgis.Warning)
                renderer.cancel()
                return None
            
            # レンダリング結果を取得
            image = renderer.renderedImage()
            if image.isNull():
                QgsMessageLog.logMessage("❌ Rendered image is null", "QMapPermalink", Qgis.Warning)
                return None
            
            # PNG形式で保存
            return self._save_image_as_png(image)
            
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Parallel rendering error: {e}", "QMapPermalink", Qgis.Critical)
            return None
    
    def _save_image_as_png(self, image):
        """QImageをPNGバイトデータに変換"""
        try:
            from PyQt5.QtCore import QByteArray, QBuffer, QIODevice
            
            byte_array = QByteArray()
            buffer = QBuffer(byte_array)
            buffer.open(QIODevice.WriteOnly)
            
            # PNG形式で保存
            success = image.save(buffer, "PNG", 100)  # 最高品質
            if not success:
                QgsMessageLog.logMessage("❌ Failed to save image as PNG", "QMapPermalink", Qgis.Warning)
                return None
            
            png_data = byte_array.data()
            QgsMessageLog.logMessage(f"💾 Image saved as PNG: {len(png_data)} bytes", "QMapPermalink", Qgis.Info)
            
            return png_data
            
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ PNG save error: {e}", "QMapPermalink", Qgis.Warning)
            return None


# 簡単に使用できるファクトリ関数
def render_independent_map(iface, width, height, bbox, crs, rotation=0.0):
    """
    独立レンダリングのためのファクトリ関数
    
    Args:
        iface: QGISInterface
        width: 画像幅（ピクセル）
        height: 画像高さ（ピクセル）
        bbox: バウンディングボックス（"minx,miny,maxx,maxy"形式）
        crs: 座標参照系（例："EPSG:3857"）
        rotation: 回転角度（度）、デフォルト0.0
    
    使用例:
        png_data = render_independent_map(iface, 512, 512, "139.6,35.6,139.8,35.8", "EPSG:4326", rotation=45.0)
    """
    renderer = PyQGISIndependentRenderer(iface)
    return renderer.render_map_image(width, height, bbox, crs, rotation)