# -*- coding: utf-8 -*-
"""
QMap Permalink Server Manager - 本格的WMS実装
QGISレイヤ状態を踏襲し、任意位置を自由にレンダリングする本格的なWMSサーバー

Version: 1.10.11
Target: QGISデスクトップのレイヤ表示状態は踏襲してもよいが、表示位置等を自由に変えれる本格的なWMS
"""

import socket
import threading
import urllib.parse
import os
import tempfile
import time
from PyQt5.QtCore import QObject, pyqtSignal


class ProfessionalWMSServer(QObject):
    """本格的なWMSサーバー - レイヤ状態踏襲＋自由位置レンダリング"""
    
    server_started = pyqtSignal()
    server_stopped = pyqtSignal()
    server_error = pyqtSignal(str)
    
    def __init__(self, iface, parent=None):
        super().__init__(parent)
        self.iface = iface
        self.server_port = 8089
        self.is_running = False
        self.server_socket = None
        self.server_thread = None
        self.plugin_version = "1.10.11"
        
        # WMSレンダリング設定
        self.professional_mode = True  # 本格的WMSモード
        self.preserve_layer_states = True  # レイヤ状態踏襲
        self.independent_positioning = True  # 独立した位置レンダリング
        # 任意のCRSを強制的にEPSG:3857として扱うオプション（デフォルト: False）
        self.force_epsg3857 = False
        
        from qgis.core import QgsMessageLog, Qgis
        QgsMessageLog.logMessage(f"🚀 Professional WMS Server initialized v{self.plugin_version}", "QMapPermalink", Qgis.Info)

    def _safe_int(self, value, default):
        """文字列から安全にintに変換する。NaNや不正値は default を返す。"""
        try:
            v = float(value)
            if v != v:
                return int(default)
            return int(v)
        except Exception:
            return int(default)
    
    def start_server(self):
        """本格的WMSサーバーを開始"""
        if self.is_running:
            return True
            
        try:
            self.server_socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            self.server_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
            # bind to all interfaces to accept non-localhost access
            self.server_socket.bind(('0.0.0.0', self.server_port))
            self.server_socket.listen(5)
            
            self.is_running = True
            self.server_thread = threading.Thread(target=self._server_loop, daemon=True)
            self.server_thread.start()
            
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"✅ Professional WMS Server started on port {self.server_port}", "QMapPermalink", Qgis.Info)
            self.server_started.emit()
            return True
            
        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"❌ Failed to start Professional WMS Server: {e}", "QMapPermalink", Qgis.Critical)
            self.server_error.emit(str(e))
            return False
    
    def stop_server(self):
        """本格的WMSサーバーを停止"""
        if not self.is_running:
            return
            
        self.is_running = False
        
        if self.server_socket:
            try:
                self.server_socket.close()
            except:
                pass
                
        if self.server_thread:
            self.server_thread.join(timeout=2.0)
            
        from qgis.core import QgsMessageLog, Qgis
        QgsMessageLog.logMessage("🛑 Professional WMS Server stopped", "QMapPermalink", Qgis.Info)
        self.server_stopped.emit()
    
    def _server_loop(self):
        """サーバーメインループ"""
        while self.is_running:
            try:
                conn, addr = self.server_socket.accept()
                self._handle_client_connection(conn, addr)
            except Exception as e:
                if self.is_running:
                    from qgis.core import QgsMessageLog, Qgis
                    QgsMessageLog.logMessage(f"⚠️ Server loop error: {e}", "QMapPermalink", Qgis.Warning)
                break
    
    def _handle_client_connection(self, conn, addr):
        """クライアント接続を処理"""
        conn.settimeout(10.0)
        
        with conn:
            request_bytes = self._read_http_request(conn)
            if not request_bytes:
                return

            request_text = request_bytes.decode('iso-8859-1', errors='replace')
            
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"🌐 Professional WMS request from {addr}", "QMapPermalink", Qgis.Info)

            try:
                request_line = request_text.splitlines()[0]
                method, target, _ = request_line.split()
                
                if method.upper() != 'GET':
                    self._send_http_response(conn, 405, "Method Not Allowed", "Only GET is supported.")
                    return

                parsed_url = urllib.parse.urlparse(target)
                params = urllib.parse.parse_qs(parsed_url.query)

                # WMSエンドポイントの処理
                if parsed_url.path == '/wms':
                    self._handle_professional_wms(conn, params)
                else:
                    self._send_http_response(conn, 404, "Not Found", "Professional WMS endpoint: /wms")
                    
            except Exception as e:
                from qgis.core import QgsMessageLog, Qgis
                QgsMessageLog.logMessage(f"❌ Request processing error: {e}", "QMapPermalink", Qgis.Critical)
                self._send_http_response(conn, 500, "Internal Server Error", str(e))
    
    def _handle_professional_wms(self, conn, params):
        """本格的WMSリクエストを処理"""
        from qgis.core import QgsMessageLog, Qgis
        
        request = params.get('REQUEST', [''])[0].upper()
        service = params.get('SERVICE', [''])[0].upper()
        
        if service != 'WMS':
            self._send_wms_error_response(conn, "InvalidParameterValue", "SERVICE parameter must be WMS")
            return
            
        QgsMessageLog.logMessage(f"🎯 Professional WMS {request} request", "QMapPermalink", Qgis.Info)
        
        if request == 'GETCAPABILITIES':
            self._handle_professional_capabilities(conn, params)
        elif request == 'GETMAP':
            self._handle_professional_getmap(conn, params)
        else:
            self._send_wms_error_response(conn, "InvalidRequest", f"Request {request} is not supported")
    
    def _handle_professional_capabilities(self, conn, params):
        """本格的WMS GetCapabilities - 動的なレイヤ情報を含む"""
        from qgis.core import QgsMessageLog, Qgis
        
        # 現在のQGISプロジェクトから動的レイヤ情報を取得
        layer_info = self._get_professional_layer_info()
        extent_info = self._get_professional_extent_info()
        
        xml_content = f"""<?xml version="1.0" encoding="UTF-8"?>
<WMS_Capabilities version="1.3.0" xmlns="http://www.opengis.net/wms" xmlns:xlink="http://www.w3.org/1999/xlink">
  <Service>
    <Name>WMS</Name>
    <Title>Professional QGIS WMS Service</Title>
    <Abstract>本格的WMS - QGISレイヤ状態踏襲＋任意位置レンダリング対応</Abstract>
    <OnlineResource xlink:href="http://localhost:{self.server_port}/wms"/>
    <ContactInformation>
      <ContactPersonPrimary>
        <ContactPerson>QMap Permalink</ContactPerson>
      </ContactPersonPrimary>
    </ContactInformation>
  </Service>
  <Capability>
    <Request>
      <GetCapabilities>
        <Format>text/xml</Format>
        <DCPType>
          <HTTP>
            <Get><OnlineResource xlink:href="http://localhost:{self.server_port}/wms"/></Get>
          </HTTP>
        </DCPType>
      </GetCapabilities>
      <GetMap>
        <Format>image/png</Format>
        <Format>image/jpeg</Format>
        <DCPType>
          <HTTP>
            <Get><OnlineResource xlink:href="http://localhost:{self.server_port}/wms"/></Get>
          </HTTP>
        </DCPType>
      </GetMap>
    </Request>
    <Layer>
      <Name>qgis_professional_map</Name>
      <Title>Professional QGIS Map Layer</Title>
      <Abstract>本格的WMS - 現在のQGISレイヤ状態を踏襲し、任意の位置・範囲をレンダリング</Abstract>
      <CRS>EPSG:4326</CRS>
      <CRS>EPSG:3857</CRS>
      <CRS>EPSG:2154</CRS>
      <CRS>EPSG:32633</CRS>
      {extent_info}
      {layer_info}
      <Style>
        <Name>default</Name>
        <Title>Default Professional Style</Title>
        <Abstract>QGISプロジェクトのデフォルトスタイル設定を使用</Abstract>
      </Style>
    </Layer>
  </Capability>
</WMS_Capabilities>"""
        
        QgsMessageLog.logMessage("📄 Sending Professional WMS Capabilities", "QMapPermalink", Qgis.Info)
        self._send_xml_response(conn, xml_content)
    
    def _handle_professional_getmap(self, conn, params):
        """本格的WMS GetMap - レイヤ状態踏襲＋任意位置レンダリング"""
        from qgis.core import QgsMessageLog, Qgis
        
        try:
            # WMSパラメータ解析
            width = self._safe_int(params.get('WIDTH', ['512'])[0], 512)
            height = self._safe_int(params.get('HEIGHT', ['512'])[0], 512)
            bbox = params.get('BBOX', [''])[0]
            # リクエストで指定されたCRSを記録
            original_crs = params.get('CRS', ['EPSG:3857'])[0]
            # オプションで任意CRSを強制的にEPSG:3857として扱う
            if getattr(self, 'force_epsg3857', False):
                crs = 'EPSG:3857'
                QgsMessageLog.logMessage("🔒 force_epsg3857 enabled: treating request CRS as EPSG:3857", "QMapPermalink", Qgis.Info)
            else:
                crs = original_crs
            layers = params.get('LAYERS', ['qgis_professional_map'])[0]
            format_type = params.get('FORMAT', ['image/png'])[0]
            
            QgsMessageLog.logMessage(f"🎨 Professional GetMap: {width}x{height}, requested CRS:{original_crs}, using CRS:{crs}, BBOX:{bbox}", "QMapPermalink", Qgis.Info)
            
            # 本格的レンダリング実行
            image_data = self._render_professional_map(width, height, bbox, crs, layers, format_type)
            
            if image_data and len(image_data) > 1000:
                QgsMessageLog.logMessage(f"✅ Professional rendering success: {len(image_data)} bytes", "QMapPermalink", Qgis.Info)
                content_type = "image/png" if format_type == "image/png" else "image/jpeg"
                self._send_binary_response(conn, 200, "OK", image_data, content_type)
            else:
                QgsMessageLog.logMessage("❌ Professional rendering failed, sending error image", "QMapPermalink", Qgis.Warning)
                error_image = self._generate_error_image(width, height, "Professional WMS Rendering Failed")
                self._send_binary_response(conn, 200, "OK", error_image, "image/png")
                
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Professional GetMap error: {e}", "QMapPermalink", Qgis.Critical)
            import traceback
            QgsMessageLog.logMessage(f"❌ Traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
            
            # エラー時は診断画像を返す
            try:
                error_image = self._generate_error_image(512, 512, f"Error: {str(e)}")
                self._send_binary_response(conn, 200, "OK", error_image, "image/png")
            except:
                self._send_http_response(conn, 500, "Internal Server Error", "Professional WMS rendering failed")
    
    def _render_professional_map(self, width, height, bbox, crs, layers, format_type):
        """本格的マップレンダリング - レイヤ状態踏襲＋任意位置"""
        from qgis.core import QgsMessageLog, Qgis
        
        QgsMessageLog.logMessage("🏆 Starting professional map rendering", "QMapPermalink", Qgis.Info)
        
        try:
            # 方法1: 完全独立レンダリング（QgsMapRendererParallelJob使用）
            image_data = self._independent_map_rendering(width, height, bbox, crs)
            if image_data and len(image_data) > 1000:
                QgsMessageLog.logMessage("✅ Independent rendering successful", "QMapPermalink", Qgis.Info)
                return image_data
            
            # 方法2: キャンバス連携レンダリング（レイヤ状態を完全踏襲）
            image_data = self._canvas_linked_rendering(width, height, bbox, crs)
            if image_data and len(image_data) > 1000:
                QgsMessageLog.logMessage("✅ Canvas-linked rendering successful", "QMapPermalink", Qgis.Info)
                return image_data
            
            # 方法3: ハイブリッドレンダリング（独立＋レイヤ状態）
            image_data = self._hybrid_rendering(width, height, bbox, crs)
            if image_data and len(image_data) > 1000:
                QgsMessageLog.logMessage("✅ Hybrid rendering successful", "QMapPermalink", Qgis.Info)
                return image_data
            
            QgsMessageLog.logMessage("❌ All professional rendering methods failed", "QMapPermalink", Qgis.Warning)
            return None
            
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Professional rendering error: {e}", "QMapPermalink", Qgis.Critical)
            return None
    
    def _independent_map_rendering(self, width, height, bbox, crs):
        """完全独立レンダリング - QgsMapRendererParallelJobを使用"""
        from qgis.core import (QgsMapSettings, QgsMapRendererParallelJob, QgsRectangle, 
                               QgsCoordinateReferenceSystem, QgsCoordinateTransform, 
                               QgsProject, QgsMessageLog, Qgis)
        from PyQt5.QtCore import QSize, QEventLoop, QTimer
        from PyQt5.QtGui import QColor
        
        try:
            QgsMessageLog.logMessage("🎯 Independent rendering with QgsMapRendererParallelJob", "QMapPermalink", Qgis.Info)
            
            # マップ設定を作成
            map_settings = QgsMapSettings()
            map_settings.setOutputSize(QSize(width, height))
            map_settings.setOutputDpi(96)
            
            # 背景色設定
            map_settings.setBackgroundColor(QColor(255, 255, 255, 0))  # 透明背景
            
            # 現在のプロジェクトからレイヤを取得（レイヤ状態を踏襲）
            project = QgsProject.instance()
            layers = []
            
            # レイヤツリーの順序と可視性を尊重
            root = project.layerTreeRoot()
            for child in root.children():
                if hasattr(child, 'layer') and child.layer():
                    layer = child.layer()
                    if child.isVisible():  # 可視レイヤのみ
                        layers.append(layer)
                        QgsMessageLog.logMessage(f"📋 Adding visible layer: {layer.name()}", "QMapPermalink", Qgis.Info)
            
            if not layers:
                # フォールバック: 全てのレイヤを取得
                layers = list(project.mapLayers().values())
                QgsMessageLog.logMessage(f"📋 Using all layers as fallback: {len(layers)} layers", "QMapPermalink", Qgis.Info)
            
            map_settings.setLayers(layers)
            
            # 座標系とエクステント設定
            target_crs = QgsCoordinateReferenceSystem(crs)
            map_settings.setDestinationCrs(target_crs)
            
            # BBOXからエクステントを設定
            if bbox:
                coords = [float(x) for x in bbox.split(',')]
                if len(coords) == 4:
                    extent = QgsRectangle(coords[0], coords[1], coords[2], coords[3])
                    map_settings.setExtent(extent)
                    QgsMessageLog.logMessage(f"📍 Set extent from BBOX: {extent.toString()}", "QMapPermalink", Qgis.Info)
                else:
                    # デフォルトエクステント
                    map_settings.setExtent(QgsRectangle(-180, -90, 180, 90))
            else:
                map_settings.setExtent(QgsRectangle(-180, -90, 180, 90))
            
            # 並列レンダリング実行
            renderer = QgsMapRendererParallelJob(map_settings)
            
            # レンダリング完了を待機
            loop = QEventLoop()
            renderer.finished.connect(loop.quit)
            
            # タイムアウト設定
            timer = QTimer()
            timer.timeout.connect(loop.quit)
            timer.start(10000)  # 10秒タイムアウト
            
            renderer.start()
            loop.exec_()
            
            if not renderer.isActive():
                # レンダリング結果を取得
                image = renderer.renderedImage()
                if not image.isNull():
                    # PNG形式で保存
                    from PyQt5.QtCore import QByteArray, QBuffer, QIODevice
                    byte_array = QByteArray()
                    buffer = QBuffer(byte_array)
                    buffer.open(QIODevice.WriteOnly)
                    
                    success = image.save(buffer, "PNG")
                    if success:
                        png_data = byte_array.data()
                        QgsMessageLog.logMessage(f"✅ Independent rendering completed: {len(png_data)} bytes", "QMapPermalink", Qgis.Info)
                        return png_data
            
            QgsMessageLog.logMessage("❌ Independent rendering failed or timed out", "QMapPermalink", Qgis.Warning)
            return None
            
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Independent rendering error: {e}", "QMapPermalink", Qgis.Warning)
            return None
    
    def _canvas_linked_rendering(self, width, height, bbox, crs):
        """キャンバス連携レンダリング - レイヤ状態を完全踏襲"""
        from qgis.core import QgsMessageLog, Qgis
        
        try:
            QgsMessageLog.logMessage("🎯 Canvas-linked rendering", "QMapPermalink", Qgis.Info)
            
            canvas = self.iface.mapCanvas()
            if not canvas:
                QgsMessageLog.logMessage("❌ No canvas available", "QMapPermalink", Qgis.Warning)
                return None
            
            # 現在のキャンバス状態を保存
            original_extent = canvas.extent()
            original_crs = canvas.mapSettings().destinationCrs()
            
            try:
                # BBOXが指定されている場合、一時的にキャンバス範囲を変更
                if bbox and crs:
                    success = self._set_canvas_extent_from_bbox(canvas, bbox, crs)
                    if not success:
                        QgsMessageLog.logMessage("⚠️ Failed to set canvas extent, using current view", "QMapPermalink", Qgis.Warning)
                
                # キャンバスから画像をキャプチャ
                png_data = self._capture_canvas_image(canvas, width, height)
                
                return png_data
                
            finally:
                # キャンバス状態を復元
                try:
                    canvas.setExtent(original_extent)
                    canvas.setDestinationCrs(original_crs)
                    canvas.refresh()
                except:
                    pass  # 復元失敗は致命的ではない
                    
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Canvas-linked rendering error: {e}", "QMapPermalink", Qgis.Warning)
            return None
    
    def _hybrid_rendering(self, width, height, bbox, crs):
        """ハイブリッドレンダリング - 独立レンダリング＋レイヤ状態踏襲"""
        from qgis.core import QgsMessageLog, Qgis
        
        QgsMessageLog.logMessage("🎯 Hybrid rendering (fallback method)", "QMapPermalink", Qgis.Info)
        
        # このメソッドは将来の拡張用
        # 現在は基本的なエラー画像を返す
        return self._generate_error_image(width, height, "Hybrid Rendering Not Yet Implemented")
    
    def _set_canvas_extent_from_bbox(self, canvas, bbox, crs):
        """キャンバスの範囲をWMS BBOXから設定"""
        from qgis.core import (QgsRectangle, QgsCoordinateReferenceSystem, 
                               QgsCoordinateTransform, QgsProject, QgsMessageLog, Qgis)
        
        try:
            coords = [float(x) for x in bbox.split(',')]
            if len(coords) != 4:
                return False
            
            minx, miny, maxx, maxy = coords
            bbox_extent = QgsRectangle(minx, miny, maxx, maxy)
            
            # 座標変換が必要な場合
            source_crs = QgsCoordinateReferenceSystem(crs)
            canvas_crs = canvas.mapSettings().destinationCrs()
            
            if source_crs.authid() != canvas_crs.authid():
                transform = QgsCoordinateTransform(source_crs, canvas_crs, QgsProject.instance())
                bbox_extent = transform.transformBoundingBox(bbox_extent)
            
            canvas.setExtent(bbox_extent)
            canvas.refresh()
            
            QgsMessageLog.logMessage(f"✅ Canvas extent updated: {bbox_extent.toString()}", "QMapPermalink", Qgis.Info)
            return True
            
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Error setting canvas extent: {e}", "QMapPermalink", Qgis.Warning)
            return False
    
    def _capture_canvas_image(self, canvas, width, height):
        """キャンバスから画像をキャプチャ"""
        from qgis.core import QgsMessageLog, Qgis
        from PyQt5.QtCore import Qt, QByteArray, QBuffer, QIODevice
        
        try:
            # キャンバスから直接ピクセルマップを取得
            pixmap = canvas.grab()
            if pixmap.isNull():
                QgsMessageLog.logMessage("❌ Failed to grab canvas", "QMapPermalink", Qgis.Warning)
                return None
            
            # 指定サイズにスケール
            if width != pixmap.width() or height != pixmap.height():
                pixmap = pixmap.scaled(width, height, Qt.KeepAspectRatio, Qt.SmoothTransformation)
            
            # PNG形式で保存
            image = pixmap.toImage()
            byte_array = QByteArray()
            buffer = QBuffer(byte_array)
            buffer.open(QIODevice.WriteOnly)
            
            success = image.save(buffer, "PNG")
            if success:
                png_data = byte_array.data()
                QgsMessageLog.logMessage(f"✅ Canvas captured: {len(png_data)} bytes", "QMapPermalink", Qgis.Info)
                return png_data
            
            return None
            
        except Exception as e:
            QgsMessageLog.logMessage(f"❌ Canvas capture error: {e}", "QMapPermalink", Qgis.Warning)
            return None
    
    def _get_professional_layer_info(self):
        """現在のQGISプロジェクトから動的レイヤ情報を取得"""
        try:
            from qgis.core import QgsProject
            
            project = QgsProject.instance()
            layers = project.mapLayers().values()
            
            layer_count = len(list(layers))
            visible_count = 0
            
            # 可視レイヤ数をカウント
            root = project.layerTreeRoot()
            for child in root.children():
                if hasattr(child, 'layer') and child.layer() and child.isVisible():
                    visible_count += 1
            
            layer_info = f"""<LayerInfo>
        <TotalLayers>{layer_count}</TotalLayers>
        <VisibleLayers>{visible_count}</VisibleLayers>
        <LastModified>{int(time.time())}</LastModified>
      </LayerInfo>"""
            
            return layer_info
            
        except Exception:
            return "<LayerInfo><TotalLayers>0</TotalLayers><VisibleLayers>0</VisibleLayers></LayerInfo>"
    
    def _get_professional_extent_info(self):
        """本格的WMS用の範囲情報を取得"""
        try:
            canvas = self.iface.mapCanvas()
            if canvas:
                extent = canvas.extent()
                crs = canvas.mapSettings().destinationCrs()
                
                # WGS84に変換
                from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject
                wgs84_crs = QgsCoordinateReferenceSystem("EPSG:4326")
                
                if crs.authid() != "EPSG:4326":
                    transform = QgsCoordinateTransform(crs, wgs84_crs, QgsProject.instance())
                    try:
                        extent = transform.transformBoundingBox(extent)
                    except:
                        pass
                
                extent_xml = f"""<EX_GeographicBoundingBox>
        <westBoundLongitude>{extent.xMinimum():.6f}</westBoundLongitude>
        <eastBoundLongitude>{extent.xMaximum():.6f}</eastBoundLongitude>
        <southBoundLatitude>{extent.yMinimum():.6f}</southBoundLatitude>
        <northBoundLatitude>{extent.yMaximum():.6f}</northBoundLatitude>
      </EX_GeographicBoundingBox>
      <BoundingBox CRS="{crs.authid()}" minx="{extent.xMinimum():.6f}" miny="{extent.yMinimum():.6f}" maxx="{extent.xMaximum():.6f}" maxy="{extent.yMaximum():.6f}"/>"""
                
                return extent_xml
                
        except Exception:
            pass
            
        return """<EX_GeographicBoundingBox>
        <westBoundLongitude>-180.0</westBoundLongitude>
        <eastBoundLongitude>180.0</eastBoundLongitude>
        <southBoundLatitude>-90.0</southBoundLatitude>
        <northBoundLatitude>90.0</northBoundLatitude>
      </EX_GeographicBoundingBox>"""
    
    def _generate_error_image(self, width, height, message):
        """エラー画像を生成"""
        from PyQt5.QtGui import QImage, QPainter, QColor, QFont
        from PyQt5.QtCore import Qt, QByteArray, QBuffer, QIODevice
        
        try:
            # 白い背景の画像を作成
            image = QImage(width, height, QImage.Format_ARGB32)
            image.fill(QColor(255, 255, 255, 255))
            
            # テキストを描画
            painter = QPainter(image)
            painter.setPen(QColor(255, 0, 0))  # 赤色
            painter.setFont(QFont("Arial", 12))
            
            # テキストを中央に配置
            rect = image.rect()
            painter.drawText(rect, Qt.AlignCenter | Qt.TextWordWrap, 
                           f"Professional WMS Error\n{message}\n{width}x{height}")
            
            painter.end()
            
            # PNG形式で保存
            byte_array = QByteArray()
            buffer = QBuffer(byte_array)
            buffer.open(QIODevice.WriteOnly)
            
            success = image.save(buffer, "PNG")
            if success:
                return byte_array.data()
                
        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(f"❌ Error generating error image: {e}", "QMapPermalink", Qgis.Warning)
        
        # フォールバック: 最小限のPNG
        return b'\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x01\x00\x00\x00\x01\x00\x08\x02\x00\x00\x00\x90wS\xde\x00\x00\x00\x12IDATx\x9cc\xf8\x0f\x00\x00\x01\x00\x01\x00\x18\xdd\x8d\xb4\x00\x00\x00\x00IEND\xaeB`\x82'
    
    def _read_http_request(self, conn):
        """HTTPリクエストを読み取り"""
        try:
            data = b''
            while b'\r\n\r\n' not in data:
                chunk = conn.recv(1024)
                if not chunk:
                    break
                data += chunk
                if len(data) > 8192:  # リクエストサイズ制限
                    break
            return data
        except:
            return b''
    
    def _send_http_response(self, conn, status_code, status_text, body):
        """HTTPレスポンスを送信"""
        try:
            body_bytes = body.encode('utf-8') if isinstance(body, str) else body
            response = f"HTTP/1.1 {status_code} {status_text}\r\n"
            response += f"Content-Length: {len(body_bytes)}\r\n"
            response += "Content-Type: text/plain; charset=utf-8\r\n"
            response += "Connection: close\r\n"
            response += "\r\n"
            
            conn.sendall(response.encode('utf-8') + body_bytes)
        except:
            pass
    
    def _send_xml_response(self, conn, xml_content):
        """XMLレスポンスを送信"""
        try:
            xml_bytes = xml_content.encode('utf-8')
            response = "HTTP/1.1 200 OK\r\n"
            response += f"Content-Length: {len(xml_bytes)}\r\n"
            response += "Content-Type: text/xml; charset=utf-8\r\n"
            response += "Connection: close\r\n"
            response += "\r\n"
            
            conn.sendall(response.encode('utf-8') + xml_bytes)
        except:
            pass
    
    def _send_binary_response(self, conn, status_code, status_text, data, content_type):
        """バイナリレスポンスを送信"""
        try:
            response = f"HTTP/1.1 {status_code} {status_text}\r\n"
            response += f"Content-Length: {len(data)}\r\n"
            response += f"Content-Type: {content_type}\r\n"
            response += "Connection: close\r\n"
            response += "\r\n"
            
            conn.sendall(response.encode('utf-8') + data)
        except:
            pass
    
    def _send_wms_error_response(self, conn, error_code, error_message):
        """WMSエラーレスポンスを送信"""
        xml_content = f"""<?xml version="1.0" encoding="UTF-8"?>
<ServiceExceptionReport version="1.3.0" xmlns="http://www.opengis.net/ogc">
  <ServiceException code="{error_code}">
    {error_message}
  </ServiceException>
</ServiceExceptionReport>"""
        
        self._send_xml_response(conn, xml_content)