# -*- coding: utf-8 -*-
"""
Lightweight WMTS-like service for QMapPermalink.

This module provides a small class that handles /wmts requests and
delegates actual rendering to the server manager's existing WMS
GetMap-with-BBOX pipeline. The class is intentionally small to avoid
duplicate rendering code and to prevent circular imports: it expects
to receive the server manager instance at construction time.
"""
import re
from qgis.core import QgsMessageLog, Qgis


class QMapPermalinkWMTSService:
    """Simple WMTS-like handler that maps XYZ tiles to a WMS GetMap BBOX.

    The service does not implement a full WMTS server — just a minimal
    GetCapabilities response and XYZ tile URL pattern /wmts/{z}/{x}/{y}.png
    which it translates into an EPSG:3857 BBOX and calls the server
    manager's _handle_wms_get_map_with_bbox method.
    """

    def __init__(self, server_manager):
        self.server_manager = server_manager

    def handle_wmts_request(self, conn, parsed_url, params, host=None):
        """Handle an incoming /wmts request.

        Args:
            conn: socket connection
            parsed_url: result of urllib.parse.urlparse(target)
            params: dict from urllib.parse.parse_qs
            host: Host header value (optional)
        """
        try:
            QgsMessageLog.logMessage(f"🌐 WMTS service handling: {parsed_url.path}", "QMapPermalink", Qgis.Info)

            # Accept WMTS GetCapabilities via REQUEST=GetCapabilities or SERVICE=WMTS
            req = params.get('REQUEST', [params.get('request', [''])[0]])[0] if params else ''
            svc = params.get('SERVICE', [params.get('service', [''])[0]])[0] if params else ''
            if (req and str(req).upper() == 'GETCAPABILITIES') or (svc and str(svc).upper() == 'WMTS'):
                try:
                    server_port = self.server_manager.http_server.getsockname()[1] if self.server_manager.http_server else self.server_manager.server_port
                except Exception:
                    server_port = self.server_manager.server_port
                if not host:
                    host = f'localhost:{server_port}'
                tile_url = f"http://{host}/wmts/{{z}}/{{x}}/{{y}}.png"
                xml = f'''<?xml version="1.0" encoding="UTF-8"?>
<Capabilities xmlns="http://www.opengis.net/wmts/1.0" version="1.0.0">
  <Contents>
    <Layer>
      <Title>QGIS Map (WMTS)</Title>
      <Identifier>qgis_map</Identifier>
      <TileMatrixSetLink>
        <TileMatrixSet>EPSG:3857</TileMatrixSet>
      </TileMatrixSetLink>
    </Layer>
    <TileMatrixSet>
      <Identifier>EPSG:3857</Identifier>
      <SupportedCRS>EPSG:3857</SupportedCRS>
      <!-- Minimal: clients should use the /wmts/{{z}}/{{x}}/{{y}}.png template -->
    </TileMatrixSet>
  </Contents>
  <ServiceMetadataURL>{tile_url}</ServiceMetadataURL>
</Capabilities>'''
                from . import http_server
                http_server.send_http_response(conn, 200, 'OK', xml, 'text/xml; charset=utf-8')
                return

            # Tile request pattern: /wmts/{z}/{x}/{y}.png
            m = re.match(r'^/wmts/(\d+)/(\d+)/(\d+)\.(png|jpg|jpeg)$', parsed_url.path)
            if m:
                z = int(m.group(1))
                x = int(m.group(2))
                y = int(m.group(3))

                # compute WebMercator bbox for XYZ tile (origin top-left)
                def tile_xyz_to_bbox(z, x, y):
                    origin = 20037508.342789244
                    tiles = 2 ** z
                    tile_size = (origin * 2) / tiles
                    minx = -origin + x * tile_size
                    maxx = -origin + (x + 1) * tile_size
                    maxy = origin - y * tile_size
                    miny = origin - (y + 1) * tile_size
                    return f"{minx},{miny},{maxx},{maxy}"

                bbox = tile_xyz_to_bbox(z, x, y)
                QgsMessageLog.logMessage(f"🔎 WMTS tile -> BBOX z={z},x={x},y={y} => {bbox}", "QMapPermalink", Qgis.Info)

                try:
                    # Delegate to server manager's WMS GetMap-with-BBOX pipeline (256x256)
                    if hasattr(self.server_manager, '_handle_wms_get_map_with_bbox'):
                        self.server_manager._handle_wms_get_map_with_bbox(conn, bbox, 'EPSG:3857', 256, 256, rotation=0.0)
                    else:
                        raise RuntimeError('WMS rendering method not available on server manager')
                except Exception as e:
                    QgsMessageLog.logMessage(f"❌ WMTS tile handler error: {e}", "QMapPermalink", Qgis.Critical)
                    from . import http_server
                    http_server.send_http_response(conn, 500, 'Internal Server Error', f'WMTS tile failed: {e}')
                return

        except Exception as e:
            QgsMessageLog.logMessage(f"❌ WMTS service error: {e}", "QMapPermalink", Qgis.Critical)
            import traceback
            QgsMessageLog.logMessage(f"❌ Error traceback: {traceback.format_exc()}", "QMapPermalink", Qgis.Critical)
            try:
                from . import http_server
                http_server.send_http_response(conn, 500, 'Internal Server Error', f'WMTS processing failed: {e}')
            except Exception:
                pass
