"""MapLibre HTML generator for QMapPermalink

This module provides a small helper to create a temporary HTML file that
loads MapLibre GL JS and centers the map based on a permalink string when
possible. If the permalink cannot be parsed, this module will raise an
error — no fallback behavior (such as opening the input as a URL) is
performed.

Coordinate transformations:
- This module will attempt to use the QGIS Python API (PyQGIS) when
	available to convert coordinates from other CRSs to WGS84.
- If PyQGIS is not available, it will fall back only to a built-in
	inverse WebMercator (EPSG:3857) conversion. It does not use `pyproj`.
	If the source CRS is not convertible by these methods, a RuntimeError
	is raised.
"""
import os
import re
import tempfile
import webbrowser
from urllib.parse import urlparse, parse_qs
import traceback


def _qgis_log(message, level='info'):
	"""Log message to QGIS message log when available, otherwise print.

	level: 'info'|'warning'|'critical'|'debug'
	"""
	try:
		# QGIS logging API
		from qgis.core import QgsMessageLog, Qgis
		if level == 'info':
			QgsMessageLog.logMessage(str(message), 'QMapPermalink', Qgis.Info)
		elif level == 'warning':
			QgsMessageLog.logMessage(str(message), 'QMapPermalink', Qgis.Warning)
		elif level == 'critical':
			QgsMessageLog.logMessage(str(message), 'QMapPermalink', Qgis.Critical)
		else:
			QgsMessageLog.logMessage(str(message), 'QMapPermalink', Qgis.Info)
	except Exception:
		# fallback to stdout so non-QGIS contexts still see messages
		try:
			print(message)
		except Exception:
			# nothing we can do
			pass


def _parse_permalink(permalink_text):
	"""Try to extract lat, lon, zoom from a permalink URL or a simple
	formatted string. Returns (lat, lon, zoom) or None on failure.
	"""
	if not permalink_text:
		return None
	# Try common query parameters: lat, lon, zoom or center, z
	try:
		parsed = urlparse(permalink_text)
		qs = parse_qs(parsed.query)
		lat = None
		lon = None
		zoom = None
		# If query contains x/y and CRS indicates EPSG:4326, treat x=lon,y=lat
		if 'x' in qs and 'y' in qs:
			crs_q = qs.get('crs', [None])[0]
			if crs_q and ('4326' in str(crs_q)):
				try:
					lon = float(qs['x'][0])
					lat = float(qs['y'][0])
					# optional zoom param
					for key in ('zoom', 'z'):
						if key in qs:
							zoom = float(qs[key][0])
					# return early; keep zoom as None if not provided so callers can
					# decide whether to derive zoom from scale later
					return (lat, lon, zoom)
				except Exception:
					# fall through to other parsing strategies on error
					pass
		for key in ('lat', 'latitude'):
			if key in qs:
				lat = float(qs[key][0])
				break
		for key in ('lon', 'lng', 'longitude'):
			if key in qs:
				lon = float(qs[key][0])
				break
		for key in ('zoom', 'z'):
			if key in qs:
				zoom = float(qs[key][0])
				break
		# Some permalinks embed center as "lon,lat,zoom" or "lat,lon,zoom"
		if ('center' in qs or 'c' in qs) and (lat is None or lon is None):
			center = qs.get('center', qs.get('c'))[0]
			parts = re.split('[,; ]+', center)
			if len(parts) >= 2:
				# try both orders
				a = float(parts[0])
				b = float(parts[1])
				# heuristics: if abs(a)>90 then assume lon,lat
				if abs(a) > 90:
					lon, lat = a, b
				else:
					lat, lon = a, b
			if len(parts) >= 3 and zoom is None:
				zoom = float(parts[2])
		if lat is not None and lon is not None:
			# keep zoom as-is (may be None) so caller can apply scale->zoom logic
			return (lat, lon, zoom)
	except Exception:
		pass

	# Try to find patterns like @lat,lon,zoomz (e.g., some mapping services)
	m = re.search(r'@\s*([0-9.+-]+),\s*([0-9.+-]+),\s*([0-9.+-]+)z', permalink_text)
	if m:
		try:
			lat = float(m.group(1))
			lon = float(m.group(2))
			zoom = float(m.group(3))
			return (lat, lon, zoom)
		except Exception:
			pass

	# fallback: try any three floats in the query/path parts of the URL
	# Avoid scanning the netloc (host:port) which can contain a port number
	# that would confuse ordering heuristics.
	try:
		p = urlparse(permalink_text)
		search_source = ' '
		# include query and path (and fragment) but not netloc
		if p.query:
			search_source += p.query + ' '
		if p.path:
			search_source += p.path + ' '
		if p.fragment:
			search_source += p.fragment
		source_to_scan = search_source
	except Exception:
		source_to_scan = permalink_text

	floats = re.findall(r'[-+]?[0-9]*\.?[0-9]+', source_to_scan)
	if len(floats) >= 3:
		try:
			a, b, c = map(float, floats[:3])
			# guess order
			if abs(a) <= 90:
				return (a, b, c)
			else:
				return (b, a, c)
		except Exception:
			pass

	return None


def open_maplibre_from_permalink(permalink_text):
	"""Generate a temporary HTML file with MapLibre and open it.

	If parsing of permalink_text fails, open the permalink_text directly in
	the default browser (it may be a full web page URL).
	"""

	# Log the received permalink as-is
	_qgis_log(f"Received permalink: {permalink_text!r}")

	parsed = _parse_permalink(permalink_text)
	if parsed is None:
		# Parsing failed — do not fallback or attempt to open the input as a URL.
		# Surface an explicit error so callers must handle invalid permalinks.
		raise ValueError(f"Cannot parse permalink: {permalink_text!r}")

	lat, lon, zoom = parsed
	_qgis_log(f"Parsed coordinates (from permalink): lat={lat}, lon={lon}, zoom={zoom}")

	# If the permalink contains a 'scale' parameter, prefer its converted zoom
	# value. We override any parsed zoom with the scale->zoom estimate when
	# possible; if scale parsing or estimator import fails we fall back to the
	# previously parsed zoom.
	try:
		from urllib.parse import urlparse as _urlparse, parse_qs as _parse_qs
		_p = _urlparse(permalink_text)
		_qs = _parse_qs(_p.query)
		if 'scale' in _qs:
			try:
				scale_val = float(_qs['scale'][0])
				# Import and use the pure-Python estimator (no QGIS dependency)
				try:
					from .scale_zoom import estimate_zoom_from_scale_maplibre
					zoom_est = estimate_zoom_from_scale_maplibre(scale_val)
					if zoom_est is not None:
						zoom = float(zoom_est)
						_qgis_log(f"scale param detected: {scale_val} -> estimated zoom={zoom}")
				except Exception:
					# best-effort: if import fails, leave zoom as parsed
					_qgis_log("scale->zoom estimator not available; using parsed zoom if any", 'debug')
			except Exception:
				# ignore float conversion errors
				_qgis_log("failed to parse scale parameter; using parsed zoom if any", 'debug')
	except Exception:
		# ignore any errors while attempting to read scale
		_qgis_log("error while reading scale parameter; using parsed zoom if any", 'debug')

	# If permalink contains x/y/crs style parameters (e.g. from QMapPermalink generate
	# output), try to detect and convert them to WGS84 (lat/lon) for MapLibre.
	# Prefer QGIS transformation APIs if available (when running inside QGIS).
	# If QGIS is not available, fall back only to the built-in
	# EPSG:3857 inverse mercator conversion. `pyproj` is intentionally not used.
	# Quick parse for x, y, crs parameters in the original permalink_text
	# examples: '.../qgis-map?x=123456&y=456789&crs=EPSG:3857'
	from urllib.parse import urlparse, parse_qs
	parsed_url = urlparse(permalink_text)
	qs = parse_qs(parsed_url.query)
	if 'x' in qs and 'y' in qs:
		x_val = float(qs['x'][0])
		y_val = float(qs['y'][0])
		crs_param = qs.get('crs', [None])[0]
		# override zoom from query if present
		if 'zoom' in qs:
			try:
				zoom = float(qs['zoom'][0])
			except Exception:
				pass
		if crs_param:
			# normalize crs string (accept 'EPSG:3857' or numeric)
			src_crs = str(crs_param)
			converted = None
			conversion_method = None
			conversion_errors = []
			# If CRS explicitly states EPSG:4326, the x/y are already lon/lat
			# (QMapPermalink uses x=lon,y=lat for EPSG:4326). Treat that as
			# a short-circuit to avoid attempting QGIS or other transforms
			# which may not be available outside QGIS.
			if '4326' in src_crs:
				try:
					lat = float(y_val)
					lon = float(x_val)
					_qgis_log(f"Detected source CRS EPSG:4326; using x/y as lon/lat -> lat={lat}, lon={lon}, zoom={zoom}")
				except Exception:
					pass
				# skip further conversion attempts
				converted = (lat, lon)
				conversion_method = 'direct'

			# try QGIS
			try:
				from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject, QgsPointXY
				src = QgsCoordinateReferenceSystem(str(src_crs))
				if src.isValid():
					dest = QgsCoordinateReferenceSystem('EPSG:4326')
					transform = QgsCoordinateTransform(src, dest, QgsProject.instance())
					pt = transform.transform(QgsPointXY(float(x_val), float(y_val)))
					converted = (float(pt.y()), float(pt.x()))
					conversion_method = 'qgis'
			except Exception:
				conversion_errors.append(('qgis', traceback.format_exc()))
			# try built-in EPSG:3857 inverse mercator
			try:
				sc = str(src_crs).upper()
				if '3857' in sc or '900913' in sc:
					import math
					R = 6378137.0
					lon_deg = (float(x_val) / R) * (180.0 / math.pi)
					lat_rad = 2 * math.atan(math.exp(float(y_val) / R)) - math.pi / 2.0
					lat_deg = lat_rad * (180.0 / math.pi)
					converted = (float(lat_deg), float(lon_deg))
					conversion_method = 'builtin'
			except Exception:
				conversion_errors.append(('builtin', traceback.format_exc()))
			# If none of the methods produced a converted point, surface an error
			if converted is None:
				err_msg = f"Point conversion failed for CRS '{src_crs}'. Attempts:\n"
				for name, tb in conversion_errors:
					err_msg += f"- {name}: {tb}\n"
				raise RuntimeError(err_msg)
			else:
				# converted is (lat, lon)
				lat, lon = converted
				_qgis_log(f"Converted point from CRS {src_crs}: method={conversion_method}, lat={lat}, lon={lon}, zoom={zoom}")



		# Detect bounding box parameters and try to convert them to WGS84 for fitBounds
	fit_bounds = None
	try:
		parsed_url = urlparse(permalink_text)
		qs = parse_qs(parsed_url.query)
		# handle explicit bbox params: x_min,y_min,x_max,y_max + optional crs
		if all(k in qs for k in ('x_min', 'y_min', 'x_max', 'y_max')):
			x_min = float(qs['x_min'][0])
			y_min = float(qs['y_min'][0])
			x_max = float(qs['x_max'][0])
			y_max = float(qs['y_max'][0])
			crs_param = qs.get('crs', [None])[0]
			# convert four corners (use same conversion routine as above)
			def _convert_point(xv, yv, src_crs):
				conversion_errors = []
				# try QGIS
				try:
					from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject, QgsPointXY
					src = QgsCoordinateReferenceSystem(str(src_crs))
					if src.isValid():
						dest = QgsCoordinateReferenceSystem('EPSG:4326')
						transform = QgsCoordinateTransform(src, dest, QgsProject.instance())
						pt = transform.transform(QgsPointXY(float(xv), float(yv)))
						return float(pt.y()), float(pt.x()), 'qgis'
				except Exception:
					conversion_errors.append(('qgis', traceback.format_exc()))
				# try built-in EPSG:3857 inverse mercator
				try:
					sc = str(src_crs).upper()
					if '3857' in sc or '900913' in sc:
						import math
						R = 6378137.0
						lon_deg = (float(xv) / R) * (180.0 / math.pi)
						lat_rad = 2 * math.atan(math.exp(float(yv) / R)) - math.pi / 2.0
						lat_deg = lat_rad * (180.0 / math.pi)
						return float(lat_deg), float(lon_deg), 'builtin'
				except Exception:
					conversion_errors.append(('builtin', traceback.format_exc()))
				# All methods failed
				err_msg = f"Point conversion failed for CRS '{src_crs}'. Attempts:\n"
				for name, tb in conversion_errors:
					err_msg += f"- {name}: {tb}\n"
				raise RuntimeError(err_msg)

			src_crs = crs_param or None
			if src_crs:
				p1_lat, p1_lon, p1_method = _convert_point(x_min, y_min, src_crs)
				p2_lat, p2_lon, p2_method = _convert_point(x_max, y_max, src_crs)
				if p1_lat is not None and p2_lat is not None:
					# fitBounds expects [[minLon,minLat],[maxLon,maxLat]]
					min_lat, min_lon = p1_lat, p1_lon
					max_lat, max_lon = p2_lat, p2_lon
					# ensure ordering
					sw = [min_lon, min_lat]
					ne = [max_lon, max_lat]
					fit_bounds = [sw, ne]
					_qgis_log(f"Converted bbox to fit_bounds: {fit_bounds} (methods: {p1_method}, {p2_method})")
	except Exception:
		fit_bounds = None

	# Build JS snippet for fitBounds if bounds were detected
	fit_js = ''
	if fit_bounds:
		try:
			sw, ne = fit_bounds
			# Ensure floats with enough precision
			fit_js = f"map.fitBounds([{sw!s},{ne!s}], {{padding:20}});"
		except Exception:
			fit_js = ''

	# If zoom is still None at this point, use a safe default
	if zoom is None:
		zoom = 10

	# Clip and format zoom for output (sane range for most tile servers)
	try:
		_zoom_val = float(zoom) if zoom is not None else None
		if _zoom_val is None:
			_zoom_out = None
		else:
			# ensure zoom is not negative; do not artificially cap the maximum zoom
			_zoom_out = max(0.0, _zoom_val)
			# round to 2 decimal places for tidy HTML output
			_zoom_out = float(f"{_zoom_out:.2f}")
	except Exception:
		_zoom_out = None

	# Log final map target used for MapLibre
	if fit_bounds:
		_qgis_log(f"MapLibre will use fitBounds: {fit_bounds}")
	else:
		_qgis_log(f"MapLibre target coordinates: lat={lat}, lon={lon}, zoom={_zoom_out}")

	# Choose tile template: prefer local WMTS when running inside QGIS so
	# the generated MapLibre HTML points at the plugin's /wmts endpoint.
	try:
		from qgis.core import QgsApplication  # existence check
		tile_template = "/wmts/{z}/{x}/{y}.png"
		_qgis_log("Using local WMTS tile template for MapLibre: /wmts/{z}/{x}/{y}.png", 'debug')
	except Exception:
		tile_template = "https://tile.openstreetmap.org/{z}/{x}/{y}.png"

	html = f'''<!doctype html>
<html>
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="initial-scale=1,maximum-scale=1,user-scalable=no" />
  <title>MapLibre Viewer</title>
	<link href="https://unpkg.com/maplibre-gl@latest/dist/maplibre-gl.css" rel="stylesheet" />
  <style>html,body,#map{{height:100%;margin:0;padding:0}}#map{{position:fixed;inset:0}}</style>
</head>
<body>
<div id="map"></div>
<button id="pitchToggle" style="position:absolute;top:10px;right:10px;z-index:1001;padding:6px 8px;background:#fff;border:1px solid #666;border-radius:4px;cursor:pointer">斜め禁止</button>
<script src="https://unpkg.com/maplibre-gl@latest/dist/maplibre-gl.js"></script>
<script>
  console.log('MapLibre script loaded');
  try {{
    // Use a minimal inline raster style (OpenStreetMap tiles)
		const style = {{
			"version": 8,
			"sources": {{
				"qmap": {{
					"type": "raster",
					"tiles": ["{tile_template}"],
					"tileSize": 256,
					"attribution": "QMapPermalink WMTS"
				}}
			}},
			"layers": [ {{ "id": "qmap", "type": "raster", "source": "qmap", "minzoom": 0 }} ]
		}};
    
			console.log('Initializing map at lat={lat}, lon={lon}, zoom={_zoom_out}');
		const map = new maplibregl.Map({{
			container: 'map',
			style: style,
			center: [{lon}, {lat}],
			zoom: {_zoom_out if _zoom_out is not None else 0}
		}});

			// Pitch toggle button: disable/enable oblique (3D) tilt interaction
			try {{
				const pitchBtn = document.getElementById('pitchToggle');
				// Start with pitch locked (斜め禁止) as requested; rotation remains allowed.
				let pitchLocked = true;
				const _enforcePitch = function() {{
					try {{
						if (map.getPitch && Math.abs(map.getPitch()) > 0.0001) {{
							map.setPitch(0);
						}}
					}} catch (e) {{ /* ignore */ }}
				}};
				function lockPitch() {{
					try {{ map.setPitch(0); }} catch(e) {{ console.warn('setPitch failed', e); }}
					try {{ if (map.on) map.on('move', _enforcePitch); }} catch(e) {{}}
					pitchLocked = true;
					pitchBtn.textContent = '斜め許可';
				}}
				function unlockPitch() {{
					try {{ if (map.off) map.off('move', _enforcePitch); }} catch(e) {{}}
					pitchLocked = false;
					pitchBtn.textContent = '斜め禁止';
				}}
				pitchBtn.addEventListener('click', function() {{
					if (!pitchLocked) lockPitch(); else unlockPitch();
				}});
				// enforce initially
				try {{ lockPitch(); }} catch(e) {{}}
			}} catch(e) {{ console.warn('pitch toggle setup failed', e); }}
		// Defer resize and fitBounds until style/tile sources are loaded to
		// avoid missing tiles or blank initial render when the container was
		// not yet ready. Call map.resize() then apply fitBounds if present.
    
		map.on('load', function() {{
			console.log('Map loaded successfully');
			try {{
				// ensure map knows its container size
				map.resize();
			}} catch (e) {{
				console.warn('map.resize() failed', e);
			}}
			try {{
				{fit_js}
			}} catch (e) {{
				console.warn('fitBounds failed', e);
			}}
		}});
    
    map.on('error', function(e) {{
      console.error('Map error:', e);
    }});
  }} catch (e) {{
    console.error('Failed to initialize map:', e);
    document.body.innerHTML = '<div style="padding:20px;font-family:sans-serif;"><h2>Map initialization failed</h2><pre>' + e.toString() + '</pre><p>Check browser console (F12) for details.</p></div>';
  }}
</script>
</body>
</html>'''

	# write to temp file
	try:
		fd, path = tempfile.mkstemp(suffix='.html', prefix='qmap_maplibre_')
		with os.fdopen(fd, 'w', encoding='utf-8') as f:
			f.write(html)
		# Log the path so callers (or QGIS Python console) can see where the file
		# was written. Return the path so callers can open it manually if needed.
		_qgis_log(f"MapLibre HTML written to: {path}")
		# open file URL using a proper file URI when possible
		try:
			from pathlib import Path
			file_uri = Path(path).as_uri()
			webbrowser.open(file_uri)
		except Exception:
			# fallback to older style if pathlib unavailable
			webbrowser.open('file://' + path)
		return path
	except Exception as e:
		# Do not silently fallback; raise an error so caller can handle it
		raise RuntimeError(f"Failed to create or open MapLibre HTML file: {e}") from e
