"""Simple MapLibre HTML generator used by the standalone server.

This module provides a small helper to create an HTML string that
initializes MapLibre centered on coordinates parsed from a permalink
style string or explicit lat/lon/zoom query parameters.

It intentionally keeps dependencies minimal so it can be used outside
of QGIS.
"""
from urllib.parse import urlparse, parse_qs
import re

def _parse_permalink(permalink_text):
    """Reuse the same parsing heuristics as the plugin's generator.

    This duplicates a subset of the original implementation so the
    standalone server does not need the full QGIS stack.
    Returns (lat, lon, zoom) or None.
    """
    if not permalink_text:
        return None
    try:
        p = urlparse(permalink_text)
        qs = parse_qs(p.query)
        lat = None; lon = None; zoom = None
        for key in ('lat','latitude'):
            if key in qs:
                lat = float(qs[key][0]); break
        for key in ('lon','lng','longitude'):
            if key in qs:
                lon = float(qs[key][0]); break
        for key in ('zoom','z'):
            if key in qs:
                try:
                    zoom = float(qs[key][0]);
                except Exception:
                    zoom = None
                break
        # center param like "lon,lat,zoom" or "lat,lon,zoom"
        if ('center' in qs or 'c' in qs) and (lat is None or lon is None):
            center = qs.get('center', qs.get('c'))[0]
            parts = re.split('[,; ]+', center)
            if len(parts) >= 2:
                a = float(parts[0]); b = float(parts[1])
                if abs(a) > 90:
                    lon, lat = a, b
                else:
                    lat, lon = a, b
            if len(parts) >= 3 and zoom is None:
                zoom = float(parts[2])
        if lat is not None and lon is not None:
            return (lat, lon, zoom)
    except Exception:
        pass

    m = re.search(r'@\s*([0-9.+-]+),\s*([0-9.+-]+),\s*([0-9.+-]+)z', permalink_text)
    if m:
        try:
            lat = float(m.group(1)); lon = float(m.group(2)); zoom = float(m.group(3))
            return (lat, lon, zoom)
        except Exception:
            pass

    # try any three floats in the text
    floats = re.findall(r'[-+]?[0-9]*\.?[0-9]+', permalink_text)
    if len(floats) >= 3:
        try:
            a,b,c = map(float, floats[:3])
            if abs(a) <= 90:
                return (a,b,c)
            else:
                return (b,a,c)
        except Exception:
            pass
    return None


def generate_maplibre_html(permalink_text=None, lat=None, lon=None, zoom=None, tile_url=None):
    """Return a full HTML string initializing MapLibre.

    Parameters can be supplied as a permalink string (permalink_text)
    or explicit lat/lon/zoom. Explicit args override the permalink.
    """
    if permalink_text and (lat is None or lon is None):
        parsed = _parse_permalink(permalink_text)
        if parsed:
            _plat, _plon, _pzoom = parsed
            if lat is None: lat = _plat
            if lon is None: lon = _plon
            if zoom is None: zoom = _pzoom

    # defaults
    try:
        lat = float(lat) if lat is not None else 35.681236
        lon = float(lon) if lon is not None else 139.767125
        zoom = float(zoom) if zoom is not None else 10
    except Exception:
        lat = 35.681236; lon = 139.767125; zoom = 10

    # ensure zoom is not negative; do not impose an artificial upper bound
    try:
        zoom = max(0.0, float(zoom))
    except Exception:
        zoom = 10

    # Decide tile URL template: if tile_url param is provided use it. Otherwise,
    # prefer a local WMTS endpoint when PyQGIS is available (plugin usage),
    # else fall back to OpenStreetMap tiles.
    if not tile_url:
        try:
            # running inside QGIS -> prefer local WMTS endpoint
            import qgis  # noqa: F401
            tile_url = "/wmts/{z}/{x}/{y}.png"
        except Exception:
            tile_url = "https://tile.openstreetmap.org/{z}/{x}/{y}.png"

    html = f'''<!doctype html>
<html>
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="initial-scale=1,maximum-scale=1,user-scalable=no" />
  <title>MapLibre Viewer</title>
  <link href="https://unpkg.com/maplibre-gl@latest/dist/maplibre-gl.css" rel="stylesheet" />
  <style>html,body,#map{{height:100%;margin:0;padding:0}}#map{{position:fixed;inset:0}}</style>
</head>
<body>
    <div id="map"></div>
    <button id="pitchToggle" style="position:absolute;top:10px;right:10px;z-index:1001;padding:6px 8px;background:#fff;border:1px solid #666;border-radius:4px;cursor:pointer">斜め禁止</button>
<script src="https://unpkg.com/maplibre-gl@latest/dist/maplibre-gl.js"></script>
<script>
  try {{
    const style = {{
      "version": 8,
                "sources": {{
                "qmap": {
                    "type": "raster",
                    "tiles": ["{tile_url}"],
                    "tileSize": 256,
                    "attribution": "QMapPermalink WMTS"
                }
            }},
            "layers": [ {{ "id": "qmap", "type": "raster", "source": "qmap" }} ]
    }};
        const map = new maplibregl.Map({{
            container: 'map',
            style: style,
            center: [{lon}, {lat}],
            zoom: {zoom}
        }});
        // Setup pitch toggle button to disable oblique tilt
            try {{
                const pitchBtn = document.getElementById('pitchToggle');
                // Keep rotation enabled but prevent pitch by forcing pitch=0 while locked
                let pitchLocked = true;
                const _enforcePitch = function() {{
                    try {{ if (map.getPitch && Math.abs(map.getPitch()) > 0.0001) map.setPitch(0); }} catch(e) {{}}
                }};
                function lockPitch() {{
                    try {{ map.setPitch(0); }} catch(e) {{}}
                    try {{ if (map.on) map.on('move', _enforcePitch); }} catch(e) {{}}
                    pitchLocked = true; pitchBtn.textContent = '斜め許可';
                }}
                function unlockPitch() {{
                    try {{ if (map.off) map.off('move', _enforcePitch); }} catch(e) {{}}
                    pitchLocked = false; pitchBtn.textContent = '斜め禁止';
                }}
                pitchBtn.addEventListener('click', function() {{ if (!pitchLocked) lockPitch(); else unlockPitch(); }});
                try {{ lockPitch(); }} catch(e) {{}}
            }} catch(e) {{ console.warn('pitch toggle setup failed', e); }}
        map.on('load', function() {{ map.resize(); }});
  }} catch (e) {{
    document.body.innerHTML = '<pre>Map init failed: ' + e.toString() + '</pre>';
  }}
</script>
</body>
</html>'''
    return html
