# -*- coding: utf-8 -*-
"""Shared lightweight HTTP utilities for QMapPermalink.

This module provides minimal request/response helpers used by the
plugin's internal socket-based HTTP servers. Keeping these utilities
centralized makes future extensions (middleware, logging, CORS, etc.)
easier.
"""
import socket


def read_http_request(conn, max_size=8192):
    """Read raw HTTP request bytes from a connected socket.

    Args:
        conn: socket-like object with recv
        max_size: maximum bytes to read (prevents unbounded memory use)

    Returns:
        bytes: raw request bytes (may be empty on error/timeout)
    """
    try:
        data = b''
        while b'\r\n\r\n' not in data and len(data) < max_size:
            chunk = conn.recv(1024)
            if not chunk:
                break
            data += chunk
        return data
    except Exception:
        return b''


def send_http_response(conn, status_code, reason, body, content_type="text/plain; charset=utf-8"):
    """Send a minimal HTTP response (text or bytes).

    Args:
        conn: socket-like object with sendall
        status_code: integer HTTP status code
        reason: status reason phrase
        body: str or bytes body
        content_type: Content-Type header value
    """
    try:
        if isinstance(body, str):
            body_bytes = body.encode('utf-8')
        else:
            body_bytes = body or b''

        header_lines = [
            f"HTTP/1.1 {status_code} {reason}",
            f"Content-Length: {len(body_bytes)}",
            f"Content-Type: {content_type}",
            "Access-Control-Allow-Origin: *",
            "Connection: close",
            "",
            "",
        ]
        header = "\r\n".join(header_lines).encode('utf-8')
        conn.sendall(header + body_bytes)
    except Exception:
        try:
            conn.close()
        except Exception:
            pass


def send_xml_response(conn, xml_content):
    """Send a 200 OK XML response encoded as UTF-8."""
    try:
        xml_bytes = xml_content.encode('utf-8')
        header_lines = [
            "HTTP/1.1 200 OK",
            f"Content-Length: {len(xml_bytes)}",
            "Content-Type: text/xml; charset=utf-8",
            "Access-Control-Allow-Origin: *",
            "Connection: close",
            "",
            "",
        ]
        header = "\r\n".join(header_lines).encode('utf-8')
        conn.sendall(header + xml_bytes)
    except Exception:
        try:
            conn.close()
        except Exception:
            pass


def send_binary_response(conn, status_code, reason, data, content_type):
    """Send a binary HTTP response (images, etc.)."""
    try:
        header_lines = [
            f"HTTP/1.1 {status_code} {reason}",
            f"Content-Length: {len(data)}",
            f"Content-Type: {content_type}",
            "Access-Control-Allow-Origin: *",
            "Connection: close",
            "",
            "",
        ]
        header = "\r\n".join(header_lines).encode('utf-8')
        conn.sendall(header + data)
    except Exception:
        try:
            conn.close()
        except Exception:
            pass


def send_wms_error_response(conn, error_code, error_message):
    """Send a basic WMS ServiceExceptionReport XML response."""
    xml_content = f"""<?xml version="1.0" encoding="UTF-8"?>
<ServiceExceptionReport version="1.3.0" xmlns="http://www.opengis.net/ogc">
  <ServiceException code="{error_code}">
    {error_message}
  </ServiceException>
</ServiceExceptionReport>"""
    send_xml_response(conn, xml_content)
