# -*- coding: utf-8 -*-
"""
Modern Material Design styles for QGIS YOLOX Plugin
"""

# Main color palette
COLORS = {
    'primary': '#2196F3',      # Blue
    'primary_dark': '#1976D2',
    'primary_light': '#BBDEFB',
    'accent': '#FF9800',       # Orange
    'success': '#4CAF50',      # Green
    'warning': '#FFC107',      # Amber
    'error': '#F44336',        # Red
    'background': '#FAFAFA',
    'surface': '#FFFFFF',
    'text_primary': '#212121',
    'text_secondary': '#757575',
    'divider': '#BDBDBD',
}

# Modern Material Design stylesheet
MODERN_STYLE = f"""
/* Main Window */
QWidget {{
    font-family: 'Segoe UI', Arial, sans-serif;
    font-size: 10pt;
    color: {COLORS['text_primary']};
}}

QMainWindow, QDialog {{
    background-color: {COLORS['background']};
}}

/* Buttons */
QPushButton {{
    background-color: {COLORS['primary']};
    color: white;
    border: none;
    border-radius: 4px;
    padding: 8px 16px;
    font-weight: 500;
    min-height: 32px;
}}

QPushButton:hover {{
    background-color: {COLORS['primary_dark']};
}}

QPushButton:pressed {{
    background-color: #0D47A1;
}}

QPushButton:disabled {{
    background-color: {COLORS['divider']};
    color: {COLORS['text_secondary']};
}}

QPushButton#secondaryButton {{
    background-color: {COLORS['surface']};
    color: {COLORS['primary']};
    border: 1px solid {COLORS['primary']};
}}

QPushButton#secondaryButton:hover {{
    background-color: {COLORS['primary_light']};
}}

QPushButton#successButton {{
    background-color: {COLORS['success']};
}}

QPushButton#warningButton {{
    background-color: {COLORS['warning']};
}}

QPushButton#errorButton {{
    background-color: {COLORS['error']};
}}

/* Input Fields */
QLineEdit, QTextEdit, QPlainTextEdit {{
    border: 1px solid {COLORS['divider']};
    border-radius: 4px;
    padding: 8px;
    background-color: {COLORS['surface']};
    selection-background-color: {COLORS['primary_light']};
}}

QLineEdit:focus, QTextEdit:focus, QPlainTextEdit:focus {{
    border: 2px solid {COLORS['primary']};
}}

QLineEdit:disabled, QTextEdit:disabled {{
    background-color: {COLORS['background']};
    color: {COLORS['text_secondary']};
}}

/* ComboBox */
QComboBox {{
    border: 1px solid {COLORS['divider']};
    border-radius: 4px;
    padding: 6px 10px;
    background-color: {COLORS['surface']};
    min-height: 28px;
}}

QComboBox:hover {{
    border: 1px solid {COLORS['primary']};
}}

QComboBox:focus {{
    border: 2px solid {COLORS['primary']};
}}

QComboBox::drop-down {{
    border: none;
    width: 20px;
}}

QComboBox::down-arrow {{
    image: url(none);
    border-left: 5px solid transparent;
    border-right: 5px solid transparent;
    border-top: 5px solid {COLORS['text_secondary']};
    margin-right: 5px;
}}

QComboBox QAbstractItemView {{
    border: 1px solid {COLORS['divider']};
    background-color: {COLORS['surface']};
    selection-background-color: {COLORS['primary_light']};
    selection-color: {COLORS['text_primary']};
}}

/* Spinbox */
QSpinBox, QDoubleSpinBox {{
    border: 1px solid {COLORS['divider']};
    border-radius: 4px;
    padding: 6px;
    background-color: {COLORS['surface']};
}}

QSpinBox:focus, QDoubleSpinBox:focus {{
    border: 2px solid {COLORS['primary']};
}}

/* Sliders */
QSlider::groove:horizontal {{
    border: none;
    height: 4px;
    background: {COLORS['divider']};
    border-radius: 2px;
}}

QSlider::handle:horizontal {{
    background: {COLORS['primary']};
    border: none;
    width: 16px;
    height: 16px;
    margin: -6px 0;
    border-radius: 8px;
}}

QSlider::handle:horizontal:hover {{
    background: {COLORS['primary_dark']};
}}

/* Progress Bar */
QProgressBar {{
    border: none;
    border-radius: 4px;
    background-color: {COLORS['divider']};
    text-align: center;
    height: 8px;
}}

QProgressBar::chunk {{
    background-color: {COLORS['primary']};
    border-radius: 4px;
}}

/* Tabs */
QTabWidget::pane {{
    border: 1px solid {COLORS['divider']};
    border-radius: 4px;
    background-color: {COLORS['surface']};
    top: -1px;
}}

QTabBar::tab {{
    background-color: {COLORS['background']};
    border: 1px solid {COLORS['divider']};
    border-bottom: none;
    padding: 10px 20px;
    margin-right: 2px;
}}

QTabBar::tab:selected {{
    background-color: {COLORS['surface']};
    border-bottom: 2px solid {COLORS['primary']};
    color: {COLORS['primary']};
    font-weight: 600;
}}

QTabBar::tab:hover {{
    background-color: {COLORS['primary_light']};
}}

/* Group Box */
QGroupBox {{
    border: 1px solid {COLORS['divider']};
    border-radius: 6px;
    margin-top: 12px;
    padding-top: 16px;
    background-color: {COLORS['surface']};
    font-weight: 600;
}}

QGroupBox::title {{
    subcontrol-origin: margin;
    subcontrol-position: top left;
    left: 12px;
    padding: 0 8px;
    color: {COLORS['primary']};
}}

/* Labels */
QLabel {{
    color: {COLORS['text_primary']};
}}

QLabel#headerLabel {{
    font-size: 16pt;
    font-weight: 600;
    color: {COLORS['primary']};
}}

QLabel#subheaderLabel {{
    font-size: 12pt;
    font-weight: 500;
    color: {COLORS['text_secondary']};
}}

QLabel#errorLabel {{
    color: {COLORS['error']};
    font-weight: 500;
}}

QLabel#successLabel {{
    color: {COLORS['success']};
    font-weight: 500;
}}

/* List Widget */
QListWidget {{
    border: 1px solid {COLORS['divider']};
    border-radius: 4px;
    background-color: {COLORS['surface']};
    padding: 4px;
}}

QListWidget::item {{
    border-radius: 4px;
    padding: 8px;
    margin: 2px;
}}

QListWidget::item:selected {{
    background-color: {COLORS['primary_light']};
    color: {COLORS['text_primary']};
}}

QListWidget::item:hover {{
    background-color: {COLORS['background']};
}}

/* Table Widget */
QTableWidget {{
    border: 1px solid {COLORS['divider']};
    border-radius: 4px;
    background-color: {COLORS['surface']};
    gridline-color: {COLORS['divider']};
}}

QTableWidget::item {{
    padding: 8px;
}}

QTableWidget::item:selected {{
    background-color: {COLORS['primary_light']};
    color: {COLORS['text_primary']};
}}

QHeaderView::section {{
    background-color: {COLORS['background']};
    color: {COLORS['text_primary']};
    padding: 8px;
    border: none;
    border-bottom: 2px solid {COLORS['divider']};
    font-weight: 600;
}}

/* Checkboxes */
QCheckBox {{
    spacing: 8px;
}}

QCheckBox::indicator {{
    width: 18px;
    height: 18px;
    border: 2px solid {COLORS['divider']};
    border-radius: 3px;
    background-color: {COLORS['surface']};
}}

QCheckBox::indicator:hover {{
    border: 2px solid {COLORS['primary']};
}}

QCheckBox::indicator:checked {{
    background-color: {COLORS['primary']};
    border: 2px solid {COLORS['primary']};
    image: url(none);
}}

/* Radio Buttons */
QRadioButton {{
    spacing: 8px;
}}

QRadioButton::indicator {{
    width: 18px;
    height: 18px;
    border: 2px solid {COLORS['divider']};
    border-radius: 9px;
    background-color: {COLORS['surface']};
}}

QRadioButton::indicator:hover {{
    border: 2px solid {COLORS['primary']};
}}

QRadioButton::indicator:checked {{
    background-color: {COLORS['primary']};
    border: 2px solid {COLORS['primary']};
}}

/* Scroll Bar */
QScrollBar:vertical {{
    border: none;
    background: {COLORS['background']};
    width: 12px;
    margin: 0;
}}

QScrollBar::handle:vertical {{
    background: {COLORS['divider']};
    border-radius: 6px;
    min-height: 20px;
}}

QScrollBar::handle:vertical:hover {{
    background: {COLORS['text_secondary']};
}}

QScrollBar::add-line:vertical, QScrollBar::sub-line:vertical {{
    border: none;
    background: none;
    height: 0px;
}}

QScrollBar:horizontal {{
    border: none;
    background: {COLORS['background']};
    height: 12px;
    margin: 0;
}}

QScrollBar::handle:horizontal {{
    background: {COLORS['divider']};
    border-radius: 6px;
    min-width: 20px;
}}

QScrollBar::handle:horizontal:hover {{
    background: {COLORS['text_secondary']};
}}

QScrollBar::add-line:horizontal, QScrollBar::sub-line:horizontal {{
    border: none;
    background: none;
    width: 0px;
}}

/* Tool Tip */
QToolTip {{
    background-color: {COLORS['text_primary']};
    color: white;
    border: none;
    padding: 6px;
    border-radius: 4px;
}}

/* Menu */
QMenuBar {{
    background-color: {COLORS['surface']};
    border-bottom: 1px solid {COLORS['divider']};
}}

QMenuBar::item {{
    padding: 8px 12px;
    background-color: transparent;
}}

QMenuBar::item:selected {{
    background-color: {COLORS['primary_light']};
}}

QMenu {{
    background-color: {COLORS['surface']};
    border: 1px solid {COLORS['divider']};
    border-radius: 4px;
    padding: 4px;
}}

QMenu::item {{
    padding: 8px 24px;
    border-radius: 4px;
}}

QMenu::item:selected {{
    background-color: {COLORS['primary_light']};
}}

/* Status Bar */
QStatusBar {{
    background-color: {COLORS['surface']};
    border-top: 1px solid {COLORS['divider']};
    padding: 4px;
}}

/* Separator */
QFrame[frameShape="4"], QFrame[frameShape="5"] {{
    color: {COLORS['divider']};
}}
"""

def get_icon_style(icon_name: str, color: str = None) -> str:
    """
    Get style for icon-like button

    Args:
        icon_name: Name of the icon (e.g., 'play', 'stop')
        color: Optional color override

    Returns:
        CSS style string
    """
    if color is None:
        color = COLORS['primary']

    return f"""
        QPushButton {{
            background-color: transparent;
            border: 2px solid {color};
            border-radius: 20px;
            width: 40px;
            height: 40px;
            font-size: 16px;
            color: {color};
        }}
        QPushButton:hover {{
            background-color: {color};
            color: white;
        }}
    """

def get_card_style() -> str:
    """Get style for card-like container"""
    return f"""
        QFrame {{
            background-color: {COLORS['surface']};
            border: 1px solid {COLORS['divider']};
            border-radius: 8px;
            padding: 16px;
        }}
    """

def get_status_style(status: str) -> str:
    """
    Get style for status indicators

    Args:
        status: 'success', 'warning', 'error', or 'info'

    Returns:
        CSS style string
    """
    color_map = {
        'success': COLORS['success'],
        'warning': COLORS['warning'],
        'error': COLORS['error'],
        'info': COLORS['primary'],
    }

    color = color_map.get(status, COLORS['primary'])

    return f"""
        QLabel {{
            background-color: {color};
            color: white;
            border-radius: 4px;
            padding: 6px 12px;
            font-weight: 500;
        }}
    """
