# -*- coding: utf-8 -*-
"""
Integration Tests for QGIS YOLOX Plugin

Tests plugin loading, MVC integration, and core workflows.
Run these tests before deploying to QGIS.
"""

import sys
import os

# Add plugin directory to path
plugin_dir = os.path.dirname(os.path.dirname(__file__))
sys.path.insert(0, plugin_dir)


def test_imports():
    """Test that all modules can be imported without errors."""
    print("Testing imports...")

    try:
        # Test utility imports
        from utils import video_utils, yolox_utils, coordinate_utils, geopackage_utils, config_utils
        print("  ✓ Utils imports successful")

        # Test model imports
        from models.video_model import VideoModel
        from models.detection_model import DetectionModel
        from models.geopackage_model import GeoPackageModel
        print("  ✓ Model imports successful")

        # Test controller imports
        from controllers.video_controller import VideoController
        from controllers.detection_controller import DetectionController
        from controllers.export_controller import ExportController
        print("  ✓ Controller imports successful")

        # Test processing provider imports
        from processing_provider.provider import YOLOXProcessingProvider
        from processing_provider.algorithms.video_detection_algorithm import VideoDetectionAlgorithm
        from processing_provider.algorithms.batch_processing_algorithm import BatchProcessingAlgorithm
        print("  ✓ Processing provider imports successful")

        return True

    except Exception as e:
        print(f"  ✗ Import failed: {e}")
        return False


def test_model_instantiation():
    """Test that all models can be instantiated."""
    print("\nTesting model instantiation...")

    try:
        from models.video_model import VideoModel
        from models.detection_model import DetectionModel

        video_model = VideoModel()
        assert hasattr(video_model, 'video_path')
        assert hasattr(video_model, 'cap')
        print("  ✓ VideoModel instantiation successful")

        detection_model = DetectionModel()
        assert hasattr(detection_model, 'detections')
        assert len(detection_model.class_names) == 80  # COCO classes
        print("  ✓ DetectionModel instantiation successful")

        return True

    except Exception as e:
        print(f"  ✗ Model instantiation failed: {e}")
        return False


def test_detection_model_operations():
    """Test DetectionModel basic operations."""
    print("\nTesting DetectionModel operations...")

    try:
        from models.detection_model import DetectionModel

        model = DetectionModel()

        # Test adding detection
        model.add_detection(
            frame_idx=0,
            bbox=[100, 100, 50, 50],
            class_id=0,  # person
            confidence=0.95,
            gps_coords=(127.0, 37.5),
            timestamp=0.0
        )

        assert model.get_detection_count() == 1
        print("  ✓ Add detection successful")

        detections = model.get_detections()
        assert len(detections) == 1
        assert detections[0]['class_name'] == 'person'
        print("  ✓ Get detections successful")

        # Test filtering
        filtered = model.filter_by_confidence(0.9)
        assert len(filtered) == 1
        print("  ✓ Filter by confidence successful")

        # Test statistics
        stats = model.get_class_statistics()
        assert stats[0] == 1  # class_id 0 has 1 detection
        print("  ✓ Get statistics successful")

        # Test clear
        model.clear()
        assert model.get_detection_count() == 0
        print("  ✓ Clear detections successful")

        return True

    except Exception as e:
        print(f"  ✗ DetectionModel operations failed: {e}")
        import traceback
        traceback.print_exc()
        return False


def test_coordinate_utils():
    """Test coordinate transformation utilities."""
    print("\nTesting coordinate utilities...")

    try:
        from utils import coordinate_utils

        # Test GPS interpolation
        gps_track = [
            {'timestamp': 0.0, 'latitude': 37.5, 'longitude': 127.0},
            {'timestamp': 2.0, 'latitude': 37.6, 'longitude': 127.1}
        ]

        result = coordinate_utils.interpolate_gps_position(gps_track, 1.0)
        assert result is not None
        lon, lat = result
        assert 127.0 <= lon <= 127.1
        assert 37.5 <= lat <= 37.6
        print("  ✓ GPS interpolation successful")

        # Test distance calculation
        point1 = (127.0, 37.5)
        point2 = (127.1, 37.6)
        distance = coordinate_utils.calculate_distance(point1, point2)
        assert distance > 0
        print(f"  ✓ Distance calculation successful ({distance:.2f}m)")

        return True

    except Exception as e:
        print(f"  ✗ Coordinate utils test failed: {e}")
        import traceback
        traceback.print_exc()
        return False


def test_yolox_utils():
    """Test YOLOX utility functions."""
    print("\nTesting YOLOX utilities...")

    try:
        from utils import yolox_utils
        import numpy as np

        # Test class names
        class_names = yolox_utils.get_coco_class_names()
        assert len(class_names) == 80
        assert class_names[0] == 'person'
        print("  ✓ Get COCO class names successful")

        # Test image preprocessing
        test_image = np.random.randint(0, 255, (480, 640, 3), dtype=np.uint8)
        preprocessed = yolox_utils.preprocess_image(test_image, (640, 640))
        assert preprocessed.shape == (1, 3, 640, 640)
        print("  ✓ Image preprocessing successful")

        # Test NMS
        bboxes = [[0, 0, 10, 10], [5, 5, 10, 10], [100, 100, 10, 10]]
        scores = [0.9, 0.8, 0.95]
        keep_indices = yolox_utils.nms(bboxes, scores, threshold=0.5)
        assert len(keep_indices) > 0
        print(f"  ✓ NMS successful (kept {len(keep_indices)}/3 boxes)")

        # Test IoU computation
        box1 = [0, 0, 10, 10]
        box2 = [5, 5, 10, 10]
        iou = yolox_utils.compute_iou(box1, box2)
        assert 0 <= iou <= 1
        print(f"  ✓ IoU computation successful (IoU={iou:.3f})")

        return True

    except Exception as e:
        print(f"  ✗ YOLOX utils test failed: {e}")
        import traceback
        traceback.print_exc()
        return False


def test_config_utils():
    """Test configuration management."""
    print("\nTesting config utilities...")

    try:
        from utils import config_utils

        # Test default config
        default_config = config_utils.get_default_config()
        assert 'model_name' in default_config
        assert default_config['model_name'] == 'yolox-s'
        print("  ✓ Get default config successful")

        return True

    except Exception as e:
        print(f"  ✗ Config utils test failed: {e}")
        return False


def test_processing_algorithms():
    """Test Processing algorithm parameter definitions."""
    print("\nTesting Processing algorithms...")

    try:
        from processing_provider.algorithms.video_detection_algorithm import VideoDetectionAlgorithm
        from processing_provider.algorithms.batch_processing_algorithm import BatchProcessingAlgorithm

        # Test VideoDetectionAlgorithm
        video_algo = VideoDetectionAlgorithm()
        assert video_algo.name() == 'video_detection'
        assert video_algo.displayName() == 'Detect Objects in Video'
        assert video_algo.group() == 'Object Detection'
        print("  ✓ VideoDetectionAlgorithm instantiation successful")

        # Test BatchProcessingAlgorithm
        batch_algo = BatchProcessingAlgorithm()
        assert batch_algo.name() == 'batch_video_detection'
        assert batch_algo.displayName() == 'Batch Process Videos'
        print("  ✓ BatchProcessingAlgorithm instantiation successful")

        return True

    except Exception as e:
        print(f"  ✗ Processing algorithms test failed: {e}")
        import traceback
        traceback.print_exc()
        return False


def run_all_tests():
    """Run all integration tests."""
    print("=" * 60)
    print("QGIS YOLOX Plugin - Integration Tests")
    print("=" * 60)

    results = []

    # Run tests
    results.append(("Import Test", test_imports()))
    results.append(("Model Instantiation", test_model_instantiation()))
    results.append(("DetectionModel Operations", test_detection_model_operations()))
    results.append(("Coordinate Utils", test_coordinate_utils()))
    results.append(("YOLOX Utils", test_yolox_utils()))
    results.append(("Config Utils", test_config_utils()))
    results.append(("Processing Algorithms", test_processing_algorithms()))

    # Print summary
    print("\n" + "=" * 60)
    print("TEST SUMMARY")
    print("=" * 60)

    passed = sum(1 for _, result in results if result)
    total = len(results)

    for test_name, result in results:
        status = "✓ PASS" if result else "✗ FAIL"
        print(f"{test_name:.<40} {status}")

    print("=" * 60)
    print(f"Total: {passed}/{total} tests passed ({passed/total*100:.1f}%)")
    print("=" * 60)

    return passed == total


if __name__ == "__main__":
    success = run_all_tests()
    sys.exit(0 if success else 1)
