# -*- coding: utf-8 -*-
"""
Video Detection Algorithm

This module contains the Processing algorithm for detecting objects
in GPS-synchronized videos using YOLOX.
"""

import os
import sys
from qgis.core import (
    QgsProcessingAlgorithm,
    QgsProcessingParameterFile,
    QgsProcessingParameterFileDestination,
    QgsProcessingParameterNumber,
    QgsProcessingParameterEnum,
    QgsProcessingParameterBoolean,
    QgsProcessingException,
    QgsCoordinateReferenceSystem,
    QgsProject
)

# Add parent directories to path for imports
sys.path.append(os.path.dirname(os.path.dirname(os.path.dirname(__file__))))

from models.video_model import VideoModel
from models.detection_model import DetectionModel
from controllers.video_controller import VideoController
from controllers.detection_controller import DetectionController
from controllers.export_controller import ExportController


class VideoDetectionAlgorithm(QgsProcessingAlgorithm):
    """
    Processing algorithm for YOLOX object detection on videos.
    """

    # Parameter names
    INPUT_VIDEO = 'INPUT_VIDEO'
    OUTPUT_GPKG = 'OUTPUT_GPKG'
    FRAME_INTERVAL = 'FRAME_INTERVAL'
    CONFIDENCE_THRESHOLD = 'CONFIDENCE_THRESHOLD'
    MODEL_NAME = 'MODEL_NAME'
    DEVICE = 'DEVICE'
    ADD_TO_MAP = 'ADD_TO_MAP'

    def __init__(self):
        """
        Initialize the algorithm.
        """
        super().__init__()

    def initAlgorithm(self, config=None):
        """
        Define algorithm parameters.

        Args:
            config: Algorithm configuration
        """
        # Input video file
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT_VIDEO,
                'Input video file',
                behavior=QgsProcessingParameterFile.File,
                fileFilter='Video files (*.mp4 *.avi *.mov *.mkv);;All files (*.*)'
            )
        )

        # Frame interval
        self.addParameter(
            QgsProcessingParameterNumber(
                self.FRAME_INTERVAL,
                'Frame interval (process every Nth frame)',
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=30,
                minValue=1
            )
        )

        # Confidence threshold
        self.addParameter(
            QgsProcessingParameterNumber(
                self.CONFIDENCE_THRESHOLD,
                'Confidence threshold',
                type=QgsProcessingParameterNumber.Double,
                defaultValue=0.5,
                minValue=0.0,
                maxValue=1.0
            )
        )

        # Model selection
        model_options = ['yolox-nano', 'yolox-tiny', 'yolox-s', 'yolox-m', 'yolox-l', 'yolox-x']
        self.addParameter(
            QgsProcessingParameterEnum(
                self.MODEL_NAME,
                'YOLOX model',
                options=model_options,
                defaultValue=2  # yolox-s
            )
        )

        # Device selection
        device_options = ['cuda', 'cpu']
        self.addParameter(
            QgsProcessingParameterEnum(
                self.DEVICE,
                'Processing device',
                options=device_options,
                defaultValue=0  # cuda
            )
        )

        # Add to map canvas
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.ADD_TO_MAP,
                'Add result layer to map',
                defaultValue=True
            )
        )

        # Output GeoPackage
        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT_GPKG,
                'Output GeoPackage',
                fileFilter='GeoPackage files (*.gpkg)'
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Execute the algorithm.

        Args:
            parameters: Algorithm parameters
            context: Processing context
            feedback: Feedback object for progress reporting

        Returns:
            dict: Output parameters
        """
        # Get parameters
        video_path = self.parameterAsFile(parameters, self.INPUT_VIDEO, context)
        output_gpkg = self.parameterAsFileOutput(parameters, self.OUTPUT_GPKG, context)
        frame_interval = self.parameterAsInt(parameters, self.FRAME_INTERVAL, context)
        confidence_threshold = self.parameterAsDouble(parameters, self.CONFIDENCE_THRESHOLD, context)

        model_options = ['yolox-nano', 'yolox-tiny', 'yolox-s', 'yolox-m', 'yolox-l', 'yolox-x']
        model_idx = self.parameterAsEnum(parameters, self.MODEL_NAME, context)
        model_name = model_options[model_idx]

        device_options = ['cuda', 'cpu']
        device_idx = self.parameterAsEnum(parameters, self.DEVICE, context)
        device = device_options[device_idx]

        add_to_map = self.parameterAsBoolean(parameters, self.ADD_TO_MAP, context)

        # Validate input
        if not os.path.exists(video_path):
            raise QgsProcessingException(f'Video file not found: {video_path}')

        feedback.pushInfo(f'Processing video: {video_path}')
        feedback.pushInfo(f'Model: {model_name}, Device: {device}')
        feedback.pushInfo(f'Frame interval: {frame_interval}, Confidence: {confidence_threshold}')

        try:
            # Create models
            video_model = VideoModel()
            detection_model = DetectionModel()

            # Create controllers
            video_controller = VideoController(video_model)
            detection_controller = DetectionController(detection_model)

            # Load video
            feedback.setProgress(5)
            feedback.pushInfo('Loading video...')
            if not video_model.load_video(video_path):
                raise QgsProcessingException('Failed to load video')

            # Load YOLOX model
            feedback.setProgress(10)
            feedback.pushInfo(f'Loading YOLOX model: {model_name}...')
            detection_controller.load_yolox_model(
                model_name=model_name,
                device=device
            )

            # Process video
            feedback.setProgress(20)
            feedback.pushInfo('Detecting objects in video...')

            # Get GPS track
            gps_track = video_model.extract_gps_metadata()
            if not gps_track or len(gps_track) == 0:
                feedback.pushWarning('No GPS data found in video. Using (0, 0) for all detections.')

            # Process frames
            detection_controller.process_video(
                video_model=video_model,
                frame_interval=frame_interval,
                confidence_threshold=confidence_threshold,
                gps_track=gps_track
            )

            feedback.setProgress(80)

            # Get detections
            detections = detection_model.get_detections()
            feedback.pushInfo(f'Found {len(detections)} detections')

            if len(detections) == 0:
                feedback.pushWarning('No objects detected in video')
                # Still create empty GeoPackage
                detections = []

            # Export to GeoPackage
            feedback.setProgress(85)
            feedback.pushInfo('Exporting to GeoPackage...')

            # Create export controller
            export_controller = ExportController(context.project().instance())

            # Export detections
            crs = QgsCoordinateReferenceSystem("EPSG:4326")
            export_controller.export_to_geopackage(
                detections=detections,
                output_path=output_gpkg,
                crs=crs,
                layer_name='yolox_detections'
            )

            feedback.setProgress(95)

            # Add to map if requested
            if add_to_map and len(detections) > 0:
                feedback.pushInfo('Adding layer to map...')
                export_controller.add_layer_to_map(output_gpkg, 'yolox_detections')

            # Cleanup
            video_model.close()

            feedback.setProgress(100)
            feedback.pushInfo('Processing complete!')

            # Return outputs
            return {self.OUTPUT_GPKG: output_gpkg}

        except Exception as e:
            raise QgsProcessingException(f'Error during processing: {str(e)}')

    def name(self) -> str:
        """
        Returns the algorithm name.

        Returns:
            str: Algorithm name (used internally)
        """
        return 'video_detection'

    def displayName(self) -> str:
        """
        Returns the translated algorithm name for display.

        Returns:
            str: Display name
        """
        return 'Detect Objects in Video'

    def group(self) -> str:
        """
        Returns the algorithm group name.

        Returns:
            str: Group name
        """
        return 'Object Detection'

    def groupId(self) -> str:
        """
        Returns the algorithm group ID.

        Returns:
            str: Group ID
        """
        return 'object_detection'

    def shortHelpString(self) -> str:
        """
        Returns a short help string for the algorithm.

        Returns:
            str: Help text
        """
        return """
        Detect objects in GPS-synchronized videos using YOLOX deep learning model.

        This algorithm processes video files frame by frame, detects objects using YOLOX,
        and exports the detection results as a GeoPackage file with GPS coordinates.

        Parameters:
        - Input video: Video file with embedded GPS metadata (DJI, GoPro, etc.)
        - Frame interval: Process every Nth frame (higher = faster but less detections)
        - Confidence threshold: Minimum detection confidence (0-1)
        - Model: YOLOX model variant (nano, tiny, s, m, l, x)
        - Device: CUDA (GPU) or CPU processing
        - Add to map: Automatically add result layer to QGIS

        Output:
        - GeoPackage file with detection points including:
          * GPS coordinates (longitude, latitude)
          * Object class and confidence
          * Frame index and timestamp
          * Bounding box coordinates
        """

    def createInstance(self):
        """
        Create a new instance of the algorithm.

        Returns:
            VideoDetectionAlgorithm: New instance
        """
        return VideoDetectionAlgorithm()
