# QGIS YOLOX Plugin - Architecture Documentation

## Table of Contents
1. [Overview](#overview)
2. [MVC Architecture](#mvc-architecture)
3. [Component Diagram](#component-diagram)
4. [Data Flow](#data-flow)
5. [Class Diagrams](#class-diagrams)
6. [Design Patterns](#design-patterns)
7. [Module Dependencies](#module-dependencies)

---

## Overview

The QGIS YOLOX plugin is built using the Model-View-Controller (MVC) architectural pattern to ensure separation of concerns, maintainability, and testability.

### Key Architectural Principles

1. **Separation of Concerns**: Models handle data, Views handle UI, Controllers orchestrate logic
2. **Signal/Slot Communication**: PyQt5 signal/slot pattern for decoupled component interaction
3. **QGIS Integration**: Implements QgsProcessingProvider for Processing framework integration
4. **Modular Design**: Each utility module has a single, well-defined responsibility
5. **Testability**: Models and Controllers are independent of QGIS UI for unit testing

---

## MVC Architecture

### Model Layer (`models/`)

**Purpose**: Manage application data and business state

#### VideoModel
- **Responsibility**: Video file management, GPS extraction, frame retrieval
- **Key Methods**:
  - `load_video(path)`: Load video and extract metadata
  - `extract_gps_metadata()`: Parse GPS track from video
  - `get_frame_at_index(idx)`: Retrieve specific frame
- **Data Storage**: OpenCV VideoCapture object, GPS data list

#### DetectionModel
- **Responsibility**: YOLOX detection results storage and querying
- **Key Methods**:
  - `add_detection()`: Store new detection
  - `filter_by_class/confidence()`: Query detections
  - `get_class_statistics()`: Compute detection statistics
- **Data Structure**: List of detection dictionaries with 80 COCO classes

#### GeoPackageModel
- **Responsibility**: GeoPackage file creation and management
- **Key Methods**:
  - `create_layer()`: Initialize GeoPackage layer
  - `add_feature()`: Add point feature with attributes
  - `save()`: Persist to file
- **Integration**: Uses QGIS QgsVectorLayer and QgsVectorFileWriter

### View Layer (`views/`)

**Purpose**: User interface components (currently skeleton implementations)

#### MainWindow
- **Responsibility**: Main plugin dialog
- **Components**: Video selector, settings, progress bar, export controls
- **Signals**: `video_loaded`, `process_started`, `export_requested`

#### VideoPlayerWidget
- **Responsibility**: Video playback display
- **Features**: Play/pause, frame seeking, detection overlay
- **Signals**: `frame_changed`

#### SettingsDialog
- **Responsibility**: Configuration management UI
- **Settings**: Model selection, thresholds, device selection

### Controller Layer (`controllers/`)

**Purpose**: Orchestrate business logic and coordinate Models

#### VideoController
- **Responsibility**: Video processing workflow
- **Operations**:
  - Load and validate video files
  - Extract frames at specified intervals
  - Manage GPS track retrieval
- **Signals**: `progress_updated`, `video_loaded`, `error_occurred`

#### DetectionController
- **Responsibility**: YOLOX detection execution
- **Operations**:
  - Load YOLOX model (with device selection)
  - Process video frames for object detection
  - Apply confidence thresholds and NMS
- **Signals**: `detection_progress`, `detection_completed`, `frame_processed`

#### ExportController
- **Responsibility**: GeoPackage export and QGIS integration
- **Operations**:
  - Create GeoPackage with detection features
  - Transform coordinates (WGS84 ↔ Project CRS)
  - Add layers to QGIS map canvas
- **Signals**: `export_progress`, `export_completed`, `layer_added`

---

## Component Diagram

```
┌─────────────────────────────────────────────────────────────────┐
│                        QGIS Application                          │
│  ┌───────────────────────────────────────────────────────────┐  │
│  │                 QgisInterface (iface)                       │  │
│  └───────────────────────────────┬─────────────────────────────┘  │
│                                  │                                │
│  ┌───────────────────────────────▼─────────────────────────────┐  │
│  │              YOLOXProcessingProvider                         │  │
│  │  ┌────────────────────────┐  ┌─────────────────────────┐   │  │
│  │  │ VideoDetectionAlgorithm │  │ BatchProcessingAlgorithm │   │  │
│  │  └────────────┬───────────┘  └──────────┬──────────────┘   │  │
│  └───────────────┼────────────────────────────┼─────────────────┘  │
└─────────────────┼────────────────────────────┼─────────────────────┘
                  │                            │
       ┌──────────▼────────────────────────────▼──────────┐
       │            QGIS YOLOX Plugin Core                 │
       │  ┌─────────────────────────────────────────────┐ │
       │  │              MainWindow (View)                │ │
       │  └────┬──────────────────────────────────┬──────┘ │
       │       │                                   │        │
       │  ┌────▼────────┐                  ┌──────▼──────┐ │
       │  │VideoPlayer  │                  │   Settings  │ │
       │  │  Widget     │                  │   Dialog    │ │
       │  └─────────────┘                  └─────────────┘ │
       │                                                    │
       │  ┌──────────────────────────────────────────────┐ │
       │  │          Controllers (Orchestration)          │ │
       │  │  ┌────────────┐ ┌───────────┐ ┌────────────┐ │ │
       │  │  │   Video    │ │ Detection │ │   Export   │ │ │
       │  │  │ Controller │ │Controller │ │ Controller │ │ │
       │  │  └────┬───────┘ └─────┬─────┘ └──────┬─────┘ │ │
       │  └───────┼────────────────┼──────────────┼───────┘ │
       │          │                │              │         │
       │  ┌───────▼────────────────▼──────────────▼───────┐ │
       │  │          Models (Data Management)              │ │
       │  │  ┌────────┐ ┌──────────┐ ┌──────────────────┐ │ │
       │  │  │ Video  │ │Detection │ │   GeoPackage     │ │ │
       │  │  │ Model  │ │  Model   │ │     Model        │ │ │
       │  │  └────┬───┘ └────┬─────┘ └────────┬─────────┘ │ │
       │  └───────┼──────────┼────────────────┼───────────┘ │
       │          │          │                │             │
       │  ┌───────▼──────────▼────────────────▼───────────┐ │
       │  │          Utilities (Reusable Functions)        │ │
       │  │  ┌────────┐ ┌────────┐ ┌──────────────────┐  │ │
       │  │  │ Video  │ │ YOLOX  │ │   Coordinate     │  │ │
       │  │  │ Utils  │ │ Utils  │ │     Utils        │  │ │
       │  │  └────────┘ └────────┘ └──────────────────┘  │ │
       │  │  ┌───────────────┐ ┌─────────────────────┐  │ │
       │  │  │  GeoPackage   │ │     Config          │  │ │
       │  │  │    Utils      │ │      Utils          │  │ │
       │  │  └───────────────┘ └─────────────────────┘  │ │
       │  └───────────────────────────────────────────────┘ │
       └────────────────────────────────────────────────────┘

External Dependencies:
  - OpenCV (cv2): Video processing
  - PyTorch: YOLOX inference
  - exiftool: GPS extraction
  - QGIS API: Geospatial operations
```

---

## Data Flow

### Video Processing Workflow

```
1. User Selection
   └─> MainWindow.on_load_video_clicked()

2. Video Loading
   └─> VideoController.load_video(path)
       └─> VideoModel.load_video(path)
           ├─> video_utils.get_video_info()  [OpenCV]
           └─> video_utils.extract_gps_from_video()  [exiftool]

3. Detection Processing
   └─> DetectionController.process_video()
       ├─> VideoModel.get_frame_at_index(i)  [Loop]
       ├─> DetectionController.detect_objects(frame)
       │   └─> yolox_utils.preprocess_image()
       │       └─> YOLOX Model Inference  [PyTorch]
       │           └─> yolox_utils.postprocess_detections()
       │               └─> yolox_utils.nms()
       └─> DetectionModel.add_detection(...)
           └─> coordinate_utils.interpolate_gps_position()

4. Export to GeoPackage
   └─> ExportController.export_to_geopackage()
       └─> GeoPackageModel.create_layer()
           └─> geopackage_utils.create_geopackage()
               └─> QgsVectorLayer creation
       └─> GeoPackageModel.add_feature()  [Loop detections]
           └─> geopackage_utils.add_detection_feature()
               └─> QgsFeature with QgsPointXY geometry
       └─> GeoPackageModel.save()
           └─> QgsVectorFileWriter.writeAsVectorFormatV3()

5. Map Integration
   └─> ExportController.add_layer_to_map()
       └─> QgsVectorLayer(uri)
           └─> QgsProject.instance().addMapLayer()
```

### GPS Coordinate Flow

```
Video File (MP4/MOV)
  └─> exiftool subprocess
      └─> JSON metadata extraction
          └─> DJI format: Camera:GPSLatitude, Camera:GPSLongitude
          └─> GoPro format: Track1:GPSLatitude, Track1:GPSLongitude
              └─> Parse GPS strings
                  └─> GPS Track: [{timestamp, latitude, longitude, altitude}, ...]
                      └─> Linear Interpolation (coordinate_utils)
                          └─> Frame GPS Position (lon, lat)
                              └─> QgsPointXY (WGS84)
                                  └─> Optional CRS Transform
                                      └─> GeoPackage Feature Geometry
```

---

## Class Diagrams

### Model Layer Classes

```
┌──────────────────────────────┐
│       VideoModel             │
├──────────────────────────────┤
│ - video_path: str            │
│ - cap: cv2.VideoCapture      │
│ - video_info: Dict           │
│ - gps_data: List[Dict]       │
│ - _is_loaded: bool           │
├──────────────────────────────┤
│ + load_video(path): bool     │
│ + extract_gps_metadata()     │
│ + get_frame_at_index(idx)    │
│ + get_frame_at_time(t)       │
│ + get_total_frames(): int    │
│ + get_fps(): float           │
│ + get_video_info(): Dict     │
│ + close()                    │
└──────────────────────────────┘

┌──────────────────────────────┐
│     DetectionModel           │
├──────────────────────────────┤
│ - detections: List[Dict]     │
│ - class_names: List[str]     │
│ - _next_id: int              │
├──────────────────────────────┤
│ + add_detection(...)         │
│ + get_detections(): List     │
│ + get_detection_count(): int │
│ + filter_by_class(ids)       │
│ + filter_by_confidence(t)    │
│ + get_class_statistics()     │
│ + clear()                    │
└──────────────────────────────┘

┌──────────────────────────────┐
│    GeoPackageModel           │
├──────────────────────────────┤
│ - output_path: str           │
│ - crs: QgsCRS                │
│ - layer: QgsVectorLayer      │
│ - features: List[QgsFeature] │
├──────────────────────────────┤
│ + create_layer(name, type)   │
│ + add_feature(lon, lat, attr)│
│ + add_features_batch(list)   │
│ + save(): bool               │
│ + get_layer(): QgsVectorLayer│
└──────────────────────────────┘
```

### Controller Layer Classes

```
┌──────────────────────────────┐
│    VideoController           │
│      (QObject)               │
├──────────────────────────────┤
│ - video_model: VideoModel    │
├──────────────────────────────┤
│ Signals:                     │
│ • progress_updated(int, str) │
│ • video_loaded(bool)         │
│ • error_occurred(str)        │
├──────────────────────────────┤
│ + load_video(path)           │
│ + extract_frames_generator() │
│ + get_gps_track(): List      │
│ + get_video_duration(): float│
│ + validate_video(path): bool │
└──────────────────────────────┘

┌──────────────────────────────┐
│   DetectionController        │
│      (QObject)               │
├──────────────────────────────┤
│ - detection_model: DetModel  │
│ - yolox_model: torch.nn.Mod  │
│ - device: str                │
│ - confidence_threshold: float│
├──────────────────────────────┤
│ Signals:                     │
│ • detection_progress(int,str)│
│ • detection_completed(list)  │
│ • frame_processed(int, int)  │
│ • error_occurred(str)        │
├──────────────────────────────┤
│ + load_yolox_model(name, dev)│
│ + process_video(video_model) │
│ + detect_objects(frame): List│
│ + set_confidence_threshold(t)│
└──────────────────────────────┘

┌──────────────────────────────┐
│    ExportController          │
│      (QObject)               │
├──────────────────────────────┤
│ - iface: QgisInterface       │
├──────────────────────────────┤
│ Signals:                     │
│ • export_progress(int, str)  │
│ • export_completed(str)      │
│ • layer_added(str)           │
│ • error_occurred(str)        │
├──────────────────────────────┤
│ + export_to_geopackage(...)  │
│ + add_layer_to_map(path)     │
│ + transform_coordinates(...)  │
│ + interpolate_gps_coords(...) │
└──────────────────────────────┘
```

---

## Design Patterns

### 1. Model-View-Controller (MVC)
- **Purpose**: Separate data, UI, and logic
- **Implementation**: Distinct `models/`, `views/`, `controllers/` packages
- **Benefits**: Testability, maintainability, reusability

### 2. Observer Pattern (Signal/Slot)
- **Purpose**: Decouple components, enable event-driven architecture
- **Implementation**: PyQt5 `pyqtSignal` and `@pyqtSlot`
- **Example**: `DetectionController.detection_progress` → `MainWindow.update_progress`

### 3. Strategy Pattern (YOLOX Model Selection)
- **Purpose**: Interchangeable algorithms (different YOLOX models)
- **Implementation**: Model name parameter in `load_yolox_model()`
- **Models**: yolox-nano, yolox-tiny, yolox-s, yolox-m, yolox-l, yolox-x

### 4. Factory Pattern (Processing Algorithms)
- **Purpose**: Create algorithm instances
- **Implementation**: `createInstance()` in each QgsProcessingAlgorithm
- **Usage**: QGIS Processing framework dynamically creates algorithm instances

### 5. Singleton Pattern (Configuration)
- **Purpose**: Single configuration instance
- **Implementation**: QSettings with application/organization name
- **Access**: `config_utils.get_default_config()`

### 6. Generator Pattern (Frame Extraction)
- **Purpose**: Memory-efficient iteration over video frames
- **Implementation**: `video_utils.extract_frames()` yields frames
- **Benefits**: Handles large videos without memory overflow

---

## Module Dependencies

### Dependency Graph

```
┌─────────────────────────────────────────────────────────────┐
│  Processing Provider                                         │
│  ├─ VideoDetectionAlgorithm                                  │
│  └─ BatchProcessingAlgorithm                                 │
└───┬─────────────────────────────────────────────────────────┘
    │
    ▼
┌───────────────────────────────────────────────────────────────┐
│  Controllers                                                   │
│  ├─ VideoController  ──────► VideoModel                       │
│  ├─ DetectionController ───► DetectionModel, YOLOX Utils      │
│  └─ ExportController  ──────► GeoPackageModel                 │
└───┬───────────────────────────────────────────────────────────┘
    │
    ▼
┌───────────────────────────────────────────────────────────────┐
│  Models                                                        │
│  ├─ VideoModel  ──────────► video_utils                       │
│  ├─ DetectionModel  ───────► yolox_utils (COCO classes)       │
│  └─ GeoPackageModel  ──────► geopackage_utils                 │
└───┬───────────────────────────────────────────────────────────┘
    │
    ▼
┌───────────────────────────────────────────────────────────────┐
│  Utilities (No internal dependencies)                          │
│  ├─ video_utils  ───────────► OpenCV, exiftool, subprocess    │
│  ├─ yolox_utils  ───────────► PyTorch, NumPy                  │
│  ├─ coordinate_utils  ──────► QGIS API (QgsCoordinateTransform)│
│  ├─ geopackage_utils  ──────► QGIS API (QgsVectorLayer)       │
│  └─ config_utils  ──────────► PyQt5.QtCore.QSettings          │
└───────────────────────────────────────────────────────────────┘
```

### External Dependencies

| Package | Version | Used By | Purpose |
|---------|---------|---------|---------|
| **PyTorch** | ≥2.0.0 | yolox_utils | YOLOX model inference |
| **torchvision** | ≥0.15.0 | yolox_utils | Image transformations |
| **OpenCV** | ≥4.8.0 | video_utils | Video I/O, frame extraction |
| **NumPy** | ≥1.24.0 | All utils | Array operations |
| **Pillow** | ≥10.0.0 | yolox_utils | Image processing |
| **QGIS API** | 3.36.2+ | All modules | Geospatial operations |
| **PyQt5** | (QGIS) | Controllers, Views | GUI and signals |
| **exiftool** | External | video_utils | GPS metadata extraction |

---

## Threading and Async Considerations

### Current Implementation (Synchronous)
- All processing happens in main thread
- Progress updates via PyQt signals
- UI may freeze during long operations

### Future Improvements
1. **QThread for video processing**: Move detection to background thread
2. **QTimer for progress**: Periodic UI updates without blocking
3. **Queue-based processing**: Producer-consumer pattern for frames

### Example Threading Pattern (Future)
```python
class DetectionWorker(QThread):
    progress = pyqtSignal(int, str)
    finished = pyqtSignal(list)

    def run(self):
        # Video processing in background
        for frame in extract_frames():
            detections = detect_objects(frame)
            self.progress.emit(progress, message)
        self.finished.emit(all_detections)

# Usage in Controller:
worker = DetectionWorker(video_model)
worker.progress.connect(self.update_progress)
worker.finished.connect(self.on_detection_complete)
worker.start()  # Non-blocking
```

---

## Error Handling Strategy

### Hierarchical Error Handling

1. **Utility Level**: Raise specific exceptions
   ```python
   if not os.path.exists(video_path):
       raise FileNotFoundError(f"Video not found: {video_path}")
   ```

2. **Model Level**: Catch and log, return None/False
   ```python
   try:
       self.cap = cv2.VideoCapture(path)
   except Exception as e:
       print(f"Error loading video: {e}")
       return False
   ```

3. **Controller Level**: Catch, emit error signal
   ```python
   try:
       detections = self.process_video(video_model)
   except Exception as e:
       self.error_occurred.emit(str(e))
   ```

4. **View Level**: Display user-friendly message
   ```python
   @pyqtSlot(str)
   def on_error(self, message):
       QMessageBox.critical(self, "Error", message)
   ```

---

## Configuration Management

### QSettings Structure

```
[QGIS YOLOX]
├─ model/
│  ├─ name = "yolox-s"
│  ├─ path = "/path/to/weights.pth"
│  └─ device = "cuda"
├─ detection/
│  ├─ confidence_threshold = 0.5
│  ├─ nms_threshold = 0.45
│  └─ frame_interval = 30
├─ export/
│  ├─ default_crs = "EPSG:4326"
│  └─ add_to_map = true
└─ paths/
   ├─ last_video_dir = "C:/Videos"
   └─ last_output_dir = "C:/Output"
```

Access via `config_utils`:
```python
config = get_default_config()
model_name = config['model_name']  # "yolox-s"

save_setting("model_name", "yolox-m")
```

---

## Testing Architecture

### Test Pyramid

```
                    ┌─────────────┐
                    │  Manual E2E │  (User testing in QGIS)
                    └──────┬──────┘
                 ┌─────────▼──────────┐
                 │  Integration Tests  │  (test_plugin_integration.py)
                 └──────────┬──────────┘
          ┌─────────────────▼─────────────────┐
          │       Unit Tests                   │
          │  • test_video_model.py             │
          │  • test_detection.py               │
          │  • test_coordinate_utils.py        │
          └────────────────────────────────────┘
```

### Test Coverage by Layer

| Layer | Test Files | Coverage |
|-------|-----------|----------|
| Utilities | test_coordinate_utils.py | Functions tested |
| Models | test_video_model.py, test_detection.py | Classes tested |
| Controllers | test_plugin_integration.py | Integration tested |
| Processing | test_plugin_integration.py | Algorithm params tested |

---

## Performance Considerations

### Bottlenecks

1. **Video Decoding**: OpenCV I/O (mitigated by frame_interval)
2. **YOLOX Inference**: GPU required for real-time (use CUDA)
3. **GPS Parsing**: exiftool subprocess (one-time cost)
4. **GeoPackage Write**: Batch feature insertion (already optimized)

### Optimization Strategies

| Aspect | Current | Optimized |
|--------|---------|-----------|
| Frame Processing | Every frame | Every 30 frames (1/sec at 30 FPS) |
| YOLOX Model | yolox-m (slow) | yolox-s (balanced) |
| Device | CPU | CUDA (10-50x faster) |
| Feature Insertion | Per-detection | Batch (add_features_batch) |

### Scalability

- **Small video** (< 1000 frames): ~30 seconds on GPU
- **Medium video** (1000-10000 frames): ~5 minutes on GPU
- **Large video** (> 10000 frames): Batch processing overnight recommended

---

## Future Architecture Improvements

### Planned Enhancements

1. **Plugin Architecture**:
   - Allow custom YOLOX models via configuration
   - Support additional object detection frameworks (YOLOv8, DETR)

2. **Caching Layer**:
   - Cache GPS tracks to avoid re-parsing
   - Cache detection results for re-export with different parameters

3. **API Layer**:
   - REST API for remote processing
   - WebSocket for real-time progress updates

4. **Database Backend**:
   - PostgreSQL/PostGIS for large-scale detections
   - Spatial indexing for fast queries

5. **Distributed Processing**:
   - Celery task queue for video processing
   - Multi-GPU support for batch jobs

---

**Document Version**: 1.0.0
**Last Updated**: 2025-12-28
**Maintainer**: QGIS YOLOX Development Team
