# -*- coding: utf-8 -*-
"""
Export Controller

This module contains the ExportController class for handling
GeoPackage export operations.
"""

from typing import List, Dict, Tuple, Optional
import os
import sys
from PyQt5.QtCore import QObject, pyqtSignal
from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsProject,
    QgsVectorLayer
)
from qgis.gui import QgisInterface

sys.path.append(os.path.dirname(os.path.dirname(__file__)))
from models.geopackage_model import GeoPackageModel
from utils import coordinate_utils


class ExportController(QObject):
    """
    Export Controller

    Handles GeoPackage creation, export, and QGIS layer addition.
    """

    # Signals
    export_progress = pyqtSignal(int, str)  # (progress_percent, message)
    export_completed = pyqtSignal(str)  # (output_path)
    layer_added = pyqtSignal(str)  # (layer_name)
    error_occurred = pyqtSignal(str)  # (error_message)

    def __init__(self, iface: QgisInterface):
        """
        Initialize the export controller.

        Args:
            iface (QgisInterface): QGIS interface instance
        """
        super().__init__()
        self.iface = iface

    def export_to_geopackage(self, detections: List[Dict], output_path: str,
                             crs: QgsCoordinateReferenceSystem = None,
                             layer_name: str = 'yolox_detections'):
        """
        Export detection results to GeoPackage.

        Args:
            detections (List[Dict]): List of detection dictionaries
            output_path (str): Path where GeoPackage will be saved
            crs (QgsCoordinateReferenceSystem): Coordinate reference system
            layer_name (str): Name of the layer

        Emits:
            export_progress: Signal with progress updates
            export_completed: Signal when export is complete
            error_occurred: Signal if export fails
        """
        if not detections or len(detections) == 0:
            self.error_occurred.emit("No detections to export")
            return

        try:
            self.export_progress.emit(0, "Creating GeoPackage...")

            # Use WGS84 if no CRS specified
            if crs is None:
                crs = QgsCoordinateReferenceSystem("EPSG:4326")

            # Create GeoPackage model
            gpkg_model = GeoPackageModel(output_path, crs)

            # Create layer
            if not gpkg_model.create_layer(layer_name, 'Point'):
                self.error_occurred.emit("Failed to create GeoPackage layer")
                return

            self.export_progress.emit(20, "Adding detection features...")

            # Add detections as features
            total_detections = len(detections)
            features_added = 0

            for idx, detection in enumerate(detections):
                # Get GPS coordinates
                gps_coords = detection.get('gps_coords', (0.0, 0.0))
                lon, lat = gps_coords

                # Prepare attributes
                attributes = {
                    'id': detection.get('id', idx + 1),
                    'frame_idx': detection['frame_idx'],
                    'timestamp': detection.get('timestamp', 0.0),
                    'class_id': detection['class_id'],
                    'class_name': detection['class_name'],
                    'confidence': detection['confidence'],
                    'bbox_x': detection['bbox'][0],
                    'bbox_y': detection['bbox'][1],
                    'bbox_width': detection['bbox'][2],
                    'bbox_height': detection['bbox'][3]
                }

                # Add feature
                if gpkg_model.add_feature(lon, lat, attributes):
                    features_added += 1

                # Update progress
                progress = 20 + int((idx / total_detections) * 70)
                self.export_progress.emit(
                    progress,
                    f"Added {features_added}/{total_detections} features"
                )

            # Save GeoPackage
            self.export_progress.emit(90, "Saving GeoPackage...")
            if gpkg_model.save():
                self.export_progress.emit(
                    100,
                    f"Export complete: {features_added} features saved"
                )
                self.export_completed.emit(output_path)
            else:
                self.error_occurred.emit("Failed to save GeoPackage")

        except Exception as e:
            error_msg = f"Error exporting to GeoPackage: {str(e)}"
            self.error_occurred.emit(error_msg)

    def create_geopackage_layer(self, detections: List[Dict],
                                output_path: str, crs: QgsCoordinateReferenceSystem,
                                layer_name: str = 'yolox_detections') -> Optional[GeoPackageModel]:
        """
        Create GeoPackage layer with detection features.

        Args:
            detections (List[Dict]): Detection results
            output_path (str): Output file path
            crs (QgsCoordinateReferenceSystem): CRS for the layer
            layer_name (str): Layer name

        Returns:
            GeoPackageModel: Created model or None if failed
        """
        try:
            # Create GeoPackage model
            gpkg_model = GeoPackageModel(output_path, crs)

            # Create layer
            if not gpkg_model.create_layer(layer_name, 'Point'):
                return None

            # Prepare features for batch addition
            features = []
            for detection in detections:
                gps_coords = detection.get('gps_coords', (0.0, 0.0))
                lon, lat = gps_coords

                attributes = {
                    'id': detection.get('id', 0),
                    'frame_idx': detection['frame_idx'],
                    'timestamp': detection.get('timestamp', 0.0),
                    'class_id': detection['class_id'],
                    'class_name': detection['class_name'],
                    'confidence': detection['confidence'],
                    'bbox_x': detection['bbox'][0],
                    'bbox_y': detection['bbox'][1],
                    'bbox_width': detection['bbox'][2],
                    'bbox_height': detection['bbox'][3]
                }

                features.append((lon, lat, attributes))

            # Add features in batch
            gpkg_model.add_features_batch(features)

            return gpkg_model

        except Exception as e:
            print(f"Error creating GeoPackage layer: {e}")
            return None

    def add_layer_to_map(self, geopackage_path: str, layer_name: str = 'yolox_detections'):
        """
        Add GeoPackage layer to QGIS map canvas.

        Args:
            geopackage_path (str): Path to the GeoPackage file
            layer_name (str): Name of the layer to add

        Emits:
            layer_added: Signal when layer is successfully added
            error_occurred: Signal if adding layer fails
        """
        try:
            # Create layer URI
            uri = f"{geopackage_path}|layername={layer_name}"

            # Create vector layer
            layer = QgsVectorLayer(uri, layer_name, "ogr")

            if not layer.isValid():
                self.error_occurred.emit(f"Failed to load layer: {layer_name}")
                return

            # Add layer to QGIS project
            QgsProject.instance().addMapLayer(layer)

            # Zoom to layer extent
            if self.iface:
                self.iface.mapCanvas().setExtent(layer.extent())
                self.iface.mapCanvas().refresh()

            self.layer_added.emit(layer_name)

        except Exception as e:
            error_msg = f"Error adding layer to map: {str(e)}"
            self.error_occurred.emit(error_msg)

    def interpolate_gps_coordinates(self, frame_idx: int, gps_track: List[Dict],
                                    fps: float) -> Optional[Tuple[float, float]]:
        """
        Interpolate GPS coordinates for a given frame.

        Args:
            frame_idx (int): Frame index
            gps_track (List[Dict]): GPS track data
            fps (float): Video frames per second

        Returns:
            tuple: (longitude, latitude) or None if unavailable
        """
        if not gps_track or len(gps_track) == 0:
            return None

        # Calculate timestamp
        timestamp = frame_idx / fps if fps > 0 else 0.0

        # Use coordinate_utils for interpolation
        gps_coords = coordinate_utils.interpolate_gps_position(gps_track, timestamp)

        return gps_coords

    def transform_coordinates(self, lon: float, lat: float,
                             source_crs: QgsCoordinateReferenceSystem,
                             target_crs: QgsCoordinateReferenceSystem) -> Tuple[float, float]:
        """
        Transform coordinates between coordinate reference systems.

        Args:
            lon (float): Longitude
            lat (float): Latitude
            source_crs (QgsCoordinateReferenceSystem): Source CRS
            target_crs (QgsCoordinateReferenceSystem): Target CRS

        Returns:
            tuple: Transformed (x, y) coordinates
        """
        try:
            # Use coordinate_utils for transformation
            point = coordinate_utils.wgs84_to_project_crs(lon, lat, target_crs)
            return (point.x(), point.y())

        except Exception as e:
            print(f"Error transforming coordinates: {e}")
            return (lon, lat)

    def export_detections_with_transform(self, detections: List[Dict],
                                        output_path: str,
                                        target_crs: QgsCoordinateReferenceSystem,
                                        layer_name: str = 'yolox_detections'):
        """
        Export detections with coordinate transformation.

        Args:
            detections (List[Dict]): Detection results
            output_path (str): Output GeoPackage path
            target_crs (QgsCoordinateReferenceSystem): Target CRS
            layer_name (str): Layer name
        """
        try:
            self.export_progress.emit(0, "Transforming coordinates...")

            # Transform coordinates in detections
            transformed_detections = []
            for detection in detections:
                det_copy = detection.copy()

                # Get original GPS coordinates (assumed WGS84)
                lon, lat = detection.get('gps_coords', (0.0, 0.0))

                # Transform if target CRS is not WGS84
                if target_crs.authid() != "EPSG:4326":
                    lon, lat = self.transform_coordinates(
                        lon, lat,
                        QgsCoordinateReferenceSystem("EPSG:4326"),
                        target_crs
                    )

                det_copy['gps_coords'] = (lon, lat)
                transformed_detections.append(det_copy)

            # Export with transformed coordinates
            self.export_to_geopackage(
                transformed_detections,
                output_path,
                target_crs,
                layer_name
            )

        except Exception as e:
            error_msg = f"Error in transform and export: {str(e)}"
            self.error_occurred.emit(error_msg)
