# -*- coding: utf-8 -*-
"""
/***************************************************************************
 QGISYOLOXPlugin
                                 A QGIS plugin
 YOLOX-based object detection for GPS-synchronized videos
 ***************************************************************************/
"""

from qgis.core import QgsApplication
from qgis.gui import QgisInterface
from PyQt5.QtWidgets import QAction, QMessageBox
from PyQt5.QtGui import QIcon
from PyQt5.QtCore import QTranslator, QCoreApplication, qVersion, QSettings

import os.path
import sys


class QGISYOLOXPlugin:
    """QGIS YOLOX Plugin Main Class"""

    def __init__(self, iface: QgisInterface):
        """
        Constructor for the plugin.

        Args:
            iface (QgisInterface): An interface instance that will be passed to this class
                which provides the hook by which you can manipulate the QGIS application
                at run time.
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # Initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # Initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'QGISYOLOXPlugin_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self._translate('&QGIS YOLOX')

        # Will be set to True in initGui()
        self.first_start = True

        # Processing provider
        self.provider = None

    def initGui(self):
        """
        Create the menu entries and toolbar icons inside the QGIS GUI.
        This method is called by QGIS when the plugin is loaded.
        """
        # Create action for plugin
        icon_path = os.path.join(self.plugin_dir, 'icon.png')
        if not os.path.exists(icon_path):
            icon_path = None

        action = self._create_action(
            text=self._translate('QGIS YOLOX - Object Detection'),
            callback=self.run,
            icon_path=icon_path,
            status_tip=self._translate('Run YOLOX object detection on GPS videos'),
            parent=self.iface.mainWindow()
        )

        # Add toolbar button and menu item
        self.iface.addToolBarIcon(action)
        self.iface.addPluginToVectorMenu(self.menu, action)

        self.actions.append(action)

        # Register Processing provider
        self._init_processing()

    def _init_processing(self):
        """Initialize the Processing provider"""
        try:
            # Add plugin directory to path for imports
            if self.plugin_dir not in sys.path:
                sys.path.insert(0, self.plugin_dir)
            from processing_provider.provider import YOLOXProcessingProvider
            self.provider = YOLOXProcessingProvider()
            QgsApplication.processingRegistry().addProvider(self.provider)
        except Exception as e:
            # If Processing provider fails to load, just show warning
            # but don't prevent plugin from loading
            print(f"Warning: Could not load YOLOX Processing provider: {e}")
            import traceback
            traceback.print_exc()

    def unload(self):
        """
        Remove the plugin menu item and icon from QGIS GUI.
        This method is called when the plugin is unloaded.
        """
        # Remove the plugin menu item and icon
        for action in self.actions:
            self.iface.removePluginVectorMenu(self.menu, action)
            self.iface.removeToolBarIcon(action)

        # Remove Processing provider
        if self.provider is not None:
            QgsApplication.processingRegistry().removeProvider(self.provider)

        # Clean up
        self.actions = []

    def run(self):
        """
        Run method that performs all the real work.
        This is called when the user clicks the toolbar button or menu item.
        """
        # Import and show main window
        try:
            # Add plugin directory to path for imports
            if self.plugin_dir not in sys.path:
                sys.path.insert(0, self.plugin_dir)
            from views.main_window import MainWindow

            # Create and show dialog
            dlg = MainWindow(self.iface)
            dlg.exec_()

        except Exception as e:
            # If GUI fails, show info dialog instead
            print(f"Error loading GUI: {e}")
            import traceback
            traceback.print_exc()

            msg = QMessageBox()
            msg.setIcon(QMessageBox.Warning)
            msg.setWindowTitle("QGIS YOLOX Plugin")
            msg.setText("GUI could not be loaded")
            msg.setInformativeText(
                f"Error: {str(e)}\n\n"
                "You can still use the plugin via Processing Toolbox:\n"
                "1. Open Processing Toolbox (Ctrl+Alt+T)\n"
                "2. Find 'YOLOX Object Detection' group\n"
                "3. Run 'Detect Objects in Video' algorithm"
            )
            msg.setStandardButtons(QMessageBox.Ok)
            msg.exec_()

    def _create_action(self, text: str, callback, icon_path: str = None,
                       status_tip: str = None, whats_this: str = None,
                       parent=None) -> QAction:
        """
        Create a QAction for the plugin.

        Args:
            text (str): Text for the action
            callback (function): Function to be called when action is triggered
            icon_path (str): Path to the icon file
            status_tip (str): Status tip for the action
            whats_this (str): What's this text for the action
            parent: Parent widget

        Returns:
            QAction: The created action
        """
        if icon_path and os.path.exists(icon_path):
            icon = QIcon(icon_path)
            action = QAction(icon, text, parent)
        else:
            action = QAction(text, parent)

        action.triggered.connect(callback)

        if status_tip:
            action.setStatusTip(status_tip)

        if whats_this:
            action.setWhatsThis(whats_this)

        return action

    def _translate(self, message: str) -> str:
        """
        Get the translation for a string using Qt translation API.

        Args:
            message (str): String for translation

        Returns:
            str: Translated string
        """
        return QCoreApplication.translate('QGISYOLOXPlugin', message)
