# -*- coding: utf-8 -*-
"""
Main Window for QGIS YOLOX Plugin
Modern Material Design interface
"""

from PyQt5.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QTabWidget,
    QPushButton, QLabel, QLineEdit, QComboBox, QSpinBox,
    QDoubleSpinBox, QProgressBar, QTextEdit, QGroupBox,
    QFileDialog, QMessageBox, QWidget, QCheckBox, QTableWidget,
    QTableWidgetItem, QHeaderView, QFrame, QSlider
)
from PyQt5.QtCore import Qt, pyqtSignal, pyqtSlot, QThread
from PyQt5.QtGui import QFont, QIcon

from qgis.core import QgsCoordinateReferenceSystem, QgsProject
from qgis.gui import QgisInterface

import os
from typing import Optional, Dict, List

from models.video_model import VideoModel
from models.detection_model import DetectionModel
from controllers.video_controller import VideoController
from controllers.detection_controller import DetectionController
from controllers.export_controller import ExportController
from views.styles import MODERN_STYLE, COLORS, get_card_style


class DetectionWorker(QThread):
    """Background worker for video detection"""
    progress = pyqtSignal(int, str)
    finished = pyqtSignal(list)
    error = pyqtSignal(str)

    def __init__(self, video_controller, detection_controller, video_model, settings):
        super().__init__()
        self.video_controller = video_controller
        self.detection_controller = detection_controller
        self.video_model = video_model
        self.settings = settings
        self._is_running = True

    def run(self):
        """Run detection in background"""
        try:
            # Connect progress signals
            self.detection_controller.detection_progress.connect(
                lambda p, m: self.progress.emit(p, m)
            )

            # Get GPS track
            gps_track = self.video_model.gps_data

            # Process video
            self.detection_controller.process_video(
                video_model=self.video_model,
                frame_interval=self.settings['frame_interval'],
                confidence_threshold=self.settings['confidence_threshold'],
                gps_track=gps_track
            )

            # Get results
            detections = self.detection_controller.detection_model.get_detections()
            self.finished.emit(detections)

        except Exception as e:
            self.error.emit(str(e))

    def stop(self):
        """Stop detection"""
        self._is_running = False
        self.quit()


class MainWindow(QDialog):
    """Main window for QGIS YOLOX Plugin"""

    def __init__(self, iface: QgisInterface, parent=None):
        super().__init__(parent)

        self.iface = iface
        self.video_model = VideoModel()
        self.detection_model = DetectionModel()
        self.video_controller = VideoController(self.video_model)
        self.detection_controller = DetectionController(self.detection_model)
        self.export_controller = ExportController(self.iface)

        self.detection_worker: Optional[DetectionWorker] = None
        self.current_detections: List[Dict] = []

        self.setup_ui()
        self.connect_signals()
        self.apply_styles()

    def setup_ui(self):
        """Setup user interface"""
        self.setWindowTitle("QGIS YOLOX - Object Detection")
        self.setMinimumSize(900, 700)

        # Main layout
        layout = QVBoxLayout(self)
        layout.setContentsMargins(20, 20, 20, 20)
        layout.setSpacing(16)

        # Header
        header = self.create_header()
        layout.addWidget(header)

        # Tab widget
        self.tabs = QTabWidget()
        self.tabs.setTabPosition(QTabWidget.North)

        # Create tabs
        self.tab_video = self.create_video_tab()
        self.tab_settings = self.create_settings_tab()
        self.tab_results = self.create_results_tab()

        self.tabs.addTab(self.tab_video, "📹 Video")
        self.tabs.addTab(self.tab_settings, "⚙️ Settings")
        self.tabs.addTab(self.tab_results, "📊 Results")

        layout.addWidget(self.tabs)

        # Status bar
        self.status_bar = self.create_status_bar()
        layout.addWidget(self.status_bar)

        # Button bar
        button_bar = self.create_button_bar()
        layout.addWidget(button_bar)

    def create_header(self) -> QWidget:
        """Create header section"""
        header = QFrame()
        header.setStyleSheet(get_card_style())

        layout = QVBoxLayout(header)

        # Title
        title = QLabel("QGIS YOLOX Object Detection")
        title.setObjectName("headerLabel")
        title.setAlignment(Qt.AlignCenter)
        layout.addWidget(title)

        # Subtitle
        subtitle = QLabel("GPS-Synchronized Video Processing with Deep Learning")
        subtitle.setObjectName("subheaderLabel")
        subtitle.setAlignment(Qt.AlignCenter)
        layout.addWidget(subtitle)

        return header

    def create_video_tab(self) -> QWidget:
        """Create video selection tab"""
        widget = QWidget()
        layout = QVBoxLayout(widget)
        layout.setSpacing(16)

        # Video file selection
        video_group = QGroupBox("Video File")
        video_layout = QVBoxLayout(video_group)

        # File path
        file_layout = QHBoxLayout()
        self.video_path_edit = QLineEdit()
        self.video_path_edit.setPlaceholderText("Select a GPS-synchronized video file...")
        self.video_path_edit.setReadOnly(True)
        file_layout.addWidget(self.video_path_edit)

        self.btn_browse = QPushButton("Browse...")
        self.btn_browse.setMinimumWidth(120)
        file_layout.addWidget(self.btn_browse)

        video_layout.addLayout(file_layout)
        layout.addWidget(video_group)

        # Video information
        info_group = QGroupBox("Video Information")
        info_layout = QVBoxLayout(info_group)

        self.video_info_text = QTextEdit()
        self.video_info_text.setReadOnly(True)
        self.video_info_text.setMaximumHeight(150)
        self.video_info_text.setPlaceholderText("Load a video to see information...")
        info_layout.addWidget(self.video_info_text)

        layout.addWidget(info_group)

        # GPS information
        gps_group = QGroupBox("GPS Data")
        gps_layout = QVBoxLayout(gps_group)

        self.gps_info_text = QTextEdit()
        self.gps_info_text.setReadOnly(True)
        self.gps_info_text.setMaximumHeight(120)
        self.gps_info_text.setPlaceholderText("GPS data will appear here...")
        gps_layout.addWidget(self.gps_info_text)

        layout.addWidget(gps_group)

        layout.addStretch()
        return widget

    def create_settings_tab(self) -> QWidget:
        """Create settings tab"""
        widget = QWidget()
        layout = QVBoxLayout(widget)
        layout.setSpacing(16)

        # Model settings
        model_group = QGroupBox("YOLOX Model")
        model_layout = QVBoxLayout(model_group)

        # Model selection
        model_select_layout = QHBoxLayout()
        model_select_layout.addWidget(QLabel("Model:"))
        self.model_combo = QComboBox()
        self.model_combo.addItems([
            "yolox-nano (Fastest)",
            "yolox-tiny (Very Fast)",
            "yolox-s (Recommended)",
            "yolox-m (Accurate)",
            "yolox-l (Very Accurate)",
            "yolox-x (Best)"
        ])
        self.model_combo.setCurrentIndex(2)  # yolox-s
        model_select_layout.addWidget(self.model_combo, 1)
        model_layout.addLayout(model_select_layout)

        # Device selection
        device_layout = QHBoxLayout()
        device_layout.addWidget(QLabel("Device:"))
        self.device_combo = QComboBox()
        self.device_combo.addItems(["CUDA (GPU)", "CPU"])
        device_layout.addWidget(self.device_combo, 1)
        model_layout.addLayout(device_layout)

        layout.addWidget(model_group)

        # Detection settings
        detection_group = QGroupBox("Detection Parameters")
        detection_layout = QVBoxLayout(detection_group)

        # Frame interval
        interval_layout = QHBoxLayout()
        interval_layout.addWidget(QLabel("Frame Interval:"))
        self.frame_interval_spin = QSpinBox()
        self.frame_interval_spin.setRange(1, 120)
        self.frame_interval_spin.setValue(30)
        self.frame_interval_spin.setSuffix(" frames")
        self.frame_interval_spin.setToolTip("Process every Nth frame (30 = 1 fps at 30 fps video)")
        interval_layout.addWidget(self.frame_interval_spin)
        interval_layout.addStretch()
        detection_layout.addLayout(interval_layout)

        # Confidence threshold
        conf_layout = QVBoxLayout()
        conf_label_layout = QHBoxLayout()
        conf_label_layout.addWidget(QLabel("Confidence Threshold:"))
        self.confidence_value_label = QLabel("0.50")
        self.confidence_value_label.setStyleSheet(f"color: {COLORS['primary']}; font-weight: bold;")
        conf_label_layout.addWidget(self.confidence_value_label)
        conf_label_layout.addStretch()
        conf_layout.addLayout(conf_label_layout)

        self.confidence_slider = QSlider(Qt.Horizontal)
        self.confidence_slider.setRange(10, 95)
        self.confidence_slider.setValue(50)
        self.confidence_slider.setTickPosition(QSlider.TicksBelow)
        self.confidence_slider.setTickInterval(10)
        self.confidence_slider.valueChanged.connect(
            lambda v: self.confidence_value_label.setText(f"{v/100:.2f}")
        )
        conf_layout.addWidget(self.confidence_slider)

        detection_layout.addLayout(conf_layout)

        layout.addWidget(detection_group)

        # Export settings
        export_group = QGroupBox("Export Options")
        export_layout = QVBoxLayout(export_group)

        self.add_to_map_check = QCheckBox("Add result layer to map automatically")
        self.add_to_map_check.setChecked(True)
        export_layout.addWidget(self.add_to_map_check)

        # CRS selection
        crs_layout = QHBoxLayout()
        crs_layout.addWidget(QLabel("Output CRS:"))
        self.crs_combo = QComboBox()
        self.crs_combo.addItems([
            "EPSG:4326 (WGS84)",
            "EPSG:3857 (Web Mercator)",
            "Project CRS"
        ])
        crs_layout.addWidget(self.crs_combo, 1)
        export_layout.addLayout(crs_layout)

        layout.addWidget(export_group)

        layout.addStretch()
        return widget

    def create_results_tab(self) -> QWidget:
        """Create results tab"""
        widget = QWidget()
        layout = QVBoxLayout(widget)
        layout.setSpacing(16)

        # Statistics
        stats_group = QGroupBox("Detection Statistics")
        stats_layout = QVBoxLayout(stats_group)

        self.stats_text = QTextEdit()
        self.stats_text.setReadOnly(True)
        self.stats_text.setMaximumHeight(120)
        self.stats_text.setPlaceholderText("Statistics will appear after detection...")
        stats_layout.addWidget(self.stats_text)

        layout.addWidget(stats_group)

        # Results table
        table_group = QGroupBox("Detection Results")
        table_layout = QVBoxLayout(table_group)

        self.results_table = QTableWidget()
        self.results_table.setColumnCount(6)
        self.results_table.setHorizontalHeaderLabels([
            "ID", "Frame", "Class", "Confidence", "GPS (Lon, Lat)", "Timestamp"
        ])
        self.results_table.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.results_table.setAlternatingRowColors(True)
        self.results_table.setEditTriggers(QTableWidget.NoEditTriggers)
        self.results_table.setSelectionBehavior(QTableWidget.SelectRows)
        table_layout.addWidget(self.results_table)

        layout.addWidget(table_group)

        # Export button
        export_layout = QHBoxLayout()
        export_layout.addStretch()

        self.btn_export = QPushButton("💾 Export to GeoPackage")
        self.btn_export.setMinimumWidth(200)
        self.btn_export.setObjectName("successButton")
        self.btn_export.setEnabled(False)
        export_layout.addWidget(self.btn_export)

        layout.addLayout(export_layout)

        return widget

    def create_status_bar(self) -> QFrame:
        """Create status bar"""
        status_frame = QFrame()
        status_frame.setMaximumHeight(60)
        status_frame.setStyleSheet(get_card_style())

        layout = QVBoxLayout(status_frame)

        # Status label
        self.status_label = QLabel("Ready")
        self.status_label.setStyleSheet(f"color: {COLORS['success']}; font-weight: 500;")
        layout.addWidget(self.status_label)

        # Progress bar
        self.progress_bar = QProgressBar()
        self.progress_bar.setVisible(False)
        self.progress_bar.setTextVisible(True)
        layout.addWidget(self.progress_bar)

        return status_frame

    def create_button_bar(self) -> QWidget:
        """Create button bar"""
        widget = QWidget()
        layout = QHBoxLayout(widget)

        self.btn_process = QPushButton("▶ Start Detection")
        self.btn_process.setMinimumWidth(150)
        self.btn_process.setMinimumHeight(40)
        self.btn_process.setEnabled(False)
        layout.addWidget(self.btn_process)

        self.btn_stop = QPushButton("⏹ Stop")
        self.btn_stop.setMinimumWidth(100)
        self.btn_stop.setMinimumHeight(40)
        self.btn_stop.setObjectName("errorButton")
        self.btn_stop.setVisible(False)
        layout.addWidget(self.btn_stop)

        layout.addStretch()

        self.btn_close = QPushButton("Close")
        self.btn_close.setMinimumWidth(100)
        self.btn_close.setMinimumHeight(40)
        self.btn_close.setObjectName("secondaryButton")
        layout.addWidget(self.btn_close)

        return widget

    def connect_signals(self):
        """Connect signals and slots"""
        # Buttons
        self.btn_browse.clicked.connect(self.on_browse_video)
        self.btn_process.clicked.connect(self.on_start_detection)
        self.btn_stop.clicked.connect(self.on_stop_detection)
        self.btn_export.clicked.connect(self.on_export_results)
        self.btn_close.clicked.connect(self.close)

        # Video controller signals
        self.video_controller.video_loaded.connect(self.on_video_loaded)
        self.video_controller.error_occurred.connect(self.on_error)

        # Export controller signals
        self.export_controller.export_completed.connect(self.on_export_completed)
        self.export_controller.error_occurred.connect(self.on_error)

    def apply_styles(self):
        """Apply modern stylesheet"""
        self.setStyleSheet(MODERN_STYLE)

    @pyqtSlot()
    def on_browse_video(self):
        """Handle browse button click"""
        file_path, _ = QFileDialog.getOpenFileName(
            self,
            "Select Video File",
            "",
            "Video Files (*.mp4 *.avi *.mov *.mkv);;All Files (*.*)"
        )

        if file_path:
            self.video_path_edit.setText(file_path)
            self.status_label.setText("Loading video...")
            self.status_label.setStyleSheet(f"color: {COLORS['warning']}; font-weight: 500;")

            # Load video in controller
            self.video_controller.load_video(file_path)

    @pyqtSlot(bool)
    def on_video_loaded(self, success: bool):
        """Handle video loaded signal"""
        if success:
            self.status_label.setText("Video loaded successfully!")
            self.status_label.setStyleSheet(f"color: {COLORS['success']}; font-weight: 500;")
            self.btn_process.setEnabled(True)

            # Display video info
            info = self.video_model.get_video_info()
            if info:
                info_text = f"""
<b>Duration:</b> {info.get('duration', 0):.2f} seconds<br>
<b>Total Frames:</b> {info.get('total_frames', 0):,}<br>
<b>FPS:</b> {info.get('fps', 0):.2f}<br>
<b>Resolution:</b> {info.get('width', 0)} x {info.get('height', 0)}<br>
<b>Codec:</b> {info.get('codec', 'Unknown')}
                """
                self.video_info_text.setHtml(info_text)

            # Display GPS info
            if self.video_model.gps_data:
                gps_text = f"""
<b>GPS Points:</b> {len(self.video_model.gps_data)}<br>
<b>First Point:</b> {self.video_model.gps_data[0].get('latitude', 0):.6f}, {self.video_model.gps_data[0].get('longitude', 0):.6f}<br>
<b>Last Point:</b> {self.video_model.gps_data[-1].get('latitude', 0):.6f}, {self.video_model.gps_data[-1].get('longitude', 0):.6f}
                """
                self.gps_info_text.setHtml(gps_text)
            else:
                self.gps_info_text.setHtml("<span style='color: orange;'>⚠ No GPS data found in video</span>")

        else:
            self.status_label.setText("Failed to load video")
            self.status_label.setStyleSheet(f"color: {COLORS['error']}; font-weight: 500;")

    @pyqtSlot()
    def on_start_detection(self):
        """Start detection process"""
        # Switch to results tab
        self.tabs.setCurrentIndex(2)

        # Get settings
        settings = {
            'model_name': self.model_combo.currentText().split()[0],  # Extract model name
            'device': 'cuda' if 'CUDA' in self.device_combo.currentText() else 'cpu',
            'frame_interval': self.frame_interval_spin.value(),
            'confidence_threshold': self.confidence_slider.value() / 100.0
        }

        # Load YOLOX model
        self.status_label.setText("Loading YOLOX model...")
        self.detection_controller.load_yolox_model(
            model_name=settings['model_name'],
            device=settings['device']
        )

        # Create and start worker
        self.detection_worker = DetectionWorker(
            self.video_controller,
            self.detection_controller,
            self.video_model,
            settings
        )

        self.detection_worker.progress.connect(self.on_detection_progress)
        self.detection_worker.finished.connect(self.on_detection_finished)
        self.detection_worker.error.connect(self.on_error)

        self.detection_worker.start()

        # Update UI
        self.btn_process.setEnabled(False)
        self.btn_stop.setVisible(True)
        self.progress_bar.setVisible(True)
        self.progress_bar.setValue(0)

    @pyqtSlot(int, str)
    def on_detection_progress(self, progress: int, message: str):
        """Update detection progress"""
        self.progress_bar.setValue(progress)
        self.status_label.setText(message)

    @pyqtSlot(list)
    def on_detection_finished(self, detections: List[Dict]):
        """Handle detection completion"""
        self.current_detections = detections

        self.status_label.setText(f"Detection complete! Found {len(detections)} objects")
        self.status_label.setStyleSheet(f"color: {COLORS['success']}; font-weight: 500;")
        self.progress_bar.setVisible(False)

        self.btn_process.setEnabled(True)
        self.btn_stop.setVisible(False)
        self.btn_export.setEnabled(True)

        # Update statistics
        self.update_statistics(detections)

        # Update results table
        self.update_results_table(detections)

    def update_statistics(self, detections: List[Dict]):
        """Update statistics display"""
        if not detections:
            return

        # Count by class
        class_counts = {}
        for det in detections:
            class_name = det['class_name']
            class_counts[class_name] = class_counts.get(class_name, 0) + 1

        # Build statistics text
        stats_html = f"<b>Total Detections:</b> {len(detections)}<br><br>"
        stats_html += "<b>Top Classes:</b><br>"

        # Sort by count
        sorted_classes = sorted(class_counts.items(), key=lambda x: x[1], reverse=True)
        for class_name, count in sorted_classes[:10]:
            stats_html += f"  • {class_name}: {count}<br>"

        self.stats_text.setHtml(stats_html)

    def update_results_table(self, detections: List[Dict]):
        """Update results table"""
        self.results_table.setRowCount(0)

        for i, det in enumerate(detections[:100]):  # Show first 100
            self.results_table.insertRow(i)

            self.results_table.setItem(i, 0, QTableWidgetItem(str(det['id'])))
            self.results_table.setItem(i, 1, QTableWidgetItem(str(det['frame_idx'])))
            self.results_table.setItem(i, 2, QTableWidgetItem(det['class_name']))
            self.results_table.setItem(i, 3, QTableWidgetItem(f"{det['confidence']:.2f}"))

            lon, lat = det['gps_coords']
            self.results_table.setItem(i, 4, QTableWidgetItem(f"{lon:.6f}, {lat:.6f}"))
            self.results_table.setItem(i, 5, QTableWidgetItem(f"{det['timestamp']:.2f}s"))

        if len(detections) > 100:
            self.results_table.insertRow(100)
            self.results_table.setItem(100, 2, QTableWidgetItem(f"... and {len(detections) - 100} more"))

    @pyqtSlot()
    def on_stop_detection(self):
        """Stop detection process"""
        if self.detection_worker:
            self.detection_worker.stop()
            self.status_label.setText("Detection stopped by user")
            self.status_label.setStyleSheet(f"color: {COLORS['warning']}; font-weight: 500;")

        self.btn_process.setEnabled(True)
        self.btn_stop.setVisible(False)
        self.progress_bar.setVisible(False)

    @pyqtSlot()
    def on_export_results(self):
        """Export results to GeoPackage"""
        if not self.current_detections:
            return

        # Get output path
        output_path, _ = QFileDialog.getSaveFileName(
            self,
            "Save GeoPackage",
            "",
            "GeoPackage (*.gpkg);;All Files (*.*)"
        )

        if not output_path:
            return

        # Get CRS
        crs_text = self.crs_combo.currentText()
        if "4326" in crs_text:
            crs = QgsCoordinateReferenceSystem("EPSG:4326")
        elif "3857" in crs_text:
            crs = QgsCoordinateReferenceSystem("EPSG:3857")
        else:
            crs = QgsProject.instance().crs()

        # Export
        self.status_label.setText("Exporting to GeoPackage...")
        self.export_controller.export_to_geopackage(
            detections=self.current_detections,
            output_path=output_path,
            crs=crs,
            layer_name='yolox_detections'
        )

        # Add to map if requested
        if self.add_to_map_check.isChecked():
            self.export_controller.add_layer_to_map(output_path, 'yolox_detections')

    @pyqtSlot(str)
    def on_export_completed(self, path: str):
        """Handle export completion"""
        self.status_label.setText(f"Export complete: {os.path.basename(path)}")
        self.status_label.setStyleSheet(f"color: {COLORS['success']}; font-weight: 500;")

        QMessageBox.information(
            self,
            "Export Complete",
            f"Results exported successfully to:\n{path}"
        )

    @pyqtSlot(str)
    def on_error(self, message: str):
        """Handle error"""
        self.status_label.setText(f"Error: {message}")
        self.status_label.setStyleSheet(f"color: {COLORS['error']}; font-weight: 500;")

        QMessageBox.critical(self, "Error", message)
