# -*- coding: utf-8 -*-
"""
Coordinate Utilities

This module contains utility functions for coordinate transformations,
GPS interpolation, and geospatial calculations.
"""

from typing import List, Dict, Tuple
import math
from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsProject,
    QgsPointXY,
    QgsGeometry,
    QgsPolygon,
    QgsLineString
)


def wgs84_to_project_crs(lon: float, lat: float,
                         target_crs: QgsCoordinateReferenceSystem) -> QgsPointXY:
    """
    Transform WGS84 coordinates to project CRS.

    Args:
        lon (float): Longitude in WGS84
        lat (float): Latitude in WGS84
        target_crs (QgsCoordinateReferenceSystem): Target CRS

    Returns:
        QgsPointXY: Transformed point in target CRS
    """
    # Create WGS84 CRS
    wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")

    # Create coordinate transform
    transform = QgsCoordinateTransform(wgs84, target_crs, QgsProject.instance())

    # Create and transform point
    point = QgsPointXY(lon, lat)
    transformed_point = transform.transform(point)

    return transformed_point


def project_crs_to_wgs84(x: float, y: float,
                         source_crs: QgsCoordinateReferenceSystem) -> Tuple[float, float]:
    """
    Transform project CRS coordinates to WGS84.

    Args:
        x (float): X coordinate in source CRS
        y (float): Y coordinate in source CRS
        source_crs (QgsCoordinateReferenceSystem): Source CRS

    Returns:
        Tuple[float, float]: (longitude, latitude) in WGS84
    """
    # Create WGS84 CRS
    wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")

    # Create coordinate transform
    transform = QgsCoordinateTransform(source_crs, wgs84, QgsProject.instance())

    # Create and transform point
    point = QgsPointXY(x, y)
    transformed_point = transform.transform(point)

    return (transformed_point.x(), transformed_point.y())


def interpolate_gps_position(gps_track: List[Dict], timestamp: float) -> Tuple[float, float]:
    """
    Interpolate GPS position for a given timestamp.

    Uses linear interpolation between two nearest GPS points.

    Args:
        gps_track (List[Dict]): List of GPS data points with keys:
            - timestamp (float)
            - latitude (float)
            - longitude (float)
        timestamp (float): Target timestamp in seconds

    Returns:
        Tuple[float, float]: Interpolated (longitude, latitude) or None if unavailable
    """
    if not gps_track or len(gps_track) == 0:
        return None

    # If only one GPS point, return it
    if len(gps_track) == 1:
        return (gps_track[0]['longitude'], gps_track[0]['latitude'])

    # Sort GPS track by timestamp
    sorted_track = sorted(gps_track, key=lambda x: x['timestamp'])

    # Find two nearest points
    before = None
    after = None

    for i, point in enumerate(sorted_track):
        if point['timestamp'] <= timestamp:
            before = point
        if point['timestamp'] >= timestamp and after is None:
            after = point
            break

    # If timestamp is before all GPS points
    if before is None and after is not None:
        return (after['longitude'], after['latitude'])

    # If timestamp is after all GPS points
    if after is None and before is not None:
        return (before['longitude'], before['latitude'])

    # If exact match
    if before and before['timestamp'] == timestamp:
        return (before['longitude'], before['latitude'])

    if after and after['timestamp'] == timestamp:
        return (after['longitude'], after['latitude'])

    # Interpolate
    if before and after:
        time_diff = after['timestamp'] - before['timestamp']
        if time_diff == 0:
            return (before['longitude'], before['latitude'])

        ratio = (timestamp - before['timestamp']) / time_diff

        lon = before['longitude'] + (after['longitude'] - before['longitude']) * ratio
        lat = before['latitude'] + (after['latitude'] - before['latitude']) * ratio

        return (lon, lat)

    return None


def calculate_camera_footprint(gps_point: Tuple[float, float], altitude: float,
                               camera_fov: float, camera_angle: float = 90.0) -> QgsGeometry:
    """
    Calculate camera footprint polygon on the ground.

    Args:
        gps_point (Tuple[float, float]): (longitude, latitude) of camera
        altitude (float): Camera altitude in meters
        camera_fov (float): Camera field of view in degrees
        camera_angle (float): Camera pitch angle (90 = straight down)

    Returns:
        QgsGeometry: Polygon geometry representing camera footprint
    """
    lon, lat = gps_point

    # Calculate footprint radius (simplified calculation)
    # For camera pointing straight down (90°), footprint is circular
    fov_rad = math.radians(camera_fov)
    radius_meters = altitude * math.tan(fov_rad / 2)

    # Convert radius from meters to degrees (approximate)
    # 1 degree latitude ≈ 111,111 meters
    # 1 degree longitude ≈ 111,111 * cos(latitude) meters
    radius_lat = radius_meters / 111111.0
    radius_lon = radius_meters / (111111.0 * math.cos(math.radians(lat)))

    # Create square footprint (simplified)
    points = [
        QgsPointXY(lon - radius_lon, lat - radius_lat),
        QgsPointXY(lon + radius_lon, lat - radius_lat),
        QgsPointXY(lon + radius_lon, lat + radius_lat),
        QgsPointXY(lon - radius_lon, lat + radius_lat),
        QgsPointXY(lon - radius_lon, lat - radius_lat)  # Close the polygon
    ]

    polygon = QgsGeometry.fromPolygonXY([points])

    return polygon


def calculate_distance(point1: Tuple[float, float], point2: Tuple[float, float]) -> float:
    """
    Calculate great circle distance between two WGS84 points.

    Uses Haversine formula.

    Args:
        point1 (Tuple[float, float]): (longitude, latitude) of first point
        point2 (Tuple[float, float]): (longitude, latitude) of second point

    Returns:
        float: Distance in meters
    """
    lon1, lat1 = point1
    lon2, lat2 = point2

    # Earth radius in meters
    R = 6371000

    # Convert to radians
    lat1_rad = math.radians(lat1)
    lat2_rad = math.radians(lat2)
    delta_lat = math.radians(lat2 - lat1)
    delta_lon = math.radians(lon2 - lon1)

    # Haversine formula
    a = (math.sin(delta_lat / 2) ** 2 +
         math.cos(lat1_rad) * math.cos(lat2_rad) *
         math.sin(delta_lon / 2) ** 2)

    c = 2 * math.atan2(math.sqrt(a), math.sqrt(1 - a))

    distance = R * c

    return distance


def create_coordinate_transform(source_crs: QgsCoordinateReferenceSystem,
                                target_crs: QgsCoordinateReferenceSystem) -> QgsCoordinateTransform:
    """
    Create a coordinate transform object.

    Args:
        source_crs (QgsCoordinateReferenceSystem): Source CRS
        target_crs (QgsCoordinateReferenceSystem): Target CRS

    Returns:
        QgsCoordinateTransform: Coordinate transform object
    """
    transform = QgsCoordinateTransform(source_crs, target_crs, QgsProject.instance())
    return transform


def dms_to_decimal(degrees: int, minutes: int, seconds: float, direction: str) -> float:
    """
    Convert degrees-minutes-seconds to decimal degrees.

    Args:
        degrees (int): Degrees
        minutes (int): Minutes
        seconds (float): Seconds
        direction (str): Direction ('N', 'S', 'E', 'W')

    Returns:
        float: Decimal degrees
    """
    decimal = float(degrees) + float(minutes) / 60.0 + float(seconds) / 3600.0

    # Apply direction
    if direction in ['S', 'W']:
        decimal = -decimal

    return decimal


def decimal_to_dms(decimal: float) -> Tuple[int, int, float, str]:
    """
    Convert decimal degrees to degrees-minutes-seconds.

    Args:
        decimal (float): Decimal degrees

    Returns:
        Tuple[int, int, float, str]: (degrees, minutes, seconds, direction)
    """
    # Determine direction
    if decimal >= 0:
        direction = 'N'  # or 'E' depending on context
    else:
        direction = 'S'  # or 'W'
        decimal = abs(decimal)

    # Calculate degrees, minutes, seconds
    degrees = int(decimal)
    minutes_decimal = (decimal - degrees) * 60
    minutes = int(minutes_decimal)
    seconds = (minutes_decimal - minutes) * 60

    return (degrees, minutes, seconds, direction)
