# -*- coding: utf-8 -*-
"""
YOLOX Processing Provider

This module contains the Processing provider for YOLOX algorithms.
"""

from qgis.core import QgsProcessingProvider
from PyQt5.QtGui import QIcon
import os
import sys

# Add parent directory to path for imports
sys.path.append(os.path.dirname(os.path.dirname(__file__)))

from processing_provider.algorithms.video_detection_algorithm import VideoDetectionAlgorithm
from processing_provider.algorithms.batch_processing_algorithm import BatchProcessingAlgorithm


class YOLOXProcessingProvider(QgsProcessingProvider):
    """
    Processing provider for YOLOX object detection algorithms.
    """

    def __init__(self):
        """
        Initialize the processing provider.
        """
        super().__init__()
        self.algorithms = []

    def id(self) -> str:
        """
        Returns the unique provider ID.

        Returns:
            str: Provider ID
        """
        return 'yolox'

    def name(self) -> str:
        """
        Returns the provider name for display.

        Returns:
            str: Provider name
        """
        return 'YOLOX Object Detection'

    def icon(self) -> QIcon:
        """
        Returns the provider icon.

        Returns:
            QIcon: Provider icon
        """
        # Try to load icon from resources folder
        icon_path = os.path.join(
            os.path.dirname(os.path.dirname(__file__)),
            'resources',
            'icons',
            'icon.png'
        )

        if os.path.exists(icon_path):
            return QIcon(icon_path)
        else:
            return QgsProcessingProvider.icon(self)

    def loadAlgorithms(self):
        """
        Load all algorithms belonging to this provider.
        """
        # Clear existing algorithms
        self.algorithms = []

        # Add video detection algorithm
        self.addAlgorithm(VideoDetectionAlgorithm())

        # Add batch processing algorithm
        self.addAlgorithm(BatchProcessingAlgorithm())

    def load(self) -> bool:
        """
        Load the provider.

        Returns:
            bool: True if loaded successfully
        """
        self.refreshAlgorithms()
        return True

    def unload(self):
        """
        Unload the provider.
        """
        pass

    def versionInfo(self) -> str:
        """
        Returns version information.

        Returns:
            str: Version string
        """
        return '1.0.0'

    def longName(self) -> str:
        """
        Returns the full provider name.

        Returns:
            str: Long name
        """
        return 'YOLOX Object Detection for GPS-Synchronized Videos'

    def supportsNonFileBasedOutput(self) -> bool:
        """
        Whether the provider supports non-file based outputs.

        Returns:
            bool: True if supported
        """
        return True
