# -*- coding: utf-8 -*-
"""
Video Data Model

This module contains the VideoModel class for managing video data,
GPS metadata extraction, and frame processing.
"""

from typing import List, Dict, Tuple, Optional
import numpy as np
import cv2
import os

import sys
sys.path.append(os.path.dirname(os.path.dirname(__file__)))
from utils import video_utils


class VideoModel:
    """
    Video Data Model

    Manages video file loading, GPS metadata extraction, and frame retrieval.
    """

    def __init__(self):
        """
        Initialize the video data model.
        """
        self.video_path: Optional[str] = None
        self.cap: Optional[cv2.VideoCapture] = None
        self.video_info: Optional[Dict] = None
        self.gps_data: Optional[List[Dict]] = None
        self._is_loaded: bool = False

    def load_video(self, video_path: str) -> bool:
        """
        Load a video file.

        Args:
            video_path (str): Path to the video file

        Returns:
            bool: True if successful, False otherwise
        """
        # Close any previously loaded video
        self.close()

        if not os.path.exists(video_path):
            print(f"Video file not found: {video_path}")
            return False

        try:
            # Get video information using utility function
            self.video_info = video_utils.get_video_info(video_path)

            # Open video capture
            self.cap = cv2.VideoCapture(video_path)

            if not self.cap.isOpened():
                print(f"Failed to open video: {video_path}")
                return False

            self.video_path = video_path
            self._is_loaded = True

            # Extract GPS metadata
            self.gps_data = video_utils.extract_gps_from_video(video_path)

            print(f"Video loaded: {video_path}")
            print(f"  Resolution: {self.get_resolution()}")
            print(f"  FPS: {self.get_fps():.2f}")
            print(f"  Duration: {self.get_duration():.2f}s")
            print(f"  Total frames: {self.get_total_frames()}")
            print(f"  GPS data: {len(self.gps_data)} points" if self.gps_data else "  GPS data: None")

            return True

        except Exception as e:
            print(f"Error loading video: {e}")
            self.close()
            return False

    def extract_gps_metadata(self) -> List[Dict]:
        """
        Extract GPS metadata from the loaded video.

        Returns:
            List[Dict]: List of GPS data dictionaries with keys:
                - timestamp (float): Video timestamp in seconds
                - latitude (float): GPS latitude
                - longitude (float): GPS longitude
                - altitude (float): GPS altitude (optional)
        """
        if not self._is_loaded:
            print("No video loaded")
            return []

        if self.gps_data is None:
            # Try to extract GPS data if not already done
            self.gps_data = video_utils.extract_gps_from_video(self.video_path)

        return self.gps_data if self.gps_data else []

    def get_frame_at_time(self, timestamp: float) -> Optional[np.ndarray]:
        """
        Get a video frame at a specific timestamp.

        Args:
            timestamp (float): Timestamp in seconds

        Returns:
            np.ndarray: Frame image as numpy array (H, W, C), or None if failed
        """
        if not self._is_loaded or self.cap is None:
            print("No video loaded")
            return None

        # Convert timestamp to frame index
        fps = self.get_fps()
        frame_idx = int(timestamp * fps)

        return self.get_frame_at_index(frame_idx)

    def get_frame_at_index(self, frame_idx: int) -> Optional[np.ndarray]:
        """
        Get a video frame at a specific frame index.

        Args:
            frame_idx (int): Frame index

        Returns:
            np.ndarray: Frame image as numpy array (H, W, C), or None if failed
        """
        if not self._is_loaded or self.cap is None:
            print("No video loaded")
            return None

        total_frames = self.get_total_frames()
        if frame_idx < 0 or frame_idx >= total_frames:
            print(f"Frame index {frame_idx} out of range (0-{total_frames-1})")
            return None

        # Set the frame position
        self.cap.set(cv2.CAP_PROP_POS_FRAMES, frame_idx)

        # Read the frame
        ret, frame = self.cap.read()

        if not ret:
            print(f"Failed to read frame at index {frame_idx}")
            return None

        return frame

    def get_total_frames(self) -> int:
        """
        Get the total number of frames in the video.

        Returns:
            int: Total frame count
        """
        if not self._is_loaded or self.video_info is None:
            return 0

        return self.video_info.get('total_frames', 0)

    def get_fps(self) -> float:
        """
        Get the frames per second (FPS) of the video.

        Returns:
            float: FPS value
        """
        if not self._is_loaded or self.video_info is None:
            return 0.0

        return self.video_info.get('fps', 0.0)

    def get_duration(self) -> float:
        """
        Get the total duration of the video in seconds.

        Returns:
            float: Duration in seconds
        """
        if not self._is_loaded or self.video_info is None:
            return 0.0

        return self.video_info.get('duration', 0.0)

    def get_resolution(self) -> Tuple[int, int]:
        """
        Get the video resolution.

        Returns:
            Tuple[int, int]: (width, height)
        """
        if not self._is_loaded or self.video_info is None:
            return (0, 0)

        width = self.video_info.get('width', 0)
        height = self.video_info.get('height', 0)

        return (width, height)

    def is_loaded(self) -> bool:
        """
        Check if a video is currently loaded.

        Returns:
            bool: True if loaded, False otherwise
        """
        return self._is_loaded

    def close(self):
        """
        Close the currently loaded video and release resources.
        """
        if self.cap is not None:
            self.cap.release()
            self.cap = None

        self.video_path = None
        self.video_info = None
        self.gps_data = None
        self._is_loaded = False
