# QGIS YOLOX Plugin - Usage Guide

## Table of Contents
1. [Quick Start](#quick-start)
2. [Basic Workflow](#basic-workflow)
3. [Processing Toolbox Usage](#processing-toolbox-usage)
4. [Advanced Features](#advanced-features)
5. [Best Practices](#best-practices)
6. [Example Workflows](#example-workflows)

---

## Quick Start

### 5-Minute Tutorial

This tutorial will guide you through processing your first GPS-synchronized video.

#### Prerequisites
- QGIS 3.36.2+ installed
- QGIS YOLOX plugin installed and enabled
- A video file with GPS metadata (e.g., from DJI drone or GoPro)

#### Steps

1. **Open QGIS**
   - Start QGIS application

2. **Open Processing Toolbox**
   - Menu: `Processing → Toolbox` (or press `Ctrl+Alt+T`)
   - Find `YOLOX Object Detection` group

3. **Run Video Detection Algorithm**
   - Double-click `Detect Objects in Video`
   - Configure parameters:
     - **Input video file**: Select your GPS-synchronized video
     - **Frame interval**: `30` (process 1 frame per second at 30 FPS)
     - **Confidence threshold**: `0.5` (50% minimum confidence)
     - **YOLOX model**: `yolox-s` (recommended)
     - **Processing device**: `cuda` (if GPU available) or `cpu`
     - **Output GeoPackage**: Choose save location (e.g., `detections.gpkg`)
   - Click `Run`

4. **View Results**
   - Processing will show progress in the dialog
   - When complete, layer automatically added to map
   - Detections shown as points on map
   - Open Attribute Table to see detection details

**Congratulations!** You've processed your first video with QGIS YOLOX.

---

## Basic Workflow

### Workflow Overview

```
1. Prepare Video
   ↓
2. Launch Plugin / Processing Algorithm
   ↓
3. Configure Detection Parameters
   ↓
4. Process Video (with progress monitoring)
   ↓
5. Review Results in QGIS
   ↓
6. Export / Analyze GeoPackage Data
```

### Step-by-Step Guide

#### Step 1: Prepare Your Video

**Requirements**:
- Video format: MP4, MOV, AVI, or MKV
- GPS metadata embedded in video
- Recommended: 30 FPS, 1080p resolution

**Verify GPS Data**:
```bash
exiftool -ee -G3 your_video.mp4 | findstr GPS
```

Expected output should show GPS coordinates:
```
[Camera]         GPSLatitude                     : 37.5 N
[Camera]         GPSLongitude                    : 127.0 E
```

#### Step 2: Choose Your Interface

**Option A: Processing Toolbox** (Recommended for most users)
- User-friendly dialog interface
- Parameter validation
- Progress feedback
- Suitable for single or batch processing

**Option B: Plugin Main Window** (Future - currently skeleton)
- Interactive video player
- Real-time detection preview
- Manual frame selection

**Option C: Python API** (For developers/automation)
- Scriptable workflows
- Integration with other tools
- Batch automation

#### Step 3: Configure Detection Parameters

Understanding each parameter:

| Parameter | Description | Recommended | Range |
|-----------|-------------|-------------|-------|
| **Frame Interval** | Process every Nth frame | 30 | 1-60 |
| **Confidence Threshold** | Minimum detection confidence | 0.5 | 0.0-1.0 |
| **YOLOX Model** | Model size vs. speed tradeoff | yolox-s | nano to x |
| **Device** | GPU (cuda) or CPU | cuda | - |

**Frame Interval Guide**:
- `1`: Every frame (slow, comprehensive)
- `10`: 3 frames/second at 30 FPS (moderate)
- `30`: 1 frame/second at 30 FPS (fast, recommended)
- `60`: 0.5 frames/second (very fast)

**Confidence Threshold Guide**:
- `0.3`: Capture more objects, more false positives
- `0.5`: Balanced (recommended)
- `0.7`: High confidence only, fewer false positives
- `0.9`: Very strict, may miss valid objects

**Model Selection**:
- `yolox-nano`: Fastest, lowest accuracy (use for quick tests)
- `yolox-s`: **Recommended** - balanced speed and accuracy
- `yolox-m/l/x`: Higher accuracy but slower

#### Step 4: Process Video

During processing, monitor:

1. **Progress Bar**: Shows current frame / total frames
2. **Status Messages**:
   - Loading video...
   - Loading YOLOX model...
   - Detecting objects in video...
   - Found X detections
   - Exporting to GeoPackage...
   - Processing complete!

**Estimated Processing Times** (example: 1 minute video, 30 FPS):

| Configuration | Time | Frames Processed |
|---------------|------|------------------|
| yolox-nano + GPU + interval=30 | 10 seconds | 60 frames |
| yolox-s + GPU + interval=30 | 20 seconds | 60 frames |
| yolox-s + CPU + interval=30 | 2-3 minutes | 60 frames |
| yolox-m + GPU + interval=10 | 2-3 minutes | 180 frames |

#### Step 5: Review Results

**Automatic Layer Loading**:
- GeoPackage layer automatically added to QGIS
- Points displayed at GPS locations
- Default styling applied

**View Attributes**:
1. Right-click layer → `Open Attribute Table`
2. See all detection details:
   - `class_name`: Object type (e.g., "person", "car")
   - `confidence`: Detection confidence (0-1)
   - `frame_idx`: Video frame number
   - `timestamp`: Time in video (seconds)
   - `bbox_*`: Bounding box coordinates in video frame

**Spatial Visualization**:
- Zoom to layer extent
- Style by class_name (different colors for different objects)
- Filter by confidence threshold
- Create heatmaps of detection density

#### Step 6: Analyze and Export

**Basic Analysis**:

1. **Count Objects by Class**:
   - Open Attribute Table
   - Click "Open field calculator"
   - Select "Update existing field"
   - Count by class_name

2. **Filter High-Confidence Detections**:
   - Right-click layer → Filter
   - Expression: `"confidence" > 0.8`

3. **Export to Other Formats**:
   - Right-click layer → Export → Save Features As
   - Formats: Shapefile, GeoJSON, CSV, KML, etc.

**Advanced Analysis**:

1. **Spatial Clustering**:
   ```
   Processing → Toolbox → Vector analysis → DBSCAN clustering
   ```

2. **Heatmap Generation**:
   ```
   Processing → Toolbox → Interpolation → Heatmap (Kernel Density Estimation)
   ```

3. **Statistics by Area**:
   ```
   Processing → Toolbox → Vector analysis → Join attributes by location
   ```

---

## Processing Toolbox Usage

### Single Video Detection

**Algorithm**: `YOLOX Object Detection → Detect Objects in Video`

#### Complete Parameter Reference

```
Input Parameters:
├─ Input video file [FILE]
│  └─ Select video with GPS metadata
│
├─ Frame interval (process every Nth frame) [INTEGER]
│  ├─ Default: 30
│  ├─ Min: 1
│  └─ Recommendation: 30 for 30 FPS video = 1 sample/second
│
├─ Confidence threshold [DOUBLE]
│  ├─ Default: 0.5
│  ├─ Range: 0.0 to 1.0
│  └─ Recommendation: 0.5 (balanced), 0.7 (strict)
│
├─ YOLOX model [ENUM]
│  ├─ Options: yolox-nano, yolox-tiny, yolox-s, yolox-m, yolox-l, yolox-x
│  ├─ Default: yolox-s (index 2)
│  └─ Recommendation: yolox-s for most users
│
├─ Processing device [ENUM]
│  ├─ Options: cuda (GPU), cpu
│  ├─ Default: cuda
│  └─ Recommendation: cuda if available
│
└─ Add result layer to map [BOOLEAN]
   ├─ Default: True
   └─ Automatically adds GeoPackage to map canvas

Output:
└─ Output GeoPackage [FILE_OUTPUT]
   └─ Path to save detection results (.gpkg)
```

#### Example: Python Script

```python
import processing

# Process drone video with GPU
result = processing.run("yolox:video_detection", {
    'INPUT_VIDEO': 'C:/Videos/drone_survey.mp4',
    'FRAME_INTERVAL': 30,
    'CONFIDENCE_THRESHOLD': 0.6,
    'MODEL_NAME': 2,  # yolox-s
    'DEVICE': 0,  # cuda
    'ADD_TO_MAP': True,
    'OUTPUT_GPKG': 'C:/Output/survey_detections.gpkg'
})

print(f"Processing complete! Output: {result['OUTPUT_GPKG']}")
```

### Batch Video Processing

**Algorithm**: `YOLOX Object Detection → Batch Process Videos`

Use this for processing multiple videos at once.

#### Parameters

```
Input Folder: Folder containing multiple video files
Output Folder: Where to save all GeoPackages
Merge Results: True = single GeoPackage for all videos
              False = one GeoPackage per video
```

#### Example Workflow

1. **Organize Videos**:
   ```
   C:/Videos/Batch/
   ├─ flight_001.mp4
   ├─ flight_002.mp4
   ├─ flight_003.mp4
   └─ flight_004.mp4
   ```

2. **Run Batch Processing**:
   - Input folder: `C:/Videos/Batch/`
   - Output folder: `C:/Output/Batch/`
   - Merge results: `False` (separate GeoPackages)
   - Other parameters: Same as single video

3. **Result**:
   ```
   C:/Output/Batch/
   ├─ flight_001.gpkg
   ├─ flight_002.gpkg
   ├─ flight_003.gpkg
   └─ flight_004.gpkg
   ```

#### Python Example

```python
import processing

result = processing.run("yolox:batch_video_detection", {
    'INPUT_FOLDER': 'C:/Videos/Survey_2024',
    'OUTPUT_FOLDER': 'C:/Output/Survey_2024',
    'FRAME_INTERVAL': 30,
    'CONFIDENCE_THRESHOLD': 0.7,
    'MODEL_NAME': 2,  # yolox-s
    'DEVICE': 0,  # cuda
    'MERGE_RESULTS': True,  # Single merged GeoPackage
    'ADD_TO_MAP': True
})

print(f"Batch processing complete!")
```

---

## Advanced Features

### Custom YOLOX Models

*Note: This feature requires YOLOX integration completion*

**Steps to use custom-trained models**:

1. Train YOLOX model on custom dataset
2. Export weights to `.pth` file
3. Place in `resources/models/` directory
4. Update configuration to reference custom weights

### Coordinate System Transformation

By default, output is in WGS84 (EPSG:4326). To use different CRS:

**Method 1: Reproject Layer**
```
Right-click layer → Export → Save Features As
CRS: Select target CRS (e.g., EPSG:32652 for UTM Zone 52N)
```

**Method 2: Python API**
```python
from controllers.export_controller import ExportController
from qgis.core import QgsCoordinateReferenceSystem

crs = QgsCoordinateReferenceSystem("EPSG:32652")  # UTM 52N
controller.export_detections_with_transform(
    detections=detections,
    output_path="output_utm.gpkg",
    target_crs=crs
)
```

### Filtering and Querying

#### Filter by Object Class

**Expression Builder** (Right-click layer → Filter):
```sql
"class_name" IN ('person', 'car', 'truck')
```

#### Filter by Confidence
```sql
"confidence" >= 0.8
```

#### Filter by Time Range
```sql
"timestamp" >= 10.0 AND "timestamp" <= 30.0
```

#### Spatial Filter (within area)
```
Processing → Toolbox → Vector selection → Extract by location
```

### Styling Detections

#### Style by Class

1. Right-click layer → Properties → Symbology
2. Change from "Single Symbol" to "Categorized"
3. Column: `class_name`
4. Click "Classify"
5. Customize colors for each class

#### Style by Confidence

1. Symbology → Graduated
2. Column: `confidence`
3. Mode: Equal Interval or Quantile
4. Color ramp: Red (low) to Green (high)

---

## Best Practices

### Video Preparation

✅ **Do**:
- Use videos with consistent GPS signal
- Prefer 30 FPS videos for standard processing
- Keep videos under 10 minutes for faster processing
- Use daylight footage for better detection
- Maintain stable camera angles

❌ **Don't**:
- Use videos without GPS metadata
- Use very high frame rates (120 FPS) unless necessary
- Process very low quality or blurry videos
- Use videos with rapid camera movements

### Parameter Selection

**For Quick Surveys**:
```
Model: yolox-nano or yolox-s
Frame Interval: 60 (0.5 fps)
Confidence: 0.7
Device: cuda
```

**For Detailed Analysis**:
```
Model: yolox-m or yolox-l
Frame Interval: 10-15 (2-3 fps)
Confidence: 0.5
Device: cuda (required)
```

**For Testing/Development**:
```
Model: yolox-nano
Frame Interval: 60
Confidence: 0.5
Device: cpu (acceptable for small videos)
```

### Performance Optimization

1. **Use GPU**: 10-50x faster than CPU
2. **Adjust Frame Interval**: Higher = faster processing
3. **Use Appropriate Model**: Don't use yolox-x unless necessary
4. **Batch Process Overnight**: For large datasets
5. **Monitor Memory**: Close other applications during processing

### Data Management

**Organize Your Outputs**:
```
Project/
├─ Videos/
│  ├─ Raw/
│  └─ Processed/
├─ Detections/
│  ├─ 2024-01-15/
│  ├─ 2024-01-20/
│  └─ Merged/
└─ Analysis/
   ├─ Reports/
   └─ Maps/
```

**Naming Convention**:
```
{date}_{location}_{model}_{threshold}_{interval}.gpkg

Examples:
20240115_site_a_yoloxs_05_30.gpkg
20240120_route_b_yoloxm_07_15.gpkg
```

---

## Example Workflows

### Example 1: Road Infrastructure Survey

**Objective**: Detect traffic signs and lights from drone survey

**Video**: DJI drone video, 5 minutes, 30 FPS

**Steps**:
1. Process with yolox-s, interval=30, confidence=0.6
2. Open attribute table
3. Filter: `"class_name" IN ('traffic light', 'stop sign')`
4. Export filtered results to "traffic_infrastructure.gpkg"
5. Create map showing infrastructure locations

**Expected Results**:
- Processing time: ~1 minute on GPU
- ~150 frames processed
- 20-50 traffic signs/lights detected

### Example 2: Vehicle Counting

**Objective**: Count vehicles on highway from aerial video

**Video**: Fixed-position camera, 10 minutes, 30 FPS

**Steps**:
1. Process with yolox-m, interval=10, confidence=0.7 (higher for accuracy)
2. Filter: `"class_name" IN ('car', 'truck', 'bus', 'motorcycle')`
3. Use Processing Toolbox → DBSCAN Clustering to group nearby detections
4. Count unique clusters (approximately unique vehicles)

**Statistical Analysis**:
```python
# In QGIS Python Console
layer = iface.activeLayer()
features = layer.getFeatures()

vehicle_classes = ['car', 'truck', 'bus', 'motorcycle']
counts = {cls: 0 for cls in vehicle_classes}

for feature in features:
    cls = feature['class_name']
    if cls in counts:
        counts[cls] += 1

print(f"Vehicle counts: {counts}")
```

### Example 3: Construction Site Monitoring

**Objective**: Monitor equipment and personnel on construction site

**Videos**: Multiple daily drone flights over 1 week

**Steps**:
1. Organize videos by date:
   ```
   Videos/
   ├─ 2024-01-15_morning.mp4
   ├─ 2024-01-15_afternoon.mp4
   ├─ 2024-01-16_morning.mp4
   ...
   ```

2. Batch process with merge=False
3. Filter each day for: `"class_name" IN ('person', 'truck', 'car')`
4. Create time-series analysis of personnel and equipment presence
5. Generate heat maps showing activity hotspots

**Temporal Analysis**:
```python
import processing
from datetime import datetime

# Process each day
dates = ['2024-01-15', '2024-01-16', '2024-01-17']

for date in dates:
    processing.run("yolox:batch_video_detection", {
        'INPUT_FOLDER': f'C:/Videos/{date}/',
        'OUTPUT_FOLDER': f'C:/Output/{date}/',
        'MERGE_RESULTS': True,
        'CONFIDENCE_THRESHOLD': 0.6
    })
```

### Example 4: Wildlife Monitoring

**Objective**: Detect and count wildlife from camera trap videos

**Customization Needed**: Train custom YOLOX model on wildlife dataset

**Steps**:
1. Prepare training data (annotated wildlife images)
2. Train YOLOX model (external process)
3. Load custom model weights in plugin
4. Process camera trap videos
5. Analyze spatial distribution and temporal patterns

---

## Keyboard Shortcuts

*Note: UI implementation pending*

Future shortcuts:
- `Ctrl+O`: Load video
- `Ctrl+P`: Start processing
- `Ctrl+E`: Export to GeoPackage
- `Ctrl+S`: Save settings
- `Space`: Play/Pause video
- `Left/Right Arrow`: Previous/Next frame

---

## Tips and Tricks

### Tip 1: Test with Short Video Clip

Before processing long videos:
1. Extract 10-second test clip
2. Process with different parameters
3. Find optimal settings
4. Apply to full video

**Extract clip with ffmpeg**:
```bash
ffmpeg -i input.mp4 -ss 00:00:00 -t 00:00:10 -c copy test_clip.mp4
```

### Tip 2: Use Processing Batch Interface

Process multiple videos with different parameters:

1. Processing Toolbox → Right-click "Detect Objects in Video"
2. Select "Execute as Batch Process"
3. Add multiple rows with different inputs
4. Click "Run"

### Tip 3: Automate with Python

Create custom workflow script:

```python
import processing
import os

video_dir = "C:/Videos/"
output_dir = "C:/Output/"

for video_file in os.listdir(video_dir):
    if video_file.endswith('.mp4'):
        input_path = os.path.join(video_dir, video_file)
        output_path = os.path.join(output_dir, video_file.replace('.mp4', '.gpkg'))

        processing.run("yolox:video_detection", {
            'INPUT_VIDEO': input_path,
            'OUTPUT_GPKG': output_path,
            'FRAME_INTERVAL': 30,
            'CONFIDENCE_THRESHOLD': 0.7
        })

print("All videos processed!")
```

### Tip 4: Monitor GPU Usage

During processing, monitor GPU usage:

**Windows**: Task Manager → Performance → GPU

**Linux**:
```bash
watch -n 1 nvidia-smi
```

This helps identify if GPU is being utilized properly.

---

## Frequently Asked Questions

**Q: Can I process videos without GPS?**
A: Yes, but all detections will have coordinates (0, 0). You can manually georeference later.

**Q: How accurate is the detection?**
A: Depends on model, video quality, and objects. YOLOX-S achieves ~40% mAP on COCO dataset. Real-world accuracy varies.

**Q: Can I detect custom objects?**
A: Yes, but requires training custom YOLOX model on your dataset (advanced).

**Q: What if video has no GPS at certain moments?**
A: Plugin interpolates between available GPS points. Gaps result in linear interpolation.

**Q: Can I run this on Mac?**
A: Yes, plugin works on macOS with QGIS 3.36.2+. GPU support requires compatible Mac (M1/M2 with MPS, or Intel with AMD GPU).

---

**Usage Guide Version**: 1.0.0
**Last Updated**: 2025-12-28
**Author**: QGIS YOLOX Development Team
