# QGIS YOLOX Plugin - API Reference

## Table of Contents
1. [Models API](#models-api)
2. [Controllers API](#controllers-api)
3. [Utilities API](#utilities-api)
4. [Processing Algorithms API](#processing-algorithms-api)
5. [Configuration API](#configuration-api)

---

## Models API

### VideoModel

**File**: `models/video_model.py`

#### Class: `VideoModel`

Manages video file loading, GPS metadata extraction, and frame retrieval.

```python
from models.video_model import VideoModel

model = VideoModel()
```

#### Methods

##### `load_video(video_path: str) -> bool`

Load a video file and extract metadata.

**Parameters**:
- `video_path` (str): Absolute path to video file

**Returns**:
- bool: True if successful, False otherwise

**Example**:
```python
model = VideoModel()
success = model.load_video("C:/Videos/drone_flight.mp4")
if success:
    print(f"Video loaded: {model.get_total_frames()} frames")
```

---

##### `extract_gps_metadata() -> List[Dict]`

Extract GPS track from video metadata using exiftool.

**Returns**:
- List[Dict]: GPS track data with keys:
  - `timestamp` (float): Video timestamp in seconds
  - `latitude` (float): GPS latitude (WGS84)
  - `longitude` (float): GPS longitude (WGS84)
  - `altitude` (float, optional): GPS altitude in meters

**Example**:
```python
gps_track = model.extract_gps_metadata()
for point in gps_track:
    print(f"t={point['timestamp']:.2f}s: ({point['latitude']}, {point['longitude']})")
```

---

##### `get_frame_at_index(frame_idx: int) -> Optional[np.ndarray]`

Retrieve a specific frame by index.

**Parameters**:
- `frame_idx` (int): Frame index (0-based)

**Returns**:
- np.ndarray: Frame as BGR numpy array (H, W, 3), or None if failed

**Example**:
```python
frame = model.get_frame_at_index(100)
if frame is not None:
    print(f"Frame shape: {frame.shape}")  # (1080, 1920, 3)
```

---

##### `get_frame_at_time(timestamp: float) -> Optional[np.ndarray]`

Retrieve frame at specific timestamp.

**Parameters**:
- `timestamp` (float): Timestamp in seconds

**Returns**:
- np.ndarray: Frame as BGR numpy array, or None if failed

---

##### `get_total_frames() -> int`

Get total number of frames in video.

**Returns**:
- int: Total frame count

---

##### `get_fps() -> float`

Get video frames per second.

**Returns**:
- float: FPS value

---

##### `get_video_info() -> Dict`

Get complete video metadata.

**Returns**:
- Dict with keys:
  - `total_frames` (int)
  - `fps` (float)
  - `duration` (float): Duration in seconds
  - `width` (int): Frame width
  - `height` (int): Frame height
  - `codec` (str): Video codec

---

##### `close()`

Release video capture resources.

**Example**:
```python
model.close()  # Always call when done with video
```

---

### DetectionModel

**File**: `models/detection_model.py`

#### Class: `DetectionModel`

Manages YOLOX detection results storage and querying.

```python
from models.detection_model import DetectionModel

model = DetectionModel()
```

#### Attributes

- `class_names` (List[str]): 80 COCO class names
- `detections` (List[Dict]): Stored detections

#### Methods

##### `add_detection(...)`

Add a new detection result.

**Parameters**:
- `frame_idx` (int): Frame index where detected
- `bbox` (List[float]): Bounding box [x, y, width, height]
- `class_id` (int): COCO class ID (0-79)
- `confidence` (float): Detection confidence (0.0-1.0)
- `gps_coords` (Tuple[float, float]): (longitude, latitude) in WGS84
- `timestamp` (float, optional): Video timestamp in seconds

**Example**:
```python
model.add_detection(
    frame_idx=150,
    bbox=[320, 240, 100, 200],  # x, y, w, h
    class_id=0,  # person
    confidence=0.92,
    gps_coords=(127.0276, 37.4979),  # lon, lat
    timestamp=5.0
)
```

---

##### `get_detections() -> List[Dict]`

Get all stored detections.

**Returns**:
- List[Dict]: Each detection has keys:
  - `id` (int): Unique detection ID
  - `frame_idx` (int)
  - `timestamp` (float)
  - `bbox` (List[float]): [x, y, width, height]
  - `class_id` (int)
  - `class_name` (str): Human-readable class name
  - `confidence` (float)
  - `gps_coords` (Tuple[float, float]): (lon, lat)

---

##### `filter_by_class(class_ids: List[int]) -> List[Dict]`

Filter detections by class IDs.

**Parameters**:
- `class_ids` (List[int]): List of COCO class IDs to include

**Returns**:
- List[Dict]: Filtered detections

**Example**:
```python
# Get only person (0), car (2), and truck (7) detections
vehicles = model.filter_by_class([0, 2, 7])
print(f"Found {len(vehicles)} vehicle/person detections")
```

---

##### `filter_by_confidence(threshold: float) -> List[Dict]`

Filter detections by confidence threshold.

**Parameters**:
- `threshold` (float): Minimum confidence (0.0-1.0)

**Returns**:
- List[Dict]: Detections with confidence ≥ threshold

**Example**:
```python
high_conf = model.filter_by_confidence(0.8)  # Only very confident detections
```

---

##### `get_class_statistics() -> List[int]`

Get detection counts per class.

**Returns**:
- List[int]: Length 80, index i = count of class ID i

**Example**:
```python
stats = model.get_class_statistics()
print(f"Persons detected: {stats[0]}")
print(f"Cars detected: {stats[2]}")
```

---

##### `get_detection_count() -> int`

Get total number of detections.

**Returns**:
- int: Total detection count

---

##### `clear()`

Remove all stored detections.

---

### GeoPackageModel

**File**: `models/geopackage_model.py`

#### Class: `GeoPackageModel`

Manages GeoPackage file creation and feature storage.

```python
from models.geopackage_model import GeoPackageModel
from qgis.core import QgsCoordinateReferenceSystem

crs = QgsCoordinateReferenceSystem("EPSG:4326")
model = GeoPackageModel("output.gpkg", crs)
```

#### Methods

##### `create_layer(layer_name: str, geometry_type: str = 'Point') -> bool`

Create a new layer in the GeoPackage.

**Parameters**:
- `layer_name` (str): Name for the layer
- `geometry_type` (str): 'Point', 'LineString', or 'Polygon'

**Returns**:
- bool: True if successful

---

##### `add_feature(lon: float, lat: float, attributes: Dict) -> bool`

Add a point feature to the layer.

**Parameters**:
- `lon` (float): Longitude (WGS84)
- `lat` (float): Latitude (WGS84)
- `attributes` (Dict): Feature attributes

**Returns**:
- bool: True if successful

**Example**:
```python
model.create_layer("detections", "Point")
model.add_feature(
    lon=127.0276,
    lat=37.4979,
    attributes={
        'id': 1,
        'class_name': 'person',
        'confidence': 0.95
    }
)
```

---

##### `add_features_batch(features: List[Tuple]) -> int`

Add multiple features efficiently.

**Parameters**:
- `features` (List[Tuple]): List of (lon, lat, attributes) tuples

**Returns**:
- int: Number of features successfully added

---

##### `save() -> bool`

Save GeoPackage to disk.

**Returns**:
- bool: True if successful

---

## Controllers API

### VideoController

**File**: `controllers/video_controller.py`

#### Class: `VideoController(QObject)`

Orchestrates video processing operations.

```python
from controllers.video_controller import VideoController
from models.video_model import VideoModel

video_model = VideoModel()
controller = VideoController(video_model)
```

#### Signals

- `progress_updated(int, str)`: Emits progress percentage and message
- `video_loaded(bool)`: Emits True when video loaded successfully
- `error_occurred(str)`: Emits error message when error occurs

#### Methods

##### `load_video(video_path: str)`

Load video with progress reporting.

**Parameters**:
- `video_path` (str): Path to video file

**Emits**:
- `progress_updated`: Multiple times during loading
- `video_loaded`: True on success, False on failure
- `error_occurred`: If loading fails

**Example**:
```python
controller.progress_updated.connect(lambda p, m: print(f"{p}%: {m}"))
controller.video_loaded.connect(lambda success: print(f"Loaded: {success}"))
controller.load_video("video.mp4")
```

---

##### `extract_frames_generator(interval: int = 1) -> Generator`

Extract frames as a generator.

**Parameters**:
- `interval` (int): Extract every Nth frame

**Yields**:
- Tuple[int, np.ndarray]: (frame_index, frame_array)

**Example**:
```python
# Process every 30th frame
for frame_idx, frame in controller.extract_frames_generator(interval=30):
    print(f"Processing frame {frame_idx}")
    # Process frame...
```

---

### DetectionController

**File**: `controllers/detection_controller.py`

#### Class: `DetectionController(QObject)`

Manages YOLOX detection execution.

```python
from controllers.detection_controller import DetectionController
from models.detection_model import DetectionModel

detection_model = DetectionModel()
controller = DetectionController(detection_model)
```

#### Signals

- `detection_progress(int, str)`: Progress updates during detection
- `detection_completed(list)`: Emits all detections when complete
- `frame_processed(int, int)`: Emits (current_frame, total_frames)
- `error_occurred(str)`: Error message

#### Methods

##### `load_yolox_model(model_name: str = 'yolox-s', device: str = 'cuda')`

Load YOLOX model for inference.

**Parameters**:
- `model_name` (str): Model variant ('yolox-nano', 'yolox-tiny', 'yolox-s', 'yolox-m', 'yolox-l', 'yolox-x')
- `device` (str): 'cuda' for GPU or 'cpu'

**Emits**:
- `error_occurred`: If model loading fails

**Example**:
```python
controller.load_yolox_model(model_name='yolox-s', device='cuda')
```

---

##### `process_video(video_model, frame_interval=30, confidence_threshold=0.5, gps_track=None)`

Process entire video for object detection.

**Parameters**:
- `video_model` (VideoModel): Video to process
- `frame_interval` (int): Process every Nth frame
- `confidence_threshold` (float): Minimum confidence for detections
- `gps_track` (List[Dict], optional): GPS track data

**Emits**:
- `detection_progress`: Regularly during processing
- `frame_processed`: After each frame
- `detection_completed`: When all frames processed
- `error_occurred`: On failure

**Example**:
```python
controller.detection_completed.connect(lambda dets: print(f"Found {len(dets)} detections"))
controller.process_video(
    video_model=video_model,
    frame_interval=30,
    confidence_threshold=0.7,
    gps_track=gps_track
)
```

---

##### `detect_objects(frame: np.ndarray) -> List[Dict]`

Detect objects in a single frame.

**Parameters**:
- `frame` (np.ndarray): Input frame (BGR format)

**Returns**:
- List[Dict]: Detections with keys 'bbox', 'class_id', 'confidence'

---

### ExportController

**File**: `controllers/export_controller.py`

#### Class: `ExportController(QObject)`

Handles GeoPackage export and QGIS integration.

```python
from controllers.export_controller import ExportController

controller = ExportController(iface)  # iface from QGIS
```

#### Signals

- `export_progress(int, str)`: Export progress updates
- `export_completed(str)`: Emits output path when done
- `layer_added(str)`: Emits layer name when added to map
- `error_occurred(str)`: Error message

#### Methods

##### `export_to_geopackage(detections, output_path, crs=None, layer_name='yolox_detections')`

Export detections to GeoPackage file.

**Parameters**:
- `detections` (List[Dict]): Detection results
- `output_path` (str): Output GeoPackage path
- `crs` (QgsCoordinateReferenceSystem, optional): CRS (defaults to WGS84)
- `layer_name` (str): Layer name in GeoPackage

**Emits**:
- `export_progress`: Multiple times during export
- `export_completed`: With output path on success
- `error_occurred`: On failure

**Example**:
```python
from qgis.core import QgsCoordinateReferenceSystem

crs = QgsCoordinateReferenceSystem("EPSG:4326")
controller.export_to_geopackage(
    detections=detections,
    output_path="C:/Output/results.gpkg",
    crs=crs,
    layer_name="drone_detections"
)
```

---

##### `add_layer_to_map(geopackage_path: str, layer_name: str = 'yolox_detections')`

Add GeoPackage layer to QGIS map canvas.

**Parameters**:
- `geopackage_path` (str): Path to GeoPackage file
- `layer_name` (str): Layer name to add

**Emits**:
- `layer_added`: With layer name on success
- `error_occurred`: On failure

---

## Utilities API

### video_utils

**File**: `utils/video_utils.py`

#### Functions

##### `extract_gps_from_video(video_path: str) -> List[Dict]`

Extract GPS track from video metadata using exiftool.

**Parameters**:
- `video_path` (str): Path to video file

**Returns**:
- List[Dict]: GPS track data

**Supported Formats**:
- DJI drones (Camera:GPSLatitude, Camera:GPSLongitude)
- GoPro cameras (Track1:GPSLatitude)
- Generic GPS metadata

**Example**:
```python
from utils import video_utils

gps_track = video_utils.extract_gps_from_video("drone_video.mp4")
print(f"GPS points: {len(gps_track)}")
```

---

##### `extract_frames(video_path: str, interval: int = 1) -> Generator`

Extract frames from video as generator.

**Parameters**:
- `video_path` (str): Path to video
- `interval` (int): Extract every Nth frame

**Yields**:
- Tuple[int, np.ndarray]: (frame_index, frame)

**Example**:
```python
# Extract every 30th frame
for idx, frame in video_utils.extract_frames("video.mp4", interval=30):
    print(f"Frame {idx}: shape={frame.shape}")
```

---

##### `get_video_info(video_path: str) -> Dict`

Get video metadata.

**Returns**:
- Dict with keys: total_frames, fps, duration, width, height, codec

---

### yolox_utils

**File**: `utils/yolox_utils.py`

#### Functions

##### `get_coco_class_names() -> List[str]`

Get 80 COCO class names.

**Returns**:
- List[str]: Class names

**Example**:
```python
from utils import yolox_utils

classes = yolox_utils.get_coco_class_names()
print(classes[0])  # 'person'
print(classes[2])  # 'car'
print(len(classes))  # 80
```

---

##### `preprocess_image(image: np.ndarray, input_size: Tuple[int, int] = (640, 640)) -> torch.Tensor`

Preprocess image for YOLOX input.

**Parameters**:
- `image` (np.ndarray): Input image (BGR format)
- `input_size` (Tuple[int, int]): Target size (width, height)

**Returns**:
- torch.Tensor: Preprocessed tensor (1, 3, H, W)

**Example**:
```python
import cv2

image = cv2.imread("frame.jpg")
tensor = yolox_utils.preprocess_image(image, (640, 640))
print(tensor.shape)  # torch.Size([1, 3, 640, 640])
```

---

##### `nms(bboxes: List, scores: List, threshold: float = 0.45) -> List[int]`

Non-Maximum Suppression to remove overlapping boxes.

**Parameters**:
- `bboxes` (List[List[float]]): Bounding boxes [[x, y, w, h], ...]
- `scores` (List[float]): Confidence scores
- `threshold` (float): IoU threshold

**Returns**:
- List[int]: Indices of boxes to keep

---

##### `compute_iou(box1: List[float], box2: List[float]) -> float`

Compute Intersection over Union between two boxes.

**Parameters**:
- `box1`, `box2` (List[float]): Boxes as [x, y, width, height]

**Returns**:
- float: IoU value (0.0-1.0)

---

### coordinate_utils

**File**: `utils/coordinate_utils.py`

#### Functions

##### `interpolate_gps_position(gps_track: List[Dict], timestamp: float) -> Tuple[float, float]`

Interpolate GPS position at given timestamp.

**Parameters**:
- `gps_track` (List[Dict]): GPS track data
- `timestamp` (float): Target timestamp in seconds

**Returns**:
- Tuple[float, float]: (longitude, latitude) or None if unavailable

**Example**:
```python
from utils import coordinate_utils

gps_track = [
    {'timestamp': 0.0, 'latitude': 37.5, 'longitude': 127.0},
    {'timestamp': 2.0, 'latitude': 37.6, 'longitude': 127.1}
]

# Interpolate at t=1.0 (midpoint)
lon, lat = coordinate_utils.interpolate_gps_position(gps_track, 1.0)
print(f"Position: ({lon}, {lat})")  # (~127.05, ~37.55)
```

---

##### `calculate_distance(point1: Tuple, point2: Tuple) -> float`

Calculate great circle distance between two GPS points.

**Parameters**:
- `point1`, `point2` (Tuple[float, float]): (longitude, latitude)

**Returns**:
- float: Distance in meters

**Example**:
```python
point1 = (127.0, 37.5)
point2 = (127.1, 37.6)
distance = coordinate_utils.calculate_distance(point1, point2)
print(f"Distance: {distance:.2f} meters")
```

---

##### `wgs84_to_project_crs(lon: float, lat: float, target_crs: QgsCoordinateReferenceSystem) -> QgsPointXY`

Transform WGS84 coordinates to target CRS.

**Parameters**:
- `lon`, `lat` (float): WGS84 coordinates
- `target_crs` (QgsCoordinateReferenceSystem): Target CRS

**Returns**:
- QgsPointXY: Transformed point

---

### geopackage_utils

**File**: `utils/geopackage_utils.py`

#### Functions

##### `create_geopackage(output_path, layer_name, crs, geometry_type='Point') -> QgsVectorLayer`

Create a new GeoPackage file with detection schema.

**Parameters**:
- `output_path` (str): Output file path
- `layer_name` (str): Layer name
- `crs` (QgsCoordinateReferenceSystem): Coordinate system
- `geometry_type` (str): 'Point', 'LineString', 'Polygon'

**Returns**:
- QgsVectorLayer: Created layer

---

##### `add_detection_feature(layer, lon, lat, attributes) -> bool`

Add a detection feature to layer.

**Parameters**:
- `layer` (QgsVectorLayer): Target layer
- `lon`, `lat` (float): GPS coordinates
- `attributes` (Dict): Feature attributes

**Returns**:
- bool: True if successful

---

### config_utils

**File**: `utils/config_utils.py`

#### Functions

##### `get_default_config() -> Dict`

Get default plugin configuration.

**Returns**:
- Dict: Configuration with keys:
  - `model_name`: 'yolox-s'
  - `device`: 'cuda'
  - `confidence_threshold`: 0.5
  - `frame_interval`: 30

---

##### `save_setting(key: str, value: Any)`

Save a configuration value to QSettings.

---

##### `load_setting(key: str, default: Any = None) -> Any`

Load a configuration value from QSettings.

---

## Processing Algorithms API

### VideoDetectionAlgorithm

**File**: `processing_provider/algorithms/video_detection_algorithm.py`

QGIS Processing algorithm for single video detection.

#### Parameters

| Name | Type | Description | Default |
|------|------|-------------|---------|
| INPUT_VIDEO | File | Input video file | Required |
| FRAME_INTERVAL | Integer | Process every Nth frame | 30 |
| CONFIDENCE_THRESHOLD | Double | Minimum confidence (0-1) | 0.5 |
| MODEL_NAME | Enum | YOLOX model variant | 2 (yolox-s) |
| DEVICE | Enum | Processing device | 0 (cuda) |
| ADD_TO_MAP | Boolean | Add result layer to map | True |
| OUTPUT_GPKG | FileDestination | Output GeoPackage path | Required |

#### Usage from Python Console

```python
import processing

result = processing.run("yolox:video_detection", {
    'INPUT_VIDEO': 'C:/Videos/drone.mp4',
    'FRAME_INTERVAL': 30,
    'CONFIDENCE_THRESHOLD': 0.7,
    'MODEL_NAME': 2,  # yolox-s
    'DEVICE': 0,  # cuda
    'ADD_TO_MAP': True,
    'OUTPUT_GPKG': 'C:/Output/detections.gpkg'
})

print(f"Output: {result['OUTPUT_GPKG']}")
```

---

### BatchProcessingAlgorithm

**File**: `processing_provider/algorithms/batch_processing_algorithm.py`

QGIS Processing algorithm for batch processing multiple videos.

#### Parameters

| Name | Type | Description | Default |
|------|------|-------------|---------|
| INPUT_FOLDER | Folder | Input folder with videos | Required |
| OUTPUT_FOLDER | FolderDestination | Output folder | Required |
| FRAME_INTERVAL | Integer | Process every Nth frame | 30 |
| CONFIDENCE_THRESHOLD | Double | Minimum confidence | 0.5 |
| MODEL_NAME | Enum | YOLOX model variant | 2 (yolox-s) |
| DEVICE | Enum | Processing device | 0 (cuda) |
| MERGE_RESULTS | Boolean | Merge all results | False |
| ADD_TO_MAP | Boolean | Add layers to map | True |

#### Usage

```python
import processing

result = processing.run("yolox:batch_video_detection", {
    'INPUT_FOLDER': 'C:/Videos',
    'OUTPUT_FOLDER': 'C:/Output',
    'FRAME_INTERVAL': 30,
    'CONFIDENCE_THRESHOLD': 0.7,
    'MERGE_RESULTS': True,  # Single GeoPackage for all videos
    'ADD_TO_MAP': True
})
```

---

## Configuration API

### QSettings Keys

Configuration is stored using PyQt5 QSettings:

```python
from PyQt5.QtCore import QSettings

settings = QSettings("QGIS", "YOLOX")

# Model settings
settings.setValue("model/name", "yolox-s")
settings.setValue("model/device", "cuda")

# Detection settings
settings.setValue("detection/confidence_threshold", 0.7)
settings.setValue("detection/frame_interval", 30)

# Paths
settings.setValue("paths/last_video_dir", "C:/Videos")
```

---

## Error Handling

All API functions follow these error handling conventions:

1. **Return Values**:
   - Functions return `None`, `False`, or empty list on failure
   - Success returns expected data type

2. **Exceptions**:
   - Utility functions may raise exceptions (FileNotFoundError, ValueError)
   - Controllers catch exceptions and emit `error_occurred` signal

3. **Signal-based Errors**:
   - Controllers emit `error_occurred(str)` signal with error message
   - Connect to this signal to handle errors in UI

**Example Error Handling**:
```python
controller = DetectionController(detection_model)

@pyqtSlot(str)
def handle_error(message):
    print(f"Error: {message}")
    QMessageBox.critical(None, "Detection Error", message)

controller.error_occurred.connect(handle_error)
```

---

**API Version**: 1.0.0
**Last Updated**: 2025-12-28
**QGIS Compatibility**: 3.36.2+
