# -*- coding: utf-8 -*-
"""
GeoPackage Utilities

This module contains utility functions for creating and managing
GeoPackage files and layers.
"""

from typing import Dict, List
from qgis.core import (
    QgsVectorLayer,
    QgsCoordinateReferenceSystem,
    QgsField,
    QgsFeature,
    QgsGeometry,
    QgsPointXY,
    QgsVectorFileWriter,
    QgsWkbTypes
)
from PyQt5.QtCore import QVariant


def create_geopackage(output_path: str, layer_name: str,
                      crs: QgsCoordinateReferenceSystem,
                      geometry_type: str = 'Point') -> QgsVectorLayer:
    """
    Create a new GeoPackage file with a vector layer.

    Args:
        output_path (str): Path where the GeoPackage will be saved
        layer_name (str): Name of the layer
        crs (QgsCoordinateReferenceSystem): Coordinate reference system
        geometry_type (str): Geometry type ('Point', 'LineString', 'Polygon')

    Returns:
        QgsVectorLayer: Created vector layer or None if failed
    """
    # Create layer URI
    uri = f"{geometry_type}?crs={crs.authid()}"

    # Create temporary layer
    layer = QgsVectorLayer(uri, layer_name, "memory")

    if not layer.isValid():
        print(f"Failed to create layer: {layer_name}")
        return None

    # Add fields
    if not add_detection_fields(layer):
        print("Failed to add fields to layer")
        return None

    # Save to GeoPackage
    options = QgsVectorFileWriter.SaveVectorOptions()
    options.driverName = "GPKG"
    options.layerName = layer_name

    error = QgsVectorFileWriter.writeAsVectorFormatV3(
        layer,
        output_path,
        layer.transformContext(),
        options
    )

    if error[0] != QgsVectorFileWriter.NoError:
        print(f"Failed to save GeoPackage: {error[1]}")
        return None

    # Load the saved layer
    gpkg_layer = QgsVectorLayer(f"{output_path}|layername={layer_name}", layer_name, "ogr")

    return gpkg_layer if gpkg_layer.isValid() else None


def add_detection_fields(layer: QgsVectorLayer) -> bool:
    """
    Add attribute fields for detection data to a layer.

    Fields added:
        - id (Integer): Unique ID
        - frame_idx (Integer): Frame index
        - timestamp (Real): Video timestamp
        - class_id (Integer): COCO class ID
        - class_name (String): Class name
        - confidence (Real): Detection confidence
        - bbox_x (Real): Bounding box X
        - bbox_y (Real): Bounding box Y
        - bbox_width (Real): Bounding box width
        - bbox_height (Real): Bounding box height

    Args:
        layer (QgsVectorLayer): Vector layer to add fields to

    Returns:
        bool: True if successful, False otherwise
    """
    fields = [
        QgsField("id", QVariant.Int),
        QgsField("frame_idx", QVariant.Int),
        QgsField("timestamp", QVariant.Double),
        QgsField("class_id", QVariant.Int),
        QgsField("class_name", QVariant.String),
        QgsField("confidence", QVariant.Double),
        QgsField("bbox_x", QVariant.Double),
        QgsField("bbox_y", QVariant.Double),
        QgsField("bbox_width", QVariant.Double),
        QgsField("bbox_height", QVariant.Double)
    ]

    layer.startEditing()
    result = layer.dataProvider().addAttributes(fields)
    layer.updateFields()
    layer.commitChanges()

    return result


def add_detection_feature(layer: QgsVectorLayer, lon: float, lat: float,
                          attributes: Dict) -> bool:
    """
    Add a detection feature to the layer.

    Args:
        layer (QgsVectorLayer): Target layer
        lon (float): Longitude (WGS84 or layer CRS)
        lat (float): Latitude (WGS84 or layer CRS)
        attributes (Dict): Attribute dictionary with field values

    Returns:
        bool: True if successful, False otherwise
    """
    # Create point geometry
    point = create_point_geometry(lon, lat)

    # Create feature
    feature = QgsFeature(layer.fields())
    feature.setGeometry(point)

    # Set attributes
    for field_name, value in attributes.items():
        field_idx = layer.fields().indexOf(field_name)
        if field_idx >= 0:
            feature.setAttribute(field_idx, value)

    # Add feature to layer
    layer.startEditing()
    result = layer.addFeature(feature)
    layer.commitChanges()

    return result


def save_layer(layer: QgsVectorLayer) -> bool:
    """
    Save (commit) changes to a vector layer.

    Args:
        layer (QgsVectorLayer): Layer to save

    Returns:
        bool: True if successful, False otherwise
    """
    if layer.isEditable():
        return layer.commitChanges()
    return True


def create_point_geometry(lon: float, lat: float) -> QgsGeometry:
    """
    Create a point geometry.

    Args:
        lon (float): Longitude
        lat (float): Latitude

    Returns:
        QgsGeometry: Point geometry
    """
    point = QgsPointXY(lon, lat)
    geometry = QgsGeometry.fromPointXY(point)
    return geometry


def load_geopackage_layer(geopackage_path: str, layer_name: str) -> QgsVectorLayer:
    """
    Load a layer from an existing GeoPackage file.

    Args:
        geopackage_path (str): Path to the GeoPackage file
        layer_name (str): Name of the layer to load

    Returns:
        QgsVectorLayer: Loaded layer or None if failed
    """
    uri = f"{geopackage_path}|layername={layer_name}"
    layer = QgsVectorLayer(uri, layer_name, "ogr")

    return layer if layer.isValid() else None


def get_layer_feature_count(layer: QgsVectorLayer) -> int:
    """
    Get the number of features in a layer.

    Args:
        layer (QgsVectorLayer): Vector layer

    Returns:
        int: Feature count
    """
    return layer.featureCount()
