# -*- coding: utf-8 -*-
"""
Configuration Utilities

This module contains utility functions for managing plugin settings
and configuration.
"""

from typing import Dict, Any
from PyQt5.QtCore import QSettings
import json
import os


# Configuration keys
CONFIG_MODEL_PATH = 'qgis_yolox/model_path'
CONFIG_MODEL_NAME = 'qgis_yolox/model_name'
CONFIG_CONFIDENCE_THRESHOLD = 'qgis_yolox/confidence_threshold'
CONFIG_NMS_THRESHOLD = 'qgis_yolox/nms_threshold'
CONFIG_FRAME_INTERVAL = 'qgis_yolox/frame_interval'
CONFIG_DEVICE = 'qgis_yolox/device'
CONFIG_OUTPUT_CRS = 'qgis_yolox/output_crs'


def load_config() -> Dict[str, Any]:
    """
    Load plugin configuration from QSettings.

    Returns:
        Dict[str, Any]: Configuration dictionary
    """
    settings = QSettings()

    config = {
        'model_path': settings.value(CONFIG_MODEL_PATH, ''),
        'model_name': settings.value(CONFIG_MODEL_NAME, 'yolox-s'),
        'confidence_threshold': float(settings.value(CONFIG_CONFIDENCE_THRESHOLD, 0.5)),
        'nms_threshold': float(settings.value(CONFIG_NMS_THRESHOLD, 0.45)),
        'frame_interval': int(settings.value(CONFIG_FRAME_INTERVAL, 30)),
        'device': settings.value(CONFIG_DEVICE, 'cuda'),
        'output_crs': settings.value(CONFIG_OUTPUT_CRS, 'EPSG:4326')
    }

    return config


def save_config(config: Dict[str, Any]):
    """
    Save plugin configuration to QSettings.

    Args:
        config (Dict[str, Any]): Configuration dictionary
    """
    settings = QSettings()

    settings.setValue(CONFIG_MODEL_PATH, config.get('model_path', ''))
    settings.setValue(CONFIG_MODEL_NAME, config.get('model_name', 'yolox-s'))
    settings.setValue(CONFIG_CONFIDENCE_THRESHOLD, config.get('confidence_threshold', 0.5))
    settings.setValue(CONFIG_NMS_THRESHOLD, config.get('nms_threshold', 0.45))
    settings.setValue(CONFIG_FRAME_INTERVAL, config.get('frame_interval', 30))
    settings.setValue(CONFIG_DEVICE, config.get('device', 'cuda'))
    settings.setValue(CONFIG_OUTPUT_CRS, config.get('output_crs', 'EPSG:4326'))


def get_default_config() -> Dict[str, Any]:
    """
    Get default configuration values.

    Returns:
        Dict[str, Any]: Default configuration dictionary
    """
    return {
        'model_path': '',
        'model_name': 'yolox-s',
        'confidence_threshold': 0.5,
        'nms_threshold': 0.45,
        'frame_interval': 30,
        'device': 'cuda',
        'output_crs': 'EPSG:4326'
    }


def get_setting(key: str, default=None) -> Any:
    """
    Get a single setting value.

    Args:
        key (str): Setting key
        default: Default value if key not found

    Returns:
        Any: Setting value
    """
    settings = QSettings()
    return settings.value(key, default)


def set_setting(key: str, value: Any):
    """
    Set a single setting value.

    Args:
        key (str): Setting key
        value (Any): Value to set
    """
    settings = QSettings()
    settings.setValue(key, value)


def reset_to_defaults():
    """
    Reset all settings to default values.
    """
    default_config = get_default_config()
    save_config(default_config)


def export_config(filepath: str):
    """
    Export configuration to a JSON file.

    Args:
        filepath (str): Path to save the configuration
    """
    config = load_config()

    with open(filepath, 'w') as f:
        json.dump(config, f, indent=2)


def import_config(filepath: str):
    """
    Import configuration from a JSON file.

    Args:
        filepath (str): Path to the configuration file
    """
    if not os.path.exists(filepath):
        raise FileNotFoundError(f"Config file not found: {filepath}")

    with open(filepath, 'r') as f:
        config = json.load(f)

    save_config(config)
