# -*- coding: utf-8 -*-
"""
GeoPackage Data Model

This module contains the GeoPackageModel class for managing
GeoPackage layer creation and feature addition.
"""

from typing import Dict, Optional, List, Tuple
import os
import sys

from qgis.core import QgsCoordinateReferenceSystem, QgsGeometry, QgsVectorLayer

sys.path.append(os.path.dirname(os.path.dirname(__file__)))
from utils import geopackage_utils


class GeoPackageModel:
    """
    GeoPackage Data Model

    Manages GeoPackage file creation, layer setup, and feature addition.
    """

    def __init__(self, output_path: str, crs: QgsCoordinateReferenceSystem):
        """
        Initialize the GeoPackage model.

        Args:
            output_path (str): Path where the GeoPackage will be saved
            crs (QgsCoordinateReferenceSystem): Coordinate reference system
        """
        self.output_path = output_path
        self.crs = crs
        self.layer: Optional[QgsVectorLayer] = None
        self.layer_name = "detections"

    def create_layer(self, layer_name: str, geometry_type: str = 'Point') -> bool:
        """
        Create a new vector layer in the GeoPackage.

        Args:
            layer_name (str): Name of the layer
            geometry_type (str): Geometry type ('Point', 'LineString', 'Polygon')

        Returns:
            bool: True if successful, False otherwise
        """
        self.layer_name = layer_name

        try:
            # Use geopackage_utils to create the GeoPackage
            self.layer = geopackage_utils.create_geopackage(
                output_path=self.output_path,
                layer_name=layer_name,
                crs=self.crs,
                geometry_type=geometry_type
            )

            if self.layer is None or not self.layer.isValid():
                print(f"Failed to create GeoPackage layer: {layer_name}")
                return False

            print(f"GeoPackage layer created: {layer_name}")
            print(f"  Path: {self.output_path}")
            print(f"  CRS: {self.crs.authid()}")
            print(f"  Geometry type: {geometry_type}")

            return True

        except Exception as e:
            print(f"Error creating GeoPackage layer: {e}")
            return False

    def add_fields(self) -> bool:
        """
        Add attribute fields to the layer for detection data.

        Fields:
            - id (Integer)
            - frame_idx (Integer)
            - timestamp (Real)
            - class_id (Integer)
            - class_name (String)
            - confidence (Real)
            - bbox_x (Real)
            - bbox_y (Real)
            - bbox_width (Real)
            - bbox_height (Real)

        Returns:
            bool: True if successful, False otherwise
        """
        if self.layer is None:
            print("No layer created yet")
            return False

        # Fields are already added by create_geopackage function
        # This method is kept for API compatibility
        return True

    def add_feature(self, lon: float, lat: float, attributes: Dict) -> bool:
        """
        Add a feature (detection) to the layer.

        Args:
            lon (float): Longitude
            lat (float): Latitude
            attributes (Dict): Attribute dictionary with field values

        Returns:
            bool: True if successful, False otherwise
        """
        if self.layer is None:
            print("No layer created yet")
            return False

        try:
            result = geopackage_utils.add_detection_feature(
                layer=self.layer,
                lon=lon,
                lat=lat,
                attributes=attributes
            )

            return result

        except Exception as e:
            print(f"Error adding feature: {e}")
            return False

    def add_features_batch(self, features: List[Tuple[float, float, Dict]]) -> bool:
        """
        Add multiple features at once.

        Args:
            features (list): List of (lon, lat, attributes) tuples

        Returns:
            bool: True if successful, False otherwise
        """
        if self.layer is None:
            print("No layer created yet")
            return False

        try:
            success_count = 0
            for lon, lat, attributes in features:
                result = self.add_feature(lon, lat, attributes)
                if result:
                    success_count += 1

            print(f"Added {success_count}/{len(features)} features")
            return success_count == len(features)

        except Exception as e:
            print(f"Error adding features batch: {e}")
            return False

    def get_layer(self) -> Optional[QgsVectorLayer]:
        """
        Get the vector layer object.

        Returns:
            QgsVectorLayer: The layer object or None
        """
        return self.layer

    def save(self) -> bool:
        """
        Save the GeoPackage file.

        Returns:
            bool: True if successful, False otherwise
        """
        if self.layer is None:
            print("No layer to save")
            return False

        try:
            result = geopackage_utils.save_layer(self.layer)

            if result:
                print(f"GeoPackage saved: {self.output_path}")
                print(f"  Total features: {self.get_feature_count()}")
            else:
                print("Failed to save GeoPackage")

            return result

        except Exception as e:
            print(f"Error saving GeoPackage: {e}")
            return False

    def get_feature_count(self) -> int:
        """
        Get the number of features in the layer.

        Returns:
            int: Feature count
        """
        if self.layer is None:
            return 0

        return geopackage_utils.get_layer_feature_count(self.layer)

    def is_valid(self) -> bool:
        """
        Check if the layer is valid.

        Returns:
            bool: True if layer exists and is valid
        """
        return self.layer is not None and self.layer.isValid()

    def load_existing_geopackage(self, layer_name: str) -> bool:
        """
        Load an existing GeoPackage layer.

        Args:
            layer_name (str): Name of the layer to load

        Returns:
            bool: True if successful, False otherwise
        """
        try:
            self.layer = geopackage_utils.load_geopackage_layer(
                geopackage_path=self.output_path,
                layer_name=layer_name
            )

            if self.layer is None:
                print(f"Failed to load GeoPackage layer: {layer_name}")
                return False

            self.layer_name = layer_name
            print(f"Loaded existing GeoPackage: {layer_name}")
            print(f"  Features: {self.get_feature_count()}")

            return True

        except Exception as e:
            print(f"Error loading GeoPackage: {e}")
            return False
