# -*- coding: utf-8 -*-
"""
Video Controller

This module contains the VideoController class for handling
video processing operations.
"""

from typing import List, Dict, Tuple, Generator
import numpy as np
import os
import sys
from PyQt5.QtCore import QObject, pyqtSignal

sys.path.append(os.path.dirname(os.path.dirname(__file__)))
from models.video_model import VideoModel
from utils import video_utils


class VideoController(QObject):
    """
    Video Controller

    Handles video loading, frame extraction, and GPS track retrieval.
    """

    # Signals
    progress_updated = pyqtSignal(int, str)  # (progress_percent, message)
    video_loaded = pyqtSignal(bool)  # (success)
    error_occurred = pyqtSignal(str)  # (error_message)

    def __init__(self, video_model: VideoModel):
        """
        Initialize the video controller.

        Args:
            video_model (VideoModel): Video data model instance
        """
        super().__init__()
        self.video_model = video_model

    def load_video(self, video_path: str):
        """
        Load a video file.

        Args:
            video_path (str): Path to the video file

        Emits:
            video_loaded: Signal with success status
            error_occurred: Signal if an error occurs
        """
        try:
            self.progress_updated.emit(0, "Loading video...")

            # Use video model to load the video
            success = self.video_model.load_video(video_path)

            if success:
                self.progress_updated.emit(100, "Video loaded successfully")
                self.video_loaded.emit(True)
            else:
                error_msg = f"Failed to load video: {video_path}"
                self.error_occurred.emit(error_msg)
                self.video_loaded.emit(False)

        except Exception as e:
            error_msg = f"Error loading video: {str(e)}"
            self.error_occurred.emit(error_msg)
            self.video_loaded.emit(False)

    def extract_frames(self, interval: int = 1) -> List[Tuple[int, np.ndarray]]:
        """
        Extract frames from the video at specified interval.

        Args:
            interval (int): Frame interval (extract every Nth frame)

        Returns:
            List[Tuple[int, np.ndarray]]: List of (frame_index, frame_image) tuples

        Emits:
            progress_updated: Signal with progress updates
        """
        if not self.video_model.is_loaded():
            self.error_occurred.emit("No video loaded")
            return []

        frames = []
        total_frames = self.video_model.get_total_frames()

        self.progress_updated.emit(0, "Extracting frames...")

        try:
            # Extract frames at specified interval
            for frame_idx in range(0, total_frames, interval):
                frame = self.video_model.get_frame_at_index(frame_idx)

                if frame is not None:
                    frames.append((frame_idx, frame))

                # Update progress
                progress = int((frame_idx / total_frames) * 100)
                self.progress_updated.emit(progress, f"Extracting frame {frame_idx}/{total_frames}")

            self.progress_updated.emit(100, f"Extracted {len(frames)} frames")
            return frames

        except Exception as e:
            error_msg = f"Error extracting frames: {str(e)}"
            self.error_occurred.emit(error_msg)
            return frames

    def extract_frames_generator(self, interval: int = 1) -> Generator[Tuple[int, np.ndarray], None, None]:
        """
        Extract frames as a generator (memory efficient).

        Args:
            interval (int): Frame interval

        Yields:
            Tuple[int, np.ndarray]: (frame_index, frame_image)
        """
        if not self.video_model.is_loaded():
            self.error_occurred.emit("No video loaded")
            return

        total_frames = self.video_model.get_total_frames()

        try:
            # Use video_utils generator for efficient frame extraction
            for frame_idx, frame in video_utils.extract_frames(self.video_model.video_path, interval):
                # Update progress
                progress = int((frame_idx / total_frames) * 100)
                self.progress_updated.emit(progress, f"Processing frame {frame_idx}/{total_frames}")

                yield (frame_idx, frame)

        except Exception as e:
            error_msg = f"Error in frame extraction generator: {str(e)}"
            self.error_occurred.emit(error_msg)

    def get_gps_track(self) -> List[Dict]:
        """
        Get GPS track from video metadata.

        Returns:
            List[Dict]: List of GPS data points

        Emits:
            progress_updated: Signal with progress updates
            error_occurred: Signal if GPS extraction fails
        """
        if not self.video_model.is_loaded():
            self.error_occurred.emit("No video loaded")
            return []

        try:
            self.progress_updated.emit(0, "Extracting GPS metadata...")

            gps_data = self.video_model.extract_gps_metadata()

            if gps_data and len(gps_data) > 0:
                self.progress_updated.emit(100, f"Found {len(gps_data)} GPS points")
            else:
                self.progress_updated.emit(100, "No GPS data found in video")

            return gps_data

        except Exception as e:
            error_msg = f"Error extracting GPS data: {str(e)}"
            self.error_occurred.emit(error_msg)
            return []

    def get_video_info(self) -> Dict:
        """
        Get video information.

        Returns:
            Dict: Video information including:
                - fps (float)
                - total_frames (int)
                - duration (float)
                - resolution (Tuple[int, int])
        """
        if not self.video_model.is_loaded():
            return {
                'fps': 0.0,
                'total_frames': 0,
                'duration': 0.0,
                'resolution': (0, 0)
            }

        return {
            'fps': self.video_model.get_fps(),
            'total_frames': self.video_model.get_total_frames(),
            'duration': self.video_model.get_duration(),
            'resolution': self.video_model.get_resolution()
        }

    def is_video_loaded(self) -> bool:
        """
        Check if a video is currently loaded.

        Returns:
            bool: True if video is loaded
        """
        return self.video_model.is_loaded()

    def close_video(self):
        """
        Close the currently loaded video.
        """
        self.video_model.close()
        self.progress_updated.emit(0, "Video closed")
