/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.osmosis.apidb.common;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.openstreetmap.osmosis.apidb.common.IdentityValueLoader;
import org.openstreetmap.osmosis.apidb.common.MysqlIdentityValueLoader;
import org.openstreetmap.osmosis.apidb.common.PostgresqlIdentityValueLoader;
import org.openstreetmap.osmosis.core.OsmosisRuntimeException;
import org.openstreetmap.osmosis.core.database.DatabaseLoginCredentials;
import org.openstreetmap.osmosis.core.database.DatabaseType;

public class DatabaseContext {
    private static final Logger LOG = Logger.getLogger(DatabaseContext.class.getName());
    private final DatabaseLoginCredentials loginCredentials;
    private Connection connection;
    private Statement statement;
    private IdentityValueLoader identityValueLoader;
    private boolean autoCommit;

    public DatabaseContext(DatabaseLoginCredentials loginCredentials) {
        this.loginCredentials = loginCredentials;
        this.autoCommit = false;
        try {
            switch (loginCredentials.getDbType()) {
                case POSTGRESQL: {
                    Class.forName("org.postgresql.Driver");
                    this.identityValueLoader = new PostgresqlIdentityValueLoader(this);
                    break;
                }
                case MYSQL: {
                    Class.forName("com.mysql.jdbc.Driver");
                    this.identityValueLoader = new MysqlIdentityValueLoader(this);
                    break;
                }
                default: {
                    throw new OsmosisRuntimeException("Unknown database type " + loginCredentials.getDbType() + ".");
                }
            }
        }
        catch (ClassNotFoundException e) {
            throw new OsmosisRuntimeException("Unable to find database driver.", (Throwable)e);
        }
    }

    private Connection getConnection() {
        if (this.connection == null) {
            switch (this.loginCredentials.getDbType()) {
                case POSTGRESQL: {
                    this.connection = this.getPostgresConnection();
                    break;
                }
                case MYSQL: {
                    this.connection = this.getMysqlConnection();
                    break;
                }
                default: {
                    throw new OsmosisRuntimeException("Unknown database type " + this.loginCredentials.getDbType() + ".");
                }
            }
        }
        return this.connection;
    }

    private Connection getPostgresConnection() {
        Connection newConnection = null;
        try {
            LOG.finer("Creating a new database connection.");
            newConnection = DriverManager.getConnection("jdbc:postgresql://" + this.loginCredentials.getHost() + "/" + this.loginCredentials.getDatabase(), this.loginCredentials.getUser(), this.loginCredentials.getPassword());
            newConnection.setAutoCommit(this.autoCommit);
        }
        catch (SQLException e) {
            throw new OsmosisRuntimeException("Unable to establish a database connection.", (Throwable)e);
        }
        return newConnection;
    }

    private Connection getMysqlConnection() {
        Connection newConnection = null;
        try {
            String url = "jdbc:mysql://" + this.loginCredentials.getHost() + "/" + this.loginCredentials.getDatabase() + "?user=" + this.loginCredentials.getUser() + "&password=" + this.loginCredentials.getPassword();
            if (this.loginCredentials.getForceUtf8()) {
                url = url + "&useUnicode=true&characterEncoding=UTF-8";
            }
            if (this.loginCredentials.getProfileSql()) {
                url = url + "&profileSql=true";
            }
            newConnection = DriverManager.getConnection(url);
            newConnection.setAutoCommit(this.autoCommit);
        }
        catch (SQLException e) {
            throw new OsmosisRuntimeException("Unable to establish a database connection.", (Throwable)e);
        }
        return newConnection;
    }

    public DatabaseType getDatabaseType() {
        return this.loginCredentials.getDbType();
    }

    public void truncateTables(List<String> tables) {
        switch (this.loginCredentials.getDbType()) {
            case POSTGRESQL: {
                StringBuilder statementBuilder = new StringBuilder();
                for (String table : tables) {
                    if (statementBuilder.length() == 0) {
                        statementBuilder.append("TRUNCATE ");
                    } else {
                        statementBuilder.append(", ");
                    }
                    statementBuilder.append(table);
                }
                statementBuilder.append(" CASCADE");
                this.executeStatement(statementBuilder.toString());
                break;
            }
            case MYSQL: {
                for (String table : tables) {
                    this.executeStatement("TRUNCATE " + table);
                }
                break;
            }
            default: {
                throw new OsmosisRuntimeException("Unknown database type " + this.loginCredentials.getDbType() + ".");
            }
        }
    }

    public void disableIndexes(List<String> tables) {
        switch (this.loginCredentials.getDbType()) {
            case POSTGRESQL: {
                break;
            }
            case MYSQL: {
                for (String table : tables) {
                    this.executeStatement("ALTER TABLE " + table + " DISABLE KEYS");
                }
                break;
            }
            default: {
                throw new OsmosisRuntimeException("Unknown database type " + this.loginCredentials.getDbType() + ".");
            }
        }
    }

    public void enableIndexes(List<String> tables) {
        switch (this.loginCredentials.getDbType()) {
            case POSTGRESQL: {
                break;
            }
            case MYSQL: {
                for (String table : tables) {
                    this.executeStatement("ALTER TABLE " + table + " ENABLE KEYS");
                }
                break;
            }
            default: {
                throw new OsmosisRuntimeException("Unknown database type " + this.loginCredentials.getDbType() + ".");
            }
        }
    }

    public void lockTables(List<String> tables) {
        switch (this.loginCredentials.getDbType()) {
            case POSTGRESQL: {
                break;
            }
            case MYSQL: {
                StringBuilder statementBuilder = new StringBuilder();
                for (String table : tables) {
                    if (statementBuilder.length() == 0) {
                        statementBuilder.append("LOCK TABLES ");
                    } else {
                        statementBuilder.append(", ");
                    }
                    statementBuilder.append(table);
                    statementBuilder.append(" WRITE");
                }
                this.executeStatement(statementBuilder.toString());
                break;
            }
            default: {
                throw new OsmosisRuntimeException("Unknown database type " + this.loginCredentials.getDbType() + ".");
            }
        }
    }

    public void unlockTables(List<String> tables) {
        switch (this.loginCredentials.getDbType()) {
            case POSTGRESQL: {
                break;
            }
            case MYSQL: {
                this.executeStatement("UNLOCK TABLES");
                break;
            }
            default: {
                throw new OsmosisRuntimeException("Unknown database type " + this.loginCredentials.getDbType() + ".");
            }
        }
    }

    public long getLastInsertId() {
        return this.identityValueLoader.getLastInsertId();
    }

    public long getLastSequenceId(String sequenceName) {
        return this.identityValueLoader.getLastSequenceId(sequenceName);
    }

    public void executeStatement(String sql) {
        try {
            if (this.statement != null) {
                this.statement.close();
            }
            this.statement = this.getConnection().createStatement();
            this.statement.execute(sql);
        }
        catch (SQLException e) {
            throw new OsmosisRuntimeException("Unable to execute statement.", (Throwable)e);
        }
    }

    public PreparedStatement prepareStatement(String sql) {
        try {
            PreparedStatement preparedStatement = this.getConnection().prepareStatement(sql);
            return preparedStatement;
        }
        catch (SQLException e) {
            throw new OsmosisRuntimeException("Unable to create database prepared statement.", (Throwable)e);
        }
    }

    private void setStatementFetchSizeForStreaming(Statement streamingStatement) {
        try {
            switch (this.loginCredentials.getDbType()) {
                case POSTGRESQL: {
                    streamingStatement.setFetchSize(10000);
                    break;
                }
                case MYSQL: {
                    streamingStatement.setFetchSize(Integer.MIN_VALUE);
                    break;
                }
                default: {
                    throw new OsmosisRuntimeException("Unknown database type " + this.loginCredentials.getDbType() + ".");
                }
            }
        }
        catch (SQLException e) {
            throw new OsmosisRuntimeException("Unable to update statement fetch size.", (Throwable)e);
        }
    }

    public PreparedStatement prepareStatementForStreaming(String sql) {
        try {
            PreparedStatement newStatement = this.getConnection().prepareStatement(sql, 1003, 1007);
            this.setStatementFetchSizeForStreaming(newStatement);
            return newStatement;
        }
        catch (SQLException e) {
            throw new OsmosisRuntimeException("Unable to create streaming resultset statement.", (Throwable)e);
        }
    }

    public ResultSet executeQuery(String sql) {
        try {
            LOG.finest("Executing query {" + sql + "}");
            if (this.statement != null) {
                this.statement.close();
            }
            this.statement = this.getConnection().createStatement(1003, 1007);
            this.setStatementFetchSizeForStreaming(this.statement);
            ResultSet resultSet = this.statement.executeQuery(sql);
            return resultSet;
        }
        catch (SQLException e) {
            throw new OsmosisRuntimeException("Unable to create resultset.", (Throwable)e);
        }
    }

    public void commit() {
        if (this.connection != null) {
            try {
                this.connection.commit();
            }
            catch (SQLException e) {
                throw new OsmosisRuntimeException("Unable to commit changes.", (Throwable)e);
            }
        }
    }

    public void release() {
        this.identityValueLoader.release();
        if (this.statement != null) {
            try {
                this.statement.close();
            }
            catch (SQLException e) {
                LOG.log(Level.WARNING, "Unable to close existing statement.", e);
            }
            this.statement = null;
        }
        if (this.connection != null) {
            try {
                this.connection.close();
            }
            catch (SQLException e) {
                LOG.log(Level.WARNING, "Unable to close database connection.", e);
            }
            this.connection = null;
        }
    }

    public void setAutoCommit(boolean autoCommit) {
        if (this.connection != null) {
            try {
                LOG.finest("Setting auto commit to " + autoCommit + ".");
                this.connection.setAutoCommit(autoCommit);
            }
            catch (SQLException e) {
                throw new OsmosisRuntimeException("Unable to commit changes.", (Throwable)e);
            }
        }
        this.autoCommit = autoCommit;
    }

    public boolean doesColumnExist(String tableName, String columnName) {
        ResultSet resultSet = null;
        try {
            LOG.finest("Checking if column {" + columnName + "} in table {" + tableName + "} exists.");
            resultSet = this.getConnection().getMetaData().getColumns(null, null, tableName, columnName);
            boolean result = resultSet.next();
            resultSet.close();
            resultSet = null;
            boolean bl = result;
            return bl;
        }
        catch (SQLException e) {
            throw new OsmosisRuntimeException("Unable to check for the existence of column " + tableName + "." + columnName + ".", (Throwable)e);
        }
        finally {
            if (resultSet != null) {
                try {
                    resultSet.close();
                }
                catch (SQLException e) {
                    LOG.log(Level.WARNING, "Unable to close column existence result set.", e);
                }
            }
        }
    }

    public boolean doesTableExist(String tableName) {
        ResultSet resultSet = null;
        try {
            LOG.finest("Checking if table {" + tableName + "} exists.");
            resultSet = this.getConnection().getMetaData().getTables(null, null, tableName, new String[]{"TABLE"});
            boolean result = resultSet.next();
            resultSet.close();
            resultSet = null;
            boolean bl = result;
            return bl;
        }
        catch (SQLException e) {
            throw new OsmosisRuntimeException("Unable to check for the existence of table " + tableName + ".", (Throwable)e);
        }
        finally {
            if (resultSet != null) {
                try {
                    resultSet.close();
                }
                catch (SQLException e) {
                    LOG.log(Level.WARNING, "Unable to close table existence result set.", e);
                }
            }
        }
    }

    protected void finalize() throws Throwable {
        this.release();
        super.finalize();
    }
}

