# Licenced under the GNU GPLv2 or later
# (c) 2023 David Koňařík
import os
import re
import json
import zlib
import random
import webbrowser
from typing import DefaultDict, cast
from qgis.PyQt.QtCore import QVariant, Qt, QUrl, QUrlQuery
from qgis.PyQt.QtGui import QPixmap, QColor, QIcon
from qgis.PyQt.QtWidgets import QAction, QDockWidget, QToolBar
from qgis.PyQt.QtNetwork import QNetworkRequest, QNetworkReply
from qgis.core import QgsCategorizedSymbolRenderer, QgsCoordinateTransform, \
    QgsCptCityColorRamp, QgsFeature, QgsPointXY, QgsProject, \
    QgsRendererCategory, QgsSimpleFillSymbolLayer, QgsSymbol, \
    QgsVectorLayer, QgsField, QgsRelation, QgsPointXY, \
    QgsCoordinateReferenceSystem, QgsNetworkAccessManager, Qgis, \
    QgsVectorLayer, QgsVectorLayerJoinInfo, QgsFeatureRequest
from qgis.gui import QgsGui, QgisInterface, QgsMapToolIdentifyFeature

from . import resources as _, webpage_dock

lxml_import_failed = True
try:
    import lxml.etree
    import lxml.html
    import lxml.html.builder as h
    lxml_import_failed = False
except: pass


def classFactory(iface):
    if lxml_import_failed:
        iface.messageBar().pushMessage(
            "Error",
            "Could not import 'lxml', a dependency of the "
            " 'Nahlížení do KN' plugin",
            level=Qgis.Critical)

    return NdknPlugin(iface)


NDKN_URL = QUrl("https://nahlizenidokn.cuzk.cz/")
MARUSHKA_URL = QUrl("https://sgi-nahlizenidokn.cuzk.cz/marushka/")
WEBPAGE_CSS = open(
    os.path.join(os.path.dirname(__file__), "webpage_style.css")) \
    .read()
LV_COLOR_RAMP = QgsCptCityColorRamp("grass/haxby", "")

def lvColor(id):
    rampPos = zlib.crc32(int.to_bytes(id, 8, "little")) / 2**32
    return LV_COLOR_RAMP.color(rampPos)


def layerByName(project, name):
    layers = project.mapLayersByName(name)
    if len(layers) == 0: return None
    else: return layers[0]


class NdknException(Exception): pass


class NdknWebpageDock(QDockWidget, webpage_dock.Ui_ndknWebpageDock):
    def __init__(self, plugin: "NdknPlugin", *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.setupUi(self)
        self.setObjectName(f"ndknWebpageDock-{random.randint(0, 10000)}")

        self.captchaFrame.setVisible(False)
        self.titleContainer.setVisible(False)
        self.captchaSubmit.clicked.connect(self.onCaptchaSubmit)
        self.captchaText.returnPressed.connect(self.onCaptchaSubmit)
        self.mainView.anchorClicked.connect(self.onAnchorClicked)

        # Reference to the base plugin for callbacks
        self.plugin = plugin

    def onAnchorClicked(self, url: QUrl):
        url = NDKN_URL.resolved(url)
        if url.host() == NDKN_URL.host() \
            and url.path().endswith("/ZobrazObjekt.aspx"):
            self.plugin.openDockedWebpage(url)
        else:
            webbrowser.open(url.toString())

    def navigate(self, url):
        req = QNetworkRequest(QUrl(url))
        req.setAttribute(QNetworkRequest.RedirectPolicyAttribute,
                         QNetworkRequest.NoLessSafeRedirectPolicy)
        resp = QgsNetworkAccessManager.instance().get(req)
        resp.finished.connect(lambda: self.onDownloaded(resp))

    def handleNetworkError(self, resp: QNetworkReply):
        if resp.error() != QNetworkReply.NetworkError.NoError:
            self.mainView.setHtml(f"""
                <b>Error fetching URL:</b><br><br>
                <code>{resp.errorString()}</code><br>
                URL requested: <code>{resp.url()}</code>
            """)
            return False
        return True

    def onDownloaded(self, resp: QNetworkReply):
        if not self.handleNetworkError(resp): return

        text = str(resp.readAll(), "utf-8")
        doc = lxml.html.fromstring(text)
        if resp.url().path().endswith("/ZadejCAPTCHA.aspx"):
            self.processCaptcha(doc)
        else:
            self.processCaptchaIframe(doc)

            if doc.xpath("//h1[text()='Informace o pozemku']") != []:
                self.processParcelPage(doc)
            elif doc.xpath("//h1[text()='Informace o stavbě']") != []:
                self.processBuildingPage(doc)
            elif doc.xpath("//h1[text()='Seznam nemovitostí na LV']") != []:
                self.processLvPage(doc)
            elif doc.xpath("//h1[text()='Informace o parcele -"
                           " sousední parcely']") != []:
                self.processOtherPage(doc)
            elif doc.xpath("//h1[text()='Informace o řízení']") != []:
                self.processOtherPage(doc)
            else:
                raise NdknException(f"Unhandled page loaded")

    def processCaptcha(self, doc):
        for script in doc.xpath("//script"):
            js = script.text_content()
            m = re.search(r"\bcaptchaOK\('([^']*)'\)", js)
            if m is not None:
                url = NDKN_URL.resolved(QUrl(m.group(1)))
                self.captchaFrame.setVisible(False)
                self.navigate(url)
                return

        imgs = doc.xpath("//img[@id='telerikC_CaptchaImageUP']")
        if len(imgs) == 0:
            raise NdknException("No CAPTCHA image, but no redirect")
        imgUrl = NDKN_URL.resolved(QUrl(imgs[0].attrib["src"]))

        req = QNetworkRequest(QUrl(imgUrl))
        resp = QgsNetworkAccessManager.instance().get(req)
        resp.finished.connect(lambda: self.onCaptchaImageDownload(resp))

        self.captchaSubmitUrl = \
            NDKN_URL.resolved(QUrl(doc.xpath("//form")[0].attrib["action"]))
        self.captchaParams = {i.attrib["name"]: i.attrib["value"]
                              for i in doc.xpath("//input")
                              if "value" in i.attrib}

    def onCaptchaImageDownload(self, resp):
        if not self.handleNetworkError(resp): return

        pixmap = QPixmap()
        if not pixmap.loadFromData(resp.readAll()):
            self.captchaImage.setText("Failed to read CAPTCHA")
        self.captchaImage.setPixmap(pixmap)
        self.captchaFrame.setVisible(True)
        self.captchaText.setText("")

    def onCaptchaSubmit(self):
        solution = self.captchaText.text()

        params = self.captchaParams | {"txtC": solution}
        query = b"&".join(
            k.encode() + b"=" + QUrl.toPercentEncoding(v)
            for k, v in params.items())
        req = QNetworkRequest(self.captchaSubmitUrl)
        req.setHeader(QNetworkRequest.ContentTypeHeader,
                      "application/x-www-form-urlencoded")
        resp = QgsNetworkAccessManager.instance().post(req, query)
        resp.finished.connect(lambda: self.onDownloaded(resp))

    def processCaptchaIframe(self, doc):
        iframes = doc.xpath("//iframe[@id='iframeCAPTCHA']")
        if len(iframes) == 0: return
        relUrl = QUrl(iframes[0].attrib["src"])
        absUrl = NDKN_URL.resolved(relUrl)
        self.navigate(absUrl)

    def processPageBase(self, doc):
        outBody = lxml.etree.fromstring("<body></body>")
        for el in doc.xpath("//*[@id='content']/*"):
            if el.attrib.get("id") == "ctl00_bodyPlaceHolder_panelPrihlaseniCaptcha":
                continue
            outBody.append(el)

        for el in outBody.xpath("//*[@class='tooltip']"):
            el.tag = "span"
            el.xpath("span")[0].addprevious(h.BR())

        for el in outBody.xpath("//*[@class='imageWMS'] | //a[@href='#']"):
            el.getparent().remove(el)

        for table in outBody.xpath("//table"):
            wrapper = h.DIV(h.CLASS("table-wrapper"))
            table.getparent().replace(table, wrapper)
            wrapper.append(table)
            for i, tr in enumerate(table.xpath(".//tr")):
                tr.attrib["class"] = "odd" if i % 2 == 1 else "even"

        for el in outBody.xpath("//*[@class='inlineButton']"):
            wrapper = h.DIV(h.CLASS("inlineButton-wrapper"))
            el.getparent().replace(el, wrapper)
            wrapper.append(el)

        for td in outBody.xpath("//td | //th"):
            if td.attrib.get("class") == "right":
                td.attrib["align"] = "right"
            if "colspan" in td.attrib:
                # Should be supported, but causes havoc with spacing
                for i in range(int(td.attrib["colspan"]) - 1):
                    td.addnext(h.TD())
                del td.attrib["colspan"]

        outBody.insert(0, h.STYLE(WEBPAGE_CSS))

        return outBody

    def processParcelPage(self, doc):
        outBody = self.processPageBase(doc)
        self.mainView.setHtml(lxml.html.tostring(outBody, encoding="unicode"))

    def processBuildingPage(self, doc):
        outBody = self.processPageBase(doc)
        self.mainView.setHtml(lxml.html.tostring(outBody, encoding="unicode"))

    def processLvPage(self, doc):
        mapLink = doc.xpath("//*[@id='linkMapa']/a/@href")[0]
        mapLinkUrl = QUrl(mapLink)
        lvId = int(QUrlQuery(mapLinkUrl.query())
                   .queryItemValue("MarQParam0"))

        req = QNetworkRequest(QUrl(
            f"{MARUSHKA_URL.toString()}/handlers/getinfo.ashx"
            f"?idquery=D6B992BE&queryProcessType=process"
            f"&qParam0={lvId}&qParamCount=1"))
        resp = QgsNetworkAccessManager.instance().get(req)
        resp.finished.connect(lambda: self.onLvMapDownload(lvId, resp))

        lvNum = doc.xpath("string(//tr[string(./td) = 'Číslo LV:']/td[2])")
        self.title.setText(f"LV {lvNum}")
        # TODO: Get color from layer style when that is changed
        self.title.setStyleSheet(self.titleStyleSheet(lvColor(lvId)))
        self.titleContainer.setVisible(True)

        ownerType = None
        owners = []
        for row in doc.xpath("//table[@summary='Vlastníci, jiní oprávnění']//tr"):
            heading = row.xpath("./th[1]/text()")
            cells = row.xpath("./td/text()")
            partner = row.xpath("./td[@colspan='2']/i/text()")
            if heading != []:
                ownerType = heading[0]
            elif partner != []:
                owners[-1]["jmeno"] += "\n" + partner[0]
            else:
                owners.append({
                    # We need actual str, not _ElementUnicodeResult
                    "druh": str(ownerType),
                    "jmeno": str(cells[0]),
                    "podil": str(cells[1]) if len(cells) > 1 else None,
                })
        if owners != []:
            self.plugin.setLvOwners(lvId, owners)

        outBody = self.processPageBase(doc)
        self.mainView.setHtml(lxml.html.tostring(outBody, encoding="unicode"))

    def onLvMapDownload(self, lvId, resp):
        doc = lxml.etree.fromstring(str(resp.readAll(), "utf-8"))
        wkbelmStr = doc.xpath("//LocalizeItem/@WKBELM")
        if len(wkbelmStr) == 0: return
        wkbelm = json.loads(wkbelmStr[0])
        parcelIds = [e["id"] for e in wkbelm]
        self.plugin.setLvMembership(lvId, parcelIds)

    def processOtherPage(self, doc):
        outBody = self.processPageBase(doc)
        self.mainView.setHtml(lxml.html.tostring(outBody, encoding="unicode"))

    def titleStyleSheet(self, color: QColor):
        return f"""
        padding: 0 10px;
        border-radius: 10px;
        background-color: {color.name()};
        """


class NdknPlugin:
    menuLabel = "&Nahlížení do KN"

    def __init__(self, iface: QgisInterface):
        self.iface = iface

    def initGui(self):
        self.toolbar: QToolBar = self.iface.addToolBar("Nahlížení do KN")
        self.toolbar.setObjectName("ndkn_toolbar")

        self.initLayersAction = QAction(
            "Nastavit &vrstvy", self.iface.mainWindow())
        self.initLayersAction.triggered.connect(self.initLayers)
        self.iface.addPluginToMenu(self.menuLabel, self.initLayersAction)

        self.parcelLookupAction = QAction(
            QIcon(":plugins/qgis_ndkn/parcel.svg"),
            "Otevřít Nahlížení pro &parcelu", self.iface.mainWindow())
        self.parcelLookupAction.setCheckable(True)
        self.parcelLookupAction.triggered.connect(self.startParcelLookup)
        self.iface.addPluginToMenu(self.menuLabel, self.parcelLookupAction)
        self.toolbar.addAction(self.parcelLookupAction)
        QgsGui.shortcutsManager().registerAction(self.parcelLookupAction)

        self.buildingLookupAction = QAction(
            QIcon(":plugins/qgis_ndkn/building.svg"),
            "Otevřít Nahlížení pro &budovu", self.iface.mainWindow())
        self.buildingLookupAction.setCheckable(True)
        self.buildingLookupAction.triggered.connect(self.startBuildingLookup)
        self.iface.addPluginToMenu(self.menuLabel, self.buildingLookupAction)
        self.toolbar.addAction(self.buildingLookupAction)
        QgsGui.shortcutsManager().registerAction(self.buildingLookupAction)

        self.collectParcelsByLvAction = QAction(
            "Vytvořit p&olygony parcel dle LV", self.iface.mainWindow())
        self.collectParcelsByLvAction.triggered.connect(lambda:
            self.collectParcelsByLv("LV"))
        self.iface.addPluginToMenu(self.menuLabel,
                                   self.collectParcelsByLvAction)

    def unload(self):
        self.iface.removePluginMenu(self.menuLabel, self.initLayersAction)
        self.iface.removePluginMenu(self.menuLabel, self.parcelLookupAction)
        self.iface.removePluginMenu(self.menuLabel, self.buildingLookupAction)
        self.iface.removePluginMenu(self.menuLabel,
                                    self.collectParcelsByLvAction)
        del self.toolbar

    def parcelyLayer(self):
        layer = cast(QgsVectorLayer,
            layerByName(QgsProject.instance() , "Parcely"))
        if layer is None:
            raise NdknException("Prerequisite layer 'Parcely' missing")
        return layer

    def stavebniObjektyLayer(self):
        layer = cast(QgsVectorLayer,
            layerByName(QgsProject.instance() , "Stavební objekty"))
        if layer is None:
            raise NdknException("Prerequisite layer 'Stavební objekty' missing")
        return layer

    def listyVlastnictviLayer(self):
        return cast(QgsVectorLayer,
                 layerByName(QgsProject.instance(), "Listy vlastnictví"))

    def vlastniciLayer(self):
        return cast(QgsVectorLayer,
                 layerByName(QgsProject.instance(), "Vlastníci"))

    def initLayers(self):
        project = QgsProject.instance()
        parcely = self.parcelyLayer()
        stavebniObjekty = self.stavebniObjektyLayer()

        listyVlastnictvi = self.listyVlastnictviLayer()
        if listyVlastnictvi is None:
            listyVlastnictvi = QgsVectorLayer(
                "None", "Listy vlastnictví", "memory")
            listyVlastnictvi.dataProvider().addAttributes([
                QgsField("id", QVariant.LongLong),
                QgsField("parcela", QVariant.LongLong),
                QgsField("budova", QVariant.LongLong)])
            listyVlastnictvi.updateFields()
            project.addMapLayer(listyVlastnictvi)

        vlastnici = self.vlastniciLayer()
        if vlastnici is None:
            vlastnici = QgsVectorLayer(
                "None", "Vlastníci", "memory")
            vlastnici.dataProvider().addAttributes([
                QgsField("lv", QVariant.LongLong),
                QgsField("druh", QVariant.String),
                QgsField("jmeno", QVariant.String),
                QgsField("podil", QVariant.String)])
            vlastnici.updateFields()
            project.addMapLayer(vlastnici)

        if not any(j.joinLayerId() == listyVlastnictvi.id()
                   for j in parcely.vectorJoins()):
            join = QgsVectorLayerJoinInfo()
            join.setJoinLayer(listyVlastnictvi)
            join.setTargetFieldName("Id")
            join.setJoinFieldName("parcela")
            join.setJoinFieldNamesSubset(["id"])
            join.setPrefix("LV_")
            join.setDynamicFormEnabled(True)
            join.setUsingMemoryCache(False)
            listyVlastnictvi.dataProvider().createAttributeIndex(
                listyVlastnictvi.fields().indexFromName("parcela"))
            parcely.addJoin(join)

        if not any(j.joinLayerId() == listyVlastnictvi.id()
                   for j in stavebniObjekty.vectorJoins()):
            join = QgsVectorLayerJoinInfo()
            join.setJoinLayer(listyVlastnictvi)
            join.setTargetFieldName("id")
            join.setJoinFieldName("budova")
            join.setJoinFieldNamesSubset(["id"])
            join.setPrefix("LV_")
            join.setDynamicFormEnabled(True)
            join.setUsingMemoryCache(False)
            listyVlastnictvi.dataProvider().createAttributeIndex(
                listyVlastnictvi.fields().indexFromName("budova"))
            stavebniObjekty.addJoin(join)

        relMan = project.relationManager()
        if len(relMan.relationsByName("Vlastníci LV")) != "":
            rel = QgsRelation()
            rel.setReferencedLayer(listyVlastnictvi.id())
            rel.setReferencingLayer(vlastnici.id())
            rel.addFieldPair("lv", "id")
            rel.setId("Vlastnici_LV")
            rel.setName("Vlastníci LV")
            relMan.addRelation(rel)

        self.updateLayerStyle()

    def updateLayerStyle(self):
        parcely = self.parcelyLayer()
        stavebniObjekty = self.stavebniObjektyLayer()
        listyVlastnictvi = self.listyVlastnictviLayer()
        if listyVlastnictvi is None: return

        lvIdFieldIdx = listyVlastnictvi.fields().indexFromName("id")
        lvIds = listyVlastnictvi.uniqueValues(lvIdFieldIdx)

        for layer in [parcely, stavebniObjekty]:
            baseSymbol = QgsSymbol.defaultSymbol(Qgis.GeometryType.Polygon)
            for i in range(baseSymbol.symbolLayerCount()):
                baseSymbol.deleteSymbolLayer(i)
            baseSymbol.appendSymbolLayer(QgsSimpleFillSymbolLayer(
                color=Qt.darkGray,
                style=Qt.NoBrush if layer == parcely else Qt.DiagCrossPattern,
                strokeColor=Qt.black,
                strokeStyle=Qt.SolidLine,
                strokeWidth=0.1))

            if not isinstance(layer.renderer(), QgsCategorizedSymbolRenderer):
                layer.setRenderer(QgsCategorizedSymbolRenderer("LV_id", [
                    QgsRendererCategory(None, baseSymbol, "Bez LV")
                ]))
            renderer = cast(QgsCategorizedSymbolRenderer, layer.renderer())
            baseCatOrig = next((c for c in renderer.categories()
                                if c.value() is None), None)
            if baseCatOrig is not None \
                and baseCatOrig.symbol().symbolLayerCount() > 0 \
                and isinstance(baseCatOrig.symbol().symbolLayer(0),
                               QgsSimpleFillSymbolLayer):
                baseSymbol = baseCatOrig.symbol()

            for lvId in lvIds:
                if renderer.categoryIndexForValue(lvId) != -1: continue
                symbol = baseSymbol.clone()
                color = lvColor(lvId)
                color.setAlpha(150)
                symbol.setColor(color)
                if layer == parcely:
                    symbol.symbolLayer(0).setBrushStyle(Qt.SolidPattern)
                renderer.addCategory(
                    QgsRendererCategory(lvId, symbol, f"LV {lvId}"))

            layer.triggerRepaint()
            self.iface.layerTreeView().refreshLayerSymbology(layer.id())

    def getNahlizeniPointUrl(self, point: QgsPointXY):
        sourceCrs = QgsProject.instance().crs()
        krovak = QgsCoordinateReferenceSystem(5514)
        transfrom = QgsCoordinateTransform(
            sourceCrs, krovak, QgsProject.instance())
        pointKrovak = transfrom.transform(point)
        return f"{NDKN_URL.toString()}/MapaIdentifikace.aspx?l=KN" \
               f"&x={int(pointKrovak.x())}&y={int(pointKrovak.y())}"

    def getNahlizeniParcelaUrl(self, id):
        return f"{NDKN_URL.toString()}/ZobrazObjekt.aspx?typ=parcela&id={id}"

    def getNahlizeniBudovaUrl(self, id):
        return f"{NDKN_URL.toString()}/ZobrazObjekt.aspx?typ=budova&id={id}"

    def openDockedWebpage(self, url):
        dock = NdknWebpageDock(self, self.iface.mainWindow())
        dock.navigate(url)
        self.iface.addTabifiedDockWidget(
            Qt.RightDockWidgetArea, dock, raiseTab=True)

    def openNahlizeni(self, layer: QgsVectorLayer, feature: QgsFeature):
        parcely = self.parcelyLayer()
        stavebniObjekty = self.stavebniObjektyLayer()

        layer.deselect(layer.selectedFeatureIds())
        layer.select(feature.id())
        if layer == parcely:
            id = feature.attribute("Id")
            self.openDockedWebpage(self.getNahlizeniParcelaUrl(id))
        elif layer == stavebniObjekty:
            id = feature.attribute("IsknBudovaId")
            self.openDockedWebpage(self.getNahlizeniBudovaUrl(id))
        else: assert False

    def startLookup(self, layer):
        canvas = self.iface.mapCanvas()
        self.lookupTool = QgsMapToolIdentifyFeature(canvas)
        self.lookupTool.setCursor(Qt.CrossCursor)
        self.lookupTool.setLayer(layer)
        self.lookupTool.featureIdentified.connect(
            lambda f: self.openNahlizeni(layer, f))
        canvas.setMapTool(self.lookupTool)

    def startParcelLookup(self):
        self.startLookup(self.parcelyLayer())
        self.lookupTool.setAction(self.parcelLookupAction)

    def startBuildingLookup(self):
        self.startLookup(self.stavebniObjektyLayer())
        self.lookupTool.setAction(self.buildingLookupAction)

    def setLvMembership(self, id, parcelIds):
        listyVlastnictvi = self.listyVlastnictviLayer()
        if listyVlastnictvi is None: return

        listyVlastnictvi.startEditing()

        for lvFeat in listyVlastnictvi.getFeatures():
            if lvFeat.attribute("id") == id:
                listyVlastnictvi.deleteFeature(lvFeat.id())

        for pid in parcelIds:
            lvFeat = QgsFeature(listyVlastnictvi.fields())
            lvFeat.setAttribute("id", id)
            lvFeat.setAttribute("parcela", pid)
            listyVlastnictvi.addFeature(lvFeat)

        listyVlastnictvi.commitChanges()
        self.updateLayerStyle()

    def setLvOwners(self, id, owners):
        vlastnici = self.vlastniciLayer()
        if vlastnici is None: return

        vlastnici.startEditing()
        for vlFeat in vlastnici.getFeatures():
            if vlFeat.attribute("lv") == id:
                vlastnici.deleteFeature(vlFeat.id())

        for owner in owners:
            feat = QgsFeature(vlastnici.fields())
            feat.setAttribute("lv", id)
            for k, v in owner.items():
                feat.setAttribute(k, v)
            vlastnici.addFeature(feat)

        vlastnici.commitChanges()

    def collectParcelsByLv(self, newLayerName, bbox=None):
        project = QgsProject.instance()
        parcels = self.parcelyLayer()
        vlastnici = self.vlastniciLayer()

        if bbox is None:
            bbox = self.iface.mapCanvas().extent()

        parcelGeomsByLv = DefaultDict(list)
        for feat in parcels.getFeatures(
                QgsFeatureRequest()
                    .setFilterExpression("LV_id IS NOT NULL")
                    .setFilterRect(bbox)):
            lvId = feat.attribute("LV_id")
            if lvId is not None:
                parcelGeomsByLv[lvId].append(feat.geometry())

        outLayer = layerByName(project, newLayerName)
        if outLayer is None:
            outLayer = QgsVectorLayer("MultiPolygon", newLayerName, "memory")
            outLayer.setCrs(parcels.crs())
            outLayer.dataProvider().addAttributes([
                QgsField("lv", QVariant.LongLong),
                QgsField("popis", QVariant.String)])
            outLayer.updateFields()
        outLayer.startEditing()
        # Clear in case we are updating an existing layer
        outLayer.deleteFeatures([f.id() for f in outLayer.getFeatures()])
        for lvId, geoms in parcelGeomsByLv.items():
            feature = QgsFeature(outLayer.fields())
            geom = geoms[0]
            for g in geoms[1:]:
                geom = geom.combine(g)
            feature.setGeometry(geom)
            feature.setAttribute("lv", lvId)
            label = "\n".join(
                of.attribute("jmeno")
                for of in vlastnici.getFeatures()
                if of.attribute("lv") == lvId
                and of.attribute("druh") == "Vlastnické právo")
            feature.setAttribute("popis", label)
            outLayer.addFeature(feature)
        outLayer.commitChanges()
        outLayer.updateExtents()
        project.addMapLayer(outLayer)
