# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Flight Planner - Horizontal Flight Simplified
                                 A QGIS plugin

 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-09-17
        copyright            : (C) 2025 by Prof Cazaroli e Leandro França
        email                : contato@geoone.com.br
***************************************************************************/
"""

__author__ = 'Prof Cazaroli e Leandro França - Participação Ilton Freitas'
__date__ = '2025-09-17'
__copyright__ = '(C) 2025 by Prof Cazaroli e Leandro França - Participação Ilton Freitas'
__revision__ = '$Format:%H$'

from qgis.core import *
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from .Funcs import verificar_plugins, simbologiaPontos, loadParametros, saveParametros
from ..images.Imgs import *
import processing
import csv
import os
import tempfile
import math
import uuid

class CSV_Simplify(QgsProcessingAlgorithm):
    def initAlgorithm(self, config=None):
        csvInSimplified, csvOutSimplified, rasterSimplified, toleranceSimplified, addPCsvSimplified, addPSimplified, addLSimplified = loadParametros("H_Simplified")
       
        self.addParameter(QgsProcessingParameterFile(
            'voo_em_csv', 
            'Generated CSV file', 
            behavior=QgsProcessingParameterFile.File, 
            fileFilter='CSV Files (*.csv)', 
            defaultValue=csvInSimplified
        ))
        self.addParameter(QgsProcessingParameterRasterLayer(
            'dem', 
            'Digital Elevation Model (DEM)',
            defaultValue=rasterSimplified
        ))
        self.addParameter(QgsProcessingParameterCrs(
            'src_projetado', 
            'Projected Coordinate Reference System (CRS)', 
            defaultValue = QgsProject.instance().crs()
        ))
        self.addParameter(QgsProcessingParameterNumber(
            'tolerancia', 
            'Tolerance', 
            type=QgsProcessingParameterNumber.Double, 
            defaultValue=toleranceSimplified
        ))
        self.addParameter(QgsProcessingParameterFileDestination(
            'csv_saida', 
            'Simplified CSV', 
            fileFilter='CSV Files (*.csv)', 
            defaultValue=csvOutSimplified 
        ))
        self.addParameter(QgsProcessingParameterBoolean(
            'adicionar_pontos_csv', 
            'Add CSV points layer to project', 
            defaultValue=addPCsvSimplified
        ))
        self.addParameter(QgsProcessingParameterBoolean(
            'adicionar_linha_trajetoria', 
            'Add flight line layer to project', 
            defaultValue=addLSimplified
        ))
        self.addParameter(QgsProcessingParameterBoolean(
            'adicionar_pontos_simplificados', 
            'Add simplified points layer to project', 
            defaultValue=addPSimplified
        ))

    def processAlgorithm(self, parameters, context, model_feedback):
        arquivo_csvIn = self.parameterAsFile(parameters, 'voo_em_csv', context)
        arquivo_csvOut = self.parameterAsFile(parameters, 'csv_saida', context)
        arquivo_DEM = self.parameterAsFile(parameters, 'dem', context)

        # Verificações
        if 'voo_em_csv' not in parameters:
            raise QgsProcessingException("❌ Generated CSV file is empty!")
        
        if arquivo_csvIn:
            if not os.path.exists(os.path.dirname(arquivo_csvIn)):
                raise QgsProcessingException("❌ Path to Generated CSV file does not exist!")
        
        if 'dem' not in parameters:
            raise QgsProcessingException("❌ DEM is empty!")
        
        if 'src_projetado' not in parameters:
            raise QgsProcessingException("❌ CRS is empty!")

        if 'tolerancia' not in parameters:
            raise QgsProcessingException("❌ Telerance is empty!")

        if 'csv_saida' not in parameters:
            raise QgsProcessingException("❌ CRS is empty!")

        if arquivo_csvOut:
            if not os.path.exists(os.path.dirname(arquivo_csvOut)):
                raise QgsProcessingException("❌ Path to Simplified CSV file does not exist!")

        # Verificar se o(s) plugin(s) instalado(s)
        plugins_verificar = ["lftools"]
        feedback = QgsProcessingFeedback()
        verificar_plugins(plugins_verificar, feedback)

        # Grava Parâmetros
        saveParametros("H_Simplified",
                        csvI=arquivo_csvIn,
                        csv=arquivo_csvOut,
                        raster=arquivo_DEM,
                        crs=parameters['src_projetado'],
                        tol=parameters['tolerancia'],
                        add1=parameters['adicionar_pontos_csv'],
                        add2=parameters['adicionar_linha_trajetoria'],
                        add3=parameters['adicionar_pontos_simplificados'])

        # ========================================================================
        # Processo
        # ========================================================================
        feedback = QgsProcessingMultiStepFeedback(15, model_feedback)
        results = {}
        outputs = {}
        
        # Lista para armazenar arquivos temporários para limpeza posterior
        temp_files = []
        
        feedback.pushInfo("=" * 50)
        feedback.pushInfo("STARTING 3D SIMPLIFICATION PROCESSING")
        feedback.pushInfo("=" * 50)

        # ETAPA 1: Criar Pontos a partir de CSV (incorporado)
        feedback.pushInfo("STEP 1: Creating points from CSV...")
        try:
            csv_path = self.parameterAsString(parameters, 'voo_em_csv', context)
            
            with open(csv_path, 'r', encoding='utf-8') as csvfile:
                reader = csv.DictReader(csvfile)
                fields = reader.fieldnames

            field_x = 'longitude'
            field_y = 'latitude'
            field_z = 'altitude(m)'

            if field_x not in fields:
                raise QgsProcessingException(f"Campo {field_x} not found in CSV")
            if field_y not in fields:
                raise QgsProcessingException(f"Campo {field_y} not found in CSV")
            if field_z not in fields:
                raise QgsProcessingException(f"Campo {field_z} not found in CSV")

            # Criar camada temporária
            temp_layer = QgsVectorLayer("PointZ?crs=EPSG:4326", "temp_points", "memory")
            provider = temp_layer.dataProvider()
            
            # Adicionar campos
            new_fields = QgsFields()
            for field in fields:
                new_fields.append(QgsField(field, QVariant.String))

            new_fields.append(QgsField('original_index', QVariant.Int))
            provider.addAttributes(new_fields)
            temp_layer.updateFields()

            # Ler e adicionar features
            features = []
            with open(csv_path, 'r', encoding='utf-8') as csvfile:
                reader = csv.DictReader(csvfile)
                for idx, row in enumerate(reader):
                    try:
                        x = float(row[field_x])
                        y = float(row[field_y])
                        z = float(row[field_z])
                        
                        point = QgsPoint(x, y, z)
                        feature = QgsFeature()
                        feature.setGeometry(QgsGeometry(point))
                        
                        attrs = [row[field] for field in fields] + [idx + 1]
                        feature.setAttributes(attrs)
                        features.append(feature)
                    except Exception as e:
                        feedback.pushWarning(f"Error in line {idx + 1}: {str(e)}")
                        continue

            provider.addFeatures(features)
            temp_layer.updateExtents()

            # Gerar nome de arquivo temporário único
            temp_file = os.path.join(tempfile.gettempdir(), f'temp_points_{uuid.uuid4().hex}.gpkg')
            temp_files.append(temp_file)
            
            # Remover arquivo se já existir
            if os.path.exists(temp_file):
                os.remove(temp_file)
                
            options = QgsVectorFileWriter.SaveVectorOptions()
            options.driverName = "GPKG"
            writer = QgsVectorFileWriter.writeAsVectorFormat(temp_layer, temp_file, options)
            
            if writer[0] != QgsVectorFileWriter.NoError:
                raise QgsProcessingException(f"Error saving temporary file: {writer}")
                
            outputs['CriarPontosAPartirDeCsv'] = {'OUTPUT': temp_file}
            feedback.pushInfo("✓ Points created successfully")

            if parameters['adicionar_pontos_csv']:
                pontos_csv_layer = QgsVectorLayer(temp_file, 'pontos_csv', 'ogr')
                if pontos_csv_layer.isValid():
                    csv_name = os.path.splitext(os.path.basename(csv_path))[0]
                    pontos_csv_layer.setName(f"{csv_name} - Original CSV")
                    QgsProject.instance().addMapLayer(pontos_csv_layer)
                    # Simbologia
                    simbologiaPontos(pontos_csv_layer)
                    feedback.pushInfo("✓ CSV point layer added to project")

        except Exception as e:
            feedback.reportError(f"Error creating points: {str(e)}")
            # Limpar arquivos temporários em caso de erro
            for temp_file in temp_files:
                try:
                    if os.path.exists(temp_file):
                        os.remove(temp_file)
                except:
                    pass
            return {}

        feedback.setCurrentStep(1)
        if feedback.isCanceled():
            return {}
        
        # ETAPA 2: Adicionar campo de índice único (já foi feito na etapa 1)
        feedback.pushInfo("STEP 2: Index field already added...")
        outputs['AdicionarIndice'] = {'OUTPUT': temp_file}
        feedback.pushInfo("✓ Index field already exists")

        feedback.setCurrentStep(2)
        if feedback.isCanceled():
            return {}

        # ETAPA 3: Amostrar valores do raster com LFTools
        feedback.pushInfo("STEP 3: Sampling raster values ​​with LFTools...")
        try:
            pontos_layer = QgsVectorLayer(outputs['AdicionarIndice']['OUTPUT'], 'pontos', 'ogr')
            
            # Gerar nome de arquivo temporário único
            temp_file_amostrado = os.path.join(tempfile.gettempdir(), f'pontos_amostrados_{uuid.uuid4().hex}.gpkg')
            temp_files.append(temp_file_amostrado)
            
            # Remover arquivo se já existir
            if os.path.exists(temp_file_amostrado):
                os.remove(temp_file_amostrado)
            
            alg_params = {
                'INPUT': parameters['dem'],
                'BAND': 1,
                'POINTS': pontos_layer,
                'RESAMPLING': 1,
                'PREFIX': 'sample_',
                'OUTPUT': temp_file_amostrado
            }
            outputs['AmostrarRaster'] = processing.run(
                "lftools:getpointvalue", 
                alg_params, 
                context=context, 
                feedback=feedback, 
                is_child_algorithm=True
            )
            
            if os.path.exists(temp_file_amostrado):
                feedback.pushInfo("✓ Raster values successfully sampled")
            else:
                feedback.reportError("Failed to create temporary file with sampled values")
                return {}
                
        except Exception as e:
            feedback.reportError(f"Error sampling raster values ​​with LFTools: {str(e)}")
            # Limpar arquivos temporários em caso de erro
            for temp_file in temp_files:
                try:
                    if os.path.exists(temp_file):
                        os.remove(temp_file)
                except:
                    pass
            return {}
        
        feedback.setCurrentStep(3)
        if feedback.isCanceled():
            return {}

        # ETAPA 4: Processar valores Z do DEM
        feedback.pushInfo("STEP 4: Processing DEM Z-values...")
        
        # Gerar nome de arquivo temporário único para os pontos com Z
        temp_file_z = os.path.join(tempfile.gettempdir(), f'pontos_z_{uuid.uuid4().hex}.gpkg')
        temp_files.append(temp_file_z)
        
        # Remover arquivo se já existir
        if os.path.exists(temp_file_z):
            os.remove(temp_file_z)
            
        try:
            extracted_points = QgsVectorLayer(temp_file_amostrado, 'extracted_points', 'ogr')
            
            if not extracted_points.isValid():
                feedback.reportError("Invalid sampled point layer")
                return {}
            
            field_names = [field.name() for field in extracted_points.fields()]
            sample_band_name = 'sample_band_1'
            if sample_band_name not in field_names:
                feedback.reportError("Field 'sample_banda1' not found in layer")
                return {}
            
            alg_params = {
                'INPUT': extracted_points,
                'Z_VALUE': QgsProperty.fromExpression(f'"{sample_band_name}"'),
                'OUTPUT': temp_file_z
            }
            outputs['DefinirValorZ'] = processing.run(
                "native:setzvalue", 
                alg_params, 
                context=context, 
                feedback=feedback, 
                is_child_algorithm=True
            )
            feedback.pushInfo("✓ Z values ​​processed successfully")
        except Exception as e:
            feedback.reportError(f"Error processing Z values: {str(e)}")
            # Limpar arquivos temporários em caso de erro
            for temp_file in temp_files:
                try:
                    if os.path.exists(temp_file):
                        os.remove(temp_file)
                except:
                    pass
            return {}
        
        feedback.setCurrentStep(4)
        if feedback.isCanceled():
            return {}

        # ETAPA 5: Reprojetar camada para SRC projetado - UTM
        feedback.pushInfo("STEP 5: Redesigning layer...")
        
        # Gerar nome de arquivo temporário único para a camada reprojetada
        temp_file_reprojetada = os.path.join(tempfile.gettempdir(), f'camada_reprojetada_{uuid.uuid4().hex}.gpkg')
        temp_files.append(temp_file_reprojetada)
        
        # Remover arquivo se já existir
        if os.path.exists(temp_file_reprojetada):
            os.remove(temp_file_reprojetada)
            
        alg_params = {
            'INPUT': temp_file_z,
            'TARGET_CRS': parameters['src_projetado'],
            'OUTPUT': temp_file_reprojetada
        }
        try:
            outputs['ReprojetarCamada'] = processing.run(
                'native:reprojectlayer', 
                alg_params, 
                context=context, 
                feedback=feedback, 
                is_child_algorithm=True
            )
            feedback.pushInfo("✓ Layer successfully redesigned")
        except Exception as e:
            feedback.reportError(f"Error redesigning layer: {str(e)}")
            # Limpar arquivos temporários em caso de erro
            for temp_file in temp_files:
                try:
                    if os.path.exists(temp_file):
                        os.remove(temp_file)
                except:
                    pass
            return {}
        
        feedback.setCurrentStep(5)
        if feedback.isCanceled():
            return {}

        # ETAPA 6: Calcular distância mínima entre pontos
        feedback.pushInfo("STEP 6: Calculating minimum distance between points...")
        try:
            pontos_reprojetados = QgsVectorLayer(temp_file_reprojetada, 'pontos_reprojetados', 'ogr')
            
            # Calcular distâncias entre pontos consecutivos
            features = list(pontos_reprojetados.getFeatures())
            distancias = []
            
            for i in range(len(features) - 1):
                geom1 = features[i].geometry()
                geom2 = features[i + 1].geometry()
                distancias.append(geom1.distance(geom2))
            
            # Calcular distância mínima
            if distancias:
                distancia_minima = min(distancias)
                raio_buffer = distancia_minima / 2
                feedback.pushInfo(f"✓ Minimum calculated distance: {distancia_minima:.2f} metros")
                feedback.pushInfo(f"✓ Buffer radius defined: {raio_buffer:.2f} metros")
            else:
                raio_buffer = 5  # Valor padrão se não for possível calcular
                feedback.pushInfo("⚠ Unable to calculate minimum distance using default value")
        except Exception as e:
            raio_buffer = 5  # Valor padrão em caso de erro
            feedback.pushInfo(f"⚠ Error calculating minimum distance: {str(e)}, using default value")

        feedback.setCurrentStep(6)
        if feedback.isCanceled():
            return {}

        # ETAPA 7: Pontos para linhas
        feedback.pushInfo("STEP 7: Converting points to lines...")
        
        # Gerar nome de arquivo temporário único para a linha de trajetória
        temp_file_linha = os.path.join(tempfile.gettempdir(), f'linha_trajetoria_{uuid.uuid4().hex}.gpkg')
        temp_files.append(temp_file_linha)
        
        # Remover arquivo se já existir
        if os.path.exists(temp_file_linha):
            os.remove(temp_file_linha)
            
        alg_params = {
            'INPUT': temp_file_reprojetada,
            'ORDER_EXPRESSION': '"original_index"',
            'OUTPUT': temp_file_linha
        }
        try:
            outputs['PontosParaLinhas'] = processing.run(
                'native:pointstopath', 
                alg_params, 
                context=context, 
                feedback=feedback, 
                is_child_algorithm=True
            )
            feedback.pushInfo("✓ Points successfully converted to lines")
            
            if parameters['adicionar_linha_trajetoria']:
                # Carregar a camada de linha de trajetória do arquivo temporário
                linha_trajetoria_layer = QgsVectorLayer(temp_file_linha, 'linha_trajetoria', 'ogr')
                if linha_trajetoria_layer.isValid():
                    csv_path = self.parameterAsString(parameters, 'voo_em_csv', context)
                    csv_name = os.path.splitext(os.path.basename(csv_path))[0]
                    linha_trajetoria_layer.setName(f"{csv_name} - Flight Line")
                    QgsProject.instance().addMapLayer(linha_trajetoria_layer)
                    feedback.pushInfo("✓ Trajectory line layer added to project")
                else:
                    feedback.reportError("Failed to load trajectory line layer")
        except Exception as e:
            feedback.reportError(f"Error converting points to line: {str(e)}")
            # Limpar arquivos temporários em caso de erro
            for temp_file in temp_files:
                try:
                    if os.path.exists(temp_file):
                        os.remove(temp_file)
                except:
                    pass
            return {}
        
        feedback.setCurrentStep(7)
        if feedback.isCanceled():
            return {}

        # ETAPA 8: Simplificar linha
        feedback.pushInfo("STEP 8: Simplifying line...")
        
        # Gerar nome de arquivo temporário único para a linha simplificada
        temp_file_simplificada = os.path.join(tempfile.gettempdir(), f'linha_simplificada_{uuid.uuid4().hex}.gpkg')
        temp_files.append(temp_file_simplificada)
        
        # Remover arquivo se já existir
        if os.path.exists(temp_file_simplificada):
            os.remove(temp_file_simplificada)
            
        alg_params = {
            'INPUT': temp_file_linha,
            'METHOD': 2, # Visvalingam
            'TOLERANCE': parameters['tolerancia'],
            'OUTPUT': temp_file_simplificada
        }
        try:
            outputs['Simplificar'] = processing.run(
                'native:simplifygeometries', 
                alg_params, 
                context=context, 
                feedback=feedback, 
                is_child_algorithm=True
            )
            feedback.pushInfo("✓ Line successfully simplified")
        except Exception as e:
            feedback.reportError(f"Error simplifying line: {str(e)}")
            # Limpar arquivos temporários em caso de erro
            for temp_file in temp_files:
                try:
                    if os.path.exists(temp_file):
                        os.remove(temp_file)
                except:
                    pass
            return {}
        
        feedback.setCurrentStep(8)
        if feedback.isCanceled():
            return {}

        # ETAPA 9: Extrair vértices da linha simplificada
        feedback.pushInfo("STEP 9: Extracting vertices...")
        
        # Gerar nome de arquivo temporário único para os vértices
        temp_file_vertices = os.path.join(tempfile.gettempdir(), f'vertices_{uuid.uuid4().hex}.gpkg')
        temp_files.append(temp_file_vertices)
        
        # Remover arquivo se já existir
        if os.path.exists(temp_file_vertices):
            os.remove(temp_file_vertices)
            
        alg_params = {
            'INPUT': temp_file_simplificada,
            'OUTPUT': temp_file_vertices
        }
        try:
            outputs['ExtrairVertices'] = processing.run(
                'native:extractvertices', 
                alg_params, 
                context=context, 
                feedback=feedback, 
                is_child_algorithm=True
            )
            feedback.pushInfo("✓ Vertices extracted successfully")
        except Exception as e:
            feedback.reportError(f"Error extracting vertices: {str(e)}")
            # Limpar arquivos temporários em caso de erro
            for temp_file in temp_files:
                try:
                    if os.path.exists(temp_file):
                        os.remove(temp_file)
                except:
                    pass
            return {}
        
        feedback.setCurrentStep(9)
        if feedback.isCanceled():
            return {}

        # ETAPA 10: Criar buffers ao redor dos vértices simplificados
        feedback.pushInfo("STEP 10: Creating buffers around vertices...")
        
        # Gerar nome de arquivo temporário único para os buffers
        temp_file_buffers = os.path.join(tempfile.gettempdir(), f'buffers_{uuid.uuid4().hex}.gpkg')
        temp_files.append(temp_file_buffers)
        
        # Remover arquivo se já existir
        if os.path.exists(temp_file_buffers):
            os.remove(temp_file_buffers)
            
        alg_params = {
            'DISSOLVE': False,
            'DISTANCE': raio_buffer,
            'END_CAP_STYLE': 0,
            'INPUT': temp_file_vertices,
            'JOIN_STYLE': 0,
            'MITER_LIMIT': 2,
            'SEGMENTS': 5,
            'OUTPUT': temp_file_buffers
        }
        try:
            outputs['CriarBuffers'] = processing.run(
                'native:buffer', 
                alg_params, 
                context=context, 
                feedback=feedback, 
                is_child_algorithm=True
            )
            feedback.pushInfo("✓ Buffers created successfully")
        except Exception as e:
            feedback.reportError(f"Error creating buffers: {str(e)}")
            # Limpar arquivos temporários em caso de erro
            for temp_file in temp_files:
                try:
                    if os.path.exists(temp_file):
                        os.remove(temp_file)
                except:
                    pass
            return {}
        
        feedback.setCurrentStep(10)
        if feedback.isCanceled():
            return {}

        # ETAPA 11: Reprojetar buffers para EPSG:4326
        feedback.pushInfo("STEP 11: Redesigning buffers...")
        
        # Gerar nome de arquivo temporário único para os buffers reprojetados
        temp_file_buffers_reprojetados = os.path.join(tempfile.gettempdir(), f'buffers_reprojetados_{uuid.uuid4().hex}.gpkg')
        temp_files.append(temp_file_buffers_reprojetados)
        
        # Remover arquivo se já existir
        if os.path.exists(temp_file_buffers_reprojetados):
            os.remove(temp_file_buffers_reprojetados)
            
        alg_params = {
            'INPUT': temp_file_buffers,
            'TARGET_CRS': QgsCoordinateReferenceSystem('EPSG:4326'),
            'OUTPUT': temp_file_buffers_reprojetados
        }
        try:
            outputs['ReprojetarBuffers'] = processing.run(
                'native:reprojectlayer', 
                alg_params, 
                context=context, 
                feedback=feedback, 
                is_child_algorithm=True
            )
            feedback.pushInfo("✓ Buffers successfully redesigned")
        except Exception as e:
            feedback.reportError(f"Error redesigning buffers: {str(e)}")
            # Limpar arquivos temporários em caso de erro
            for temp_file in temp_files:
                try:
                    if os.path.exists(temp_file):
                        os.remove(temp_file)
                except:
                    pass
            return {}
        
        feedback.setCurrentStep(11)
        if feedback.isCanceled():
            return {}

        # ETAPA 12: Unir atributos por localização (pontos originais dentro dos buffers)
        feedback.pushInfo("STEP 12: Merging attributes by location...")
        
        # Gerar nome de arquivo temporário único para a união de atributos
        temp_file_atributos_unidos = os.path.join(tempfile.gettempdir(), f'atributos_unidos_{uuid.uuid4().hex}.gpkg')
        temp_files.append(temp_file_atributos_unidos)
        
        # Remover arquivo se já existir
        if os.path.exists(temp_file_atributos_unidos):
            os.remove(temp_file_atributos_unidos)
            
        alg_params = {
            'DISCARD_NONMATCHING': True,
            'INPUT': outputs['AdicionarIndice']['OUTPUT'],  # Pontos originais com índice
            'JOIN': temp_file_buffers_reprojetados,  # Buffers reprojetados
            'PREDICATE': [0],  # Intersect
            'OUTPUT': temp_file_atributos_unidos
        }
        try:
            outputs['UnirAtributos'] = processing.run(
                'native:joinattributesbylocation', 
                alg_params, 
                context=context, 
                feedback=feedback, 
                is_child_algorithm=True
            )
            feedback.pushInfo("✓ Attributes successfully united")
        except Exception as e:
            feedback.reportError(f"Error joining attributes: {str(e)}")
            # Limpar arquivos temporários em caso de erro
            for temp_file in temp_files:
                try:
                    if os.path.exists(temp_file):
                        os.remove(temp_file)
                except:
                    pass
            return {}
        
        feedback.setCurrentStep(12)
        if feedback.isCanceled():
            return {}

        # ETAPA 13: Extrair índices dos pontos simplificados
        feedback.pushInfo("STEP 13: Extracting simplified indexes...")
        try:
            # Carregar a camada simplificada diretamente do arquivo temporário
            simplified_layer = QgsVectorLayer(temp_file_atributos_unidos, 'simplified_layer', 'ogr')
            
            if simplified_layer is None or not simplified_layer.isValid():
                feedback.reportError("Unable to load simplified layer")
                feedback.reportError("Simplified layer invalid after all attempts")
                return {}
            
            # Verificar se o campo 'original_index' existe
            field_names = [field.name() for field in simplified_layer.fields()]
            if 'original_index' not in field_names:
                feedback.reportError("Field 'original_index' not found in simplified layer")
                return {}
            
            # Coletar índices originais dos pontos simplificados
            indices_simplificados = set()
            feature_count = 0
            for feature in simplified_layer.getFeatures():
                index = feature.attribute('original_index')
                if index is not None:
                    indices_simplificados.add(int(index))
                feature_count += 1
            
            feedback.pushInfo(f"✓ Processed {feature_count} features")
            feedback.pushInfo(f"✓ Found {len(indices_simplificados)} simplified indices")
            
            if parameters['adicionar_pontos_simplificados']:
                csv_path = self.parameterAsString(parameters, 'voo_em_csv', context)
                csv_name = os.path.splitext(os.path.basename(csv_path))[0]
                simplified_layer.setName(f"{csv_name} - Simplified CSV")
                QgsProject.instance().addMapLayer(simplified_layer)
                feedback.pushInfo("✓ Simplified points layer added to project")
                    
        except Exception as e:
            feedback.reportError(f"Error extracting indexes: {str(e)}")
            # Limpar arquivos temporários em caso de erro
            for temp_file in temp_files:
                try:
                    if os.path.exists(temp_file):
                        os.remove(temp_file)
                except:
                    pass
            return {}
        
        feedback.setCurrentStep(13)
        if feedback.isCanceled():
            return {}

        # ETAPA 14: Filtrar CSV original com base nos índices
        feedback.pushInfo("STEP 14: Filtering original CSV...")
        try:
            csv_path = self.parameterAsString(parameters, 'voo_em_csv', context)
            csv_saida = self.parameterAsString(parameters, 'csv_saida', context)
            
            with open(csv_path, 'r', encoding='utf-8') as arquivo_entrada:
                leitor = csv.reader(arquivo_entrada)
                cabecalho = next(leitor)
                linhas = list(leitor)
                
                with open(csv_saida, 'w', newline='', encoding='utf-8') as arquivo_saida:
                    escritor = csv.writer(arquivo_saida)
                    escritor.writerow(cabecalho)
                    
                    # Escrever linhas correspondentes aos índices simplificados
                    for idx, linha in enumerate(linhas, start=1):
                        if idx in indices_simplificados:
                            escritor.writerow(linha)
            
            feedback.pushInfo(f"✓ Simplified CSV saved in: {csv_saida}")
            results['CSV_Simplificado'] = csv_saida
            
        except Exception as e:
            feedback.reportError(f"Error filtering CSV: {str(e)}")
            # Limpar arquivos temporários em caso de erro
            for temp_file in temp_files:
                try:
                    if os.path.exists(temp_file):
                        os.remove(temp_file)
                except:
                    pass
            return {}
        
        # Limpar arquivos temporários
        for temp_file in temp_files:
            try:
                if os.path.exists(temp_file):
                    os.remove(temp_file)
            except:
                pass
        
        feedback.pushInfo("=" * 50)
        feedback.pushInfo("PROCESSING COMPLETED SUCCESSFULLY!")
        feedback.pushInfo("=" * 50)

        return results

    def name(self):
        return 'SimplifyWaypoints'.lower()

    def displayName(self):
        return self.tr('1. Simplify Waypoints')

    def group(self):
        return 'CSV Tools'

    def groupId(self):
        return 'CSVTools'.lower()

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return CSV_Simplify()

    def tags(self):
        return self.tr('Flight Plan,Measure,Topography,Plano voo,Plano de voo,voo,drones,GeoOne,simplify,simplificar,generalize').split(',')

    def icon(self):
        return QIcon(os.path.join(os.path.dirname(os.path.dirname(__file__)), 'images/CSV.png'))
    
    figura2 = 'images/csv_simplipy.jpg'

    def shortHelpString(self):
        texto = f"""
        <p>
        This tool optimizes the flight plan by reducing the number of waypoints through a three-dimensional tolerance relative to the original flight line. The algorithm preserves the photo spacing defined during the planning stage, ensuring that the <strong>number of photos and final coverage remain unchanged</strong>.
        </p>
        <p> The result is a leaner flight plan with fewer waypoints, without compromising photogrammetric quality. This approach is particularly useful for drones with <em>limitations on the maximum number of waypoints</em> — such as the Phantom 4 and older DJI Mini models — while also making mission execution more stable, reducing the risk of <em>freezes or slowdowns</em> during the flight.</p>
        <p>It generates a new <b>CSV</b> with simplified flight plan witch compatible with the <b>Litchi</b> app.</p>
        <p>It can also be used with other flight applications, utilizing the generated layers for flight lines and waypoints.</p>
        """

        corpo = '''<div align="center">
                      <img src="'''+ os.path.join(os.path.dirname(os.path.dirname(__file__)), self.figura2) +'''">
                      </div>
                      <div align="right">
                      <p><b>Learn more:</b></p>
                        <ul>
                            <li><a href="https://geoone.com.br/pvplanodevoo">Sign up for GeoFlight Planner course</a></li>
                            <li><a href="https://portal.geoone.com.br/m/lessons/planodevoo?classId=6025">Click here to access the class with all the details about this tool!</a></li>
                        </ul>
                      <p align="right">
                      <b>Autores: Prof Cazaroli, Leandro França and Ilton Freitas</b>
                      </p>
                      <a target="_blank" rel="noopener noreferrer" href="https://geoone.com.br/"><img title="GeoOne" src="data:image/png;base64,'''+ GeoOne +'''"></a>
					  <p><i>"Automated, easy and straight to the point mapping is at GeoOne!"</i></p>
                      </div>
                    </div>'''

        return self.tr(texto) + corpo