# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Flight Planner H Linear - Manually placed Side and Front distance data
                                 A QGIS plugin
 Flight Planner H Linear
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-11-21
        copyright            : (C) 2025 by Prof Cazaroli e Leandro França
        email                : contato@geoone.com.br
***************************************************************************/
"""

__author__ = 'Prof Cazaroli and Leandro França'
__date__ = '2025-11-21'
__copyright__ = '(C) 2025 by Prof Cazaroli and Leandro França'
__revision__ = '$Format:%H$'

from qgis.core import *
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from .Funcs import gerar_CSV, set_Z_value, reprojeta_camada_WGS84, simbologiaLinhaVoo, simbologiaPontos, simbologiaPontos3D, verificarCRS, loadParametros, saveParametros, arredondar_para_cima, pontos3D
from ..images.Imgs import *
import os

class PlanoVoo_H_Line(QgsProcessingAlgorithm):
    def initAlgorithm(self, config=None):
        hVooL, abGroundL, dlL, nLinL, dfL, velocL, tStayL, gimbalL, rasterL, csvL = loadParametros("H_Line")

        self.addParameter(QgsProcessingParameterVectorLayer('linha', 'Axis_Line', types=[QgsProcessing.TypeVectorLine]))
        
        self.addParameter(QgsProcessingParameterNumber('altura','Flight Height (m)',
                                                       type=QgsProcessingParameterNumber.Double, minValue=2,defaultValue=hVooL))
        self.addParameter(QgsProcessingParameterBoolean('aboveGround', 'Above Ground (Follow Terrain)',defaultValue=abGroundL))

        frontal_opts = [self.tr('2 Lines'), self.tr('3 Lines (Axis included)'), self.tr('4 Lines'), self.tr('5 Lines (Axis include)')]
        self.addParameter(QgsProcessingParameterEnum('nLinhas', self.tr('Number of Flight Routes (from 1 to 5)'), options = frontal_opts, defaultValue= nLinL))

        self.addParameter(QgsProcessingParameterNumber('bf','Lateral Buffer (m)',
                                                       type=QgsProcessingParameterNumber.Double, minValue=0.5,defaultValue=dlL))
        
        self.addParameter(QgsProcessingParameterNumber('df','Front Spacing Between Photos (m)',
                                                       type=QgsProcessingParameterNumber.Double, minValue=1,defaultValue=dfL))
        self.addParameter(QgsProcessingParameterNumber('velocidade','Flight Speed (m/s)',
                                                       type=QgsProcessingParameterNumber.Double, minValue=1,maxValue=20,defaultValue=velocL))
        self.addParameter(QgsProcessingParameterNumber('tempo','Time to Wait for Photo (seconds)',
                                                       type=QgsProcessingParameterNumber.Integer, minValue=0,maxValue=10,defaultValue=tStayL))
        self.addParameter(QgsProcessingParameterNumber('gimbalAng','Gimbal Angle (degrees)',
                                                       type=QgsProcessingParameterNumber.Integer, minValue=-90, maxValue=70, defaultValue=gimbalL))
        self.addParameter(QgsProcessingParameterRasterLayer('raster','Input Raster (if any)', defaultValue=rasterL, optional=True))
        #self.addParameter(QgsProcessingParameterFolderDestination('saida_kml', 'Output Folder for kml (Google Earth)', defaultValue=skml, optional=True))
        self.addParameter(QgsProcessingParameterFileDestination('saida_csv', 'Output CSV File (Litchi)', fileFilter='CSV files (*.csv)', defaultValue=csvL))

    def processAlgorithm(self, parameters, context, feedback):
        teste = False # Quando True mostra camadas intermediárias

        # =====Parâmetros de entrada para variáveis==============================
        linha_layer = self.parameterAsVectorLayer(parameters, 'linha', context)
        if not linha_layer:
            raise QgsProcessingException("❌ Axis line layer is required.")
    
        camadaMDE = self.parameterAsRasterLayer(parameters, 'raster', context)

        altVoo = parameters['altura']
        terrain = parameters['aboveGround']
        # incluir_eixo  → controla se a linha central participa do voo
        # dois_buffers  → controla se existe 2º offset de cada lado

        n_linhas = parameters['nLinhas']
        if n_linhas == 0:        # 2 linhas
            incluir_eixo = False
            dois_buffers = False
        elif n_linhas == 1:      # 3 linhas (com eixo)
            incluir_eixo = True
            dois_buffers = False
        elif n_linhas == 2:      # 4 linhas
            incluir_eixo = False
            dois_buffers = True
        elif n_linhas == 3:      # 5 linhas (com eixo)
            incluir_eixo = True
            dois_buffers = True
        else:
            raise QgsProcessingException("❌ Invalid number of flight lines option.")

        deltaLat = parameters['bf']          # Distância Buffer de voo paralelas - sem cálculo
        deltaFront = parameters['df']        # Espaçamento Frontal entre as fotografias- sem cálculo
        velocidade = parameters['velocidade']
        tempo = parameters['tempo']
        gimbalAng = parameters['gimbalAng']
        raster_layer = self.parameterAsRasterLayer(parameters, 'raster', context)
        arquivo_csv = self.parameterAsFile(parameters, 'saida_csv', context)

        # ===== Verificações =====================================================
        # Verificar se as camadas estão salvas e fora da edição
        for lyr, nome in [(linha_layer, 'Axis_Line')]:
            if lyr.isEditable():
                raise QgsProcessingException(f"❌ Layer '{nome}' is in edit mode. Please save and exit editing before continuing.")
            
            # Detecta camada temporária ou não salva
            storage_type = lyr.dataProvider().storageType().lower()
            uri = lyr.dataProvider().dataSourceUri().lower()
            if storage_type == '' or 'memory:' in uri or '/temporary/' in uri:
                raise QgsProcessingException(f"❌ Layer '{nome}' is not saved. Save the layer to disk before using it.")

        # Verificar caminho das pastas
        if 'saida_csv' not in parameters:
            raise QgsProcessingException("❌ Path to CSV file is empty!")

        if arquivo_csv:
            if not os.path.exists(os.path.dirname(arquivo_csv)):
                raise QgsProcessingException("❌ Path to CSV file does not exist!")
            
        # Verificar as Geometrias
        if linha_layer.featureCount() != 1:
            raise QgsProcessingException("❌ The Axis must contain only one line.")
        
        # Verificar o SRC das Camadas
        crs = linha_layer.crs()

        if "UTM" in crs.description().upper():
            feedback.pushInfo(f"The layer 'Axis' is already in CRS UTM.")
        elif "WGS 84" in crs.description().upper() or "SIRGAS 2000" in crs.description().upper():
            crs = verificarCRS(linha_layer, feedback)
            nome = linha_layer.name() + "_reproject"
            linha_layer = QgsProject.instance().mapLayersByName(nome)[0]
        else:
            raise QgsProcessingException(f"❌ Layer must be WGS84 or SIRGAS2000 or UTM. Other ({crs.description().upper()}) not supported")
        
        linha_features = next(linha_layer.getFeatures()) # dados do Eixo (Axis)
        geom = linha_features.geometry()

        if geom.isEmpty():
            raise QgsProcessingException("❌ Axis geometry is empty.")
    
        # Grava Parâmetros
        saveParametros("H_Line",
                        h=parameters['altura'],
                        v=parameters['velocidade'],
                        t=parameters['tempo'],
                        gimbal=parameters['gimbalAng'],
                        raster=raster_layer.source() if raster_layer else "",
                        csv=arquivo_csv,
                        abGround=parameters['aboveGround'],
                        dl=parameters['bf'],
                        df=parameters['df'],
                        dfop=parameters['nLinhas']
                        )

        # ===== Normalizar para LineString ==============================================
        def to_linestring(g: QgsGeometry) -> QgsGeometry:
            if g.isMultipart():
                ml = g.asMultiPolyline()
                if ml:
                    longest = max(ml, key=lambda pts: QgsGeometry.fromPolylineXY([QgsPointXY(p.x(), p.y()) for p in pts]).length())
                    return QgsGeometry.fromPolylineXY([QgsPointXY(p.x(), p.y()) for p in longest])
            pl = g.asPolyline()
            if pl:
                return QgsGeometry.fromPolylineXY([QgsPointXY(p.x(), p.y()) for p in pl])
            return g.convertToSingleType()

        linha_geom = to_linestring(geom)
        
        # ===== Offsets laterais (direita/esquerda) =====
        def valid_or_fallback(curve: QgsGeometry) -> QgsGeometry:
            if curve and not curve.isEmpty() and curve.isGeosValid():
                return to_linestring(curve)
            return linha_geom
        
        if incluir_eixo:
            linha_direita1 = linha_geom.offsetCurve(-deltaLat, segments=8, joinStyle=QgsGeometry.JoinStyleRound, miterLimit=2.0)
            linha_esquerda1 = linha_geom.offsetCurve(deltaLat, segments=8, joinStyle=QgsGeometry.JoinStyleRound, miterLimit=2.0)
        else:
            linha_direita1 = linha_geom.offsetCurve(-deltaLat/2, segments=8, joinStyle=QgsGeometry.JoinStyleRound, miterLimit=2.0)
            linha_esquerda1 = linha_geom.offsetCurve(deltaLat/2, segments=8, joinStyle=QgsGeometry.JoinStyleRound, miterLimit=2.0)

        linha_direita1 = valid_or_fallback(linha_direita1)
        linha_esquerda1 = valid_or_fallback(linha_esquerda1)

        linha_direita2 = None
        linha_esquerda2 = None
        if dois_buffers:
            if incluir_eixo:
                linha_direita2 = valid_or_fallback(linha_geom.offsetCurve(-2*deltaLat, segments=8, joinStyle=QgsGeometry.JoinStyleRound, miterLimit=2.0))
                linha_esquerda2 = valid_or_fallback(linha_geom.offsetCurve(2*deltaLat, segments=8, joinStyle=QgsGeometry.JoinStyleRound, miterLimit=2.0))
            else:
                linha_direita2 = valid_or_fallback(linha_geom.offsetCurve(-2*deltaLat + deltaLat/2, segments=8, joinStyle=QgsGeometry.JoinStyleRound, miterLimit=2.0))
                linha_esquerda2 = valid_or_fallback(linha_geom.offsetCurve(2*deltaLat - deltaLat/2, segments=8, joinStyle=QgsGeometry.JoinStyleRound, miterLimit=2.0))

        # ===== Determinar lado inicial da linha-eixo conforme orientação da linha =====
        linha_pts = linha_geom.asPolyline()
        inicio = QgsPointXY(linha_pts[0])
        fim = QgsPointXY(linha_pts[-1])

        dx = fim.x() - inicio.x()
        dy = fim.y() - inicio.y()

        lado_inicial = "direita" if abs(dx) >= abs(dy) else "esquerda"

        # ===== Geração de pontos ao longo das linhas =====
        def gerar_pontos(geom_line: QgsGeometry) -> list:
            comp = geom_line.length()
            dist = 0.0
            pontos = []
            while dist < comp:
                pontos.append(geom_line.interpolate(dist))
                dist += deltaFront
            pontos.append(geom_line.interpolate(comp))
            return pontos

        if lado_inicial == "esquerda":
            pontos_esquerda2 = gerar_pontos(linha_esquerda2) if dois_buffers else []
            pontos_esquerda1 = gerar_pontos(linha_esquerda1)
            pontos_eixo = gerar_pontos(linha_geom) if incluir_eixo else []
            pontos_direita1 = gerar_pontos(linha_direita1)
            pontos_direita2 = gerar_pontos(linha_direita2) if dois_buffers else []
        else:
            pontos_direita2 = gerar_pontos(linha_direita2) if dois_buffers else []
            pontos_direita1 = gerar_pontos(linha_direita1)
            pontos_eixo = gerar_pontos(linha_geom) if incluir_eixo else []
            pontos_esquerda1 = gerar_pontos(linha_esquerda1)
            pontos_esquerda2 = gerar_pontos(linha_esquerda2) if dois_buffers else []

        # ===== Ajuste da ordem dos pontos =====
        def ajustar_ordem(pontos_atual, pontos_anterior):
            if not pontos_atual or not pontos_anterior:
                return pontos_atual
            dist_inicio = pontos_atual[0].distance(pontos_anterior[-1])
            dist_fim = pontos_atual[-1].distance(pontos_anterior[-1])
            if dist_fim < dist_inicio:
                pontos_atual.reverse()
            return pontos_atual

        seq = []

        def append_seg(lista_pontos, nome):
            # Não adiciona listas vazias
            if not lista_pontos:
                return
            # Ajusta ordem em relação ao último segmento já adicionado
            if seq:
                lista_pontos = ajustar_ordem(lista_pontos, seq[-1][0])
            seq.append((lista_pontos, nome))

        # Detectar quantidade de buffers com base nas listas calculadas
        dois_buffers = bool(pontos_direita2) and bool(pontos_esquerda2)

        if lado_inicial == "esquerda":
            if dois_buffers:
                append_seg(pontos_esquerda2, 'esquerda2')
                append_seg(pontos_esquerda1, 'esquerda1')
                if incluir_eixo and pontos_eixo:
                    append_seg(pontos_eixo, 'eixo')
                append_seg(pontos_direita1, 'direita1')
                append_seg(pontos_direita2, 'direita2')
            else:
                append_seg(pontos_esquerda1, 'esquerda1')
                if incluir_eixo and pontos_eixo:
                    append_seg(pontos_eixo, 'eixo')
                append_seg(pontos_direita1, 'direita1')
        else:  # lado_inicial == "direita"
            if dois_buffers:
                append_seg(pontos_direita2, 'direita2')
                append_seg(pontos_direita1, 'direita1')
                if incluir_eixo and pontos_eixo:
                    append_seg(pontos_eixo, 'eixo')
                append_seg(pontos_esquerda1, 'esquerda1')
                append_seg(pontos_esquerda2, 'esquerda2')
            else:
                append_seg(pontos_direita1, 'direita1')
                if incluir_eixo and pontos_eixo:
                    append_seg(pontos_eixo, 'eixo')
                append_seg(pontos_esquerda1, 'esquerda1')      

        # ===== Camada de pontos =====
        pontos_layer = QgsVectorLayer(f'Point?crs={crs.authid()}', 'Photo Points', 'memory')
        prov = pontos_layer.dataProvider()
        prov.addAttributes([
            QgsField("id", QVariant.Int),
            QgsField("latitude", QVariant.Double),
            QgsField("longitude", QVariant.Double),
            QgsField("altitude", QVariant.Double),
            QgsField("height", QVariant.Double)
        ])
        pontos_layer.updateFields()

        # ===== Inserir features numeradas =====
        transformador = QgsCoordinateTransform(pontos_layer.crs(), QgsCoordinateReferenceSystem('EPSG:4326'), QgsProject.instance())

        contador = 1
        for lista, tipo in seq:
            for ponto in lista:
                if not ponto or ponto.isEmpty():
                    continue
                geom = ponto.asPoint()
                geom_utm = QgsGeometry.fromPointXY(QgsPointXY(geom.x(), geom.y()))
                geom_wgs = transformador.transform(QgsPointXY(geom.x(), geom.y()))

                f = QgsFeature()
                f.setGeometry(geom_utm)
                f.setAttributes([
                    contador,
                    geom_wgs.y(),   # latitude
                    geom_wgs.x(),   # longitude
                    None,           # altitude preenchida depois
                    altVoo               # height (altura de voo)
                ])
                prov.addFeature(f)
                contador += 1

        pontos_layer.updateExtents()
        feedback.pushInfo(f"✅ {contador-1} Photo Points generated.")
        
        # ===== Altitude via MDE =====
        if camadaMDE:
            transformadorMDE = QgsCoordinateTransform(pontos_layer.crs(), camadaMDE.crs(), QgsProject.instance())
            pontos_layer.startEditing()
            for f in pontos_layer.getFeatures():
                pt = f.geometry().asPoint()
                pt_transf = transformadorMDE.transform(QgsPointXY(pt.x(), pt.y()))
                value, result = camadaMDE.dataProvider().sample(pt_transf, 1)
                if result:
                    f["altitude"] = value + altVoo
                    f["height"] = altVoo
                    pontos_layer.updateFeature(f)
            pontos_layer.commitChanges()

        # ===== Reprojetar para WGS84 e aplicar Z =====
        crs_wgs = QgsCoordinateReferenceSystem('EPSG:4326')
        pontos_reproj = reprojeta_camada_WGS84(pontos_layer, crs_wgs, transformador)

        # Point para PointZ
        if camadaMDE:
            pontos_reproj = set_Z_value(pontos_reproj, z_field="altitude")
            pontos_reproj = pontos3D(pontos_reproj)
            simbologiaPontos3D(pontos_reproj, "VH_Line")
        else:
            pontos_reproj = set_Z_value(pontos_reproj, z_field="height")
            simbologiaPontos(pontos_reproj)
            
            QgsProject.instance().addMapLayer(pontos_reproj)

        # ===== Camada de linhas unindo os pontos de cada linha de voo =====
        linhas_voo_layer = QgsVectorLayer(
            f'LineString?crs={crs.authid()}',
            'Flight Lines',
            'memory'
        )
        prov_linhas = linhas_voo_layer.dataProvider()

        prov_linhas.addAttributes([
            QgsField("id", QVariant.Int),
            QgsField("tipo", QVariant.String)
        ])
        linhas_voo_layer.updateFields()

        id_linha = 1

        for lista_pontos, tipo in seq:
            if not lista_pontos or len(lista_pontos) < 2:
                continue

            pts = []
            for p in lista_pontos:
                if not p or p.isEmpty():
                    continue
                pt = p.asPoint()
                pts.append(QgsPointXY(pt.x(), pt.y()))

            if len(pts) < 2:
                continue

            feat = QgsFeature()
            feat.setGeometry(QgsGeometry.fromPolylineXY(pts))
            feat.setAttributes([id_linha, tipo])
            prov_linhas.addFeature(feat)

            id_linha += 1

        linhas_voo_layer.updateExtents()

        # ===== Simbologia das linhas de voo =====
        simbologiaLinhaVoo('L', linhas_voo_layer)

        # ===== Adiciona a camada de linhas ao projeto =====
        QgsProject.instance().addMapLayer(linhas_voo_layer)

        feedback.pushInfo("")
        feedback.pushInfo("✅ Flight Line and Photo Spots completed.")
        
        # =============L I T C H I==========================================================

        feedback.pushInfo("")

        if arquivo_csv and arquivo_csv.endswith('.csv'): # Verificar se o caminho CSV está preenchido
            gerar_CSV("L", pontos_reproj, arquivo_csv, velocidade, tempo, arredondar_para_cima(deltaFront, 2), 360, altVoo, gimbalAng, terrain, n_linhas)

            feedback.pushInfo("✅ CSV file successfully generated.")
        else:
            feedback.pushInfo("❌ CSV path not specified. Export step skipped.")

        
        # ============= Mensagem de Encerramento =====================================================
        feedback.pushInfo("")
        feedback.pushInfo("✅ Horizontal Flight Plan successfully executed.")
        feedback.pushInfo("")
        
        return {}

    def name(self):
        return 'Flight_Plan_H_Line'

    def displayName(self):
        return self.tr('4. Following terrain - Line')

    def group(self):
        return 'Horizontal Flight'

    def groupId(self):
        return 'horizontal'
    
    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return PlanoVoo_H_Line()

    def tags(self):
        return self.tr('Flight Plan,Measure,Topography,Plano voo,Plano de voo,voo,drone,GeoOne,Linear,Track,Road,Line,Channel').split(',')

    def icon(self):
        return QIcon(os.path.join(os.path.dirname(os.path.dirname(__file__)), 'images/Horizontal.png'))

    texto = """This tool generates a <b>horizontal linear flight plan</b> from a single <b>axis line</b>, creating <b>2 to 5 parallel flight routes</b> with automatically placed photo waypoints.
<p>
You define the <b>lateral offset</b>, <b>front photo spacing</b>, flight height, speed and gimbal angle, and the tool builds a <b>continuous mission</b> along the line.
</p>
<p>
When a <b>DEM</b> is provided, the flight can <b>follow the terrain</b>, adjusting waypoint altitudes to maintain a consistent height above ground.
</p>
<p>
The result is a <b>Litchi-compatible CSV</b> and optional flight line and waypoint layers in QGIS, ideal for mapping roads, corridors and other linear features.
</p>
"""

    figura2 = 'images/linear_flight.jpg'

    def shortHelpString(self):
        corpo = '''<div align="center">
                      <img src="'''+ os.path.join(os.path.dirname(os.path.dirname(__file__)), self.figura2) +'''">
                      </div>
                      <div align="right">
                      <p><b>Learn more:</b><o:p></o:p></p>
                        <ul style="margin-top: 0cm;" type="disc">
                        <li><a href="https://geoone.com.br/pvplanodevoo">Sign up for GeoFlight Planner course</a><o:p></o:p></li>
                        <li><a href="https://portal.geoone.com.br/m/lessons/planodevoo?classId=6022">Click here to access the class with all the details about this tool!</a><o:p></o:p></li>
                        </ul>
                      <p align="right">
                      <b>Autores: Prof Cazaroli & Leandro França</b>
                      </p>
                      <a target="_blank" rel="noopener noreferrer" href="https://geoone.com.br/"><img title="GeoOne" src="data:image/png;base64,'''+ GeoOne +'''"></a>
					  <p><i>"Automated, easy and straight to the point mapping is at GeoOne!"</i></p>
                      </div>
                    </div>'''
        return self.tr(self.texto) + corpo