# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Flight Planner - Vertical Flight Facade
                                 A QGIS plugin
 Flight Planner VF
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-12-02
        copyright            : (C) 2024 by Prof Cazaroli e Leandro França
        email                : contato@geoone.com.br
***************************************************************************/
"""

__author__ = 'Prof Cazaroli e Leandro França'
__date__ = '2024-12-02'
__copyright__ = '(C) 2024 by Prof Cazaroli e Leandro França'
__revision__ = '$Format:%H$'

from qgis.core import *
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from ..images.Imgs import *
import processing
import os
import math
import numpy as np

from .Funcs import (
    gerar_CSV,
    set_Z_value,
    reprojeta_camada_WGS84,
    pontos3D,
    simbologiaPontos,
    simbologiaPontos3D,
    verificarCRS,
    loadParametros,
    saveParametros,
    calculaDistancia_Linha_Ponto,
    removeLayersReproj
)

class PlanoVoo_V_F(QgsProcessingAlgorithm):
    def initAlgorithm(self, config=None):
        hObjVF, altMinVF, dlVF, dfVF, velocVF, tStayVF, gimbalVF, rasterVF, csvVF = loadParametros("VF")

        self.addParameter(QgsProcessingParameterVectorLayer('linha_base','Flight Base Line', types=[QgsProcessing.TypeVectorLine]))
        self.addParameter(QgsProcessingParameterVectorLayer('ponto_base','Position Point of the Facade', types=[QgsProcessing.TypeVectorPoint]))
        self.addParameter(QgsProcessingParameterNumber('altura','Facade Height (m)',
                                                       type=QgsProcessingParameterNumber.Double, minValue=2,defaultValue=hObjVF))
        self.addParameter(QgsProcessingParameterNumber('alturaMin','Start Height (m)',
                                                       type=QgsProcessingParameterNumber.Double, minValue=0.5,defaultValue=altMinVF))
        self.addParameter(QgsProcessingParameterNumber('dl','Spacing between Flight Lines (m)',
                                                       type=QgsProcessingParameterNumber.Double,
                                                       minValue=0.5,defaultValue=dlVF))
        self.addParameter(QgsProcessingParameterNumber('df','Spacing between Photos (m)',
                                                       type=QgsProcessingParameterNumber.Double,
                                                       minValue=0.5,defaultValue=dfVF))
        self.addParameter(QgsProcessingParameterNumber('velocidade','Flight Speed (m/s)',
                                                       type=QgsProcessingParameterNumber.Double, minValue=1,maxValue=20,defaultValue=velocVF))
        self.addParameter(QgsProcessingParameterNumber('tempo','Wait time for Photo (seconds)',
                                                       type=QgsProcessingParameterNumber.Integer, minValue=0,maxValue=10,defaultValue=tStayVF))
        self.addParameter(QgsProcessingParameterNumber('gimbalAng','Gimbal Angle (degrees)',
                                                       type=QgsProcessingParameterNumber.Integer, minValue=-90, maxValue=70, defaultValue=gimbalVF))
        self.addParameter(QgsProcessingParameterRasterLayer('raster','Input Raster (if any)', defaultValue=rasterVF, optional=True))
        #self.addParameter(QgsProcessingParameterFolderDestination('saida_kml', 'Output Folder for kml (Google Earth)', defaultValue=skml, optional=True))
        self.addParameter(QgsProcessingParameterFileDestination('saida_csv', 'Output CSV File (Litchi)', fileFilter='CSV files (*.csv)', defaultValue=csvVF))

    def processAlgorithm(self, parameters, context, feedback):
        teste = False # Quando True mostra camadas intermediárias

        # ===== Parâmetros de entrada para variáveis ====================================================
        linha_base = self.parameterAsVectorLayer(parameters, 'linha_base', context)

        ponto_base = self.parameterAsVectorLayer(parameters, 'ponto_base', context)

        camadaMDE = self.parameterAsRasterLayer(parameters, 'raster', context)

        H = parameters['altura']
        h = parameters['alturaMin']
        deltaLat = parameters['dl']   # Distância das linhas de voo paralelas - sem cálculo
        deltaFront = parameters['df'] # Espaçamento Frontal entre as fotografias- sem cálculo
        velocidade = parameters['velocidade']
        tempo = parameters['tempo']
        gimbalAng = parameters['gimbalAng']
        raster_layer = self.parameterAsRasterLayer(parameters, 'raster', context)
        arquivo_csv = self.parameterAsFile(parameters, 'saida_csv', context)

        # ===== Verificações =============================================================================
        # Verificar se as camadas estão salvas e fora da edição
        for lyr, nome in [(linha_base, 'Flight Base Line'), (ponto_base, 'Position Point of the Facade')]:
            if lyr.isEditable():
                raise QgsProcessingException(f"❌ Layer '{nome}' is in edit mode. Please save and exit editing before continuing.")
            
            # Detecta camada temporária ou não salva
            storage_type = lyr.dataProvider().storageType().lower()
            uri = lyr.dataProvider().dataSourceUri().lower()
            if storage_type == '' or 'memory:' in uri or '/temporary/' in uri:
                raise QgsProcessingException(f"❌ Layer '{nome}' is not saved. Save the layer to disk before using it.")

        # Verificar caminho das pastas
        if 'saida_csv' not in parameters:
            raise QgsProcessingException("❌ Path to CSV file is empty!")

        if arquivo_csv:
            if not os.path.exists(os.path.dirname(arquivo_csv)):
                raise QgsProcessingException("❌ Path to CSV file does not exist!")
            
        
        # Verificar as Geometrias
        if linha_base.featureCount() != 1:
            raise QgsProcessingException("❌ Flight Base Line must contain only one line.")

        if ponto_base.featureCount() != 1:
            raise QgsProcessingException("❌ Position of the Facade must contain only one point.")
        

        # Verificar o SRC das Camadas
        crs = linha_base.crs()
        crsP = ponto_base.crs() # não usamos o crsP, apenas para verificar a camada
        if crs != crsP:
            raise QgsProcessingException("❌ Both layers must be from the same CRS.")

        if "UTM" in crs.description().upper():
            feedback.pushInfo(f"The layer 'Flight Base Line' is already in CRS UTM.")
        elif "WGS 84" in crs.description().upper() or "SIRGAS 2000" in crs.description().upper():
            crs = verificarCRS(linha_base, feedback)
            nome = linha_base.name() + "_reproject"
            linha_base = QgsProject.instance().mapLayersByName(nome)[0]
        else:
            raise QgsProcessingException(f"❌ Layer must be WGS84 or SIRGAS2000 or UTM. Other ({crs.description().upper()}) not supported")

        if "UTM" in crsP.description().upper():
            feedback.pushInfo(f"The layer 'Position of the Facade' is already in CRS UTM.")
        elif "WGS 84" in crsP.description().upper() or "SIRGAS 2000" in crsP.description().upper():
            verificarCRS(ponto_base, feedback)
            nome = ponto_base.name() + "_reproject"
            ponto_base = QgsProject.instance().mapLayersByName(nome)[0]
        else:
            raise QgsProcessingException(f"❌ Layer must be WGS84 or SIRGAS2000 or UTM. Other ({crs.description().upper()}) not supported")


        linha = next(linha_base.getFeatures())
        linha_base_geom = linha.geometry()
        if linha_base_geom.isMultipart():
            linha_base_geom = linha_base_geom.asGeometryCollection()[0]

        p = next(ponto_base.getFeatures())
        ponto_base_geom = p.geometry()
        if ponto_base_geom.isMultipart():
            ponto_base_geom = ponto_base_geom.asGeometryCollection()[0]

        # Obtem a distância da Linha de Voo ao ponto_base
        dist_ponto_base = calculaDistancia_Linha_Ponto(linha_base_geom, ponto_base_geom)

        if dist_ponto_base <= 10:
            raise QgsProcessingException(f"❌ Horizontal distance ({round(dist_ponto_base, 2)}) is 10 meters or less.")

        # ===== Grava Parâmetros =====================================================
        saveParametros("VF",
                        h=parameters['altura'],
                        v=parameters['velocidade'],
                        t=parameters['tempo'],
                        gimbal=parameters['gimbalAng'],
                        raster=raster_layer.source() if raster_layer else "",
                        csv=arquivo_csv,
                        altMin=parameters['alturaMin'],
                        dl=parameters['dl'],
                        df=parameters['df'])

        # Mostra valores parciais
        feedback.pushInfo(f"✅ Flight Line to Facade Distance: {round(dist_ponto_base, 2)}     Facade Height: {round(H, 2)}")

        # ===== Sobreposições digitadas manualmente ====================================================

        feedback.pushInfo(f"✅ Lateral Spacing: {round(deltaLat,2)}, Frontal Spacing: {round(deltaFront,2)}")

        # Obtem as alturas das linhas de Voo (range só para números inteiros)
        alturas = [i for i in np.arange(h, H + h + 1, deltaLat)]

        # Obtem as distâncias nas linhas de Voo
        comprimento_linha_base = linha_base_geom.length() # comprimento da linha
        distancias = [i for i in np.arange(0, comprimento_linha_base, deltaFront)]

        # ===============================================================================
        # Reprojetar para WGS 84 (EPSG:4326), usado pelo OpenTopography
        crs_wgs = QgsCoordinateReferenceSystem('EPSG:4326')
        transformador = QgsCoordinateTransform(crs, crs_wgs, QgsProject.instance())
        # ===============================================================================

        # =============================================================================================
        # ===== Criar a camada Pontos de Fotos ========================================================

        # Criar uma camada Pontos com os deltaFront sobre a linha Base e depois empilhar com os deltaFront
        pontos_fotos = QgsVectorLayer('Point?crs=' + crs.authid(), 'Photo Points', 'memory')
        pontos_provider = pontos_fotos.dataProvider()

        # Definir campos
        campos = QgsFields()
        campos.append(QgsField("id", QVariant.Int))
        campos.append(QgsField("linha", QVariant.Int))
        campos.append(QgsField("latitude", QVariant.Double))
        campos.append(QgsField("longitude", QVariant.Double))
        campos.append(QgsField("altitude", QVariant.Double))
        campos.append(QgsField("height", QVariant.Double))
        pontos_provider.addAttributes(campos)
        pontos_fotos.updateFields()

        pontoID = 1

        # Verificar a posição da linha base em relação ao ponto_base que se quer medir
        linha = linha_base_geom.asPolyline()

        # Coordenadas da linha base
        p1 = linha[0]
        p2 = linha[-1]

        # Ângulo em relação ao norte (em graus)
        dx = p2.x() - p1.x()
        dy = p2.y() - p1.y()
        angulo_linha_base = math.degrees(math.atan2(dx, dy))

        # Calcular a perpendicular (90 graus)
        angulo_perpendicular = (angulo_linha_base + 90) % 360

        # Verificar orientação do ponto em relação à linha base
        ponto_base_point = ponto_base_geom.asPoint()

        # Calcular a equação da linha base (Ax + By + C = 0)
        A = p2.y() - p1.y()
        B = p1.x() - p2.x()
        C = p2.x() * p1.y() - p1.x() * p2.y()

        # Verificar o sinal ao substituir as coordenadas do ponto de orientação
        orientacao = A * ponto_base_point.x() + B * ponto_base_point.y() + C

        # Ajustar o ângulo da perpendicular com base na orientação
        if orientacao < 0:
            angulo_perpendicular += 180
            angulo_perpendicular %= 360

        feedback.pushInfo(f"✅ Angle of the perpendicular in relation to the North: {angulo_perpendicular:.2f}°")

        # Criar as carreiras de pontos
        for idx, altura in enumerate(alturas, start=1):  # Cada altura representa uma "linha"
            # Alternar o sentido
            if idx % 2 == 0:  # "Linha de vem" (segunda, quarta, ...)
                dist_horiz = reversed(distancias)
            else:  # "Linha de vai" (primeira, terceira, ...)
                dist_horiz = distancias

            for d in dist_horiz:
                if d == comprimento_linha_base:  # Ajuste para evitar problemas com interpolate
                    d = comprimento_linha_base

                ponto = linha_base_geom.interpolate(d).asPoint()
                ponto_geom = QgsGeometry.fromPointXY(QgsPointXY(ponto))

                # Obter altitude do MDE
                param_kml = 'relativeToGround'
                if camadaMDE:
                    param_kml = 'absolute'
                    transformadorMDE = QgsCoordinateTransform(linha_base.crs(), camadaMDE.crs(), QgsProject.instance())
                    ponto_mde = transformadorMDE.transform(QgsPointXY(ponto.x(), ponto.y()))
                    value, result = camadaMDE.dataProvider().sample(QgsPointXY(ponto_mde), 1)  # Resolução de amostragem
                    a = value if result else 0
                else:
                    a = 0

                # Criar o recurso de ponto
                Ponto_Geo = transformador.transform(ponto)
                ponto_feature = QgsFeature()
                ponto_feature.setFields(campos)
                ponto_feature.setAttribute("id", pontoID)
                ponto_feature.setAttribute("linha", idx)  # Linha correspondente à altura
                ponto_feature.setAttribute("latitude", Ponto_Geo.y())
                ponto_feature.setAttribute("longitude",Ponto_Geo.x())
                ponto_feature.setAttribute("altitude", a+float(altura))
                ponto_feature.setAttribute("height", float(altura))
                ponto_feature.setGeometry(ponto_geom)
                pontos_provider.addFeature(ponto_feature)

                pontoID += 1

        # Atualizar a camada
        pontos_fotos.updateExtents()
        pontos_fotos.commitChanges()

        # Reprojetar camada Pontos Fotos de UTM para WGS84 (4326)
        pontos_reproj = reprojeta_camada_WGS84(pontos_fotos, crs_wgs, transformador)

        # Point para PointZ
        if param_kml == 'absolute':
            pontos_reproj = set_Z_value(pontos_reproj, z_field="height")
            pontos_reproj = pontos3D(pontos_reproj)
            simbologiaPontos3D(pontos_reproj, "VF")
        else:
            pontos_reproj = set_Z_value(pontos_reproj, z_field="height")
            simbologiaPontos(pontos_reproj)
            
            QgsProject.instance().addMapLayer(pontos_reproj)

        feedback.pushInfo("")
        feedback.pushInfo("✅ Photo Spots generated.")

        # ===============================================================================
        # ===== Criar a camada "Linha de Voo" ===========================================

        linhas_facade_layer = QgsVectorLayer('LineStringZ?crs=' + crs.authid(), 'Flight Line', 'memory')
        linhas_facade_provider = linhas_facade_layer.dataProvider()

        # Definir campos
        campos = QgsFields()
        campos.append(QgsField("id", QVariant.Int))
        campos.append(QgsField("height", QVariant.Double))
        linhas_facade_provider.addAttributes(campos)
        linhas_facade_layer.updateFields()

        linhas_facade_layer.startEditing()

        novas_features = []
        linha_id = 1  # Inicializa ID das linhas

        for altura in alturas:
            nova_linha_geom = QgsGeometry.fromWkt(linha_base_geom.asWkt())

            # Extrair coordenadas da linha base e adicionar o valor de Z (altura)
            coordenadas = nova_linha_geom.asPolyline()
            coordenadas_z = [QgsPoint(pt.x(), pt.y(), altura) for pt in coordenadas]

            nova_linha_geom = QgsGeometry.fromPolyline(coordenadas_z)

            feature = QgsFeature()
            feature.setGeometry(nova_linha_geom)
            feature.setAttributes([linha_id, float(altura)])

            novas_features.append(feature)

            linha_id += 1

        linhas_facade_provider.addFeatures(novas_features)

        linhas_facade_layer.updateExtents()
        linhas_facade_layer.commitChanges()

         # Reprojetar linha Voo para WGS84 (4326)
        linha_voo_reproj = reprojeta_camada_WGS84(linhas_facade_layer, crs_wgs, transformador)

        # LineString paraLineStringZ
        linha_voo_reproj = set_Z_value(linha_voo_reproj, z_field="height")

        # ===== LINHA VOO =================================
        QgsProject.instance().addMapLayer(linha_voo_reproj)

        feedback.pushInfo("")
        feedback.pushInfo("✅ Flight Line and Photo Spots completed.")

        # =============L I T C H I==========================================================

        feedback.pushInfo("")

        if arquivo_csv and arquivo_csv.endswith('.csv'): # Verificar se o caminho CSV está preenchido
            gerar_CSV("VF", pontos_reproj, arquivo_csv, velocidade, tempo, deltaFront, angulo_perpendicular, H, gimbalAng)

            feedback.pushInfo("✅ CSV file successfully generated.")
        else:
            feedback.pushInfo("❌ CSV path not specified. Export step skipped.")

        # ============= Remover Camadas Reproject ===================================================

        removeLayersReproj('_reproject')

        # ============= Mensagem de Encerramento =====================================================
        feedback.pushInfo("")
        feedback.pushInfo("✅ Facade Vertical Flight Plan successfully executed.")
        feedback.pushInfo("")

        return {}

    def name(self):
        return 'Flight_Plan_V_F'

    def displayName(self):
        return self.tr('1. Facade')

    def group(self):
        return 'Vertical Flight'

    def groupId(self):
        return 'vertical'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return PlanoVoo_V_F()

    def tags(self):
        return self.tr('Flight Plan,Measure,Topography,Fachada,Vertical,Front,View,GeoOne').split(',')

    def icon(self):
        return QIcon(os.path.join(os.path.dirname(os.path.dirname(__file__)), 'images/Vertical.png'))

    texto = """This tool is designed for creating vertical flight plans tailored for mapping building facades, ideal for architectural projects and building inspections.
It enables the planning of a precise vertical trajectory with appropriate overlap and stop times for the drone, ensuring high-quality photographs and detailed mapping.</span></p>
<p class="MsoNormal"><b>Configuration Details:</b></p>
<ul style="margin-top: 0cm;" type="disc">
  <li><b><span>Estimated Facade Height:</span></b><span> Specifies the highest point of the facade to be mapped.</span></li>
  <li><b><span>Flight Base Line:</span></b><span> The path along which the drone will fly in front of the facade.</span></li>
  <li><b><span>Position of the Facade:</span></b><span> A reference point on the facade used to calculate overlap distances.</span></li>
</ul>
<p><span>The outputs are <b>CSV</b> file compatible with the <b>Litchi app</b>. and 2 Layers - <b>Flight Line</b> and <b>Photos Points</b>.
<p>It can also be used with other flight applications, utilizing the 2 genereted Layers for flight lines and waypoints.</p>
<p><b>
"""

    figura = 'images/Facade.jpg'

    def shortHelpString(self):
        corpo = '''<div align="center">
                      <img src="'''+ os.path.join(os.path.dirname(os.path.dirname(__file__)), self.figura) +'''">
                      </div>
                      <div align="right">
                      <p><b>Learn more:</b><o:p></o:p></p>
                        <ul style="margin-top: 0cm;" type="disc">
                        <li><a href="https://geoone.com.br/pvplanodevoo">Sign up for GeoFlight Planner course</a><o:p></o:p></li>
                        <li><a href="https://portal.geoone.com.br/m/lessons/planodevoo?classId=6023">Click here to access the class with all the details about this tool!</a><o:p></o:p></li>
                        </ul>
                      <p align="right">
                      <b>Autores: Prof Cazaroli & Leandro França</b>
                      </p>
                      <a target="_blank" rel="noopener noreferrer" href="https://geoone.com.br/"><img title="GeoOne" src="data:image/png;base64,'''+ GeoOne +'''"></a>
					  <p><i>"Automated, easy and straight to the point mapping is at GeoOne!"</i></p>
                      </div>
                    </div>'''
        return self.tr(self.texto) + corpo