# -*- coding: utf-8 -*-

"""
/******************************************************************************************
 Calculator GSD
 
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-07-04
        copyright            : (C) 2025 by Prof Cazaroli and Leandro França and Prof Ilton
        email                : contato@geoone.com.br
*******************************************************************************************/
"""

__author__ = 'Prof Cazaroli and Leandro França and Prof Ilton Freitas'
__date__ = '2025-07-04'
__copyright__ = '(C) 2025 by Prof Cazaroli and Leandro França and Prof Ilton Freitas'
__revision__ = '$Format:%H$'

import os
import json
from qgis.core import QgsSettings

from qgis.PyQt.QtWidgets import (QDialog, QScrollArea, QWidget, QVBoxLayout, QFormLayout, QHBoxLayout,
    QGroupBox, QLabel, QComboBox, QPushButton, QDoubleSpinBox, QSpinBox, QMessageBox, QTableWidget, QTableWidgetItem)
from qgis.gui import QgsCollapsibleGroupBox
from qgis.PyQt.QtGui import QDesktopServices
from qgis.PyQt.QtCore import QUrl, Qt

# Compat PyQt5/PyQt6 para ItemFlag
def _qt_item_flag(name: str):
    bucket = getattr(Qt, "ItemFlag", None)  # PyQt6 tem Qt.ItemFlag
    if bucket is not None and hasattr(bucket, name):
        return getattr(bucket, name)
    # fallback PyQt5: flags ficam direto em Qt
    return getattr(Qt, name)

ITEM_IS_EDITABLE = _qt_item_flag("ItemIsEditable")


# Calculation functions
def calculate_gsd_by_sensor(altitude_m, sensor_width_mm, sensor_height_mm, image_width_px, image_height_px, focal_mm):
    if altitude_m <= 0 or focal_mm <= 0:
        raise ValueError("Altitude needs to be defined.")
    
    footprint_w = altitude_m * sensor_width_mm / focal_mm
    footprint_h = altitude_m * sensor_height_mm / focal_mm
    gsd_h = footprint_w / image_width_px
    gsd_v = footprint_h / image_height_px
    gsd_m = max(gsd_h, gsd_v)

    return gsd_m * 100.0

def calculate_altitude_from_gsd(gsd_cm, sensor_width_mm, sensor_height_mm, image_width_px, image_height_px, focal_mm):
    if gsd_cm <= 0 or focal_mm <= 0:
        raise ValueError("GSD and focal length must be positive.")
    
    gsd_m = gsd_cm / 100.0
    pixel_size_w = sensor_width_mm / image_width_px
    pixel_size_h = sensor_height_mm / image_height_px
    pixel_size = max(pixel_size_w, pixel_size_h)
    altitude = (gsd_m * focal_mm) / pixel_size

    return altitude

def calculate_spacing(altitude_m, sensor_width_mm, sensor_height_mm, focal_mm, front_overlap_pct, side_overlap_pct):
    if altitude_m <= 0 or focal_mm <= 0:
        raise ValueError("Altitude needs to be defined.")
    
    footprint_w = altitude_m * sensor_width_mm / focal_mm
    footprint_h = altitude_m * sensor_height_mm / focal_mm
    forward_spacing = footprint_h * (1 - front_overlap_pct / 100.0)
    side_spacing = footprint_w * (1 - side_overlap_pct / 100.0)

    return forward_spacing, side_spacing

def calculate_overlap(altitude_m, sensor_width_mm, sensor_height_mm, focal_mm, spacing_forward, spacing_side):
    if altitude_m <= 0 or focal_mm <= 0:
        raise ValueError("Altitude needs to be defined.")
    
    footprint_w = altitude_m * sensor_width_mm / focal_mm
    footprint_h = altitude_m * sensor_height_mm / focal_mm
    front_overlap = max(0.0, 1 - spacing_forward / footprint_h) * 100.0 if footprint_h > 0 else 0.0
    side_overlap = max(0.0, 1 - spacing_side / footprint_w) * 100.0 if footprint_w > 0 else 0.0

    return front_overlap, side_overlap

class Calculator_Dialog(QDialog):
    def __init__(self, iface):
        super().__init__()

        self.setWindowTitle("GSD & Overlap - Mission Calculator")
        self.resize(550, 540)

        layout = QVBoxLayout(self)
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        layout.addWidget(scroll)

        container = QWidget()
        scroll.setWidget(container)
        container_layout = QVBoxLayout(container)

        # Drone selection
        self.drone_data = {}
        self.droneCombo = QComboBox()
        self.droneCombo.addItem("--- Select a Drone Model ---")
        try:
            with open(os.path.join(os.path.dirname(__file__), 'drone_data.json'), 'r', encoding='utf-8') as f:
                data = json.load(f)
                for name, specs in data.items():
                    self.drone_data[name] = specs
                    self.droneCombo.addItem(name)
        except Exception as e:
            QMessageBox.warning(self, "Warning", f"Failed to load drone data:\n{e}")

        self.saveButton = QPushButton("Save Custom Drone Data")
        self.saveButton.setVisible(False)
        self.saveButton.clicked.connect(self._save_custom_drone)

        drone_group = QGroupBox("GeoFlight Planner")
        form = QFormLayout(drone_group)
        
        # Combobox + SAVE button side by side
        drone_row_widget = QWidget()
        drone_row_layout = QHBoxLayout(drone_row_widget)
        drone_row_layout.setContentsMargins(0, 0, 0, 0)
        drone_row_layout.addWidget(self.droneCombo)
        drone_row_layout.addWidget(self.saveButton)
        form.addRow("Drone Model:", drone_row_widget)

        # Sensor info table
        self.table = QTableWidget(0, 2)
        self.table.setHorizontalHeaderLabels(["Parameter", "Value"])
        self.table.verticalHeader().setVisible(False)
        self.table.setColumnWidth(0, 150)
        self.table.setColumnWidth(1, 100)
        self.table.setMinimumHeight(150)
        form.addRow(self.table)

        container_layout.addWidget(drone_group)
        self.droneCombo.currentIndexChanged.connect(self._update_sensor_info)

        # Save buttom, data Drone in QGIS Environment
        self.saveToQGISButton = QPushButton("Save Drone Data to Flight Planner")
        self.saveToQGISButton.clicked.connect(self._save_to_qgis_environment)
        form.addRow(self.saveToQGISButton)

        # GSD Section
        self.altitudeInput = QDoubleSpinBox()
        self.altitudeInput.setRange(2.5, 500)
        self.gsdResult = QLabel("GSD: --")
        gsd_btn = QPushButton("Calculate GSD")
        gsd_btn.clicked.connect(self.calculate_gsd)
        gsd_box = QgsCollapsibleGroupBox("Ground Sample Distance (GSD) Estimation")
        gsd_box.setCollapsed(True)
        gsd_form = QFormLayout(gsd_box)
        self.altitudeInput.setValue(120)
        gsd_form.addRow("Flight Altitude (2.5 to 500 m):", self.altitudeInput)
        gsd_form.addRow("Result:", self.gsdResult)
        gsd_form.addRow(gsd_btn)
        container_layout.addWidget(gsd_box)

        # Altitude from GSD Section
        self.gsdInput = QDoubleSpinBox()
        self.gsdInput.setRange(0.5, 20)
        self.altitudeFromGSDResult = QLabel("Altitude: --")
        altitude_btn = QPushButton("Calculate Altitude")
        altitude_btn.clicked.connect(self.calculate_altitude_from_gsd)
        altitude_box = QgsCollapsibleGroupBox("Compute Altitude for desired GSD")
        altitude_box.setCollapsed(True)
        altitude_form = QFormLayout(altitude_box)
        self.gsdInput.setValue(5.0)
        altitude_form.addRow("GSD: (0.5 to 20 cm)", self.gsdInput)
        altitude_form.addRow("Result:", self.altitudeFromGSDResult)
        altitude_form.addRow(altitude_btn)
        container_layout.addWidget(altitude_box)

        # Spacing Section
        self.altitudeSpacing = QDoubleSpinBox()
        self.altitudeSpacing.setRange(2.5, 500)
        self.frontOverlap = QSpinBox()
        self.frontOverlap.setRange(50, 95)
        self.frontOverlap.setValue(85)
        self.sideOverlap = QSpinBox()
        self.sideOverlap.setRange(30, 95)
        self.sideOverlap.setValue(75)
        self.spacingResult = QLabel("Spacing: --")
        spacing_btn = QPushButton("Calculate Spacing")
        spacing_btn.clicked.connect(self.calculate_spacing)
        spacing_box = QgsCollapsibleGroupBox("Calculate Spacing from Overlaps")
        spacing_box.setCollapsed(True)
        spacing_form = QFormLayout(spacing_box)
        self.altitudeSpacing.setValue(120)
        spacing_form.addRow("Flight Altitude (2.5 to 500 m):", self.altitudeSpacing)
        spacing_form.addRow("Frontal Overlap (50 to 95%):", self.frontOverlap)
        spacing_form.addRow("Side Overlap (30 to 95%):", self.sideOverlap)
        spacing_form.addRow("Result:", self.spacingResult)
        spacing_form.addRow(spacing_btn)
        container_layout.addWidget(spacing_box)

        # Overlap Section
        self.altitudeOverlap = QDoubleSpinBox()
        self.altitudeOverlap.setRange(2.5, 500)
        self.spacingForward = QDoubleSpinBox()
        self.spacingForward.setRange(1, 100)
        self.spacingSide = QDoubleSpinBox()
        self.spacingSide.setRange(1, 100)
        self.overlapResult = QLabel("Overlap: --")
        overlap_btn = QPushButton("Calculate Overlap")
        overlap_btn.clicked.connect(self.calculate_overlap)
        overlap_box = QgsCollapsibleGroupBox("Calculate Overlap from Spacing")
        overlap_box.setCollapsed(True)
        overlap_form = QFormLayout(overlap_box)
        self.altitudeOverlap.setValue(120)
        overlap_form.addRow("Flight Altitude (2.5 to 500 m):", self.altitudeOverlap)
        overlap_form.addRow("Forward Photo Spacing (1 to 100 m):", self.spacingForward)
        overlap_form.addRow("Side Line Spacing (1 to 100 m):", self.spacingSide)
        overlap_form.addRow("Result:", self.overlapResult)
        overlap_form.addRow(overlap_btn)
        container_layout.addWidget(overlap_box)

        # Help Button
        help_btn = QPushButton("User Guide")
        help_btn.clicked.connect(self.open_help)
        container_layout.addWidget(help_btn)

    def _get_specs(self):
        selected = self.droneCombo.currentText()
        return self.drone_data.get(selected)

    def _update_sensor_info(self):  # Info Data Drone selected for Table Data
        selected = self.droneCombo.currentText()
        self.table.setRowCount(0)

        specs = self._get_specs()

        # Mostrar botão de salvar apenas quando for "Custom"
        self.saveButton.setVisible(selected == "Custom")

        if not specs:
            # Sem specs: mostra placeholders educativos (opcional)
            placeholders = {
                "sensor_width": 13.2,
                "sensor_height": 8.8,
                "focal_length": 8.8,
                "image_width": 4056,
                "image_height": 3040,
                "min_angle_cam": 0,
                "max_angle_cam": 90,
            }
            units = {
                "sensor_width": "mm",
                "sensor_height": "mm",
                "focal_length": "mm",
                "image_width": "px",
                "image_height": "px",
                "min_angle_cam": "deg",
                "max_angle_cam": "deg",
            }
            for row, (key, value) in enumerate(placeholders.items()):
                self.table.insertRow(row)

                # Coluna Parameter (sempre não editável)
                key_item = QTableWidgetItem(str(key))
                key_item.setFlags(key_item.flags() & ~ITEM_IS_EDITABLE)
                self.table.setItem(row, 0, key_item)

                # Coluna Value (não editável — só para ilustrar)
                unit = units.get(key, "")
                display_value = f"{value} {unit}" if unit else str(value)
                value_item = QTableWidgetItem(display_value)
                value_item.setFlags(value_item.flags() & ~ITEM_IS_EDITABLE)
                self.table.setItem(row, 1, value_item)
            return

        # Temos specs reais: preencher a tabela com base nelas
        units = {
            "sensor_width": "mm",
            "sensor_height": "mm",
            "focal_length": "mm",
            "image_width": "px",
            "image_height": "px",
            "min_angle_cam": "deg",
            "max_angle_cam": "deg",
        }

        for row, (key, value) in enumerate(specs.items()):
            self.table.insertRow(row)

            # Coluna Parameter (sempre não editável)
            key_item = QTableWidgetItem(str(key))
            key_item.setFlags(key_item.flags() & ~ITEM_IS_EDITABLE)  # <-- usar helper
            self.table.setItem(row, 0, key_item)

            # Coluna Value (editável apenas se for Custom)
            unit = units.get(key, "")
            display_value = f"{value} {unit}" if unit else str(value)
            value_item = QTableWidgetItem(display_value)
            if selected != "Custom":
                value_item.setFlags(value_item.flags() & ~ITEM_IS_EDITABLE)  # <-- usar helper
            self.table.setItem(row, 1, value_item)

        # selected = self.droneCombo.currentText()
        # self.table.setRowCount(0)
        # self.saveButton.setVisible(selected == "Custom")


        specs = self._get_specs()
        if specs:
            units = {
                "sensor_width": "mm",
                "sensor_height": "mm",
                "focal_length": "mm",
                "image_width": "px",
                "image_height": "px",
                "min_angle_cam": "degrees",
                "max_angle_cam": "degrees"
            }
            for row, (key, value) in enumerate(specs.items()):
                self.table.insertRow(row)

                # Coluna Parameter (sempre não editável)
                key_item = QTableWidgetItem(str(key))
                key_item.setFlags(key_item.flags() & ~ITEM_IS_EDITABLE)
                self.table.setItem(row, 0, key_item)

                # Coluna Value (editável apenas se for Custom)
                unit = units.get(key, "")
                display_value = f"{value} {unit}" if unit else str(value)
                value_item = QTableWidgetItem(display_value)
                if selected != "Custom":
                    value_item.setFlags(value_item.flags() & ~ITEM_IS_EDITABLE)
                self.table.setItem(row, 1, value_item)

    def _save_custom_drone(self): # Save Custom Drone in JSON file
        custom_specs = {}
        for row in range(self.table.rowCount()):
            key_item = self.table.item(row, 0)
            value_item = self.table.item(row, 1)
            if key_item and value_item:
                key = key_item.text()
                value_text = value_item.text().split()[0]  # remove unidade
                try:
                    value = float(value_text)
                except ValueError:
                    QMessageBox.warning(self, "Invalid Input", f"Value for '{key}' is not a valid number.")
                    return
                custom_specs[key] = value

        try:
            json_path = os.path.join(os.path.dirname(__file__), 'drone_data.json')
            with open(json_path, 'r', encoding='utf-8') as f:
                data = json.load(f)
            data["Custom"] = custom_specs
            with open(json_path, 'w', encoding='utf-8') as f:
                json.dump(data, f, indent=4)
            QMessageBox.information(self, "Saved", "Custom drone data saved successfully.")
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Failed to save custom drone data:\n{e}")
            
    def _save_to_qgis_environment(self): # Save Data for Flighting
        selected = self.droneCombo.currentText()
        if selected == "--- Select a Drone Model ---":
            QMessageBox.warning(self, "Warning", "Please select a valid drone model.")
            return

        # Mapeamento entre chave da tabela e nome da variável no QGIS
        key_map = {
            "sensor_width": "sensorH",
            "sensor_height": "sensorV",
            "focal_length": "dFocal",
            "image_width": "imageW",
            "image_height": "imageH",
            "min_angle_cam": "minAngleCam",
            "max_angle_cam": "maxAngleCam"
        }

        values_to_save = {}

        for row in range(self.table.rowCount()):
            key_item = self.table.item(row, 0)
            value_item = self.table.item(row, 1)

            if not key_item or not value_item:
                continue

            key = key_item.text()
            value_text = value_item.text().split()[0]  # remove unidade

            # Validação obrigatória para Custom
            if selected == "Custom":
                if value_text.strip() == "":
                    QMessageBox.warning(self, "Incomplete Data", f"Missing value for '{key}'. Please fill all fields before saving.")
                    return
                try:
                    float(value_text)
                except ValueError:
                    QMessageBox.warning(self, "Invalid Input", f"Value for '{key}' must be numeric.")
                    return

            try:
                value = float(value_text)
            except ValueError:
                value = 0  # fallback silencioso para drones não-Custom

            if key in key_map:
                values_to_save[key_map[key]] = value

        try:
            settings = QgsSettings()
            settings.setValue("qgis-drone-flight-planner/nameDrone", selected)
            for var_name, var_value in values_to_save.items():
                settings.setValue(f"qgis-drone-flight-planner/{var_name}", var_value)

            QMessageBox.information(self, "Saved", f"Drone '{selected}' parameters saved to QGIS environment.")
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Failed to save parameters:\n{e}")
               
    def calculate_gsd(self):
        specs = self._get_specs()
        if not specs:
            QMessageBox.warning(self, "Error", "Please select a valid drone model.")
            return
        try:
            alt = self.altitudeInput.value()
            gsd = calculate_gsd_by_sensor(
                alt,
                float(specs["sensor_width"]),
                float(specs["sensor_height"]),
                int(specs["image_width"]),
                int(specs["image_height"]),
                float(specs["focal_length"])
            )
            self.gsdResult.setText(f"GSD = {gsd:.2f} cm/pixel")
        except Exception as e:
            QMessageBox.critical(self, "Calculation Error", str(e))

    def calculate_altitude_from_gsd(self):
        specs = self._get_specs()
        if not specs:
            QMessageBox.warning(self, "Error", "Please select a valid drone model.")
            return
        try:
            gsd_cm = self.gsdInput.value()
            alt = calculate_altitude_from_gsd(
                gsd_cm,
                float(specs["sensor_width"]),
                float(specs["sensor_height"]),
                int(specs["image_width"]),
                int(specs["image_height"]),
                float(specs["focal_length"])
            )
            self.altitudeFromGSDResult.setText(f"Altitude = {alt:.2f} m")
        except Exception as e:
            QMessageBox.critical(self, "Calculation Error", str(e))

    def calculate_spacing(self):
        specs = self._get_specs()
        if not specs:
            QMessageBox.warning(self, "Error", "Please select a valid drone model.")
            return
        try:
            alt = self.altitudeSpacing.value()
            fw, sd = calculate_spacing(
                alt,
                float(specs["sensor_width"]),
                float(specs["sensor_height"]),
                float(specs["focal_length"]),
                self.frontOverlap.value(),
                self.sideOverlap.value()
            )
            self.spacingResult.setText(f"Forward Photos = {fw:.1f} m; Side Lines = {sd:.1f} m")
        except Exception as e:
            QMessageBox.critical(self, "Calculation Error", str(e))

    def calculate_overlap(self):
        specs = self._get_specs()
        if not specs:
            QMessageBox.warning(self, "Error", "Please select a valid drone model.")
            return
        try:
            alt = self.altitudeOverlap.value()
            front, side = calculate_overlap(
                alt,
                float(specs["sensor_width"]),
                float(specs["sensor_height"]),
                float(specs["focal_length"]),
                self.spacingForward.value(),
                self.spacingSide.value()
            )
            self.overlapResult.setText(f"Front Photos = {front:.1f} %; Side Lines = {side:.1f} %")
        except Exception as e:
            QMessageBox.critical(self, "Calculation Error", str(e))


    def open_help(self):
        url = QUrl("https://geoone.com.br/entendendo-o-gsd-e-a-sobreposicao-no-voo-com-drones/")
        # help_file_path = os.path.join(os.path.dirname(__file__), '..', 'calculator', 'calculator.html')
        # if os.path.exists(help_file_path):
        #     QDesktopServices.openUrl(QUrl.fromLocalFile(help_file_path))
        # else:
        #     QMessageBox.warning(self, "Help", f"Help file not found: {help_file_path}")
        if url.isValid():
            QDesktopServices.openUrl(url)
        else:
            QMessageBox.warning(self, "Help", "The documentation link appears to be offline.")
