# -*- coding: utf-8 -*-
"""
/***************************************************************************
 camera2geo
                                 A QGIS plugin
 Camera to geographic space image conversion
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-10-19
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Kanoa Lindiwe LLC
        email                : cankanoa@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtWidgets import QAction, QToolButton, QMenu
from qgis.gui import QgsMapToolIdentifyFeature, QgsMapToolIdentify
from qgis.PyQt.QtCore import Qt, QSettings, QTranslator, QCoreApplication, QVariant
from qgis.core import QgsRasterLayer, QgsProject, QgsMessageLog, Qgis, QgsApplication
from qgis import processing
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QTemporaryDir
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

from .camera2geo.main import camera2geo as camera2geo_function
from .provider import Camera2GeoProvider

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .camera2geo_dialog import camera2geoDialog
import os.path


class AttributeInspectTool(QgsMapToolIdentifyFeature):
    def __init__(self, iface, on_hit):
        super().__init__(iface.mapCanvas())
        self.iface = iface
        self.on_hit = on_hit

    def canvasReleaseEvent(self, event):
        if event.button() != Qt.LeftButton:
            return
        layer = self.iface.activeLayer()
        if not layer or layer.type() != layer.VectorLayer:
            return

        # identify at click (screen coords)
        res = self.identify(event.pos().x(), event.pos().y(),
                            [layer], QgsMapToolIdentify.TopDownStopAtFirst)
        if not res:
            return

        feat = res[0].mFeature
        names = [f.name() for f in layer.fields()]
        attrs = dict(zip(names, feat.attributes()))

        if callable(self.on_hit):
            self.on_hit(attrs, feat, layer)

def extract_path(
    attrs: dict,
    layer=None,
    target_name="photo",
    iface=None,
):
    # try exact field name match
    for k, v in attrs.items():
        if k.lower() == target_name:
            return v.strip() if isinstance(v, str) else (None if v in (None, QVariant()) else v)

    # try field alias match
    if layer:
        for f in layer.fields():
            if f.alias().strip().lower() == target_name and f.name() in attrs:
                v = attrs[f.name()]
                return v.strip() if isinstance(v, str) else (None if v in (None, QVariant()) else v)

    # handle missing field inside this function
    if iface:
        iface.messageBar().pushWarning(
            "Camera2Geo", f"No '{target_name}' field found on the selected feature."
        )

    return None

class Camera2GeoPlugin:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        self.provider = None
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'camera2geo_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Camera2geo')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None
        self.toolbar = None
        self.split_btn = None
        self.tool = None
        self.dlg = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('camera2geo', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        # Toolbar
        self.toolbar = self.iface.addToolBar("Camera2Geo")
        self.toolbar.setObjectName("Camera2GeoToolbar")

        # Main action (activates map tool)
        icon_path = os.path.join(self.plugin_dir, "icon_low.png")
        self.actRun = QAction(QIcon(icon_path), self.tr('Camera2geo'), self.iface.mainWindow())
        self.actRun.setCheckable(True)
        self.actRun.triggered.connect(self.activate_tool)

        # Settings action (opens your existing dialog)
        self.actSettings = QAction(self.tr('Picker Settings…'), self.iface.mainWindow())
        self.actSettings.triggered.connect(self.open_settings_dialog)

        # Dropdown menu
        menu = QMenu(self.iface.mainWindow())

        # Settings
        menu.addAction(self.actSettings)

        # Divider
        menu.addSeparator()

        # Add Bulk Tools
        self.actProcessCamera2Geo = QAction("Camera 2 Geo", self.iface.mainWindow())
        self.actProcessCamera2Geo.triggered.connect(lambda: self.open_processing_tool("camera2geo:camera2geo"))
        menu.addAction(self.actProcessCamera2Geo)

        self.actProcessSearch = QAction("Search Camera and Lens", self.iface.mainWindow())
        self.actProcessSearch.triggered.connect(lambda: self.open_processing_tool("camera2geo:camera_lens_search"))
        menu.addAction(self.actProcessSearch)

        self.actProcessApplyMetadata = QAction("Apply Metadata", self.iface.mainWindow())
        self.actProcessApplyMetadata.triggered.connect(lambda: self.open_processing_tool("camera2geo:apply_metadata"))
        menu.addAction(self.actProcessApplyMetadata)

        self.actProcessReadMetadata = QAction("Read Metadata", self.iface.mainWindow())
        self.actProcessReadMetadata.triggered.connect(lambda: self.open_processing_tool("camera2geo:read_metadata"))
        menu.addAction(self.actProcessReadMetadata)

        # Save to folder name
        # one or mulitple at the same time

        # Split button
        self.split_btn = QToolButton(self.toolbar)
        self.split_btn.setDefaultAction(self.actRun)
        self.split_btn.setMenu(menu)
        self.split_btn.setPopupMode(QToolButton.MenuButtonPopup)
        self.toolbar.addWidget(self.split_btn)

        # Keep old menu entry if you want
        self.iface.addPluginToMenu(self.menu, self.actRun)
        self.iface.addPluginToMenu(self.menu, self.actSettings)

        self.iface.mapCanvas().mapToolSet.connect(self._on_map_tool_changed)
        self.first_start = True

        # Register Processing Provider
        self.provider = Camera2GeoProvider()
        QgsApplication.processingRegistry().addProvider(self.provider)

    def open_processing_tool(self, alg_id):
        alg = QgsApplication.processingRegistry().algorithmById(alg_id)
        if alg:
            processing.execAlgorithmDialog(alg)
        else:
            self.iface.messageBar().pushWarning("Camera2Geo", f"Processing tool not found: {alg_id}")

    def activate_tool(self):
        def on_hit(attrs, feat, layer):
            s = QSettings()

            path_field_name = s.value("camera2geo/path_field_name", "photo")
            path = extract_path(attrs, layer, path_field_name)

            if getattr(self, "_output_tmpdir", None) is None or not self._output_tmpdir.isValid():
                self._output_tmpdir = QTemporaryDir()

            out_dir = self._output_tmpdir.path()

            elevation_data_value = s.value("camera2geo/elevation_data", "plane")
            if elevation_data_value == "plane": # No elevation data
                elevation_data_value = False
            elif elevation_data_value == "online": # Online elevation
                elevation_data_value = True
            else: # File elevation
                pass

            out_path = camera2geo_function(
                path, out_dir,
                sensor_width_mm=s.value("camera2geo/sensor_width_mm", None, type=float),
                sensor_height_mm=s.value("camera2geo/sensor_height_mm", None, type=float),
                epsg=s.value("camera2geo/epsg", 4326, type=int),
                correct_magnetic_declination=s.value("camera2geo/correct_magnetic_declination", False, type=bool),
                cog=s.value("camera2geo/cog", False, type=bool),
                lens_correction=s.value("camera2geo/lens_correction", False, type=bool),
                elevation_data=elevation_data_value,
            )[0]

            if not out_path or not os.path.exists(out_path):
                self.iface.messageBar().pushCritical("Camera2Geo", "Processing failed: no output file.")
                return

            layer_name = os.path.basename(out_path)
            rlayer = QgsRasterLayer(out_path, layer_name, "gdal")

            if not rlayer.isValid():
                self.iface.messageBar().pushCritical("Camera2Geo", f"Failed to load: {out_path}")
                return

            project = QgsProject.instance()
            root = project.layerTreeRoot()

            group_name = s.value("camera2geo/output_group", "photos")

            # find or create group
            group = root.findGroup(group_name)
            if group is None:
                group = root.addGroup(group_name)

            # Remove other layers in group if checked
            remove_prev = s.value("camera2geo/remove_previous_photos", False, type=bool)
            if remove_prev:
                for child in list(group.children()):
                    if hasattr(child, 'layerId'):
                        project.removeMapLayer(child.layerId())

            # add layer without stealing focus
            project.addMapLayer(rlayer, addToLegend=False)
            group.insertLayer(-0, rlayer)

            # Simply add the layer
            # QgsProject.instance().addMapLayer(rlayer)

        self.tool = AttributeInspectTool(self.iface, on_hit)
        self.tool.setAction(self.actRun)
        self.iface.mapCanvas().setMapTool(self.tool)
        self.actRun.setChecked(True)

    def _on_map_tool_changed(self, tool):
        if tool is not self.tool:
            self.actRun.setChecked(False)

    def open_settings_dialog(self):
        if self.dlg is None:
            self.dlg = camera2geoDialog()
        self.dlg.show()
        self.dlg.raise_()
        self.dlg.activateWindow()

    def unload(self):
        self.actRun.setChecked(False)
        for action in self.actions:
            self.iface.removePluginMenu(self.menu, action)
            self.iface.removeToolBarIcon(action)
        if self.toolbar:
            self.iface.mainWindow().removeToolBar(self.toolbar)
            self.toolbar = None
        if self.provider:
            QgsApplication.processingRegistry().removeProvider(self.provider)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = camera2geoDialog()

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass
