# -*- coding: utf-8 -*-
"""
/***************************************************************************
 AdjustStyle
                                 A QGIS plugin
 Adjust color, line thickness, font, font size etc. of all symbols/labels.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-05-05
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Florian Neukirchen
        email                : mail@riannek.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import re
from qgis.core import *
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon, QColor, QPalette
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QMessageBox
# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .adjust_style_dockwidget import AdjustStyleDockWidget, AdjustStyleLayoutDockWidget, AdjustStyleLayoutHandler
from .adjust_style_font_dialog import ReplaceFontDialog
import os.path




class AdjustStyle:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'adjust_style_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.layouts = []
        self.menu = self.tr(u'&Adjust Syle ')
        icon_path = ':/plugins/adjust_style/icon.png'
        self.icon = QIcon(icon_path)

        #print "** INITIALIZING AdjustStyle"

        self.pluginIsActive = False
        self.dockwidget = None



    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('AdjustStyle', message)


    def add_action(
        self,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        action = QAction(self.icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        self.add_action(
            text=self.tr(u'Adjust Style'),
            callback=self.run,
            parent=self.iface.mainWindow())
              
        self.iface.layoutDesignerOpened.connect(self.whenOpenedDesignerInterface)
        self.iface.layoutDesignerWillBeClosed.connect(self.whenClosedDesignerInterface)
        

    #--------------------------------------------------------------------------

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        #print "** CLOSING AdjustStyle"

        # disconnects
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        # self.dockwidget = None

        self.pluginIsActive = False


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        #print "** UNLOAD AdjustStyle"

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Adjust Syle '),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        # del self.toolbar

        for layout in self.layouts:
            layout.unload()
        self.layouts=[]

    #--------------------------------------------------------------------------

    def whenOpenedDesignerInterface(self, designer):
        # The signal returns the instance of QgsLayoutDesignerInterface

        layout_handler = AdjustStyleLayoutHandler(self, designer)
        self.layouts.append(layout_handler)



    def whenClosedDesignerInterface(self, designer):
        for layout in self.layouts:
            if layout.designer == designer:
                layout.unload()
                self.layouts.remove(layout)



    #--------------------------------------------------------------------------

    # Functions connected to buttons

    def hueBtn(self):
        self.value = self.dockwidget.spinBox.value()
        self.change_color = self.rotate_hue
        self.mapToLayers(self.layer_change_color)

    def saturationPlusBtn(self):
        self.value = self.dockwidget.changeSpinBox.value()
        self.change_color = self.change_saturation
        self.mapToLayers(self.layer_change_color)       

    def saturationMinusBtn(self):
        self.value = self.dockwidget.changeSpinBox.value() * -1
        self.change_color = self.change_saturation
        self.mapToLayers(self.layer_change_color)  

    def hsvValuePlusBtn(self):
        self.value = self.dockwidget.changeSpinBox.value()
        self.change_color = self.change_hsv_value
        self.mapToLayers(self.layer_change_color)  

    def hsvValueMinusBtn(self):
        self.value = self.dockwidget.changeSpinBox.value() * -1
        self.change_color = self.change_hsv_value
        self.mapToLayers(self.layer_change_color)  

    def strokeWidthPlusBtn(self):
        self.value = self.dockwidget.changeSpinBox.value() / 100
        self.mapToLayers(self.layer_change_stroke)  

    def strokeWidthMinusBtn(self):
        self.value = self.dockwidget.changeSpinBox.value() * -1 / 100
        self.mapToLayers(self.layer_change_stroke)  

    def fontSizePlusBtn(self):
        self.value = self.dockwidget.changeSpinBox.value() / 100
        self.mapToLayers(self.layer_font_size)

    def fontSizeMinusBtn(self):
        self.value = self.dockwidget.changeSpinBox.value() * -1 / 100
        self.mapToLayers(self.layer_font_size)

    def saveStylesBtn(self):
        self.counter = 0
        self.overwrite = False
        self.url = QgsProject.instance().readPath("./")
        self.url = QFileDialog.getExistingDirectory(
            self.dockwidget, 'Select a directory', self.url
        )

        if not self.url:
            # Canceled
            return

        if not os.access(self.url, os.W_OK):
            self.iface.messageBar().pushWarning(self.tr('Save Styles'), self.tr("Can't write to directory {}").format(self.url))
            return
        self.iface.mainWindow().statusBar().showMessage(self.tr('Save styles to: {}'.format(self.url)), 3000) 

        self.mapToLayers(self.save_layer_style)

        # Also save projects background color if checkCanvas is checked and background ist not white
        # as simple txt file

        # Two options for later feedback
        bf_feetback = ''
        bf_feetback_saved = ' ' + self.tr('Canvas background color has been saved.')

        if self.dockwidget.checkCanvas.isChecked():
            bg = QgsProject.instance().backgroundColor().name()


            url = os.path.join(self.url, 'backgroundcolor.txt')

            # Check if file exits
            if os.path.exists(url) and not self.overwrite:
                choice = QMessageBox.question(
                    self.dockwidget,
                    self.tr('File exists'),
                    self.tr('File backgroundcolor.txt already exists. Do you want to overwrite it?'),
                    QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No
                )
                if choice == QMessageBox.StandardButton.Yes:
                    with open(url, 'w') as f:
                        f.write(bg)
                    bf_feetback = bf_feetback_saved
            else:
                with open(url, 'w') as f:
                    f.write(bg)
                bf_feetback = bf_feetback_saved

        # Feedback for user
        if self.counter > 0:
            self.iface.messageBar().pushInfo(self.tr('Save Styles'), self.tr('Succesfully saved styles of {} selected layers.').format(self.counter) + bf_feetback)
        else:
            self.iface.messageBar().pushWarning(self.tr('Error'), self.tr('Could not save styles.'))

    
    def loadStylesBtn(self):
        self.counter = 0
        self.counter_fail = 0
        self.url = QgsProject.instance().readPath("./")
        self.url = QFileDialog.getExistingDirectory(
            self.dockwidget, 'Select a directory', self.url
        )

        if not self.url:
            # Canceled
            return

        if not os.access(self.url, os.R_OK):
            self.iface.messageBar().pushWarning(self.tr('Load Styles'), self.tr("Can't read directory {}").format(self.url))
            return
        
        self.iface.mainWindow().statusBar().showMessage(self.tr('Load styles from: {}'.format(self.url)), 3000) 

        self.mapToLayers(self.load_layer_style)

        # Eventually load canvas background color
        url = os.path.join(self.url, 'backgroundcolor.txt')
        
        if self.dockwidget.checkCanvas.isChecked() and os.path.exists(url):
            bg = ''
            with open(url, 'r') as f:
                bg = f.read()

            bgcolor = QColor(bg.strip())
            if bgcolor.isValid():
                QgsProject.instance().setBackgroundColor(bgcolor)


        # Feedback
        if self.counter == 0:
            self.iface.messageBar().pushWarning(self.tr('Load styles'), self.tr('Could not load any style for any layer.'))
        elif self.counter > 0 and self.counter_fail == 0:
            self.iface.messageBar().pushInfo(self.tr('Load styles'), f'Succesfully loaded styles of all {self.counter} selected layers.')
        else:
            self.iface.messageBar().pushInfo(
                self.tr('Load styles'), 
                self.tr('Succesfully loaded styles of {} layers but failed on {} selected layers.').format(self.counter, self.counter_fail)
                )


    # Use the choice of layers and map the corresponding function to them

    def mapToLayers(self, func):
        layerchoice = self.dockwidget.buttonGroup.checkedId()
        if layerchoice == 1:
            # Radiobutton "Active layer" is checked
            layer = self.iface.activeLayer()
            func(layer)

        elif layerchoice == 2:
            # Selected Layers is checked
            for layer in self.iface.layerTreeView().selectedLayers():
                func(layer)

        elif layerchoice == 3:
            # Visible layers is checked
            for layer in self.iface.mapCanvas().layers():
                func(layer)

        elif layerchoice == 4:
            # All layers is checked
            for layer in QgsProject.instance().mapLayers().values():
                func(layer)


        if self.dockwidget.checkAnnotation.isChecked():
            # Also change the main annotation layer
            layer = QgsProject.instance().mainAnnotationLayer()
            func(layer)


        if self.dockwidget.checkCanvas.isChecked():
            # Also change canvas background color
            if func == self.layer_change_color:
                color = QgsProject.instance().backgroundColor()
                color = self.change_color(color, self.value)
                QgsProject.instance().setBackgroundColor(color)

        return

    # Functions to change colors

    def rotate_hue(self, qcolor, degree):
        h, s, v, a = qcolor.getHsv()
        # QColor uses h = -1 for achromatic colors
        if h > -1:
            h = h + degree
            if h >= 360:
                h = h - 360
            qcolor.setHsv(h, s, v, a)
        return qcolor

    def change_hsv_value(self, qcolor, increment):
        h, s, v, a = qcolor.getHsv()
        v = v + increment
        if v > 255:
            v = 255
        elif v < 0:
            v = 0
        qcolor.setHsv(h, s, v, a)
        return qcolor        

    def change_saturation(self, qcolor, increment):
        h, s, v, a = qcolor.getHsv()
        if s > 0:
            s = s + increment
            if s > 255:
                s = 255
            elif s < 0:
                s = 0
            qcolor.setHsv(h, s, v, a)
        return qcolor 

    def layer_change_color(self, layer):
        # Layer groups have type None
        if not layer:
            return

        # Do nothing on Null Symbol Renderer
        if isinstance(layer, QgsNullSymbolRenderer):
            return
        
        # Handle Annotation layers
        if isinstance(layer, QgsAnnotationLayer):
            self.change_annotationlayer_colors(layer)
            # Quit function, nothing else to do
            return

        # Mesh Layer
        if isinstance(layer, QgsMeshLayer):
            renderer_settings = layer.rendererSettings()
            # Vector Settings (Arrows)
            group = renderer_settings.activeVectorDatasetGroup()
            settings = renderer_settings.vectorSettings(group)
            settings = self.change_interpolated_color(settings)
            renderer_settings.setVectorSettings(group, settings)

            # Scalar Settings (Mesh Contour Color)
            group = renderer_settings.activeScalarDatasetGroup()
            settings = renderer_settings.scalarSettings(group)
            settings = self.change_interpolated_color(settings)
            renderer_settings.setScalarSettings(group, settings)

            layer.setRendererSettings(renderer_settings)

            QgsProject.instance().setDirty()
            layer.triggerRepaint()
            layer.emitStyleChanged()
            return



        # QGIS 3.24 introduces QgsGroupLayer and it does not have a renderer
        try:
            if isinstance(layer, QgsGroupLayer):
                return
        except NameError:
            pass

        # Make function callable with layer AND renderers with embedded renderer
        # So we can call the function recursively with embedded renderers.
        # "layer" might be a renderer in this case
        if type(layer) in (QgsFeatureRenderer, QgsInvertedPolygonRenderer):
            renderer = layer.embeddedRenderer()
        else:
            try:
                renderer = layer.renderer()
            except AttributeError:
                print(layer, "has no renderer")
                return
        
        # Vector layers: Change symbols depending on renderer type
        if isinstance(renderer, QgsSingleSymbolRenderer):
            symbol = renderer.symbol()
            self.change_symbol_color(symbol)

        elif isinstance(renderer, QgsRuleBasedRenderer):
            for rule in renderer.rootRule().children():
                symbol = rule.symbol()
                self.change_symbol_color(symbol)

        elif isinstance(renderer, QgsCategorizedSymbolRenderer) or isinstance(renderer, QgsGraduatedSymbolRenderer):
            ramp = renderer.sourceColorRamp() # Can be None
            if isinstance(ramp, QgsGradientColorRamp):
                ramp = ramp.clone()
                self.change_ramp_colors(ramp)
                renderer.updateColorRamp(ramp)
            elif isinstance(ramp, QgsCptCityColorRamp):
                ramp = ramp.cloneGradientRamp()
                self.change_ramp_colors(ramp)
                renderer.updateColorRamp(ramp)
            elif (not ramp or isinstance(ramp, QgsRandomColorRamp)) and isinstance(renderer, QgsCategorizedSymbolRenderer): 
                for index, cat in enumerate(renderer.categories()):
                    symbol = cat.symbol().clone()
                    self.change_symbol_color(symbol)
                    renderer.updateCategorySymbol(index, symbol)
            elif (not ramp or isinstance(ramp, QgsRandomColorRamp)) and isinstance(renderer, QgsGraduatedSymbolRenderer): 
                for index, range in enumerate(renderer.ranges()):
                    symbol = range.symbol().clone()
                    self.change_symbol_color(symbol)
                    renderer.updateRangeSymbol(index, symbol)
            else:
                # Color Brewer Ramps and maybe other types
                pass

        elif isinstance(renderer, QgsHeatmapRenderer):
            ramp = renderer.colorRamp()
            if isinstance(ramp, QgsGradientColorRamp):
                ramp = ramp.clone()
                self.change_ramp_colors(ramp)
                renderer.setColorRamp(ramp)
            elif isinstance(ramp, QgsCptCityColorRamp):
                ramp = ramp.cloneGradientRamp()
                self.change_ramp_colors(ramp)
                renderer.setColorRamp(ramp)
       

        # inverted polygon or dissolved renderer
        elif type(renderer) in (QgsFeatureRenderer, QgsInvertedPolygonRenderer):
            self.layer_change_color(renderer)



        # Raster Layer with pseudo color
        elif isinstance(renderer, QgsSingleBandPseudoColorRenderer):
            shader = renderer.shader()
            func = shader.rasterShaderFunction()
            ramp = func.sourceColorRamp()

            if isinstance(ramp, QgsGradientColorRamp):
                ramp = ramp.clone()
            elif isinstance(ramp, QgsCptCityColorRamp):
                ramp = ramp.cloneGradientRamp()
            else:
                # Nothing else to do
                return
            
            self.change_ramp_colors(ramp)
            func.setSourceColorRamp(ramp)
            func.classifyColorRamp()

            shader.setRasterShaderFunction(func)
            
        # Raster layer with contour renderer
        elif isinstance(renderer, QgsRasterContourRenderer):
            symbol = renderer.contourSymbol().clone()
            self.change_symbol_color(symbol)
            renderer.setContourSymbol(symbol)
            symbol = renderer.contourIndexSymbol().clone()
            self.change_symbol_color(symbol)
            renderer.setContourIndexSymbol(symbol)

        # Raster layers with single color renderer (New in QGIS 3.38)
        try:
            if isinstance(renderer, QgsRasterSingleColorRenderer):
                color = renderer.color()
                color = self.change_color(color, self.value)
                renderer.setColor(color)
        except NameError:
            pass

        # Raster layers with "colorize" in the render pipeline
        if isinstance(layer, QgsRasterLayer):
            huesat = layer.pipe().hueSaturationFilter() # May be none
            if huesat and huesat.colorizeOn():
                huesat = huesat.clone()
                color = huesat.colorizeColor()
                color = self.change_color(color, self.value)
                huesat.setColorizeColor(color)
                layer.pipe().set(huesat)

        # Vector layers: paint effects
        if isinstance(layer, QgsVectorLayer):
            effects = renderer.paintEffect()
            if effects.enabled():
                self.change_effect_colors(effects)

        # Labels
     
        if isinstance(layer, QgsVectorLayer) and layer.labelsEnabled():
            labeling = layer.labeling() # Returns QgsVectorLayerSimpleLabeling or QgsRuleBasedLabeling

            if isinstance(labeling, QgsVectorLayerSimpleLabeling):
                settings = labeling.settings() # Returns QgsPalLayerSettings
                settings = self.change_font_color(settings)
                labeling.setSettings(settings)

            if isinstance(labeling, QgsRuleBasedLabeling):
                for rule in labeling.rootRule().children():
                    settings = rule.settings()
                    settings = self.change_font_color(settings)
                    rule.setSettings(settings)

        #  Only for real layers:
        if not isinstance(layer, QgsFeatureRenderer):
            # Set dirty flag, trigger repaint
            QgsProject.instance().setDirty()
            layer.triggerRepaint()

            # Also show the changes in "Layer Styling" panel and TOC
            layer.emitStyleChanged()
            self.iface.layerTreeView().refreshLayerSymbology(layer.id())


    def change_symbol_color(self, symbol):
        if not symbol:
            return
        
        # used by Line Pattern Fill (new in QGIS 3.36)
        if isinstance(symbol, QgsFillSymbol):
            for symlayer in symbol.symbolLayers():
                self.change_symbol_color(symlayer)

        # Iterate through Symbol Layers
        try:
            symlayers = symbol.symbolLayers()
        except AttributeError:
            # No Symbol Layers in Symbol
            symlayers = []

        for symlayer in symlayers:
            # Marker symbols with subsymbol
            try:
                subsymbol = symlayer.subSymbol()
                self.change_symbol_color(subsymbol)
            except AttributeError:
                pass 

            # Interpolated line (new in QGIS 3.20)
            try:
                if isinstance(symlayer, QgsInterpolatedLineSymbolLayer):
                    ipc = symlayer.interpolatedColor()
                    ipc = self.change_interpolated_color(ipc)
                    symlayer.setInterpolatedColor(ipc)
            except NameError:
                pass

            # Most symbols
            else:
            # Fill color
                color = symlayer.color() 
                color = self.change_color(color, self.value)
                symlayer.setColor(color)
                
                # Stroke color
                color = symlayer.strokeColor() 
                color = self.change_color(color, self.value)
                symlayer.setStrokeColor(color)

            # Gradient layers (QgsLineburstSymbolLayer are new in QGIS 3.24)
            try:
                if isinstance(symlayer, QgsGradientFillSymbolLayer) or isinstance(symlayer, QgsShapeburstFillSymbolLayer) or isinstance(symlayer, QgsLineburstSymbolLayer):
                    # Set color 2 for 2-colored gradients
                    color = symlayer.color2()
                    color = self.change_color(color, self.value)
                    symlayer.setColor2(color)

                    # change ramp for ramp based gradient
                    ramp = symlayer.colorRamp()
                    if isinstance(ramp, QgsGradientColorRamp):
                        ramp = ramp.clone()
                        self.change_ramp_colors(ramp)
                        symlayer.setColorRamp(ramp)
                    elif isinstance(ramp, QgsCptCityColorRamp):
                        ramp = ramp.cloneGradientRamp()
                        self.change_ramp_colors(ramp)
                        symlayer.setColorRamp(ramp)
            except NameError:
                pass
            

            # Effects
            effects = symlayer.paintEffect()
            if effects and effects.enabled():
                self.change_effect_colors(effects)

        return

    def change_interpolated_color(self, ipc):
        # ipc can be QgsInterpolatedLineColor or QgsMeshRendererVectorSettings or QgsMeshRendererScalarSettings

        try:
            coloring_method = ipc.coloringMethod()
        except AttributeError:
            coloring_method = 1 # Color Ramp
        if coloring_method == 0:
            # Single color
            if isinstance(ipc, QgsInterpolatedLineColor):
                color = ipc.singleColor()
            else:
                color = ipc.color() 
            color = self.change_color(color, self.value)
            ipc.setColor(color)
            return ipc
        
        # ColorRamp
        crsh = ipc.colorRampShader()
        ramp = crsh.sourceColorRamp()
        if isinstance(ramp, QgsGradientColorRamp):
            ramp = ramp.clone()
            self.change_ramp_colors(ramp)
            crsh.setSourceColorRamp(ramp)
            crsh.classifyColorRampV2()
        elif isinstance(ramp, QgsCptCityColorRamp):
            ramp = ramp.cloneGradientRamp()
            self.change_ramp_colors(ramp)
            crsh.setSourceColorRamp(ramp)
            crsh.classifyColorRampV2()

        if isinstance(ipc, QgsInterpolatedLineColor):
            ipc.setColor(crsh)
        elif isinstance(ipc, QgsMeshRendererVectorSettings):
            ipc.setColorRampShader(crsh)
        elif isinstance(ipc, QgsMeshRendererScalarSettings):
            ipc.setColorRampShader(crsh)

        return ipc
    

    def change_annotationlayer_colors(self, layer):
        for item in layer.items().values():
            if isinstance(item, QgsAnnotationPointTextItem) or isinstance(item, QgsAnnotationLineTextItem):
                self.change_font_color(item)
            else:
                symbol = item.symbol()
                self.change_symbol_color(symbol)

            effects = layer.paintEffect()
            if effects.enabled():
                self.change_effect_colors(effects)

            # Set dirty flag, trigger repaint
            QgsProject.instance().setDirty()
            layer.triggerRepaint()

            # Also show the changes in "Layer Styling" panel and TOC
            layer.emitStyleChanged()



    def change_effect_colors(self, effectstack):
        for effect in effectstack.effectList():
            if effect.enabled() and type(effect) in (QgsInnerGlowEffect, QgsDropShadowEffect, QgsInnerShadowEffect, QgsOuterGlowEffect):
                color = effect.color()
                color = self.change_color(color, self.value)
                effect.setColor(color)

                # Color ramp of glow effects

                try:
                    ramp = effect.ramp()
                    if isinstance(ramp, QgsGradientColorRamp):
                        ramp = ramp.clone()
                        self.change_ramp_colors(ramp)
                        effect.setRamp(ramp)
                    elif isinstance(ramp, QgsCptCityColorRamp):
                        ramp = ramp.cloneGradientRamp()
                        self.change_ramp_colors(ramp)
                        effect.setRamp(ramp)
                except AttributeError:
                    # Only glow effects can have ramps
                    pass
                



    def change_ramp_colors(self, ramp):
        color = ramp.color1()
        color = self.change_color(color, self.value)
        ramp.setColor1(color)

        color = ramp.color2()
        color = self.change_color(color, self.value)
        ramp.setColor2(color)

        new_stops = []
        for stop in ramp.stops():
            offset = stop.offset
            color = stop.color
            color = self.change_color(color, self.value)
            new_stops.append(QgsGradientStop(offset, color))
        
        ramp.setStops(new_stops)



    def change_font_color(self, settings):
        
        if isinstance(settings, QgsTextFormat):
            # When called from layout items
            format = settings
        else:
            format = settings.format() # Returns QgsTextFormat

        # Font
        color = format.color()
        color = self.change_color(color, self.value)
        format.setColor(color)

        # Buffer
        bf = format.buffer() # Returns QgsTextBufferSettings
        if bf.enabled():
            color = bf.color()
            color = self.change_color(color, self.value)
            bf.setColor(color)


        # Background
        bg = format.background() # Returns QgsTextBackgroundSettings
        if bg.enabled():
            symbol = bg.fillSymbol()
            self.change_symbol_color(symbol)

        if isinstance(settings, QgsTextFormat):
            return format
        
        settings.setFormat(format)

        return settings
    
    # Change stroke width

    def layer_change_stroke(self, layer):
        # Layer groups have type None
        if not layer:
            return

        # Do nothing on Null Symbol Renderer
        if isinstance(layer, QgsNullSymbolRenderer):
            return
        
        # Handle Annotation layers
        if isinstance(layer, QgsAnnotationLayer):
            for item in layer.items().values():
                if not (isinstance(item, QgsAnnotationPointTextItem) or isinstance(item, QgsAnnotationLineTextItem)):
                    symbol = item.symbol()
                    self.change_symbol_stroke(symbol)

            QgsProject.instance().setDirty()
            layer.triggerRepaint()
            layer.emitStyleChanged()
            self.iface.layerTreeView().refreshLayerSymbology(layer.id())
            return
        # Mesh Layer
        if isinstance(layer, QgsMeshLayer):
            renderer_settings = layer.rendererSettings()
            # Vector Settings (Arrows)
            group = renderer_settings.activeVectorDatasetGroup()
            settings = renderer_settings.vectorSettings(group)
            width = settings.lineWidth()
            width = width + width * self.value
            settings.setLineWidth(width)
            renderer_settings.setVectorSettings(group, settings)

            layer.setRendererSettings(renderer_settings)

            QgsProject.instance().setDirty()
            layer.triggerRepaint()
            layer.emitStyleChanged()
            return



        # QGIS 3.24 introduces QgsGroupLayer and it does not have a renderer
        try:
            if isinstance(layer, QgsGroupLayer):
                return
        except NameError:
            pass

        try:
            renderer = layer.renderer()
        except AttributeError:
            print(layer, "has no renderer")
            return


        # Use embedded renderer for "inverted polygon" and dissolved renderer 
        if type(renderer) in (QgsFeatureRenderer, QgsInvertedPolygonRenderer):
            renderer = renderer.embeddedRenderer()
        
        # Normal renderer types       
        if isinstance(renderer, QgsSingleSymbolRenderer):
            symbol = renderer.symbol()
            self.change_symbol_stroke(symbol)
        
        elif isinstance(renderer, QgsCategorizedSymbolRenderer):
            for index, cat in enumerate(renderer.categories()):
                symbol = cat.symbol().clone()
                self.change_symbol_stroke(symbol)
                renderer.updateCategorySymbol(index, symbol)

        elif isinstance(renderer, QgsGraduatedSymbolRenderer):
            for index, range in enumerate(renderer.ranges()):
                symbol = range.symbol().clone()
                self.change_symbol_stroke(symbol)
                renderer.updateRangeSymbol(index, symbol)

        elif isinstance(renderer, QgsRuleBasedRenderer):
            for rule in renderer.rootRule().children():
                symbol = rule.symbol()
                self.change_symbol_stroke(symbol)

        elif isinstance(renderer, QgsRasterContourRenderer):
            symbol = renderer.contourSymbol().clone()
            self.change_symbol_stroke(symbol)
            renderer.setContourSymbol(symbol)
            symbol = renderer.contourIndexSymbol().clone()
            self.change_symbol_stroke(symbol)
            renderer.setContourIndexSymbol(symbol)

        else:
            # Do not trigger repaint
            return

        QgsProject.instance().setDirty()
        layer.triggerRepaint()
        layer.emitStyleChanged()
        self.iface.layerTreeView().refreshLayerSymbology(layer.id())
        return


    def change_symbol_stroke(self, symbol):
        if not symbol:
            return
        
        # Iterate through Symbol Layers
        try:
            symlayers = symbol.symbolLayers()
        except AttributeError:
            # No Symbal Layers in Symbol
            symlayers = []

        for symlayer in symlayers:

            # Interpolated Line, Lineburst (New in QGIS)
            try:
                if isinstance(symlayer, QgsInterpolatedLineSymbolLayer):
                    ipw = symlayer.interpolatedWidth()

                    # Fixed stroke
                    width = ipw.fixedStrokeWidth()
                    width = width + width * self.value
                    ipw.setFixedStrokeWidth(width)

                    # Variable stroke
                    width = ipw.minimumWidth()
                    width = width + width * self.value
                    ipw.setMinimumWidth(width)

                    width = ipw.maximumWidth()
                    width = width + width * self.value
                    ipw.setMaximumWidth(width)

                    symlayer.setInterpolatedWidth(ipw)

                elif isinstance(symlayer, QgsLineburstSymbolLayer):
                    width = symlayer.width()
                    width = width + width * self.value
                    symlayer.setWidth(width)

            except NameError:
                pass

            # Line symbols
            if isinstance(symlayer, QgsSimpleLineSymbolLayer):
                width = symlayer.width()
                width = width + width * self.value
                symlayer.setWidth(width)

            # Marker symbols with subsymbol
            try:
                subsymbol = symlayer.subSymbol()
                self.change_symbol_stroke(subsymbol)
            except AttributeError:
                pass

            # Most other symbols
            else:
                try:
                    width = symlayer.strokeWidth()
                    width = width + width * self.value
                    symlayer.setStrokeWidth(width)
                except AttributeError as e:
                    # print('Adjust style: ' + str(e))
                    pass

            
        return
    
    # Change font size

    def layer_font_size(self, layer):
        if isinstance(layer, QgsVectorLayer) and layer.labelsEnabled():
            labeling = layer.labeling() # Returns QgsVectorLayerSimpleLabeling or QgsRuleBasedLabeling

            if isinstance(labeling, QgsVectorLayerSimpleLabeling):
                settings = labeling.settings() # Returns QgsPalLayerSettings
                settings = self.change_font_size(settings)
                labeling.setSettings(settings)

            if isinstance(labeling, QgsRuleBasedLabeling):
                for rule in labeling.rootRule().children():
                    settings = rule.settings()
                    settings = self.change_font_size(settings)
                    rule.setSettings(settings)

            QgsProject.instance().setDirty()
            layer.triggerRepaint()
            layer.emitStyleChanged()
            self.iface.layerTreeView().refreshLayerSymbology(layer.id())

        elif isinstance(layer, QgsAnnotationLayer):
            for item in layer.items().values():
                if isinstance(item, QgsAnnotationPointTextItem) or isinstance(item, QgsAnnotationLineTextItem):
                    self.change_font_size(item)
            QgsProject.instance().setDirty()
            layer.triggerRepaint()
            layer.emitStyleChanged()
        
        return

    def change_font_size(self, settings):
        if isinstance(settings, QgsTextFormat):
            # When called from layout items
            format = settings
        else:
            format = settings.format() # Returns QgsTextFormat

        size = format.size()
        size = size + size * self.value
        if size < 0:
            size = 0
        format.setSize(size)

        if isinstance(settings, QgsTextFormat):
            return format

        settings.setFormat(format)
        return settings

    # Replace Font

    def replace_font_dlg(self, layout=None):
        

        # Get a set of all fonts in the project
        
        if layout:
            self.fontset = layout.collect_fonts()
            parent = layout.designer.window()
        else:
            self.fontset = set()
            self.mapToLayers(self.collect_fonts)
            parent = self.iface.mainWindow()

        if not self.fontset:
            # set is empty
            if layout:
                messagebar = layout.designer.messageBar()
            else:
                messagebar = self.iface.messageBar()
            messagebar.pushInfo(self.tr('Replace Font'), self.tr('The selection does not contain any fonts.'))
            return
        
        self.dlg = ReplaceFontDialog(parent=parent)
        
        # Populate combo box
        self.dlg.currentFontsComboBox.clear()
        self.fontset = list(sorted(self.fontset))
        self.dlg.currentFontsComboBox.addItems(self.fontset)
        
        # Show dialog, run dialog event loop
        self.dlg.show()
        result = self.dlg.exec()

        # See if OK was pressed
        if result:
            i = self.dlg.currentFontsComboBox.currentIndex()
            self.oldfont = self.fontset[i] # string
            self.newfont = self.dlg.fontComboBox.currentFont() # QFont
            if layout:
                layout.replace_font()
            else:
                self.mapToLayers(self.replace_font)

 
    def collect_fonts(self, layer):
        if isinstance(layer, QgsVectorLayer) and layer.labelsEnabled():
            labeling = layer.labeling() # Returns QgsVectorLayerSimpleLabeling or QgsRuleBasedLabeling

            if isinstance(labeling, QgsVectorLayerSimpleLabeling):
                # catch attribute error, Issue #8
                try:
                    format = labeling.settings().format() # Returns QgsTextFormat
                    self.fontset.add(format.font().family())
                except AttributeError:
                    pass

            if isinstance(labeling, QgsRuleBasedLabeling):
                for rule in labeling.rootRule().children():
                    try:
                        format = rule.settings().format()
                        self.fontset.add(format.font().family())
                    except AttributeError:
                        pass
        
        elif isinstance(layer, QgsAnnotationLayer):
            for item in layer.items().values():
                if isinstance(item, QgsAnnotationPointTextItem) or isinstance(item, QgsAnnotationLineTextItem):
                    format = item.format()
                    self.fontset.add(format.font().family())




    def replace_font(self, layer):
        if isinstance(layer, QgsVectorLayer) and layer.labelsEnabled():
            labeling = layer.labeling() # Returns QgsVectorLayerSimpleLabeling or QgsRuleBasedLabeling

            if isinstance(labeling, QgsVectorLayerSimpleLabeling):
                settings = labeling.settings()
                # It seems possible to get none here, Issue #8
                if settings:
                    format = settings.format() # Returns QgsTextFormat
                    if format.font().family() == self.oldfont:
                        format.setFont(self.newfont)
                        settings.setFormat(format)
                        labeling.setSettings(settings)


            if isinstance(labeling, QgsRuleBasedLabeling):
                for rule in labeling.rootRule().children():
                    settings = rule.settings()
                    if settings:
                        format = settings.format()
                        if format.font().family() == self.oldfont:
                            format.setFont(self.newfont)
                            settings.setFormat(format)
                            rule.setSettings(settings)

            QgsProject.instance().setDirty()
            layer.triggerRepaint() 
            self.iface.layerTreeView().refreshLayerSymbology(layer.id())
            layer.emitStyleChanged()     

        elif isinstance(layer, QgsAnnotationLayer):
            for item in layer.items().values():
                if isinstance(item, QgsAnnotationPointTextItem) or isinstance(item, QgsAnnotationLineTextItem):
                    format = item.format()
                    if format.font().family() == self.oldfont:
                            format.setFont(self.newfont)
                            item.setFormat(format)

            QgsProject.instance().setDirty()
            layer.triggerRepaint() 
            layer.emitStyleChanged()             

    # Load and save styles

    def save_layer_style(self, layer):
        # Remove bad characters from layer name
        clean_name = re.sub(r'[^\w_.-]', '_', layer.name()) 

        # Check if this is the main annotation layer
        if layer == QgsProject.instance().mainAnnotationLayer():
            filename = "Main_Annotation_Layer.qml"

        # Check if there are more layers of the same name
        else:
            listoflayers = QgsProject.instance().mapLayersByName(layer.name())
            if len(listoflayers) == 1:
                filename = clean_name + '.qml'
            else:
                i = listoflayers.index(layer)
                filename = clean_name + '__(' + str(i).zfill(2) + ').qml'

        url = os.path.join(self.url, filename)

        # Check if file exits
        if os.path.exists(url) and not self.overwrite:
            
            choice = QMessageBox.question(
                self.dockwidget,
                self.tr('File exists'),
                self.tr('File {} already exists. Do you want to overwrite it?').format(filename),
                QMessageBox.StandardButton.YesToAll | QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No
            )
            if choice == QMessageBox.StandardButton.Yes:
                pass
            elif choice == QMessageBox.StandardButton.YesToAll:
                self.overwrite = True
            else:
                return

        # Save Style    
        status = layer.saveNamedStyle(url)

        # status is a tuple (str, bool)
        if not status[1]:
            self.iface.messageBar().pushWarning(self.tr('Save Style {} failed:').format(layer.name()), status[0])
        else:
            self.counter += 1


    def load_layer_style(self, layer):
        # Remove bad characters from layer name
        clean_name = re.sub(r'[^\w_.-]', '_', layer.name()) 

        url = os.path.join(self.url, clean_name + '.qml')

        # Check if this is the main annotation layer
        if layer == QgsProject.instance().mainAnnotationLayer():
            filename = "Main_Annotation_Layer.qml"
            url = os.path.join(self.url, filename)

        # Handle the dublicate layer name problem
        else:
            listoflayers = QgsProject.instance().mapLayersByName(layer.name())
            if len(listoflayers) > 1:
                i = listoflayers.index(layer)
                filename = clean_name + '__(' + str(i).zfill(2) + ').qml'
                url1 = os.path.join(self.url, filename) 
                if os.path.exists(url1):
                    url = url1
                else:
                    # Use a filename without number if it exists, else try (00)
                    if not os.path.exists(url):
                        url = os.path.join(self.url, clean_name + '__(00).qml')
            else:
                # If filename without number does not exist, try (00)
                if not os.path.exists(url):
                        url = os.path.join(self.url, clean_name + '__(00).qml')

        # Load the style
        # status = layer.loadNamedStyle(url, True) 
        status = layer.loadNamedStyle(url)
        
        if status[1]:
            self.counter += 1
        else:
            self.counter_fail += 1

        QgsProject.instance().setDirty()
        layer.triggerRepaint()
        layer.emitStyleChanged()
        self.iface.layerTreeView().refreshLayerSymbology(layer.id())





    #--------------------------------------------------------------------------

    def run(self):
        """Run method that loads and starts the plugin"""

        if not self.pluginIsActive:
            self.pluginIsActive = True

            #print "** STARTING AdjustStyle"

            # dockwidget may not exist if:
            #    first run of plugin
            #    removed on close (see self.onClosePlugin method)
            if self.dockwidget == None:
                # Create the dockwidget (after translation) and keep reference
                self.dockwidget = AdjustStyleDockWidget()

            # connect to provide cleanup on closing of dockwidget
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            self.value = 20

            # Connect buttons
            self.dockwidget.hueButton.clicked.connect(self.hueBtn)
            self.dockwidget.plusSatButton.clicked.connect(self.saturationPlusBtn)
            self.dockwidget.minusSatButton.clicked.connect(self.saturationMinusBtn)
            self.dockwidget.plusValueButton.clicked.connect(self.hsvValuePlusBtn)
            self.dockwidget.minusValueButton.clicked.connect(self.hsvValueMinusBtn)
            self.dockwidget.plusStrokeWidthButton.clicked.connect(self.strokeWidthPlusBtn)
            self.dockwidget.minusStrokeWidthButton.clicked.connect(self.strokeWidthMinusBtn)
            self.dockwidget.plusFontSizeButton.clicked.connect(self.fontSizePlusBtn)
            self.dockwidget.minusFontSizeButton.clicked.connect(self.fontSizeMinusBtn)
            self.dockwidget.replaceFontButton.clicked.connect(self.replace_font_dlg)
            self.dockwidget.saveStylesButton.clicked.connect(self.saveStylesBtn)
            self.dockwidget.loadStylesButton.clicked.connect(self.loadStylesBtn)
            
            # set IDs for radio buttons
            self.dockwidget.buttonGroup.setId(self.dockwidget.radioActiveLayer, 1)
            self.dockwidget.buttonGroup.setId(self.dockwidget.radioSelectedLayers, 2)
            self.dockwidget.buttonGroup.setId(self.dockwidget.radioVisibleLayers, 3)
            self.dockwidget.buttonGroup.setId(self.dockwidget.radioAllLayers, 4)

            # show the dockwidget
            self.iface.addDockWidget(Qt.DockWidgetArea.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()
