# -*- coding: utf-8 -*-

"""qgis2fds"""

__author__ = "Emanuele Gissi"
__date__ = "2020-05-04"
__copyright__ = "(C) 2020 by Emanuele Gissi"
__revision__ = "$Format:%H$"  # replaced with git SHA1

import time, os
from qgis.core import Qgis, QgsProject
from qgis.utils import pluginMetadata
from . import utils


class FDSCase:
    def __init__(
        self,
        feedback,
        path,
        name,
        utm_crs,
        wgs84_origin,
        pixel_size,
        dem_layer,
        domain,
        terrain,
        texture,
        wind,
    ) -> None:
        self.feedback = feedback
        self.name = name  # chid
        self.utm_crs = utm_crs
        self.wgs84_origin = wgs84_origin
        self.pixel_size = pixel_size
        self.dem_layer = dem_layer
        self.domain = domain
        self.terrain = terrain
        self.texture = texture
        self.wind = wind

        self.filename = f"{name}.fds"
        self.filepath = os.path.join(path, self.filename)

    def get_fds(self):
        # Init
        plugin_version = pluginMetadata("qgis2fds", "version")
        qgis_version = Qgis.QGIS_VERSION.encode("ascii", "ignore").decode("ascii")
        qgis_filepath = QgsProject.instance().fileName() or "not saved"
        date = time.strftime("%a, %d %b %Y, %H:%M:%S", time.localtime())

        landuse_layer_desc = f"{self.terrain.landuse_layer and self.terrain.landuse_layer.name() or 'none'}"
        landuse_type_filepath = f"{self.terrain.landuse_type.filepath and utils.shorten(self.terrain.landuse_type.filepath) or 'none'}"
        fire_layer_desc = (
            f"{self.terrain.fire_layer and self.terrain.fire_layer.name() or 'none'}"
        )
        wind_filepath = (
            f"{self.wind.filepath and utils.shorten(self.wind.filepath) or 'none'}"
        )

        # Prepare fds case
        return f"""\
! Generated by qgis2fds {plugin_version} on QGIS {qgis_version}
! QGIS file: {utils.shorten(qgis_filepath)}
! Date: {date}
{self.domain.get_comment()}
Desired resolution: {self.pixel_size:.1f}m
DEM layer: {self.dem_layer.name()}
Landuse layer: {landuse_layer_desc}
Landuse type file: {landuse_type_filepath}
Fire layer: {fire_layer_desc}
FDS DEVCs layer: FIXME
Wind file: {wind_filepath}

&HEAD CHID='{self.name}' TITLE='Description of {self.name}' /

MISC LEVEL_SET_MODE parameter
 1: Wind not affected by the terrain. No fire.
 2: Wind field established over the terrain, then frozen. No fire.
 3: Wind field following the terrain, no fire.
 4: Wind and fire fully-coupled.

&MISC ORIGIN_LAT={self.wgs84_origin.y():.7f}
      ORIGIN_LON={self.wgs84_origin.x():.7f}
      NORTH_BEARING=0.
      {self.texture.get_fds()}
      LEVEL_SET_MODE=1 
      THICKEN_OBSTRUCTIONS=T /

&TIME T_END=0. /

Example REAC, used when LEVEL_SET_MODE=4
_REAC ID='Wood' SOOT_YIELD=0.005 O=2.5 C=3.4 H=6.2
      HEAT_OF_COMBUSTION=17700. /
{self.domain.get_fds()}
{self.terrain.landuse_type.get_fds()}

Output quantities
&SLCF AGL_SLICE=5. QUANTITY='LEVEL SET VALUE' /
&SLCF AGL_SLICE=5. QUANTITY='TEMPERATURE' VECTOR=T /
&SLCF PBX={0.:.2f} QUANTITY='TEMPERATURE' VECTOR=T /
&SLCF PBY={0.:.2f} QUANTITY='TEMPERATURE' VECTOR=T /
{self.wind.get_fds()}
{self.terrain.get_fds()}

&TAIL /
"""

    def save(self):
        self.feedback.pushInfo(f"Write the fds case to <{self.filepath}>...")
        utils.write_file(
            feedback=self.feedback,
            filepath=self.filepath,
            content=self.get_fds(),
        )
