# -*- coding: utf-8 -*-
"""
/***************************************************************************
 SmartSDI
                                 A QGIS plugin
 Description
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-07-19
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Geo-System
        email                : geo-system@geo-system.com.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QUrl, QByteArray
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis.PyQt.QtNetwork import QNetworkRequest

from qgis.core import QgsProject, QgsMessageLog, Qgis, QgsNetworkAccessManager, QgsVectorLayer, QgsWkbTypes, QgsProcessingParameterFileDestination, QgsMapLayerProxyModel
from qgis.gui import QgsMessageBar

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .smartsdi_dialog import SmartSDIDialog
import os.path
import tempfile


class SmartSDI:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'SmartSDI_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&SmartSDI')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('SmartSDI', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = self.plugin_dir +'/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Pobierz działki i budynki'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&SmartSDI'),
                action)
            self.iface.removeToolBarIcon(action)
    
    def bbox(self):
        """Sets the BBOX and SRS fields from the current map extent"""
        
        bbox=self.iface.mapCanvas().extent()
        epsg=self.iface.mapCanvas().mapSettings().destinationCrs().authid()
        self.dlg.BBtop.setText(str(bbox.yMaximum()))
        self.dlg.BBbottom.setText(str(bbox.yMinimum()))
        self.dlg.BBright.setText(str(bbox.xMaximum()))
        self.dlg.BBleft.setText(str(bbox.xMinimum()))
        self.dlg.BBepsg.setText(epsg)
 
    def load(self,path,layer):
        """Loads the downloaded data as a new layer"""
        
        self.download_count+=1
        name='SmartSDI '+layer+str(self.download_count)
        vlayer = QgsVectorLayer(path, "SmartSDI "+layer,  "ogr")
        if not vlayer.isValid():
            QgsMessageLog.logMessage('Błąd wczytywania warstwy '+layer, name, Qgis.Warning)
        else:
            QgsProject.instance().addMapLayer(vlayer)
            #load the associated style
            style=os.path.join(self.plugin_dir,layer+'.qml')
            vlayer.loadNamedStyle(style)
            QgsMessageLog.logMessage('Wczytano warstwę '+name, 'SmartSDI', Qgis.Success)
    
    def download(self,epsg,bbox,layer):
        """Downloads the data from the web service"""
        
        #use POST request for long bboxes (large polygons/lines)
        if len(bbox)>512:
            url='https://smartsdi.pl/qgis/download.php'
            postData = QByteArray()
            postData.append("layer").append('=').append(layer).append("&")
            postData.append("epsg").append('=').append(epsg).append("&")
            postData.append("bbox").append('=').append(bbox).append("&")
            post=1
        else:
            url='https://smartsdi.pl/qgis/download.php?layer='+layer+'&epsg='+epsg+'&bbox='+bbox
            post=0
        
        workingUrl = QUrl(url)
        request = QNetworkRequest(workingUrl)
        networkAccessManager = QgsNetworkAccessManager()
        #POST
        if post:
            response = networkAccessManager.blockingPost(request,postData)
        #GET for short requests
        else:
            response = networkAccessManager.blockingGet(request)
        status_code = response.attribute(QNetworkRequest.HttpStatusCodeAttribute)
        #all OK
        if status_code == 200:
            #save data to temp file
            dest = QgsProcessingParameterFileDestination(name=layer)
            path = dest.generateTemporaryDestination()
            print(path)
            f = open(path, "wb")
            f.write(response.content())
            f.close()
            #load temp file as new layer
            self.load(path,layer)
        #HTTP status not OK - show the message
        elif status_code == 403:
            info=str(response.content(), 'utf-8')
            self.iface.messageBar().pushMessage("Błąd", info, level=Qgis.Critical)
        else:
            QgsMessageLog.logMessage(response.errorString(), 'SmartSDI', Qgis.Warning)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.download_count = 0;
            self.dlg = SmartSDIDialog()

        # show the dialog
        self.dlg.show()
        self.bbox()
        self.dlg.BBget.clicked.connect(self.bbox)
        #filter the layer list - only lines and polygons
        self.dlg.layerList.setFilters(QgsMapLayerProxyModel.PolygonLayer|QgsMapLayerProxyModel.LineLayer)
        
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            selectedLayer=self.dlg.layerList.currentLayer()
            #use the BBOX
            if selectedLayer is None:
                epsg=self.dlg.BBepsg.text().replace('EPSG:','')
                bbox=self.dlg.BBleft.text()
                bbox+=','+self.dlg.BBbottom.text()
                bbox+=','+self.dlg.BBright.text()
                bbox+=','+self.dlg.BBtop.text()
            #layer was selected - use it as a boundary for downloading
            else:
                epsg=selectedLayer.crs().authid().replace('EPSG:','')
                features=selectedLayer.getFeatures()
                for f in features:
                    bbox=f.geometry().asWkt()
                    break
            #which data to download (parcels, buildings, addresses)
            if(self.dlg.checkDz.isChecked()):
                self.download(epsg,bbox,'dzialki')
            if(self.dlg.checkBud.isChecked()):
                self.download(epsg,bbox,'budynki')
            if(self.dlg.checkAdr.isChecked()):
                self.download(epsg,bbox,'adresy')

