# -*- coding: utf-8 -*-
"""
/***************************************************************************
 FastPointInspection
                                 A QGIS plugin
 Fast Point Inspection is a QGIS plugin that streamlines the process of classifying point geometries in a layer.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-09-27
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Tharles de Sousa Andrade
        email                : irtharles@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis.PyQt.QtGui import QColor
from qgis.PyQt.QtCore import QVariant
from qgis.core import (
    Qgis,
    QgsProject,
    QgsField,
    QgsMarkerSymbol,
    QgsVectorLayer,
    QgsRuleBasedRenderer,
    QgsFeatureRequest
)
from PyQt5.QtWidgets import (
    QWidget,
    QLabel,
    QProgressBar,
    QHBoxLayout,
    QVBoxLayout,
    QGridLayout,
    QPushButton,
    QScrollArea
)
from qgis import utils
# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .fpi_dockwidget import FastPointInspectionDockWidget
from .db import setup_database, get_classes_from_db
from .editor import ClassesEditorWidget
import os.path


class FastPointInspection:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'FastPointInspection_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Fast Point Inspection')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'FastPointInspection')
        self.toolbar.setObjectName(u'FastPointInspection')

        # print "** INITIALIZING FastPointInspection"

        self.pluginIsActive = False
        self.dockwidget = None

        self.classes = None
        self.cls_selected = None
        self.progress_message_bar = None
        self.progress_bar = None
        self.current_class_label = None
        self.point_layer = None
        self.class_buttons = []
        self.class_layout = None
        self.layout = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('FastPointInspection', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/fpi/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Fast Point Inspection'),
            callback=self.run,
            parent=self.iface.mainWindow())

    # --------------------------------------------------------------------------

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        # print "** CLOSING FastPointInspection"

        # disconnects
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        self.dockwidget = None
        self.classes = None
        self.cls_selected = None
        self.progress_message_bar = None
        self.progress_bar = None
        self.current_class_label = None
        self.point_layer = None
        self.class_buttons = []
        self.class_layout = None
        self.layout = None
        self.pluginIsActive = False

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        # print "** UNLOAD FastPointInspection"

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Fast Point Inspection'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    # --------------------------------------------------------------------------

    def run(self):
        """Run method that loads and starts the plugin"""
        try:
            if not self.pluginIsActive:
                self.pluginIsActive = True

                # print "** STARTING FastPointInspection"

                # dockwidget may not exist if:
                #    first run of plugin
                #    removed on close (see self.onClosePlugin method)
                if self.dockwidget is None:
                    # Create the dockwidget (after translation) and keep reference
                    self.dockwidget = FastPointInspectionDockWidget()

                    setup_database()

                    self.layout = QVBoxLayout()

                    # Get the point layer from the project
                    self.point_layer = self.get_point_layer()

                    if self.point_layer is None:
                        self.iface.messageBar().pushMessage(
                            'POINTS GRID',
                            'The grid with points was not found.',
                            level=Qgis.Critical,
                            duration=5,
                        )

                    if self.point_layer is not None:
                        self.zoom()
                        self.add_class_field()
                        self.load_classes()

                        self.set_feature_color()
                        self.point_layer.selectionChanged.connect(self.set_class_for_selected_features)

                    # Label to show the current selected class
                    self.current_class_label = QLabel("Class not selected")
                    self.current_class_label.setAlignment(Qt.AlignCenter)
                    self.current_class_label.setStyleSheet("font-size: 25px;")

                    self.layout.addWidget(self.current_class_label)

                    # Clear classification button
                    clear_btn = QPushButton("Clear Classification")
                    clear_btn.clicked.connect(self.clear_classification)
                    clear_btn.setCursor(Qt.PointingHandCursor)

                    editor = ClassesEditorWidget(self)

                    self.layout.addWidget(editor)
                    self.layout.addLayout(self.class_layout)
                    self.layout.addWidget(clear_btn)

                    contentWidget = QWidget()
                    contentLayout = QVBoxLayout(contentWidget)

                    contentLayout.addLayout(self.layout)

                    self.scrollArea = QScrollArea()
                    self.scrollArea.setWidgetResizable(True)
                    self.scrollArea.setWidget(contentWidget)

                    self.dockwidget.setWidget(self.scrollArea)

                    QgsProject.instance().layerWasAdded.connect(self.handle_layer_added)
                # connect to provide cleanup on closing of dockwidget
                self.dockwidget.closingPlugin.connect(self.onClosePlugin)

                # show the dockwidget
                self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
                self.dockwidget.show()
        except Exception as e:
            self.iface.messageBar().pushMessage(
                'run',
                f"An error occurred while init plugin. Error -> {e}",
                level=Qgis.Critical,
                duration=5,
            )
            pass

    def handle_layer_added(self, layer):
        # Check the layer's CRS
        layer_crs = layer.crs().authid()
        if not layer_crs:
            new_crs = QgsProject.instance().crs().authid()
            layer.setCrs(new_crs)

    def reload_plugin(self):
        name = 'qgis-fpi'

        # Find the plugin
        plugin = utils.plugins.get(name)
        if plugin is None:
            raise ValueError(f"Plugin {name} not found.")

        # Unload the plugin
        utils.unloadPlugin(name)

        # Load the plugin again
        utils.loadPlugin(name)
        self.onClosePlugin()
        utils.startPlugin(name)

    def reset(self):
        self.classes = None
        self.cls_selected = None
        self.progress_message_bar = None
        self.progress_bar = None
        self.widget = None

    def load_classes(self):
        if not hasattr(self, 'class_layout') or self.class_layout is None:
            self.class_layout = QGridLayout()
        # Clear existing buttons from the layout
        for i in reversed(range(self.class_layout.count())):
            widget = self.class_layout.itemAt(i).widget()
            if widget is not None:
                widget.deleteLater()

        self.classes = get_classes_from_db()
        self.class_buttons = []
        row, col = 0, 0
        for cls in self.classes:
            btn = QPushButton(cls["class"])
            btn.setStyleSheet(f"background-color: rgb({','.join(cls['rgba'].split(',')[:3])})")
            btn.clicked.connect(lambda _, cls=cls: self.on_class_selected(cls))
            btn.setCursor(Qt.PointingHandCursor)
            self.class_buttons.append(btn)

            self.class_layout.addWidget(btn, row, col)
            col += 1
            if col >= 2:  # Set the number of columns in the grid
                col = 0
                row += 1

    def start_processing(self):
        try:
            # Remove existing progress bar if any
            if self.progress_message_bar:
                self.finish_progress()

            self.progress_message_bar = self.iface.messageBar().createMessage("")

            self.progress_bar = QProgressBar()
            self.progress_bar.setMaximum(100)

            # Create a QWidget to hold the progress bar
            container = QWidget()
            layout = QHBoxLayout(container)
            layout.setContentsMargins(0, 0, 0, 0)
            layout.addWidget(QLabel("Processing..."))
            layout.addWidget(self.progress_bar)
            container.setLayout(layout)

            self.progress_message_bar.layout().addWidget(container)
            self.iface.messageBar().pushWidget(self.progress_message_bar, Qgis.Info)
        except Exception as e:
            self.iface.messageBar().pushMessage(
                'start_processing',
                f"Error -> {e}",
                level=Qgis.Critical,
                duration=5,
            )
            pass

    def update_progress(self, value):
        if self.progress_bar:
            self.progress_bar.setValue(value)
        else:
            self.start_processing()
            self.progress_bar.setValue(value)

    def finish_progress(self):
        if self.progress_message_bar:
            # Hide and remove the progress bar
            self.iface.messageBar().clearWidgets()
            self.progress_bar = None
            self.progress_message_bar = None

    def zoom(self):
        try:
            # Set the extent to the layer's extent
            self.iface.mapCanvas().setExtent(self.point_layer.extent())
            self.point_layer.extent()

            # Set the scale to 1:5000
            self.iface.mapCanvas().zoomScale(5000)

            self.iface.mapCanvas().setDestinationCrs(self.point_layer.crs())

            # Refresh the canvas
            self.iface.mapCanvas().refreshAllLayers()
        except Exception as e:
            self.iface.messageBar().pushMessage(
                'ON_SET_CLASS',
                "An error occurred while attempting to save the class.",
                level=Qgis.Critical,
                duration=5,
            )
            pass

    def clear_classification(self):
        if self.point_layer is None:
            return
        self.iface.actionSelectFreehand().trigger()
        self.current_class_label.setText("Removing Classification")
        self.current_class_label.setStyleSheet("background-color: none; font-size: 25px;")
        self.cls_selected = {
            "class": None,
            "rgba": "0,0,0,20",
            "class_id": None
        }

    def get_point_layer(self):
        for layer in QgsProject.instance().mapLayers().values():
            if isinstance(layer, QgsVectorLayer):
                return layer
        return None

    def add_class_field(self):
        provider = self.point_layer.dataProvider()
        if provider.fieldNameIndex("class") == -1 and provider.fieldNameIndex("class_id") == -1:
            class_id = QgsField("class_id", QVariant.Int)
            _class = QgsField("class", QVariant.String)
            provider.addAttributes([class_id, _class])
            self.point_layer.updateFields()

    def on_class_selected(self, cls):
        self.cls_selected = cls
        self.current_class_label.setText(f"{cls['class']}")
        self.current_class_label.setStyleSheet(
            f"background-color: rgb({','.join(cls['rgba'].split(',')[:3])}); font-size: 25px;")
        self.iface.actionSelectFreehand().trigger()

    def set_class_for_selected_features(self):
        try:
            self.start_processing()
            cls = self.cls_selected

            if cls is None:
                self.iface.messageBar().pushMessage(
                    'ON_SET_CLASS',
                    "Please choose a class; none has been selected yet.",
                    level=Qgis.Critical,
                    duration=5,
                )
                self.update_progress(100)
                self.finish_progress()
                return

            selected_features = [feature.id() for feature in self.point_layer.selectedFeatures()]
            request = QgsFeatureRequest()
            request.setFilterFids(selected_features)

            rgba = [int(i) for i in cls['rgba'].split(',')]

            self.iface.mapCanvas().setSelectionColor(QColor(rgba[0], rgba[1], rgba[2], rgba[3]))

            all_features = list(self.point_layer.getFeatures(request))
            class_id_idx = self.point_layer.fields().indexOf('class_id')
            class_idx = self.point_layer.fields().indexOf('class')

            total_features = len(all_features)

            if total_features == 0:
                # Handle the case where there are no selected features
                self.iface.messageBar().pushMessage(
                    'ON_SET_CLASS',
                    "No features selected.",
                    level=Qgis.Critical,
                    duration=5,
                )
                self.update_progress(100)
                self.finish_progress()
                return
            else:
                progress_per_feature = 70.0 / total_features

            self.point_layer.startEditing()

            attribute_map = {}

            for i, feature in enumerate(all_features):
                feature_id = feature.id()
                attributes = {class_id_idx: cls['class_id'], class_idx: cls['class']}
                attribute_map[feature_id] = attributes

                self.update_progress(int(10 + (i + 1) * progress_per_feature))

            self.update_progress(80)
            self.point_layer.dataProvider().changeAttributeValues(attribute_map)

            self.point_layer.commitChanges()
            self.update_progress(90)
            self.set_feature_color()
            self.update_progress(100)
            self.finish_progress()
        except Exception as e:
            self.update_progress(100)
            self.finish_progress()
            self.iface.messageBar().pushMessage(
                'ON_SET_CLASS',
                f"An error occurred while attempting to save the class. Error -> {e}",
                level=Qgis.Critical,
                duration=5,
            )
            pass

    def set_feature_color(self):
        try:
            # Create default symbol and renderer
            symbol = QgsMarkerSymbol.createSimple({'name': 'square'})

            symbol_layer = symbol.symbolLayer(0)
            symbol_layer.setStrokeStyle(Qt.SolidLine)
            symbol_layer.setStrokeColor(QColor(0, 0, 0, 255))

            renderer = QgsRuleBasedRenderer(symbol)

            # Prepare rules
            rules = []
            for cls in self.classes:
                rgba = cls['rgba'].split(',')
                color = QColor(int(rgba[0]), int(rgba[1]), int(rgba[2]), int(rgba[3]))
                rules.append(
                    (cls['class'], f""" "class" = '{cls['class']}' """, color)
                )

            # Additional rule for NOT DEFINED
            rules.append(('NOT DEFINED', ' "class" is NULL ', QColor(0, 0, 0, 20)))

            # Set up the rule-based symbology
            root_rule = renderer.rootRule()

            for label, expression, color in rules:
                rule = root_rule.children()[0].clone()
                rule.setLabel(label)
                rule.setFilterExpression(expression)

                # Set color
                rule.symbol().setColor(color)

                # Remove border for all except 'NOT DEFINED'
                if label == 'NOT DEFINED':
                    rule.symbol().symbolLayer(0).setStrokeStyle(Qt.SolidLine)
                    rule.symbol().symbolLayer(0).setStrokeColor(QColor(0, 0, 0, 20))
                else:
                    rule.symbol().symbolLayer(0).setStrokeStyle(Qt.NoPen)

                root_rule.appendChild(rule)

            # Remove default rule
            root_rule.removeChildAt(0)

            # Apply the renderer to the layer
            if self.point_layer is not None:
                self.point_layer.setRenderer(renderer)
                self.iface.layerTreeView().refreshLayerSymbology(self.point_layer.id())
                self.point_layer.triggerRepaint()
        except Exception as e:
            self.iface.messageBar().pushMessage(
                'ON_SET_FEATURE_COLOR',
                f"Failed to Set Feature Color. Error -> {e}",
                level=Qgis.Critical,
                duration=5,
            )
            pass
