# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ElementTable
                                 A QGIS plugin
 This plugin lets to set/unset visibility of layers with chemical data with
 help of element table widget
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-12-07
        git sha              : $Format:%H$
        copyright            : (C) 2020 by Petras Jokubauskas
        email                : p.jokubauskas@uw.edu.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from .qpet import element_table
from qgis.core import QgsProject, QgsLayerTreeGroup
# Import the modified class for the DockWidget:
from .chemical_selector_dockwidget import ElementTableDockWidget
import os.path


class ElementTable:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):

        self.iface = iface

        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'ElementTable_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Element table')
        # This is a bit hacky way for placing button on the mini toolbar
        # in the Layers panel; this references the toolbar for adding
        # the button of table:
        self.legendWidget = self.iface.layerTreeView().parent()
        self.toolbar = self.legendWidget.children()[1]

        self.pluginIsActive = False
        self.dockwidget = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ElementTable', message)

    def add_action(self,
                   icon_path,
                   text,
                   callback,
                   enabled_flag=True,
                   add_to_menu=True,
                   add_to_toolbar=True,
                   status_tip=None,
                   whats_this=None,
                   parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        # get shortcuts to project and layerTreeRoot - the main containers of
        # the layers:
        self.proj = QgsProject.instance()
        self.ltr = self.proj.layerTreeRoot()
        icon_path = os.path.join(self.plugin_dir, 'pt-colorful.svg')
        self.add_action(
            icon_path,
            text=self.tr(u'Element Table'),
            callback=self.run,
            parent=self.iface.mainWindow())

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        # disconnects
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        # self.dockwidget = None

        self.pluginIsActive = False

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'Element Table'),
                action)
            self.toolbar.removeAction(action)
        # remove the REFERENCE to toolbar
        del self.toolbar
        del self.legendWidget

    # --------------------------------------------------------------------------

    def run(self):
        """Run method that loads and starts the plugin
        (method is called when plugin button is clicked"""

        if not self.pluginIsActive:
            self.initTable()
        elif self.dockwidget.isVisible():
            self.dockwidget.hide()
        elif self.dockwidget.isHidden():
            self.dockwidget.show()

    def initTable(self):
        self.pluginIsActive = True
        if self.dockwidget is None:
            self.dockwidget = ElementTableDockWidget()
        # create dictionary where tie layer_ids with chemical information
        # it would be easier to do with bidict, but lets keep dependencies min.
        # Element is key, ids will be in lists as value
        self.chemical_dict = {i: list()
                              for i in element_table.geo_groups['ALL']}
        # layer_id will be key, element will be value
        self.layers_dict = dict()
        # disabling all buttons on the widget
        for i in list(self.chemical_dict.keys()):
            self.dockwidget.et.update_state(i, enable=False)
        self.scan_added_layers_for_elements(self.proj.mapLayers())
        # connect to provide cleanup on closing of dockwidget
        self.dockwidget.closingPlugin.connect(self.onClosePlugin)

        self.iface.addDockWidget(Qt.LeftDockWidgetArea, self.dockwidget)
        self.dockwidget.show()
        self.proj.legendLayersAdded.connect(
            self.scan_added_layers_for_elements)
        self.proj.layersWillBeRemoved.connect(self.update_at_layers_removed)
        self.ltr.visibilityChanged.connect(
            self.update_element_table_check_state)
        self.dockwidget.et.elementChecked.connect(
            self.check_elements_in_legend)
        self.dockwidget.et.elementUnchecked.connect(
            self.uncheck_elements_in_legend)
        self.dockwidget.et.allElementsCleared.connect(
            self.uncheck_all_elements_in_legend)
        self.ltr.nameChanged.connect(self.scan_renamed_layer)

    def scan_renamed_layer(self, layer_node, new_name):
        if type(layer_node) == QgsLayerTreeGroup:
            return
        layer_id = layer_node.layerId()
        new_element = self.element_in_layer_name(layer_id)
        if layer_id in self.layers_dict:
            old_element = self.layers_dict[layer_id]
            if new_element == old_element:
                return
            self.update_at_layers_removed([layer_id])
        if new_element is not None:
            self.scan_added_layers_for_elements(layer_id)

    def scan_added_layers_for_elements(self, map_layers):
        """scan added layers to TOC if names contain abbreviation(s)
           or element name(s) and add its id to the dictionary for tracking
           their state. Finally, if added layer is detected to be chemical
           then update the state of corresponding button on the widget"""
        if type(map_layers) == list:
            map_layers = [i.id() for i in map_layers]
        elif type(map_layers) == dict:
            map_layers = list(map_layers.keys())
        elif type(map_layers) == str:
            map_layers = [map_layers]
        for layer_id in map_layers:
            element = self.element_in_layer_name(layer_id)
            if element is not None:
                self.chemical_dict[element].append(layer_id)
                self.layers_dict[layer_id] = element
                if len(self.chemical_dict[element]) == 1:  # enable button once
                    self.dockwidget.et.update_state(element, enable=True)
                if self.ltr.findLayer(layer_id).isVisible():
                    self.dockwidget.et.switch_button_wo_trigger([element],
                                                                press=True)

    def update_at_layers_removed(self, map_layers):
        for layer_id in map_layers:
            if layer_id in self.layers_dict:
                element = self.layers_dict[layer_id]
                self.chemical_dict[element].remove(layer_id)
                del self.layers_dict[layer_id]
                if not any([self.ltr.findLayer(j).isVisible()
                            for j in self.chemical_dict[element]]):
                    self.dockwidget.et.switch_button_wo_trigger([element],
                                                                press=False)
                if len(self.chemical_dict[element]) == 0:
                    self.dockwidget.et.update_state(element, enable=False)

    def update_element_table_check_state(self, layer_tree_node):
        # groups are also layer tree nodes, if it is group emiting,
        # then method needs to be skipped
        if type(layer_tree_node) == QgsLayerTreeGroup:
            return
        layer_id = layer_tree_node.layer().id()
        if layer_id in self.layers_dict:
            element = self.layers_dict[layer_id]
            if any([self.ltr.findLayer(j).isVisible()
                    for j in self.chemical_dict[element]]):
                state = True
            else:
                state = False
            self.dockwidget.et.switch_button_wo_trigger([element], press=state)

    def uncheck_elements_in_legend(self, element):
        for i in self.chemical_dict[element]:
            self.ltr.findLayer(i).setItemVisibilityChecked(False)

    def uncheck_all_elements_in_legend(self):
        for element in self.chemical_dict:
            self.uncheck_elements_in_legend(element)

    def check_elements_in_legend(self, element):
        for i in self.chemical_dict[element]:
            self.ltr.findLayer(i).setItemVisibilityCheckedParentRecursive(True)

    def element_in_layer_name(self, layer_id):
        """return element abbreviation from layers name using robust parser
        implemented in element_table module; for this application
        the result lenght is checked so that only element from a meaningful
        layer name would be returned"""
        name = self.proj.mapLayer(layer_id).name()
        elements = element_table.parse_elements(name)  # this returns a set
        if len(elements) == 1:  # we pass a set with single element
            return elements.pop()
        elif len(elements) == 2:  # if oxide, i.e. string "Fe2O3" {"Fe","O"}
            if "O" in elements:
                elements.remove("O")
                return elements.pop()
