# -*- coding: utf-8 -*-
"""
/***************************************************************************
 easydemDialog
                                 A QGIS plugin
 Get Digital Elevation Model (DEM) data from Google Earth Engine and plot as raster layer it contour lines, make elevation maps.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2024-11-13
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Caio Arantes
        email                : caiosimplicioarantes@gmail.com
        ICON SOURCE: <a href="https://www.flaticon.com/free-icons/topography" title="topography icons">Topography icons created by Freepik - Flaticon</a>
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
# Standard library imports
import os
import sys
import importlib
import platform
import subprocess
import zipfile
import json
import webbrowser
import io
import array
import shutil
import urllib.request
from datetime import datetime, timedelta
from dateutil.relativedelta import relativedelta

# Third-party imports
import geopandas as gpd
import requests
import pandas as pd
import numpy as np
from scipy.signal import savgol_filter
import plotly.express as px
import plotly.graph_objects as go
from plotly.subplots import make_subplots

# PyQt5 imports
from PyQt5 import uic, QtWidgets
from PyQt5.QtCore import QSettings, Qt, QDate
from PyQt5.QtWidgets import (
    QDialog,
    QMessageBox,
    QFileDialog,
    QApplication,
    QGridLayout,
    QWidget,
    QDesktopWidget,
    QVBoxLayout,
    QCheckBox,
    QDialogButtonBox,
    QPushButton,
    QLineEdit,
)
from PyQt5.QtGui import QColor

# QGIS imports
import qgis
from qgis.core import (
    QgsProject,
    QgsRasterLayer,
    QgsRasterShader,
    QgsColorRampShader,
    QgsSingleBandPseudoColorRenderer,
    QgsStyle,
    QgsRasterBandStats,
    QgsMapLayer,
    QgsVectorLayer,
    QgsColorRamp,
    QgsLayerTreeLayer,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsMultiBandColorRenderer,
    QgsContrastEnhancement,
    QgsProcessingFeedback,
    QgsApplication,
    QgsRectangle,
    QgsFeature,
    QgsGeometry,
    QgsField,
    QgsVectorFileWriter,
)
from qgis.utils import iface
from qgis import processing
from qgis.PyQt.QtCore import QVariant
from qgis.analysis import QgsNativeAlgorithms


def get_installed_version():
    """Return the installed Earth Engine API version, or None if not installed."""
    try:
        import ee
        return ee.__version__
    except ImportError:
        return None

def get_latest_version():
    """Query PyPI for the latest Earth Engine API version."""
    try:
        url = "https://pypi.org/pypi/earthengine-api/json"
        with urllib.request.urlopen(url) as response:
            data = json.load(response)
        return data["info"]["version"]
    except Exception as e:
        print("Error fetching latest version from PyPI:", e)
        return None

def install_earthengine_api():
    """Install or upgrade the Earth Engine API to the latest version using pip's internal API."""
    try:
        # Attempt to use pip.main (for older pip versions)
        import pip
        print("Using pip version:", pip.__version__)
        pip_args = ['install', '--upgrade', 'earthengine-api']
        pip.main(pip_args)
        print("Earth Engine API installed/upgraded successfully (using pip.main).")
    except AttributeError:
        # Fallback for newer pip versions that do not expose pip.main
        try:
            from pip._internal.cli.main import main as pip_main
            pip_main(['install', '--upgrade', 'earthengine-api'])
            print("Earth Engine API installed/upgraded successfully (using pip._internal).")
        except Exception as e:
            print("An error occurred during installation:", e)
    except Exception as e:
        print("An error occurred during installation:", e)

# Determine installed and latest versions.
installed_version = get_installed_version()
latest_version = get_latest_version()

if installed_version:
    print("Installed Earth Engine API version:", installed_version)
else:
    print("Earth Engine API is not installed.")

if latest_version:
    print("Latest Earth Engine API version available on PyPI:", latest_version)
else:
    print("Could not determine the latest Earth Engine API version from PyPI.")

# If there's no installation or the installed version differs from the latest, install/upgrade.
if (installed_version is None) or (latest_version is not None and installed_version != latest_version):
    print("Upgrading/Installing Earth Engine API to the latest version...")
    install_earthengine_api()
    # Invalidate caches so that the newly installed package is found.
    importlib.invalidate_caches()
else:
    print("Latest version is already installed. Importing Earth Engine API...")

# Import the Earth Engine API and print its version.
try:
    importlib.import_module('ee')
    import ee
    print("Final Earth Engine API version:", ee.__version__)
except ImportError:
    print("Earth Engine API could not be imported after installation.")


# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'easy_dialog_base.ui'))

class easydemDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, parent=None):
        super(easydemDialog, self).__init__(parent)
        self.setupUi(self)

        self.dem_datasets = { "NASADEM": {
                "ID": "NASA/NASADEM_HGT/001",
                "Resolution": [30],
                "Coverage": "Global",
                "Description": "A refined reprocessing of SRTM data, incorporating auxiliary datasets for improved accuracy and void reduction.",
                "Info": "<b>NASADEM</b> <br>"
                        "<b>ID:</b> NASA/NASADEM_HGT/001 <br>"
                        "<b>Resolution:</b> 30 meters <br>"
                        "<b>Coverage:</b> Global <br>"
                        "NASADEM integrates auxiliary data from ASTER GDEM, ICESat GLAS, and PRISM datasets. Significant improvements include enhanced phase unwrapping and void reduction using ICESat GLAS data as control. "
                        "These refinements make it ideal for detailed topographic studies and high-resolution terrain analysis. "
                        "(<a href='https://earthdata.nasa.gov/esds/competitive-programs/measures/nasadem'>Source</a>)"
            },
            "ASTER Global Digital Elevation Model (GDEM)": {
                "ID": "NASA/ASTER_GED/AG100_003",
                "Resolution": [100],
                "Coverage": "Global",
                "Description": "The Advanced Spaceborne Thermal Emission and Reflection Radiometer Global Emissivity Database (ASTER-GED) is a comprehensive product developed by NASA's JPL and Caltech. It includes elevation data, mean emissivity, LST, NDVI, and standard deviations for ASTER Thermal Infrared bands.",
                "Info": "<b>ASTER GDEM</b> <br>"
                        "<b>ID:</b> NASA/ASTER_GED/AG100_003 <br>"
                        "<b>Resolution:</b> 100 meters <br>"
                        "<b>Coverage:</b> Global <br>"
                        "Derived from clear-sky pixels using the ASTER Temperature Emissivity Separation (TES) algorithm with WVS atmospheric correction. "
                        "(<a href='https://developers.google.com/earth-engine/datasets/catalog/NASA_ASTER_GED_AG100_003/'>Source</a>)"
            },
            "Copernicus Global Digital Elevation Model (GLO-30)": {
                "ID": "COPERNICUS/DEM/GLO30",
                "Resolution": [30],
                "Coverage": "Global",
                "Description": "A Digital Surface Model (DSM) that includes Earth's surface features such as buildings, infrastructure, and vegetation, derived from the WorldDEM product.",
                "Info": "<b>Copernicus GLO-30</b> <br>"
                        "<b>ID:</b> COPERNICUS/DEM/GLO30 <br>"
                        "<b>Resolution:</b> 30 meters <br>"
                        "<b>Coverage:</b> Global <br>"
                        "Derived from the WorldDEM product, based on radar data from the TanDEM-X mission, a partnership between DLR and Airbus Defence and Space. "
                        "Earth Engine assets are ingested from DGED files. "
                        "(<a href='https://spacedata.copernicus.eu/collections/elevation'>Source</a>)"
            },
            "JAXA ALOS Global Digital Surface Model (AW3D30)": {
                "ID": "JAXA/ALOS/AW3D30/V3_2",
                "Resolution": [30],
                "Coverage": "Global",
                "Description": "A global digital surface model (DSM) dataset at ~30-meter resolution, derived from the high-resolution (5-meter) World 3D Topographic Data.",
                "Info": "<b>JAXA ALOS DSM (AW3D30)</b> <br>"
                        "<b>ID:</b> JAXA/ALOS/AW3D30/V3_2 <br>"
                        "<b>Resolution:</b> 30 meters <br>"
                        "<b>Coverage:</b> Global <br>"
                        "Version 3.2 (January 2021) includes updates to high-latitude formats, auxiliary data, and processing methods. "
                        "It uses stereo optical image matching for elevation calculation, with improvements in detecting anomalous values and incorporating updated auxiliary datasets such as coastline and AW3D version 3 data for Japan. "
                        "Clouds, snow, and ice are masked during processing, but some errors may persist near their edges. "
                        "Due to variable resolutions, this dataset is an image collection, requiring reprojection for slope computations. "
                        "(<a href='https://www.eorc.jaxa.jp/ALOS/en/aw3d30/index.htm'>Source</a>)"
            },
            "GMTED2010 (Global Multi-resolution Terrain Elevation Data 2010)": {
                "ID": "USGS/GMTED2010_FULL",
                "Resolution": [250, 500, 1000],
                "Coverage": "Global",
                "Description": "A global elevation dataset derived from multiple sources, replacing the GTOPO30 Elevation Model, with coverage at multiple resolutions.",
                "Info": "<b>GMTED2010</b> <br>"
                        "<b>ID:</b> USGS/GMTED2010_FULL <br>"
                        "<b>Resolutions:</b> 250, 500, and 1000 meters <br>"
                        "<b>Coverage:</b> Global <br>"
                        "Developed using NGA's SRTM DTED® (1-arc-second void-filled data) as the primary source. Additional data sources include non-SRTM DTED®, Canadian CDED, SPOT 5 Reference3D, US NED, Australia's GEODATA, and DEMs for Antarctica and Greenland. "
                        "This dataset offers improved global elevation data and replaces the GTOPO30 Elevation Model. "
                        "(<a href='https://topotools.cr.usgs.gov/gmted_viewer/viewer.htm'>Source</a>)"
            }

        }

        self.textEdit.setReadOnly(True)  # Prevent editing
        self.textEdit.setTextInteractionFlags(Qt.TextBrowserInteraction)
        self.textEdit.anchorClicked.connect(self.open_link)

        self.tabWidget.setCurrentIndex(0)

        self.folder_set = False
        self.aio_set = True
        self.autentication = False

        # Call update_dem_datasets after initialization to avoid accessing dem_datasets before it's defined.
        self.update_dem_datasets()
        #self.load_vector_layers()
        self.dem_dataset_combobox.currentIndexChanged.connect(self.update_dem_info)
        self.load_vector_layers_button.clicked.connect(self.load_vector_layers)
        self.vector_layer_combobox.currentIndexChanged.connect(self.get_selected_layer_path)
        self.autenticacao.clicked.connect(self.auth)
        self.desautenticacao.clicked.connect(self.auth_clear)
        self.elevacao.clicked.connect(self.elevacao_clicked)
        self.mQgsFileWidget.fileChanged.connect(self.on_file_changed)
        self.pushButtonNext.clicked.connect(self.next_button_clicked)
        self.tabWidget.currentChanged.connect(self.on_tab_changed)

        self.project_QgsPasswordLineEdit.setEchoMode(QtWidgets.QLineEdit.Normal)

        # Ensure this sets up self.project_QgsPasswordLineEdit (or rename to something like self.projectIdLineEdit)
        self.loadProjectId()
        # Connect the textChanged signal to automatically save changes.
        self.project_QgsPasswordLineEdit.textChanged.connect(self.autoSaveProjectId)


    def open_link(self, url):
        """Open the clicked link in the default web browser."""
        print(f"Opening URL: {url.toString()}")
        webbrowser.open(url.toString())

    def next_clicked(self):
        self.tabWidget.setCurrentIndex((self.tabWidget.currentIndex() + 1) % self.tabWidget.count())

    def back_clicked(self):
        self.tabWidget.setCurrentIndex((self.tabWidget.currentIndex() - 1) % self.tabWidget.count())

    def load_path_sugestion(self):
        """
        Load the path suggestion based on the user's operating system.
        """
        system = platform.system()
        if system == 'Windows':
            self.output_folder = os.path.join(os.environ['USERPROFILE'], 'Downloads')
        elif system == 'Linux':
            self.output_folder = os.path.join(os.environ['HOME'], 'Downloads')
        elif system == 'Darwin':  # MacOS
            self.output_folder = os.path.join(os.environ['HOME'], 'Downloads')

        # Pre-configure with a suggested directory
        self.mQgsFileWidget.setFilePath(self.output_folder)

    def loadProjectId(self):
        """
        Loads the saved project ID from QSettings and sets it in the widget.
        This will run every time the plugin is opened.
        """
        settings = QSettings()
        # Retrieve the project ID from QSettings. The key "MyPlugin/projectID" is arbitrary.
        saved_project_id = settings.value("MyPlugin/projectID", "", type=str)
        self.project_QgsPasswordLineEdit.setText(saved_project_id)
        print("Loaded project ID:", saved_project_id)
        self.autenticacao.setEnabled(bool(self.project_QgsPasswordLineEdit.text()))

    def autoSaveProjectId(self, new_text):
        """
        Automatically saves the project ID to QSettings whenever the text changes.
        This ensures that the project ID remains available even after QGIS is closed and reopened.
        """
        settings = QSettings()
        settings.setValue("MyPlugin/projectID", new_text)
        print("Project ID auto-saved:", new_text)
        self.autenticacao.setEnabled(bool(self.project_QgsPasswordLineEdit.text()))

    def pop_aviso_auth(self, aviso):
        """
        Displays a warning message box with the given message and Ok button.
        Args:
            aviso (str): The warning message to display in the message box.
        Returns:
            None
        Note:
            This method restores the override cursor before displaying the message box.
        """
        QApplication.restoreOverrideCursor()
        msg = QMessageBox(self)
        msg.setWindowTitle("Warning!")
        msg.setIcon(QMessageBox.Warning)
        msg.setText(aviso)
        
        # Set buttons with Ok on the right
        msg.setStandardButtons(QMessageBox.Ok)
        
        # Access the buttons to set custom text
        ok_button = msg.button(QMessageBox.Ok)
        ok_button.setText("Ok")
        
        msg.exec_()

    def on_tab_changed(self, index):
        print(f"Tab changed to index: {index}")
        if index == 2 and (self.pushButtonNext.isEnabled() == False):
            self.tabWidget.setCurrentIndex(1)
            return
        
        if index == 1 and (self.autentication == False):
            self.tabWidget.setCurrentIndex(0)

        if index == 1:
            self.load_vector_layers()


    def next_button_clicked(self):
        self.tabWidget.setCurrentIndex(self.tabWidget.currentIndex() + 1)

    def on_file_changed(self, file_path):
        """Slot called when the selected file changes."""
        print(f"File selected: {file_path}")
        self.output_folder = file_path
        self.folder_set = True
        self.check_next_button()

    def check_next_button(self):
        """Enables the Next button if all required inputs are set."""
        if self.folder_set and self.aio_set:
            self.pushButtonNext.setEnabled(True)
        else:
            self.pushButtonNext.setEnabled(False)

    def update_dem_datasets(self):
        print(list(self.dem_datasets.keys()))
        self.dem_dataset_combobox.addItems(list(self.dem_datasets.keys()))
        self.update_dem_info()

    def get_unique_filename(self, base_file_name):
        """
        Generates a unique filename by checking if the file already exists
        and adding a numerical suffix to it if needed.

        Parameters:
        base_file_name (str): The base filename to use.

        Returns:
        str: The unique filename.
        """
        output_file = self.output_folder+f'/{base_file_name}.tif'
        counter = 1

        while os.path.exists(output_file):
            output_file = self.output_folder +f'/{base_file_name}_{counter}.tif'
            counter += 1

        print(f"Unique filename: {output_file}")
        return output_file

    def load_vector_layers(self) -> None:
        layers = QgsProject.instance().mapLayers().values()
        self.vector_layer_combobox.clear()
        self.vector_layer_ids = {}
        
        for layer in layers:
            if layer.type() == QgsMapLayer.VectorLayer:
                layer_name = layer.name()
                print(f"Adding layer: {layer_name}")  # Debug: Show added layer names
                self.vector_layer_combobox.addItem(layer_name)
                self.vector_layer_ids[layer_name] = layer.id()
        
        # Debug: Show the layer dictionary after loading
        print(f"Loaded vector layers: {self.vector_layer_ids}")
        self.get_selected_layer_path()


    def get_selected_layer_path(self):
        """
        Retrieves the path of the currently selected layer in the combobox and triggers further processing.
        """
        # Get the currently selected layer name from the combobox
        layer_name = self.vector_layer_combobox.currentText()
        self.zoom_to_layer(layer_name)
        print(f"Selected layer name: {layer_name}")  # Debug: Show selected layer name

        # Get the corresponding layer ID
        layer_id = self.vector_layer_ids.get(layer_name)

        # Get the layer using its ID
        layer = QgsProject.instance().mapLayer(layer_id)
        if layer:
            print(f"Layer found: {layer.name()}, ID: {layer_id}")  # Debug: Confirm layer is found
            self.selected_aio_layer_path = layer.dataProvider().dataSourceUri().split('|')[0]
            print(f"Selected layer path: {self.selected_aio_layer_path}")  # Debug: Show selected layer path
            
            # Trigger the processing function
            self.load_vector_function()

            # Enable next steps if necessary
            return None
        else:
            print(f"Layer '{layer_name}' with ID '{layer_id}' not found in the project.")
            return None

    def update_dem_info(self):
        dem_name = self.dem_dataset_combobox.currentText()
        dem_info = self.dem_datasets[dem_name]["Info"]
        self.dem_info_textbox.setHtml(dem_info)
        self.dem_resolution_combobox.clear()
        self.dem_resolution_combobox.addItems([str(res) for res in self.dem_datasets[dem_name]["Resolution"]])


    def zoom_to_layer(self, layer_name, margin_ratio=0.1):
        """
        Zoom to the specified layer with an optional margin.

        :param layer_name: Name of the layer to zoom to.
        :param margin_ratio: Fraction of the extent to add as margin (default is 0.1, or 10%).
        """
        project = QgsProject.instance()
        layers = project.mapLayersByName(layer_name)  # Get layers matching the name
        
        if not layers:
            print(f"Layer '{layer_name}' not found.")
            return
        
        layer = layers[0]  # Use the first matching layer
        iface = qgis.utils.iface  # Access the QGIS interface
        canvas = iface.mapCanvas()  # Get the active map canvas
        
        # Ensure the canvas CRS matches the layer CRS
        canvas.setDestinationCrs(layer.crs())
        
        # Get the layer's extent and add a margin
        layer_extent = layer.extent()
        x_margin = layer_extent.width() * margin_ratio
        y_margin = layer_extent.height() * margin_ratio
        
        expanded_extent = QgsRectangle(
            layer_extent.xMinimum() - x_margin,
            layer_extent.yMinimum() - y_margin,
            layer_extent.xMaximum() + x_margin,
            layer_extent.yMaximum() + y_margin
        )
        
        # Set the expanded extent to the canvas
        canvas.setExtent(expanded_extent)
        canvas.refresh()
        
        print(f"Zoomed to layer extent with margin: {expanded_extent.toString()}")

    def auth(self):
        """
        Authenticates Earth Engine and validates the default project.
        Warnings are displayed only if the default project is invalid.
        """
        try:
            # Step 1: Authenticate and initialize Earth Engine
            print("Authenticating Earth Engine...")
            ee.Authenticate()
            ee.Initialize(project=self.project_QgsPasswordLineEdit.text())
            print("Authentication successful!")

            # Step 2: Test default project
            print("Testing default project...")
            default_project_path = f"projects/{self.project_QgsPasswordLineEdit.text()}/assets/"  # Replace with your default project's path if known

            # Attempt to list assets in the default project
            try:
                assets = ee.data.listAssets({'parent': default_project_path})
                print(f"Assets in default project: {assets}")

                if assets.get('assets') is not None:  # Valid project detected
                    print("Default project is valid.")
                    self.pop_aviso_auth("Authentication successful!")
                    self.autentication = True
                    self.load_vector_layers()
                    self.load_path_sugestion()
                    self.next_clicked()
                else:
                    print("Default project is valid but contains no assets.")  # No warning needed for this case
            except ee.EEException as e:
                # Invalid project or access issue
                print(f"Default project validation failed: {e}")
                self.pop_aviso_auth(f"Default project validation failed: {e}\nFollow the instructions to have a valid Google Cloud project.")
                self.auth_clear(True)


        except ee.EEException as e:
            # Handle Earth Engine-specific errors
            print(f"Earth Engine error: {e}")
            if "Earth Engine client library not initialized" in str(e):
                message = "Authentication failed. Please authenticate again."
                print(message)
                self.pop_aviso_auth(message)
            else:
                message = f"An error occurred during authentication or initialization: {e}"
                print(message)
                self.pop_aviso_auth(message)
                self.auth_clear(True)


        except Exception as e:
            # Handle unexpected errors
            message = f"An unexpected error occurred: {e}"
            print(message)
            self.pop_aviso_auth(message)

    def auth_clear(self, silent=False):
        """
        Completely clears Earth Engine authentication by deleting the entire
        Earth Engine configuration directory, including credentials and cached data.
        """
        self.project_QgsPasswordLineEdit.clear()
        self.autenticacao.setEnabled(False)
        self.autentication = False


        system = platform.system()
        
        # Determine the Earth Engine configuration directory based on OS.
        if system == 'Windows':
            config_dir = os.path.join(os.environ['USERPROFILE'], '.config', 'earthengine')
        elif system in ['Linux', 'Darwin']:  # Linux or MacOS (Darwin)
            config_dir = os.path.join(os.environ['HOME'], '.config', 'earthengine')
        else:
            raise Exception(f"Unsupported operating system: {system}")
        
        # Check if the configuration directory exists and delete it.
        if os.path.exists(config_dir):
            try:
                shutil.rmtree(config_dir)
                if not silent:
                    message = "Earth Engine configuration cleared successfully (all files deleted)."
                    print(message)
                    self.pop_aviso_auth(message)
            except Exception as e:
                message = f"Error clearing Earth Engine configuration: {e}"
                print(message)
                self.pop_aviso_auth(message)
        else:
            message = "No Earth Engine configuration found to clear."
            print(message)
            self.pop_aviso_auth(message)

    def pop_aviso(self, aviso):
        QApplication.restoreOverrideCursor()
        msg = QMessageBox(parent=self)
        msg.setWindowTitle("Alerta!")
        msg.setIcon(QMessageBox.Warning)
        msg.setText(aviso)
        msg.setStandardButtons(QMessageBox.Ok | QMessageBox.Cancel)  # Add Ok and Cancel buttons

        ret = msg.exec_()  # Get the result of the dialog

        if ret == QMessageBox.Ok:
            
            # Handle Ok button click
            print("Ok button clicked")
            # Add your code here for what to do when Ok is clicked
            return True
        elif ret == QMessageBox.Cancel:
            
            # Handle Cancel button click
            print("Cancel button clicked")
            # Add your code here for what to do when Cancel is clicked
            return False
    
    def load_vector_function(self):
        """
        Loads the vector layer from the selected file path, reprojects it to EPSG:4326,
        dissolves multiple features if necessary, and converts it into an Earth Engine
        FeatureCollection representing the AOI.
        """
        shapefile_path = self.selected_aio_layer_path
        self.aoi = None  # Ensure the attribute exists to avoid AttributeError

        try:
            # Load the shapefile, handling both .zip archives and regular files.
            if shapefile_path.endswith('.zip'):
                with zipfile.ZipFile(shapefile_path, 'r') as zip_ref:
                    shapefile_found = False
                    for file in zip_ref.namelist():
                        if file.endswith('.shp'):
                            shapefile_found = True
                            shapefile_within_zip = file
                            break
                    if not shapefile_found:
                        print("No .shp file found inside the zip archive.")
                        return

                    # Read shapefile directly from the zip archive.
                    self.aoi = gpd.read_file(f"zip://{shapefile_path}/{shapefile_within_zip}")
            else:
                self.aoi = gpd.read_file(shapefile_path)

            # Reproject the GeoDataFrame to EPSG:4326 to ensure correct coordinates for Earth Engine.
            self.aoi = self.aoi.to_crs(epsg=4326)

            if self.aoi.empty:
                print("The shapefile does not contain any geometries.")
                return

            # Dissolve multiple features into a single geometry if necessary.
            if len(self.aoi) > 1:
                self.aoi = self.aoi.dissolve()

            # Extract the first geometry.
            geometry = self.aoi.geometry.iloc[0]

            # Validate the geometry type.
            if geometry.geom_type not in ['Polygon', 'MultiPolygon']:
                print("The geometry is not a valid type (Polygon or MultiPolygon).")
                return

            # Convert the geometry to GeoJSON format.
            geojson = geometry.__geo_interface__

            # Remove any third dimension from the coordinates.
            if geojson['type'] == 'Polygon':
                geojson['coordinates'] = [list(map(lambda coord: coord[:2], ring)) for ring in geojson['coordinates']]
            elif geojson['type'] == 'MultiPolygon':
                geojson['coordinates'] = [
                    [list(map(lambda coord: coord[:2], ring)) for ring in polygon]
                    for polygon in geojson['coordinates']
                ]

            # Create an Earth Engine geometry object.
            ee_geometry = ee.Geometry(geojson)
            feature = ee.Feature(ee_geometry)
            self.aoi = ee.FeatureCollection([feature])

            print("AOI defined successfully.")
            self.aio_set = True
            self.check_next_button()

        except Exception as e:
            print(f"Error in load_vector_function: {e}")
            return


    def elevacao_clicked(self):
        aoi = self.aoi  # Assuming 'self.aoi' holds the Earth Engine FeatureCollection

        DEM_source_key = self.dem_dataset_combobox.currentText()
        DEM_source_id = self.dem_datasets[DEM_source_key]["ID"]
        DEM_resolution = int(self.dem_resolution_combobox.currentText())
        print(f"Selected DEM source: {DEM_source_key} ({DEM_source_id})", DEM_resolution)

        # Replace invalid characters in DEM source ID for filenames
        safe_dem_source_id = DEM_source_id.replace("/", "_").replace("\\", "_")

        # Fetch DEM image based on selected source
        if DEM_source_id == 'COPERNICUS/DEM/GLO30':
            dem = ee.ImageCollection(DEM_source_id).select('DEM').mosaic().clip(aoi)
        elif DEM_source_id == 'JAXA/ALOS/AW3D30/V3_2':
            dem = ee.ImageCollection(DEM_source_id).select('DSM').mosaic().clip(aoi)
        elif DEM_source_id == 'NASA/NASADEM_HGT/001':
            dem = ee.Image(DEM_source_id).select('elevation').clip(aoi)
        elif DEM_source_id == 'USGS/GMTED2010_FULL':
            dem = ee.Image(DEM_source_id).select('min').clip(aoi)
        elif DEM_source_id == 'ASTER/ASTGTM':
            dem = ee.Image(DEM_source_id).select('elevation').clip(aoi)
        else:
            dem = ee.Image(DEM_source_id).clip(aoi).select('elevation')

        try:
            url = dem.getDownloadUrl({
                'scale': DEM_resolution,
                'region': aoi.geometry().bounds().getInfo(),
                'format': 'GeoTIFF'
            })

            

            # Include DEM source ID in file name, replacing invalid characters
            base_file_name = f'elevation_profile_{safe_dem_source_id}'
            output_file = self.get_unique_filename(base_file_name)

            response = requests.get(url)
            if response.status_code == 200:
                with open(output_file, 'wb') as file:
                    file.write(response.content)
                print(f"DEM image downloaded as {output_file}")
            else:
                print(f"Failed to download DEM image: {response.status_code}")
                return

        except Exception as e:
            print(f"Error during download: {e}")
            return

        # Load the vector layer for clipping
        vector_layer = QgsVectorLayer(self.selected_aio_layer_path, "Vector Layer", "ogr")

        if not vector_layer.isValid():
            print(f"Error: Vector layer '{self.selected_aio_layer_path}' is invalid.")
            return

        # Generate a unique name for the clipped output, including DEM source ID
        output_path = self.get_unique_filename(f'clipped_elevation_{safe_dem_source_id}')

        # Clip the raster using the vector layer
        try:
            processing.run("gdal:cliprasterbymasklayer", {
                'INPUT': output_file,
                'MASK': vector_layer,
                'NODATA': -9999,  # Ensure this is the right value for your dataset
                'CROP_TO_CUTLINE': True,
                'OUTPUT': output_path
            })
            print(f"Clipped raster saved to: {output_path}")

            # Load and add the clipped raster to the map canvas
            clipped_raster_layer = QgsRasterLayer(output_path, self.vector_layer_combobox.currentText() + f' - {safe_dem_source_id}')
            if clipped_raster_layer.isValid():
                QgsProject.instance().addMapLayer(clipped_raster_layer)
                print("Clipped raster added to canvas.")
            else:
                print("Failed to load clipped raster.")

        except Exception as e:
            print(f"Error during clipping: {str(e)}")
