
from typing import List, Union

from PyQt5.QtWidgets import (
    QAbstractItemView,
    QButtonGroup,
    QCheckBox,
    QComboBox,
    QDialog,
    QDoubleSpinBox,
    QFileDialog,
    QGridLayout,
    QGroupBox,
    QHBoxLayout,
    QLabel,
    QLineEdit,
    QPushButton,
    QRadioButton,
    QSpinBox,
    QTextEdit,
    QTreeWidget,
    QTreeWidgetItem,
    QVBoxLayout,
    QWidget,
    QSpacerItem,
)

from PyQt5.QtCore import (
    Qt,
    QFileInfo,
)

from PyQt5.QtGui import (
    QColor,
)

from matplotlib.figure import Figure
from matplotlib.backends.backend_qt5agg import FigureCanvasQTAgg as FigureCanvas
import matplotlib.pyplot as plt


from qgis.gui import QgsColorButton, QgsOpacityWidget, QgsRubberBand
from qgis.core import (
    QgsWkbTypes,
    QgsPointXY,
    QgsVectorLayer,
    QgsProject
)

from ...apps.qt.colors import *
from ...apps.qt.tools import *
from ...apps.qt.messages import *
from ...apps.qt.filesystem import *

from ...apps.qgis.messages import *
from ...apps.qgis.project import *
from ...apps.qgis.canvas import *
from ...apps.qgis.symbology import *

from ...utils.geoprofiler.export import *

from geogst.core.profiles.geoprofiles import *
from geogst.plots.parameters import *
from geogst.core.geometries.lines import *

from .structures import *


class FigureWindow(QDialog):

    def __init__(self,
                 figure: Figure
                 ):

        super().__init__()

        self.figure = figure

        layout = QVBoxLayout()

        self.export_button = QPushButton("Save figure")
        self.export_button.clicked.connect(self.save_figure)
        layout.addWidget(self.export_button)

        canvas = FigureCanvas(figure)
        layout.addWidget(canvas)

        self.setLayout(layout)

        self.setWindowFlags(
            Qt.Window | Qt.WindowMinimizeButtonHint | Qt.WindowMaximizeButtonHint | Qt.WindowCloseButtonHint)

        self.setSizeGripEnabled(True)

        self.resize(600, 500)  # Dimensioni iniziali
        self.setMinimumSize(200, 200)  # Dimensioni minime
        self.setMaximumSize(16777215, 16777215)  # Massima ampiezza disponibile

        self.show()

    def export_as_svg(self):
        """Combina tutte le figure e le esporta come un unico file SVG."""
        # Apri un QFileDialog per scegliere il file di output
        file_path, _ = QFileDialog.getSaveFileName(
            self,
            "Esporta come SVG",
            "",
            "SVG Files (*.svg);;All Files (*)"
        )

        if file_path:
            # Calcola le proporzioni relative di ogni figura
            total_height = sum(canvas.size().height() for canvas in self.fig_canvases)
            figure_ratios = [canvas.size().height() / total_height for canvas in self.fig_canvases]

            # Crea una nuova figura combinata con dimensioni proporzionali
            combined_figure = plt.figure(figsize=(8, 8))  # Larghezza fissa, altezza regolata
            current_height = 0

            for i, (canvas, ratio) in enumerate(zip(self.figure, figure_ratios)):
                ax_combined = combined_figure.add_axes(
                    [0.1, current_height, 0.8, ratio]  # [left, bottom, width, height]
                )
                original_figure = canvas.figure
                for ax in original_figure.axes:
                    for line in ax.get_lines():
                        ax_combined.plot(
                            line.get_xdata(),
                            line.get_ydata(),
                            label=line.get_label()
                        )
                ax_combined.set_title(f"Figura combinata {i + 1}")
                ax_combined.legend()
                current_height += ratio  # Aggiorna la posizione verticale

            # Salva la figura combinata come SVG
            combined_figure.savefig(file_path, format="svg")
            print(f"Figura combinata salvata come SVG in {file_path}")

    def save_figure(self):
        """Salva il canvas della figura in un file."""
        # Apri un QFileDialog per scegliere il file
        file_path, _ = QFileDialog.getSaveFileName(
            self,
            "Salva Figura",
            "",
            "PDF Files (*.pdf);;SVG Files (*.svg);;TIFF Files (*.tif *.tiff);;JPEG Files (*.jpg *.jpeg);;PNG Files (*.png);;All Files (*)"
        )

        if file_path:  # Se un file è stato selezionato
            # Salva la figura nel formato selezionato
            try:
                self.figure.savefig(file_path)
                print(f"Figura salvata in {file_path}")
            except Exception as e:
                print(f"Errore durante il salvataggio: {e!r}")


class GraphsWindow(QDialog):

    def __init__(self,
        figs: List[Figure]):

        super().__init__()

        self.figs = figs

        # Layout per contenere il canvas
        layout = QVBoxLayout()

        # Aggiungi un pulsante per esportare in SVG
        self.export_button = QPushButton("Export as SVG")
        self.export_button.clicked.connect(self.export_as_svg)
        layout.addWidget(self.export_button)

        # Aggiunta del canvas al layout

        self.fig_canvases = []
        for fig in figs:
            canvas = FigureCanvas(fig)
            layout.addWidget(canvas)
            self.fig_canvases.append(canvas)

        self.setLayout(layout)

        # Permetti il resize
        #self.setWindowFlags(self.windowFlags() | Qt.WindowMaximizeButtonHint)
        self.setWindowFlags(Qt.Window | Qt.WindowMinimizeButtonHint | Qt.WindowMaximizeButtonHint | Qt.WindowCloseButtonHint)

        self.setSizeGripEnabled(True)

        # Imposta dimensioni minime o iniziali
        self.resize(600, 500)  # Dimensioni iniziali
        self.setMinimumSize(200, 200)  # Dimensioni minime
        self.setMaximumSize(16777215, 16777215)  # Massima ampiezza disponibile

        self.show()

    def export_as_svg(self):
        """Combina tutte le figure e le esporta come un unico file SVG."""
        # Apri un QFileDialog per scegliere il file di output
        file_path, _ = QFileDialog.getSaveFileName(
            self,
            "Esporta come SVG",
            "",
            "SVG Files (*.svg);;All Files (*)"
        )

        if file_path:
            # Calcola le proporzioni relative di ogni figura
            total_height = sum(canvas.size().height() for canvas in self.fig_canvases)
            figure_ratios = [canvas.size().height() / total_height for canvas in self.fig_canvases]

            # Crea una nuova figura combinata con dimensioni proporzionali
            combined_figure = plt.figure(figsize=(8, 8))  # Larghezza fissa, altezza regolata
            current_height = 0

            for i, (canvas, ratio) in enumerate(zip(self.figs, figure_ratios)):
                ax_combined = combined_figure.add_axes(
                    [0.1, current_height, 0.8, ratio]  # [left, bottom, width, height]
                )
                original_figure = canvas.figure
                for ax in original_figure.axes:
                    for line in ax.get_lines():
                        ax_combined.plot(
                            line.get_xdata(),
                            line.get_ydata(),
                            label=line.get_label()
                        )
                ax_combined.set_title(f"Figura combinata {i + 1}")
                ax_combined.legend()
                current_height += ratio  # Aggiorna la posizione verticale

            # Salva la figura combinata come SVG
            combined_figure.savefig(file_path, format="svg")
            print(f"Figura combinata salvata come SVG in {file_path}")

    def save_figure(self):
        """Salva il canvas della figura in un file."""
        # Apri un QFileDialog per scegliere il file
        file_path, _ = QFileDialog.getSaveFileName(
            self,
            "Salva Figura",
            "",
            "PDF Files (*.pdf);;SVG Files (*.svg);;TIFF Files (*.tif *.tiff);;JPEG Files (*.jpg *.jpeg);;PNG Files (*.png);;All Files (*)"
        )

        if file_path:  # Se un file è stato selezionato
            # Salva la figura nel formato selezionato
            try:
                self.figure.savefig(file_path)
                print(f"Figura salvata in {file_path}")
            except Exception as e:
                print(f"Errore durante il salvataggio: {e!r}")


class SourceDEMSelectionDialog(QDialog):

    def __init__(self,
                 plugin_name,
                 raster_layers,
                 parent=None
                 ):

        super(SourceDEMSelectionDialog, self).__init__(parent)

        self.plugin_name = plugin_name

        self.singleband_raster_layers_in_project = raster_layers

        self.listDEMs_treeWidget = QTreeWidget()
        self.listDEMs_treeWidget.setColumnCount(1)
        self.listDEMs_treeWidget.headerItem().setText(0, "Name")
        self.listDEMs_treeWidget.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        self.listDEMs_treeWidget.setDragEnabled(False)
        self.listDEMs_treeWidget.setDragDropMode(QAbstractItemView.NoDragDrop)
        self.listDEMs_treeWidget.setAlternatingRowColors(True)
        self.listDEMs_treeWidget.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.listDEMs_treeWidget.setSelectionMode(QAbstractItemView.SingleSelection)
        self.listDEMs_treeWidget.setTextElideMode(Qt.ElideLeft)

        self.populate_raster_layer_treewidget()

        self.listDEMs_treeWidget.resizeColumnToContents(0)

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()
        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        layout = QGridLayout()

        layout.addWidget(
            self.listDEMs_treeWidget,
            0, 0, 1, 3)
        layout.addLayout(
            buttonLayout,
            1, 0, 1, 3)

        self.setLayout(layout)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        self.setWindowTitle("Define elevation source")

    def populate_raster_layer_treewidget(self):

        self.listDEMs_treeWidget.clear()

        for raster_layer in self.singleband_raster_layers_in_project:
            tree_item = QTreeWidgetItem(self.listDEMs_treeWidget)
            tree_item.setText(0, raster_layer.name())


class DefineSourceLineLayerDialog(QDialog):

    def __init__(self,
                 plugin_name,
                 current_line_layers,
                 parent=None
                 ):

        super(DefineSourceLineLayerDialog, self).__init__(parent)

        self.plugin_name = plugin_name
        self.current_line_layers = current_line_layers

        layout = QGridLayout()

        layout.addWidget(
            QLabel(self.tr("Input line layer:")),
            0, 0, 1, 1)

        self.LineLayers_comboBox = QComboBox()
        layout.addWidget(
            self.LineLayers_comboBox,
            0, 1, 1, 3)
        self.refresh_input_profile_layer_combobox()

        # Ok/Cancel choices

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()
        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        layout.addLayout(
            buttonLayout,
            1, 0, 1, 3)

        self.setLayout(layout)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        self.setWindowTitle("Define source line layer")

    def refresh_input_profile_layer_combobox(self):

        self.LineLayers_comboBox.clear()

        for layer in self.current_line_layers:
            self.LineLayers_comboBox.addItem(layer.name())

        shape_qgis_ndx = self.LineLayers_comboBox.currentIndex()
        self.line_shape = self.current_line_layers[shape_qgis_ndx]

    def refresh_order_field_combobox(self):

        self.Trace2D_order_field_comboBox.clear()
        self.Trace2D_order_field_comboBox.addItem('--optional--')

        shape_qgis_ndx = self.LineLayers_comboBox.currentIndex()
        self.line_shape = self.current_line_layers[shape_qgis_ndx]

        line_layer_field_list = self.line_shape.dataProvider().fields().toList()
        for field in line_layer_field_list:
            self.Trace2D_order_field_comboBox.addItem(field.name())

    def refresh_name_field_combobox(self):

        self.Trace2D_name_field_comboBox.clear()
        self.Trace2D_name_field_comboBox.addItem('--optional--')

        shape_qgis_ndx = self.LineLayers_comboBox.currentIndex()
        self.line_shape = self.current_line_layers[shape_qgis_ndx]

        line_layer_field_list = self.line_shape.dataProvider().fields().toList()
        for field in line_layer_field_list:
            self.Trace2D_name_field_comboBox.addItem(field.name())

    def refresh_label_field_combobox(self):

        self.Trace2D_label_field_comboBox.clear()
        self.Trace2D_label_field_comboBox.addItem('--optional--')

        shape_qgis_ndx = self.LineLayers_comboBox.currentIndex()
        self.line_shape = self.current_line_layers[shape_qgis_ndx]

        line_layer_field_list = self.line_shape.dataProvider().fields().toList()
        for field in line_layer_field_list:
            self.Trace2D_label_field_comboBox.addItem(field.name())


class GeologicalDataIntersectLinesDialog(QDialog):
    """
    Intersects the profiles with a line layer: defines input parameters.
    """

    def __init__(self,
                 plugin_name,
                 current_line_layers,
                 parent=None
                 ):

        super(GeologicalDataIntersectLinesDialog, self).__init__(parent)

        self.plugin_name = plugin_name
        self.current_line_layers = current_line_layers

        # GUI

        layout = QGridLayout()

        layout.addWidget(
            QLabel(self.tr("Input line layer:")),
            0, 0, 1, 1)

        self.LineLayers_comboBox = QComboBox()
        layout.addWidget(
            self.LineLayers_comboBox,
            0, 1, 1, 3)
        self.refresh_input_profile_layer_combobox()

        layout.addWidget(QLabel("Id field"), 1, 0, 1, 1)
        self.inters_input_id_fld_line_comboBox = QComboBox()
        layout.addWidget(
            self.inters_input_id_fld_line_comboBox,
            1, 1, 1, 3
        )

        self.refresh_id_field_combobox()
        self.LineLayers_comboBox.currentIndexChanged.connect(self.refresh_id_field_combobox)

        # Ok/Cancel choices

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()
        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        layout.addLayout(
            buttonLayout,
            3, 0, 1, 3)

        self.setLayout(layout)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        self.setWindowTitle("Intersection line layer")

    def refresh_input_profile_layer_combobox(self):

        self.LineLayers_comboBox.clear()

        for layer in self.current_line_layers:
            self.LineLayers_comboBox.addItem(layer.name())

        shape_qgis_ndx = self.LineLayers_comboBox.currentIndex()
        self.line_shape = self.current_line_layers[shape_qgis_ndx]

    def refresh_id_field_combobox(self):

        self.inters_input_id_fld_line_comboBox.clear()
        #self.inters_input_id_fld_line_comboBox.addItem('--optional--')

        shape_qgis_ndx = self.LineLayers_comboBox.currentIndex()
        self.line_shape = self.current_line_layers[shape_qgis_ndx]

        line_layer_field_list = self.line_shape.dataProvider().fields().toList()
        for field in line_layer_field_list:
            self.inters_input_id_fld_line_comboBox.addItem(field.name())


class GeologicalDataIntersectPolygonsDialog(QDialog):
    """
    Intersects the profiles with a polygon layer:
    defines input parameters.
    """

    def __init__(self,
                 plugin_name,
                 current_polygon_layers,
                 parent=None
                 ):

        super(GeologicalDataIntersectPolygonsDialog, self).__init__(parent)

        self.plugin_name = plugin_name
        self.current_polygon_layers = current_polygon_layers

        # GUI

        layout = QGridLayout()

        layout.addWidget(
            QLabel(self.tr("Input polygon layer:")),
            0, 0, 1, 1)

        self.PolygonLayers_comboBox = QComboBox()
        layout.addWidget(
            self.PolygonLayers_comboBox,
            0, 1, 1, 3)
        self.refresh_input_profile_layer_combobox()

        self.refresh_id_field_combobox()
        self.PolygonLayers_comboBox.currentIndexChanged.connect(self.refresh_id_field_combobox)

        # Ok/Cancel choices

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()
        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        layout.addLayout(
            buttonLayout,
            3, 0, 1, 3)

        self.setLayout(layout)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        self.setWindowTitle("Intersection polygon layer")

    def refresh_input_profile_layer_combobox(self):

        self.PolygonLayers_comboBox.clear()

        for layer in self.current_polygon_layers:
            self.PolygonLayers_comboBox.addItem(layer.name())

        shape_qgis_ndx = self.PolygonLayers_comboBox.currentIndex()
        self.pol_shape = self.current_polygon_layers[shape_qgis_ndx]

    def refresh_id_field_combobox(self):

        shape_qgis_ndx = self.PolygonLayers_comboBox.currentIndex()
        self.pol_shape = self.current_polygon_layers[shape_qgis_ndx]


class BaseDataLoadPointListDialog(QDialog):

    def __init__(self, plugin_name, parent=None):

        super(BaseDataLoadPointListDialog, self).__init__(parent)

        self.plugin_name = plugin_name
        layout = QGridLayout()

        layout.addWidget(
            QLabel(self.tr("Point list, with at least two points.")),
            0, 0, 1, 1)
        layout.addWidget(
            QLabel(self.tr("Each point is defined by a comma-separated, x-y coordinate pair (same CRS as project), one for each row")), 1, 0,
            1, 1)
        layout.addWidget(
            QLabel(self.tr("Example:\n549242.7, 242942.2\n578370.3, 322634.5")),
            2, 0, 1, 1)

        self.point_list_qtextedit = QTextEdit()
        layout.addWidget(
            self.point_list_qtextedit,
            3, 0, 1, 1)

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()
        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        layout.addLayout(
            buttonLayout,
            4, 0, 1, 3)

        self.setLayout(layout)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        self.setWindowTitle("Point list")


class StylesElevationsParamsDialog(QDialog):

    def __init__(self,
                 plugin_name,
                 current_graphical_params,
                 parent=None
                 ):

        super(StylesElevationsParamsDialog, self).__init__(parent)

        self.plugin_name = plugin_name
        self.current_graphical_params = current_graphical_params

        self.qtwdElevationLayers = QTreeWidget()
        self.qtwdElevationLayers.setColumnCount(2)
        self.qtwdElevationLayers.headerItem().setText(0, "Type")
        self.qtwdElevationLayers.headerItem().setText(1, "Color")
        self.qtwdElevationLayers.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        self.qtwdElevationLayers.setDragEnabled(False)
        self.qtwdElevationLayers.setDragDropMode(QAbstractItemView.NoDragDrop)
        self.qtwdElevationLayers.setAlternatingRowColors(True)
        self.qtwdElevationLayers.setSelectionMode(QAbstractItemView.SingleSelection)
        self.qtwdElevationLayers.setTextElideMode(Qt.ElideLeft)

        self.populate_elevation_layer_treewidget()

        self.qtwdElevationLayers.resizeColumnToContents(0)
        self.qtwdElevationLayers.resizeColumnToContents(1)

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()
        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        layout = QGridLayout()

        layout.addWidget(
            self.qtwdElevationLayers,
            0, 0, 1, 3)
        layout.addLayout(
            buttonLayout,
            1, 0, 1, 3)

        self.setLayout(layout)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        self.setWindowTitle("Elevation style")

    def populate_elevation_layer_treewidget(self):

        self.qtwdElevationLayers.clear()

        for ndx, layer_name in enumerate(["Elevation"]):
            tree_item = QTreeWidgetItem(self.qtwdElevationLayers)
            tree_item.setText(0, layer_name)
            color_button = QgsColorButton()
            qcolor = create_qcolor(self.current_graphical_params.elevations.color)
            color_button.setColor(qcolor)
            self.qtwdElevationLayers.setItemWidget(tree_item, 1, color_button)


class GeologicalDataProjectPointsDialog(QDialog):

    def __init__(self,
                 plugin_name,
                 current_point_layers: List,
                 parent=None
                 ):

        super(GeologicalDataProjectPointsDialog, self).__init__(parent)

        self.plugin_name = plugin_name
        self.current_point_layers = current_point_layers

        # gui definition

        main_layout = QGridLayout()

        # input point geological layer

        main_layout.addWidget(QLabel("Layer "),
                                       0, 0, 1, 1)

        self.point_layer_combobox = QComboBox()

        main_layout.addWidget(self.point_layer_combobox, 0, 1, 1, 1)

        # Creazione del gruppo di RadioButton per rendere le opzioni mutuamente esclusive
        self.radio_group = QButtonGroup()

        # Opzione 1: RadioButton per attributi Z

        self.radio_attributi_z = QRadioButton("Point attributes have z field")
        self.radio_group.addButton(self.radio_attributi_z)
        main_layout.addWidget(self.radio_attributi_z, 1, 0, 1, 1)

        self.radio_attributi_z.setChecked(True)

        self.z_fld_comboBox = QComboBox()
        main_layout.addWidget(self.z_fld_comboBox, 1, 1, 1, 1)

        # Opzione 2: RadioButton per punti 3D
        self.radio_punti_3d = QRadioButton("Points are already 3D")
        self.radio_group.addButton(self.radio_punti_3d)
        main_layout.addWidget(self.radio_punti_3d, 2, 0, 1, 3)

        main_layout.addWidget(QLabel("Max point distance from profile trace"), 3, 0, 1, 1)
        self.max_profile_distance_qdoublespinbox = QDoubleSpinBox()
        self.max_profile_distance_qdoublespinbox.setMaximum(10000)
        self.max_profile_distance_qdoublespinbox.setValue(200)
        main_layout.addWidget(
            self.max_profile_distance_qdoublespinbox,
            3, 1, 1, 1)

        main_layout.addWidget(QLabel("Label points using "), 4, 0, 1, 1)

        self.point_label_fld_comboBox = QComboBox()
        main_layout.addWidget(self.point_label_fld_comboBox, 4, 1, 1, 1)

        self.refresh_point_layer_combobox()
        self.refresh_layer_related_comboboxes()
        self.point_layer_combobox.currentIndexChanged.connect(self.refresh_layer_related_comboboxes)

        # ok/cancel setup

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()
        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        main_layout.addLayout(buttonLayout, 5, 0, 1, 2)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        # widget final setup

        self.setLayout(main_layout)
        self.setWindowTitle("Project points")

    def refresh_point_layer_combobox(
            self,
    ):

        refresh_combobox(
            self.point_layer_combobox,
            None,
            [layer.name() for layer in self.current_point_layers]
        )

    def refresh_layer_related_comboboxes(self):

        if len(self.current_point_layers) == 0:
            warn_qgis(self.plugin_name, "No point layer available")
            return

        layer_ndx = self.point_layer_combobox.currentIndex()

        layer = self.current_point_layers[layer_ndx]
        fields = layer.dataProvider().fields()
        field_names = [field.name() for field in fields.toList()]

        self.point_label_fld_comboBox.clear()
        self.z_fld_comboBox.clear()
        self.point_label_fld_comboBox.addItems(field_names)
        self.z_fld_comboBox.addItems(field_names)


class GeologicalDataProjectAttitudesDialog(QDialog):
    """
    Projects geological attitudes onto profiles.
    """

    def __init__(self,
                 plugin_name,
                 current_point_layers: List,
                 parent=None
                 ):

        super(GeologicalDataProjectAttitudesDialog, self).__init__(parent)

        self.plugin_name = plugin_name
        self.current_point_layers = current_point_layers

        # gui definition

        main_layout = QVBoxLayout()

        ## input section

        input_groupbox = QGroupBox(self)
        input_groupbox.setTitle('Input parameters')

        input_layout = QGridLayout()

        # input point geological layer

        input_layout.addWidget(
            QLabel("Point layer storing attitudes"),
            0, 0, 1, 1)

        self.point_layer_combobox = QComboBox()
        input_layout.addWidget(self.point_layer_combobox, 0, 1, 1, 3)

        input_layout.addWidget(QLabel("Attitude azimuth field "), 1, 0, 1, 1)

        self.orient_fld_combobox = QComboBox()
        input_layout.addWidget(self.orient_fld_combobox, 1, 1, 1, 1)

        self.use_dipdir_qradiobuttom = QRadioButton("Dip dir.")
        self.use_dipdir_qradiobuttom.setChecked(True)
        input_layout.addWidget(self.use_dipdir_qradiobuttom, 1, 2, 1, 1)

        self.use_rhr_strike_qradiobutton = QRadioButton("RHR str.")
        input_layout.addWidget(self.use_rhr_strike_qradiobutton, 1, 3, 1, 1)

        input_layout.addWidget(QLabel("Attitude dip angle field"), 2, 0, 1, 1)
        self.dipangle_fld_combobox = QComboBox()
        input_layout.addWidget(self.dipangle_fld_combobox, 2, 1, 1, 3)


        input_layout.addWidget(QLabel("Record id/label field"), 3, 0, 1, 1)

        self.point_label_fld_comboBox = QComboBox()
        input_layout.addWidget(self.point_label_fld_comboBox, 3, 1, 1, 3)


        # Creazione del gruppo di RadioButton per rendere le opzioni mutuamente esclusive
        self.radio_group = QButtonGroup()

        # Opzione 1: RadioButton per selezione DEM
        self.radio_dem = QRadioButton("Derive point heights from DEM")
        self.radio_group.addButton(self.radio_dem)
        input_layout.addWidget(self.radio_dem, 4, 0, 1, 1)

        self.dem_combobox = QComboBox()

        input_layout.addWidget(self.dem_combobox, 4, 1, 1, 3)
        self.current_raster_layers = get_dict_of_monoband_raster_layers()
        self.dem_combobox.addItems(self.current_raster_layers.keys())

        # Opzione 2: RadioButton per attributi Z
        self.radio_attributi_z = QRadioButton("Point attributes have z field")
        self.radio_group.addButton(self.radio_attributi_z)
        input_layout.addWidget(self.radio_attributi_z, 5, 0, 1, 1)

        self.combobox_attributi_z = QComboBox()
        input_layout.addWidget(self.combobox_attributi_z, 5, 1, 1, 3)

        # Opzione 3: RadioButton per punti 3D
        self.radio_punti_3d = QRadioButton("Points are already 3D")
        self.radio_group.addButton(self.radio_punti_3d)
        input_layout.addWidget(self.radio_punti_3d, 6, 0, 1, 3)

        # Impostare un'opzione predefinita (opzionale)
        self.radio_dem.setChecked(True)


        input_layout.addWidget(QLabel("Max point distance from profile trace"), 7, 0, 1, 1)

        self.max_profile_distance_qdoublespinbox = QDoubleSpinBox()
        self.max_profile_distance_qdoublespinbox.setMaximum(10000)
        self.max_profile_distance_qdoublespinbox.setValue(200)
        input_layout.addWidget(
            self.max_profile_distance_qdoublespinbox,
            7, 1, 1, 3)

        input_groupbox.setLayout(input_layout)
        main_layout.addWidget(input_groupbox)

        # Configurazione spaziatura del layout
        input_layout.setContentsMargins(10, 10, 10, 10)
        input_layout.setSpacing(5)

        self.refresh_point_layer_combobox()
        self.refresh_layer_related_comboboxes()
        self.point_layer_combobox.currentIndexChanged.connect(self.refresh_layer_related_comboboxes)

        # ok/cancel setup

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()
        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        main_layout.addLayout(buttonLayout, 3)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        # widget final setup

        self.setLayout(main_layout)
        self.setWindowTitle("Project geological attitudes")

    def refresh_point_layer_combobox(
            self,
    ):

        refresh_combobox(
            self.point_layer_combobox,
            None,
            [layer.name() for layer in self.current_point_layers]
        )

    def refresh_layer_related_comboboxes(self):

        if len(self.current_point_layers) == 0:
            warn_qgis(self.plugin_name, "No point layer available")
            return

        layer_ndx = self.point_layer_combobox.currentIndex()

        layer = self.current_point_layers[layer_ndx]
        fields = layer.dataProvider().fields()
        field_names = [field.name() for field in fields.toList()]

        self.point_label_fld_comboBox.clear()
        self.orient_fld_combobox.clear()
        self.dipangle_fld_combobox.clear()
        self.combobox_attributi_z.clear()

        self.point_label_fld_comboBox.addItems(field_names)
        self.orient_fld_combobox.addItems(field_names)
        self.dipangle_fld_combobox.addItems(field_names)
        self.combobox_attributi_z.addItems(field_names)


class StylesTotalGraphicalParamsDialog(QDialog):

    def __init__(self,
        plugin_name: str,
        geoprofiles: GeoProfiles,
        current_graphical_params: GraphicalParameters,
        polygons_intersection_parameters: PolygonsIntersectionParameters,
        parent=None
    ):

        super(StylesTotalGraphicalParamsDialog, self).__init__(parent)

        self.plugin_name = plugin_name
        self.geoprofiles = geoprofiles
        self.current_graphical_params = current_graphical_params
        self.polygons_intersection_parameters = polygons_intersection_parameters

        # Prepare the dialog

        main_layout = QVBoxLayout()

        # Figure parameters

        groupbox_0 = QGroupBox("Figure parameters")

        layout_0 = QGridLayout()

        layout_0.addWidget(
            QLabel(self.tr("Figure width")),
            0, 0, 1, 1)

        self.figure_width_qlineedit = QLineEdit()
        self.figure_width_qlineedit.setText(f"{self.current_graphical_params.figure.width}")
        layout_0.addWidget(
            self.figure_width_qlineedit,
            0, 1, 1, 1)

        layout_0.addWidget(
            QLabel(self.tr(" inches")),
            0, 2, 1, 1)

        layout_0.addWidget(
            QLabel(self.tr("Figure height")),
            1, 0, 1, 1)

        self.figure_height_qlineedit = QLineEdit()
        self.figure_height_qlineedit.setText(f"{self.current_graphical_params.figure.height}")
        layout_0.addWidget(
            self.figure_height_qlineedit,
            1, 1, 1, 1)

        layout_0.addWidget(
            QLabel(self.tr(" inches")),
            1, 2, 1, 1)

        # groupbox final settings

        groupbox_0.setLayout(layout_0)

        main_layout.addWidget(groupbox_0)

        # Plot elevation parameters

        groupbox_1 = QGroupBox("Plot elevation parameters")

        layout_1 = QGridLayout()

        layout_1.addWidget(
            QLabel(self.tr("Vertical exaggeration")),
            0, 0, 1, 1)

        self.vertical_exxageration_ratio_qlineedit = QLineEdit()
        self.vertical_exxageration_ratio_qlineedit.setText(f"{self.current_graphical_params.axis.vertical_exaggeration}")
        layout_1.addWidget(
            self.vertical_exxageration_ratio_qlineedit,
            0, 1, 1, 1)

        layout_1.addWidget(
            QLabel(self.tr("z max value")),
            1, 0, 1, 1)

        self.z_max_value_qlineedit = QLineEdit()
        self.z_max_value_qlineedit.setText(f"{self.current_graphical_params.axis.z_max}")
        layout_1.addWidget(
            self.z_max_value_qlineedit,
            1, 1, 1, 1)

        layout_1.addWidget(
            QLabel(self.tr("z min value")),
            2, 0, 1, 1)

        self.z_min_value_qlineedit = QLineEdit()
        self.z_min_value_qlineedit.setText(f"{self.current_graphical_params.axis.z_min}")
        layout_1.addWidget(
            self.z_min_value_qlineedit,
            2, 1, 1, 1)

        # groupbox final settings

        groupbox_1.setLayout(layout_1)

        main_layout.addWidget(groupbox_1)

        # Style parameters

        groupbox_2 = QGroupBox("Plot styles")

        layout_2 = QGridLayout()

        # topographic elevations

        self.topographic_elevations_qpushbutton = QPushButton(self.tr("Topographic elevations"))

        self.topographic_elevations_qpushbutton.clicked.connect(self.define_elevations_style)

        layout_2.addWidget(
            self.topographic_elevations_qpushbutton,
            0, 0, 1, 3)

        # point projections

        self.point_projections_qpushbutton = QPushButton(self.tr("Point projections"))

        self.point_projections_qpushbutton.clicked.connect(self.define_point_projections_style)

        layout_2.addWidget(
            self.point_projections_qpushbutton,
            1, 0, 1, 3)

        # attitudes projections

        self.attitude_projections_qpushbutton = QPushButton(self.tr("Attitude projections"))

        self.attitude_projections_qpushbutton.clicked.connect(self.define_attitude_projections_style)

        layout_2.addWidget(
            self.attitude_projections_qpushbutton,
            2, 0, 1, 3)

        # line intersections

        self.line_intersections_qpushbutton = QPushButton(self.tr("Line intersections"))

        self.line_intersections_qpushbutton.clicked.connect(self.define_line_intersections_style)

        layout_2.addWidget(
            self.line_intersections_qpushbutton,
            3, 0, 1, 3)

        # polygon intersections

        self.polygon_intersections_qpushbutton = QPushButton(self.tr("Polygon intersections"))

        self.polygon_intersections_qpushbutton.clicked.connect(self.define_polygon_intersections_style)

        layout_2.addWidget(
            self.polygon_intersections_qpushbutton,
            4, 0, 1, 3)

        # groupbox final settings

        groupbox_2.setLayout(layout_2)

        main_layout.addWidget(groupbox_2)

        # ok/cancel section

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()
        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        main_layout.addLayout(buttonLayout)

        self.setLayout(main_layout)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        self.setWindowTitle("Graphical parameters")

    def extract_elevation_plot_style(self,
                                     dialog):

        curr_item = dialog.qtwdElevationLayers.topLevelItem(0)
        return dialog.qtwdElevationLayers.itemWidget(curr_item, 1).color()

    def define_elevations_style(self):

        dialog = StylesElevationsParamsDialog(
            self.plugin_name,
            self.current_graphical_params
        )

        if dialog.exec_():
            self.elevation_color = self.extract_elevation_plot_style(dialog)
        else:
            return

    def extract_point_plot_style(self,
                                 dialog
                                 ) -> Union[None, PointPlotParams]:

        try:

            marker = ltMarkerStyles[dialog.marker_type_combobox.currentText()]
            marker_size = int(dialog.marker_size_qspinbox.value())
            color = qcolor2rgbmpl(dialog.marker_color_qgscolorbutton.color())
            alpha = dialog.marker_alpha_qgsopacitywidget.opacity()
            labels = dialog.labels.isChecked()

            return PointPlotParams(
                marker=marker,
                markersize=marker_size,
                color=color,
                alpha=alpha,
                labels=labels,
            )

        except:

            return None

    def define_point_projections_style(self):

        dialog = StylesPointDialog(
            self.plugin_name,
            "Point projection style",
            self.current_graphical_params.point_projections
        )

        if dialog.exec_():
            self.point_projections_style = self.extract_point_plot_style(dialog)
        else:
            return

    def define_line_intersections_style(self):

        if not self.geoprofiles:
            warn_qt(self,
                    self.plugin_name,
                    f"Before defining line style you must define line intersections sources.")
            return

        dialog = StylesPointDialog(
            self.plugin_name,
            "Line intersections style",
            self.current_graphical_params.line_intersections,
        )

        if dialog.exec_():
            self.line_intersections_style = self.extract_point_plot_style(dialog)
        else:
            return

    def extract_attitude_plot_style(self,
                                    dialog):

        try:

            marker = ltMarkerStyles[dialog.marker_type_combobox.currentText()]
            marker_size = int(dialog.marker_size_qspinbox.value())
            color = qcolor2rgbmpl(dialog.marker_color_qgscolorbutton.color())
            alpha = dialog.marker_alpha_qgsopacitywidget.opacity()
            label_orientations = dialog.label_orientations.isChecked()
            label_ids = dialog.label_ids.isChecked()

            return AttitudePlotParams(
                marker=marker,
                markersize=marker_size,
                color=color,
                alpha=alpha,
                label_orientations=label_orientations,
                label_ids=label_ids
            )

        except:

            pass

    def define_attitude_projections_style(self):

        dialog = StylesAttitudeProjectionsParamsDialog(
            self.plugin_name,
            self.current_graphical_params.attitude_projections,
        )

        if dialog.exec_():
            self.attitude_projections_style = self.extract_attitude_plot_style(dialog)
        else:
            return

    def extract_classification_colors(self, dialog):

        polygon_classification_colors_dict = dict()

        for classification_ndx in range(dialog.polygon_classifications_treeWidget.topLevelItemCount()):
            class_itemwidget = dialog.polygon_classifications_treeWidget.topLevelItem(classification_ndx)
            classification = str(class_itemwidget.text(0))
            # get color
            color = qcolor2rgbmpl(dialog.polygon_classifications_treeWidget.itemWidget(class_itemwidget, 1).color())
            polygon_classification_colors_dict[classification] = color

        return polygon_classification_colors_dict

    def initialize_polygons_intersections_style(self):

        symbology_field_name = get_symbology_field(self.polygons_intersection_parameters.polygon_layer)

        #fo.write(f"\nSymbology field name: {symbology_field_name}")

        symbology_colors = extract_qgis_colors(self.polygons_intersection_parameters.polygon_layer)

        #fo.write(f"\nSymbology colors: {symbology_colors}")

        self.current_graphical_params.polygon_intersections = symbology_colors

    def define_polygon_intersections_style(self):

        if self.current_graphical_params.polygon_intersections is None:
            self.initialize_polygons_intersections_style()

        # create windows for user_definition of intersection colors in profile

        dialog = StylesPolygonIntersectionsParamsDialog(
            self.plugin_name,
            self.current_graphical_params,
            #polygon_classification_set,
            #polygons_symbology_colors,
        )

        if dialog.exec_():
            polygon_classification_colors_dict = self.extract_classification_colors(dialog)
        else:
            return

        if len(polygon_classification_colors_dict) == 0:
            warn_qt(self,
                    self.plugin_name,
                    "No defined colors")
            return
        else:
            self.current_graphical_params.polygon_intersections = polygon_classification_colors_dict


class StylesPointDialog(QDialog):
    """
    Point plot style.
    """

    def __init__(self,
                 plugin_name,
                 window_title,
                 current_point_style,
                 parent=None
                 ):

        super(StylesPointDialog, self).__init__(parent)

        self.plugin_name = plugin_name
        self.current_point_style = current_point_style

        # gui definition

        main_layout = QGridLayout()

        # input point geological layer

        main_layout.addWidget(
            QLabel("Marker type"),
            0, 0, 1, 1)

        self.marker_type_combobox = QComboBox()

        self.marker_type_combobox.insertItems(0, list(ltMarkerStyles.keys()))
        self.marker_type_combobox.setCurrentIndex(list(ltMarkerStyles.values()).index(self.current_point_style.marker))
        main_layout.addWidget(self.marker_type_combobox, 0, 1, 1, 1)

        main_layout.addWidget(
            QLabel("Marker size"),
            1, 0, 1, 1)

        self.marker_size_qspinbox = QSpinBox()
        self.marker_size_qspinbox.setValue(self.current_point_style.markersize)
        main_layout.addWidget(self.marker_size_qspinbox, 1, 1, 1, 1)

        main_layout.addWidget(
            QLabel("Marker color"),
            2, 0, 1, 1)

        self.marker_color_qgscolorbutton = QgsColorButton()
        self.marker_color_qgscolorbutton.setColor(create_qcolor(self.current_point_style.color))
        main_layout.addWidget(self.marker_color_qgscolorbutton, 2, 1, 1, 1)

        main_layout.addWidget(
            QLabel("Marker alpha"),
            3, 0, 1, 1)

        self.marker_alpha_qgsopacitywidget = QgsOpacityWidget()
        self.marker_alpha_qgsopacitywidget.setOpacity(self.current_point_style.alpha)
        main_layout.addWidget(self.marker_alpha_qgsopacitywidget, 3, 1, 1, 1)

        self.labels = QCheckBox("Add labels")
        self.labels.setChecked(self.current_point_style.labels)

        main_layout.addWidget(self.labels, 4, 0, 1, 1)

        # ok/cancel setup

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()
        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        main_layout.addLayout(buttonLayout, 5, 0, 1, 2)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        # widget final setup

        self.setLayout(main_layout)
        self.setWindowTitle(window_title)


class StylesAttitudeProjectionsParamsDialog(QDialog):
    """
    Definition of attitude projection style.
    """

    def __init__(self,
                 plugin_name,
                 current_attitude_projections_style_params,
                 parent=None
                 ):

        super(StylesAttitudeProjectionsParamsDialog, self).__init__(parent)

        self.plugin_name = plugin_name
        self.current_attitude_projections_style_params = current_attitude_projections_style_params

        # gui definition

        main_layout = QGridLayout()

        # input point geological layer

        main_layout.addWidget(
            QLabel("Marker type"),
            0, 0, 1, 1)

        self.marker_type_combobox = QComboBox()

        self.marker_type_combobox.insertItems(0, list(ltMarkerStyles.keys()))
        self.marker_type_combobox.setCurrentIndex(list(ltMarkerStyles.values()).index(self.current_attitude_projections_style_params.marker))
        main_layout.addWidget(self.marker_type_combobox, 0, 1, 1, 1)

        main_layout.addWidget(
            QLabel("Marker size"),
            1, 0, 1, 1)

        self.marker_size_qspinbox = QSpinBox()
        self.marker_size_qspinbox.setValue(self.current_attitude_projections_style_params.markersize)
        main_layout.addWidget(self.marker_size_qspinbox, 1, 1, 1, 1)

        main_layout.addWidget(
            QLabel("Marker color"),
            2, 0, 1, 1)

        self.marker_color_qgscolorbutton = QgsColorButton()
        self.marker_color_qgscolorbutton.setColor(create_qcolor(self.current_attitude_projections_style_params.color))
        main_layout.addWidget(self.marker_color_qgscolorbutton, 2, 1, 1, 1)

        main_layout.addWidget(
            QLabel("Marker alpha"),
            3, 0, 1, 1)

        self.marker_alpha_qgsopacitywidget = QgsOpacityWidget()
        self.marker_alpha_qgsopacitywidget.setOpacity(self.current_attitude_projections_style_params.alpha)
        main_layout.addWidget(self.marker_alpha_qgsopacitywidget, 3, 1, 1, 1)

        main_layout.addWidget(
            QLabel("Add labels"),
            4, 0, 1, 1)

        self.label_orientations = QCheckBox("Orientations")
        self.label_orientations.setChecked(self.current_attitude_projections_style_params.label_orientations)

        main_layout.addWidget(self.label_orientations, 4, 1, 1, 1)

        self.label_ids = QCheckBox("Ids")
        self.label_ids.setChecked(self.current_attitude_projections_style_params.label_ids)

        main_layout.addWidget(self.label_ids, 5, 1, 1, 1)

        # ok/cancel setup

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()
        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        main_layout.addLayout(buttonLayout, 6, 0, 1, 2)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        # widget final setup

        self.setLayout(main_layout)
        self.setWindowTitle("Project attitude style")


class ExportTopographicProfileDialog(QDialog):

    def __init__(self,
                 plugin_name,
                 parent=None
                 ):

        super(ExportTopographicProfileDialog, self).__init__(parent)

        self.plugin_name = plugin_name

        layout = QVBoxLayout()

        # Output type

        output_type_groupBox = QGroupBox(self.tr("Output format"))

        output_type_layout = QGridLayout()

        self.outtype_shapefile_point_QRadioButton = QRadioButton(self.tr("shapefile - point"))
        output_type_layout.addWidget(self.outtype_shapefile_point_QRadioButton, 0, 0, 1, 1)
        self.outtype_shapefile_point_QRadioButton.setChecked(True)

        self.outtype_shapefile_line_QRadioButton = QRadioButton(self.tr("shapefile - line"))
        output_type_layout.addWidget(
            self.outtype_shapefile_line_QRadioButton,
            1, 0, 1, 1)

        output_type_groupBox.setLayout(output_type_layout)

        layout.addWidget(output_type_groupBox)

        # Output name/path

        output_path_groupBox = QGroupBox(self.tr("Output file"))

        output_path_layout = QGridLayout()

        self.outpath_QLineEdit = QLineEdit()
        output_path_layout.addWidget(
            self.outpath_QLineEdit,
            0, 0, 1, 1)

        self.outpath_QPushButton = QPushButton("....")
        self.outpath_QPushButton.clicked.connect(self.define_outpath)
        output_path_layout.addWidget(
            self.outpath_QPushButton,
            0, 1, 1, 1)

        self.load_output_checkBox = QCheckBox("load output shapefile in project")
        self.load_output_checkBox.setChecked(True)
        output_path_layout.addWidget(
            self.load_output_checkBox,
            1, 0, 1, 2)

        output_path_groupBox.setLayout(output_path_layout)

        layout.addWidget(output_path_groupBox)

        decide_QWiget = QWidget()

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        decide_QWiget.setLayout(buttonLayout)

        layout.addWidget(decide_QWiget)

        self.setLayout(layout)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        self.setWindowTitle("Export topographic profile")

    def define_outpath(self):

        if self.outtype_shapefile_line_QRadioButton.isChecked() or self.outtype_shapefile_point_QRadioButton.isChecked():
            outfile_path = new_file_path(
                self,
                "Save file",
                "",
                "Shapefiles (*.shp)"
            )
        elif self.outtype_csv_QRadioButton.isChecked():
            outfile_path = new_file_path(
                self,
                "Save file",
                "",
                "Csv (*.csv)"
            )
        else:
            error_qt(
                self,
                self.plugin_name,
                self.tr("Output type definition error")
            )
            return

        self.outpath_QLineEdit.setText(outfile_path)


class CreateNewGeopackageDialog(QDialog):

    def __init__(self,
                 plugin_name,
                 parent=None
                 ):

        super(CreateNewGeopackageDialog, self).__init__(parent)

        self.plugin_name = plugin_name

        layout = QGridLayout()

        self.new_geopackage_QLineEdit = QLineEdit()
        layout.addWidget(
            self.new_geopackage_QLineEdit,
            0, 0, 1, 1)

        self.outpath_QPushButton = QPushButton("....")
        self.outpath_QPushButton.clicked.connect(self.define_outpath)
        layout.addWidget(
            self.outpath_QPushButton,
            0, 1, 1, 1)

        layout.addWidget(QLabel("New layer"), 1, 0, 1, 1)

        self.new_layer_QLineEdit = QLineEdit()
        layout.addWidget(
            self.new_layer_QLineEdit,
            1, 1, 1, 1)

        decide_QWiget = QWidget()

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        decide_QWiget.setLayout(buttonLayout)

        layout.addWidget(decide_QWiget)

        self.setLayout(layout)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        self.setWindowTitle("Define new geopackage")

    def define_outpath(self):

        outfile_path = new_file_path(
            self,
            "New geopackage",
            "",
            "Geopackages (*.gpkg)"
        )

        self.new_geopackage_QLineEdit.setText(outfile_path)


class OpenGeopackageDialog(QDialog):

    def __init__(self,
                 plugin_name,
                 parent=None
                 ):

        super(OpenGeopackageDialog, self).__init__(parent)

        self.plugin_name = plugin_name

        layout = QGridLayout()

        self.existing_geopackage_QLineEdit = QLineEdit()
        layout.addWidget(
            self.existing_geopackage_QLineEdit,
            0, 0, 1, 1)

        self.outpath_QPushButton = QPushButton("....")
        self.outpath_QPushButton.clicked.connect(self.choose_existing_geopackage_path)
        layout.addWidget(
            self.outpath_QPushButton,
            0, 1, 1, 1)

        layout.addWidget(QLabel("Layer"), 1, 0, 1, 1)

        self.existing_layer_QLineEdit = QLineEdit()
        layout.addWidget(
            self.existing_layer_QLineEdit,
            1, 1, 1, 1)

        decide_QWiget = QWidget()

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        decide_QWiget.setLayout(buttonLayout)

        layout.addWidget(decide_QWiget)

        self.setLayout(layout)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        self.setWindowTitle("Choose geopackage")

    def choose_existing_geopackage_path(self):

        outfile_path = old_file_path(
            self,
            "Define geopackage",
            "",
            "Geopackages (*.gpkg)"
        )

        self.existing_geopackage_QLineEdit.setText(outfile_path)


class ExportLineDataDialog(QDialog):

    def __init__(self,
                 plugin_name,
                 parent=None
                 ):

        super(ExportLineDataDialog, self).__init__(parent)

        self.plugin_name = plugin_name

        layout = QVBoxLayout()

        # Output type

        output_type_groupBox = QGroupBox(self.tr("Output format"))

        output_type_layout = QGridLayout()

        self.outtype_shapefile_line_QRadioButton = QRadioButton(self.tr("shapefile - line"))
        output_type_layout.addWidget(self.outtype_shapefile_line_QRadioButton, 0, 0, 1, 1)
        self.outtype_shapefile_line_QRadioButton.setChecked(True)

        self.outtype_csv_QRadioButton = QRadioButton(self.tr("csv"))
        output_type_layout.addWidget(
            self.outtype_csv_QRadioButton,
            0, 1, 1, 1)

        output_type_groupBox.setLayout(output_type_layout)

        layout.addWidget(output_type_groupBox)

        # Output name/path

        output_path_groupBox = QGroupBox(self.tr("Output file"))

        output_path_layout = QGridLayout()

        self.outpath_QLineEdit = QLineEdit()
        output_path_layout.addWidget(
            self.outpath_QLineEdit,
            0, 0, 1, 1)

        self.outpath_QPushButton = QPushButton("....")
        self.outpath_QPushButton.clicked.connect(self.define_outpath)
        output_path_layout.addWidget(
            self.outpath_QPushButton,
            0, 1, 1, 1)

        self.load_output_checkBox = QCheckBox("load output in project")
        self.load_output_checkBox.setChecked(True)
        output_path_layout.addWidget(
            self.load_output_checkBox,
            1, 0, 1, 2)

        output_path_groupBox.setLayout(output_path_layout)

        layout.addWidget(output_path_groupBox)

        decide_QWiget = QWidget()

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        decide_QWiget.setLayout(buttonLayout)

        layout.addWidget(decide_QWiget)

        self.setLayout(layout)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        self.setWindowTitle("Export")

    def define_outpath(self):

        if self.outtype_shapefile_line_QRadioButton.isChecked():
            outfile_path = new_file_path(
                self,
                "Save file",
                "",
                "Shapefiles (*.shp)"
            )
        elif self.outtype_csv_QRadioButton.isChecked():
            outfile_path = new_file_path(
                self,
                "Save file",
                "",
                "Csv (*.csv)"
            )
        else:
            error_qt(
                self,
                self.plugin_name,
                self.tr("Output type definition error")
            )
            return

        self.outpath_QLineEdit.setText(outfile_path)


class BaseDataDigitizeLineDialog(QDialog):

    def __init__(
            self,
            plugin_name,
            canvas,
            parent=None
    ):

        super(BaseDataDigitizeLineDialog, self).__init__(parent)

        self.plugin_name = plugin_name
        self.canvas = canvas
        self.previous_maptool = self.canvas.mapTool()  # Save the standard map tool for restoring it at the end

        layout = QVBoxLayout()

        self.qpbtDigitizeLine = QPushButton(self.tr("Digitize trace"))
        self.qpbtDigitizeLine.setToolTip(
            "Digitize the trace on the map.\n"
            "Left click: add point\n"
            "Right click: end adding point"
        )
        self.qpbtDigitizeLine.clicked.connect(self.digitize_line)

        layout.addWidget(self.qpbtDigitizeLine)

        self.qpbtClearLine = QPushButton(self.tr("Clear"))
        self.qpbtClearLine.clicked.connect(self.clear_rubberband)
        layout.addWidget(self.qpbtClearLine)

        self.qpbtClearLine = QPushButton(self.tr("Save"))
        self.qpbtClearLine.clicked.connect(self.save_rubberband)
        layout.addWidget(self.qpbtClearLine)

        self.setLayout(layout)

        self.setWindowTitle("Digitize line")

    '''
    def connect_digitize_maptool(self):

        self.digitize_maptool.moved.connect(self.canvas_refresh_profile_line)
        self.digitize_maptool.leftClicked.connect(self.profile_add_point)
        self.digitize_maptool.rightClicked.connect(self.canvas_end_profile_line)
    '''

    def digitize_line(self):

        self.clear_rubberband()

        info_qgis(
            self.plugin_name,
            "Now you can digitize a line on the map.\nLeft click: add point\nRight click: end adding point"
        )

        self.rubberband = QgsRubberBand(self.canvas)
        self.rubberband.setWidth(2)
        self.rubberband.setColor(QColor(Qt.red))

        self.digitize_maptool = MapDigitizeTool(self.canvas)
        self.canvas.setMapTool(self.digitize_maptool)

        self.digitize_maptool.moved.connect(self.canvas_refresh_profile_line)
        self.digitize_maptool.leftClicked.connect(self.profile_add_point)
        self.digitize_maptool.rightClicked.connect(self.canvas_end_profile_line)

    def canvas_refresh_profile_line(self, position):

        x, y = xy_from_canvas(self.canvas, position)

        self.refresh_rubberband(
            self.profile_canvas_points_x + [x],
            self.profile_canvas_points_y + [y]
        )

    def profile_add_point(self, position):

        x, y = xy_from_canvas(self.canvas, position)

        self.profile_canvas_points_x.append(x)
        self.profile_canvas_points_y.append(y)

    def canvas_end_profile_line(self):

        self.refresh_rubberband(
            self.profile_canvas_points_x,
            self.profile_canvas_points_y)

        self.digitized_profile_line2dt = None
        if len(self.profile_canvas_points_x) > 1:
            raw_line = Ln(list(zip(self.profile_canvas_points_x, self.profile_canvas_points_y))).remove_coincident_points()
            if raw_line.num_points() > 1:
                self.digitized_profile_line2dt = raw_line

        self.profile_canvas_points_x = []
        self.profile_canvas_points_y = []

        self.restore_previous_map_tool()

    def restore_previous_map_tool(self):

        self.canvas.unsetMapTool(self.digitize_maptool)
        self.canvas.setMapTool(self.previous_maptool)

    def refresh_rubberband(self,
                           xy_list
                           ):

        self.rubberband.reset(QgsWkbTypes.LineGeometry)
        for x, y in xy_list:
            self.rubberband.addPoint(QgsPointXY(x, y))

    def clear_rubberband(self):

        self.profile_canvas_points_x = []
        self.profile_canvas_points_y = []

        self.digitized_profile_line2dt = None

        try:

            self.rubberband.reset()

        except:

            pass

    def save_rubberband(self):

        def output_profile_line(
                output_format,
                output_filepath,
                pts2dt,
                proj_sr
        ):

            points = [[n, pt2dt.x, pt2dt.y] for n, pt2dt in enumerate(pts2dt)]
            if output_format == "csv":
                success, msg = write_generic_csv(output_filepath,
                                                 ['id', 'x', 'y'],
                                                 points)
                if not success:
                    warn_qt(
                        self,
                        self.plugin_name,
                        msg
                    )
            elif output_format == "shapefile - line":
                success, msg = write_rubberband_profile_lnshp(
                    output_filepath,
                    ['id'],
                    points,
                    proj_sr)
                if not success:
                    warn_qt(
                        self,
                        self.plugin_name,
                        msg
                    )
            else:
                error_qt(
                    self,
                    self.plugin_name,
                    "Debug: error in export format"
                )
                return

            if success:
                ok_qgis(
                    self.plugin_name,
                    "Line saved"
                )

        def extract_format_type():

            if dialog.outtype_shapefile_line_QRadioButton.isChecked():
                return "shapefile - line"
            elif dialog.outtype_csv_QRadioButton.isChecked():
                return "csv"
            else:
                return ""

        if self.digitized_profile_line2dt is None:

            warn_qt(
                self,
                self.plugin_name,
                "No available line to save [1]"
            )
            return

        elif self.digitized_profile_line2dt.num_points() < 2:

            warn_qt(
                self,
                self.plugin_name,
                "No available line to save [2]"
            )
            return

        dialog = ExportLineDataDialog(self.plugin_name)
        if dialog.exec_():
            output_format = extract_format_type()
            if output_format == "":
                warn_qt(
                    self,
                    self.plugin_name,
                    "Error in output format"
                )
                return
            output_filepath = dialog.outpath_QLineEdit.text()
            if len(output_filepath) == 0:
                warn_qt(
                    self,
                    self.plugin_name,
                    "Error in output path"
                )
                return
            add_to_project = dialog.load_output_checkBox.isChecked()
        else:
            warn_qt(
                self,
                self.plugin_name,
                "No export defined"
            )
            return

        # get project CRS information
        project_crs_osr = proj4str()

        output_profile_line(
            output_format,
            output_filepath,
            self.digitized_profile_line2dt.pts,
            project_crs_osr)

        # add theme to QGis project
        if output_format == "shapefile - line" and add_to_project:
            try:

                digitized_line_layer = QgsVectorLayer(output_filepath,
                                                      QFileInfo(output_filepath).baseName(),
                                                      "ogr")
                QgsProject.instance().addMapLayer(digitized_line_layer)

            except:

                error_qt(
                    self,
                    self.plugin_name,
                    "Unable to load layer in project"
                )
                return


class BaseDataProfilerSettingsDialog(QDialog):

    def __init__(
            self,
            plugin_name,
            parent=None
    ):

        super(BaseDataProfilerSettingsDialog, self).__init__(parent)

        self.plugin_name = plugin_name

        layout = QGridLayout()

        layout.addWidget(QLabel("Number of parallel profiles"), 0, 0, 1, 1)

        self.num_parallel_profiles_wdgt = QSpinBox()
        self.num_parallel_profiles_wdgt.setValue(1)
        self.num_parallel_profiles_wdgt.setMinimum(1)
        self.num_parallel_profiles_wdgt.setSingleStep(2)
        layout.addWidget(self.num_parallel_profiles_wdgt, 0, 1, 1, 1)

        layout.addWidget(QLabel("Spacing between parallel profiles"), 1, 0, 1, 1)

        self.spacing_wdgt = QDoubleSpinBox()
        self.spacing_wdgt.setValue(0.0)
        self.spacing_wdgt.setMinimum(0.0)
        self.spacing_wdgt.setMaximum(10000.0)
        self.spacing_wdgt.setSingleStep(50.0)

        layout.addWidget(self.spacing_wdgt, 1, 1, 1, 1)

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")
        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        verticalSpacer = QSpacerItem(20, 30, QSizePolicy.Minimum, QSizePolicy.Expanding)
        layout.addItem(verticalSpacer)

        layout.addWidget(okButton,
                         2, 0, 1, 1)
        layout.addWidget(cancelButton,
                         2, 1, 1, 1)

        self.setLayout(layout)

        self.setWindowTitle("Profiles settings")
        self.setMinimumSize(400, 200)


class StylesPolygonIntersectionsParamsDialog(QDialog):

    '''
    colors = ["darkseagreen", "darkgoldenrod", "darkviolet", "hotpink", "powderblue", "yellowgreen", "palevioletred",
              "seagreen", "darkturquoise", "beige", "darkkhaki", "red", "yellow", "magenta", "blue", "cyan",
              "chartreuse"]
    '''

    def __init__(
            self,
            plugin_name,
            current_graphical_params,
            #polygons_symbology_colors: dict,
            parent=None
    ):

        super(StylesPolygonIntersectionsParamsDialog, self).__init__(parent)

        self.plugin_name = plugin_name
        self.current_graphical_params = current_graphical_params
        #self.polygon_classifications = list(polygons_symbology_colors)

        self.polygon_classifications_treeWidget = QTreeWidget()
        self.polygon_classifications_treeWidget.setColumnCount(2)
        self.polygon_classifications_treeWidget.headerItem().setText(0, "Name")
        self.polygon_classifications_treeWidget.headerItem().setText(1, "Color")
        self.polygon_classifications_treeWidget.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        self.polygon_classifications_treeWidget.setDragEnabled(False)
        self.polygon_classifications_treeWidget.setDragDropMode(QAbstractItemView.NoDragDrop)
        self.polygon_classifications_treeWidget.setAlternatingRowColors(True)
        self.polygon_classifications_treeWidget.setTextElideMode(Qt.ElideLeft)

        self.update_classification_colors_treewidget()

        self.polygon_classifications_treeWidget.resizeColumnToContents(0)
        self.polygon_classifications_treeWidget.resizeColumnToContents(1)

        okButton = QPushButton("&OK")
        cancelButton = QPushButton("Cancel")

        buttonLayout = QHBoxLayout()
        buttonLayout.addStretch()
        buttonLayout.addWidget(okButton)
        buttonLayout.addWidget(cancelButton)

        layout = QGridLayout()

        layout.addWidget(self.polygon_classifications_treeWidget, 0, 0, 1, 3)
        layout.addLayout(buttonLayout, 1, 0, 1, 3)

        self.setLayout(layout)

        okButton.clicked.connect(self.accept)
        cancelButton.clicked.connect(self.reject)

        self.setWindowTitle("Polygon intersection colors")

    def update_classification_colors_treewidget(self):

        self.polygon_classifications_treeWidget.clear()

        for classification_id, color in self.current_graphical_params.polygon_intersections.items():

            tree_item = QTreeWidgetItem(self.polygon_classifications_treeWidget)
            tree_item.setText(0, str(classification_id))

            color_QgsColorButton = QgsColorButton()

            if classification_id in self.current_graphical_params.polygon_intersections:
                color = QColor.fromRgbF(*color)
            color_QgsColorButton.setColor(QColor(color))
            self.polygon_classifications_treeWidget.setItemWidget(tree_item, 1, color_QgsColorButton)


class HelpDialog(QDialog):

    def __init__(self,
                 plugin_name,
                 plugin_folder,
                 parent=None):

        super(HelpDialog, self).__init__(parent)

        layout = QVBoxLayout()

        # About section

        helpTextBrwsr = QTextBrowser(self)

        url_path = f"file:///{plugin_folder}/help/geoprofiler/help.html"
        helpTextBrwsr.setSource(QUrl(url_path))
        helpTextBrwsr.setSearchPaths([f'{plugin_folder}/help/geoprofiler/images'])
        helpTextBrwsr.setMinimumSize(700, 600)

        layout.addWidget(helpTextBrwsr)

        self.setLayout(layout)

        self.setWindowTitle("{} - Points-plane distances help".format(plugin_name))



