# -*- coding: utf-8 -*-
"""
/***************************************************************************
 gestion_crise
 Plugin QGIS de gestion de crise
 Version finale modernisée (pathlib-only)
 - priorité disque externe (vérifie dossiers finaux des partages à la racine)
 - fallback réseau
 - projet modèle embarqué (Gestion_crise.qgz)
 - configuration partages réseau
 - avertissement automatique (asynchrone)
 - compatibilité PyQt5 / PyQt6
 - DEBUG_MODE optionnel + logging fiable (QgsMessageLog + print flush)
 ***************************************************************************/
"""

# --- Librairies système ---
import platform
import threading
import unicodedata
import os
import time
import datetime
import subprocess
from pathlib import Path
from ctypes import windll

# --- Librairies Windows spécifiques ---
import win32api
import win32file
import win32net
import win32netcon
import win32con
import pywintypes

# --- QGIS / PyQt ---
from qgis.PyQt.QtCore import (
    QLocale, QObject, pyqtSignal, QThread, QTimer,
    QSettings, QTranslator, QCoreApplication, Qt,
    PYQT_VERSION_STR, QFile, QVariant, QMetaObject, Q_ARG
)
from qgis.PyQt.QtGui import QIcon, QMovie

from qgis.PyQt.QtWidgets import (
    QAction, QMessageBox, QInputDialog, QLineEdit, QDialog,
    QLabel, QPushButton, QVBoxLayout, QGridLayout, QHBoxLayout,
    QApplication, QProgressBar, QFrame, QDialogButtonBox,
    QTextBrowser, QTextEdit, QSizePolicy
)

from qgis.core import *
from qgis.gui import *

# --- Plugin resources ---
from . import resources
from .Gestion_crise_dialog import GestionCriseDialog

# --- Compatibilité PyQt5 / PyQt6 ---
if PYQT_VERSION_STR.startswith("5"):
    qmessagebox_information = QMessageBox.Information
    qt_windowsmodal = Qt.WindowModal
    qt_windowstaysontophint = Qt.WindowStaysOnTopHint
elif PYQT_VERSION_STR.startswith("6"):
    qmessagebox_information = QMessageBox.Icon.Information
    qt_windowsmodal = Qt.WindowModality.WindowModal
    qt_windowstaysontophint = Qt.WindowType.WindowStaysOnTopHint

# --- Réglage par défaut (sera surchargé par QSettings à l'init) ---
import sys
import warnings
DEBUG_MODE = False

# --- neutralisation complète des prints et warnings si DEBUG désactivé ---
if not DEBUG_MODE:
    def _silent_print(*args, **kwargs):
        pass
    print = _silent_print  # type: ignore
    warnings.filterwarnings("ignore")

def _debug_sink(msg: str):
    """Écrit dans le journal QGIS + console Python (avec flush)."""
    try:
        QgsMessageLog.logMessage(msg, "Gestion_crise", Qgis.Info)
    except Exception:
        pass
    try:
        print(f"[DEBUG] {msg}", flush=True)
    except Exception:
        pass

def debug(msg: str):
    """Affiche un message de debug si DEBUG_MODE est activé (fiable)."""
    if DEBUG_MODE:
        _debug_sink(msg)



# ============================================================================
# WORKERS
# ============================================================================
class FullProgressSearchWorker(QObject):

    def tr(self, text):
        from qgis.PyQt.QtCore import QCoreApplication
        return QCoreApplication.translate('Gestion_crise', text)

    progress = pyqtSignal(int)
    status  = pyqtSignal(str)
    finished = pyqtSignal(dict)

    def __init__(self, plugin):
        super().__init__()
        self.plugin = plugin

    def run(self):

        result = {
            "disque": False,
            "drive": None,
            "reseau": False,
            "readers": [],
            "ip_dispo": False
        }

        lecteurs = [d for d in win32api.GetLogicalDriveStrings().split("\x00") if d]

        # Progression : lecteurs + ping + test réseau + fin
        total = len(lecteurs) + 2
        step  = 0

        # -------------------------------
        # 1️⃣ Extraction des noms attendus
        # -------------------------------
        noms_attendus = {
            self.plugin._extract_folder_name_from_unc(c)
            for c in self.plugin.partages.values()
            if isinstance(c, str) and c.startswith("\\\\")
        }
        noms_attendus = {x for x in noms_attendus if x}

        # -------------------------------
        # 2️⃣ Recherche disque externe
        # -------------------------------
        for drive in lecteurs:
            step += 1

            self.status.emit(
                self.tr("Test du lecteur %1…").replace("%1", drive)
            )

            self.progress.emit(int(100 * step / total))

            try:
                dtype = win32file.GetDriveType(drive)
                if dtype not in (win32con.DRIVE_FIXED, win32con.DRIVE_REMOVABLE):
                    continue

                base = Path(drive)
                label = win32api.GetVolumeInformation(drive)[0]

                # par label
                if label == self.plugin.disque_ext:
                    result["disque"] = True
                    result["drive"]  = drive
                    self.finished.emit(result)
                    return

                # par structure
                dossiers = {
                    p.name.lower()
                    for p in base.iterdir()
                    if p.is_dir()
                }

                if noms_attendus.issubset(dossiers):
                    result["disque"] = True
                    result["drive"]  = drive
                    self.finished.emit(result)
                    return

            except Exception:
                continue

        # -------------------------------
        # 3️⃣ Test IP serveurs
        # -------------------------------
        step += 1
        self.status.emit(self.tr("Test des serveurs…"))
        self.progress.emit(int(100 * step / total))

        ip_ok = self.plugin._serveurs_disponibles()
        result["ip_dispo"] = ip_ok

        # -------------------------------
        # 4️⃣ Détection réseau (IP uniquement)
        # -------------------------------
        step += 1
        self.status.emit(self.tr("Recherche du réseau interne…"))
        self.progress.emit(int(100 * step / total))

        readers = []
        for lecteur, chemin in self.plugin.partages.items():
            if isinstance(chemin, str) and chemin.startswith("\\\\"):
                try:
                    ip = chemin[2:].split("\\")[0]
                except:
                    continue
                if self.plugin._ping_ip(ip):
                    readers.append(lecteur)

        if readers:
            result["reseau"] = True
            result["readers"] = readers

        # -------------------------------
        # 5️⃣ Fin
        # -------------------------------
        self.status.emit(self.tr("Analyse terminée."))
        self.progress.emit(100)
        self.finished.emit(result)


# ============================================================================
# worker recherche disque externe
# ============================================================================
class DiskSearchWorker(QObject):
    progress = pyqtSignal(str, int, int)
    finished = pyqtSignal(bool)

    def __init__(self, plugin):
        super().__init__()
        self.plugin = plugin

    def run(self):
        # Liste des lecteurs
        try:
            lecteurs = [d for d in win32api.GetLogicalDriveStrings().split("\x00") if d]
        except Exception:
            lecteurs = []

        total = max(1, len(lecteurs))
        index = 0

        # -------------------------------------------------------
        # 1️⃣ Extraction des noms de dossiers attendus
        #     → Toujours depuis la CONFIG utilisateur
        # -------------------------------------------------------
        noms_attendus = {
            self.plugin._extract_folder_name_from_unc(chemin)
            for chemin in self.plugin.partages.values()
            if isinstance(chemin, str) and chemin.startswith("\\\\")
        }
        noms_attendus = {n for n in noms_attendus if n}  # nettoyage

        # -------------------------------------------------------
        # 2️⃣ Recherche réelle du disque externe
        # -------------------------------------------------------
        for drive in lecteurs:
            index += 1

            # Signal progression → throbber
            self.progress.emit(drive, index, total)

            # Laisser respirer le thread (fluidité GIF)
            QThread.msleep(60)

            try:
                drive_type = win32file.GetDriveType(drive)
                if drive_type not in (win32con.DRIVE_FIXED, win32con.DRIVE_REMOVABLE):
                    continue

                # Liste des dossiers présents à la racine du lecteur
                dossiers = {
                    p.name.lower()
                    for p in Path(drive).iterdir()
                    if p.is_dir()
                }

                # Match strict : tous les dossiers attendus doivent exister
                if noms_attendus.issubset(dossiers):
                    # 🔥 disque externe trouvé !
                    self.finished.emit(True)
                    return

            except Exception:
                continue

        # -------------------------------------------------------
        # 3️⃣ Aucun disque trouvé
        # -------------------------------------------------------
        self.finished.emit(False)


# ============================================================================
# worker recherche complête disques externes et réseau interne
# ============================================================================
class FullSearchWorker(QObject):
    progress = pyqtSignal(int)        # valeur 0–100 %
    status = pyqtSignal(str)          # message dynamique
    finished = pyqtSignal(dict)       # résultat final

    def __init__(self, plugin):
        super().__init__()
        self.plugin = plugin

    def run(self):
        result = {
            "disque": False,
            "reseau": False,
            "readers": [],
            "ip_dispo": False,
            "drive": None
        }

        # -------------------------------------------------------
        # Étape 1️⃣ : Recherche disque externe
        # -------------------------------------------------------
        lecteurs = [d for d in win32api.GetLogicalDriveStrings().split("\x00") if d]

        total_pas = len(lecteurs) + 3   # +3 = IP + réseau + fin
        pas = 0

        # --- Extraction fiable des noms attendus depuis la CONFIG utilisateur ---
        noms_attendus = {
            self.plugin._extract_folder_name_from_unc(chemin)
            for chemin in self.plugin.partages.values()
            if isinstance(chemin, str) and chemin.startswith("\\\\")
        }
        noms_attendus = {n for n in noms_attendus if n}

        # -------------------------------------------------------
        # Test de chaque lecteur
        # -------------------------------------------------------
        for drive in lecteurs:
            pas += 1

            self.status.emit(
                self.tr("Test du lecteur %1…").replace("%1", drive)
            )

            self.progress.emit(int((pas / total_pas) * 100))

            try:
                drive_type = win32file.GetDriveType(drive)
                if drive_type not in (win32con.DRIVE_FIXED, win32con.DRIVE_REMOVABLE):
                    continue

                base = Path(drive)
                volume_label = win32api.GetVolumeInformation(drive)[0]

                # 1️⃣ Détection par label exact
                if volume_label == self.plugin.disque_ext:
                    result["disque"] = True
                    result["drive"] = drive
                    break

                # 2️⃣ Détection par structure dossier
                dossiers_presents = {
                    p.name.strip().lower()
                    for p in base.iterdir()
                    if p.is_dir()
                }

                if noms_attendus.issubset(dossiers_presents):
                    result["disque"] = True
                    result["drive"] = drive
                    break

            except Exception:
                continue

        # -------------------------------------------------------
        # Si disque externe trouvé → fin immédiate
        # -------------------------------------------------------
        if result["disque"]:
            self.status.emit(self.tr("Disque externe détecté."))
            self.progress.emit(100)
            self.finished.emit(result)
            return

        # -------------------------------------------------------
        # 2️⃣ Test IP serveurs
        # -------------------------------------------------------
        pas += 1
        self.status.emit(self.tr("Test des serveurs…"))
        self.progress.emit(int((pas / total_pas) * 100))

        ip_dispo = self.plugin._serveurs_disponibles()
        result["ip_dispo"] = ip_dispo

        # -------------------------------------------------------
        # 3️⃣ Recherche réseau : test IP uniquement (aucun accès UNC)
        # -------------------------------------------------------
        pas += 1
        self.status.emit(self.tr("Recherche du réseau interne…"))
        self.progress.emit(int((pas / total_pas) * 100))

        readers = []

        for lecteur, chemin in self.plugin.partages.items():
            if not isinstance(chemin, str) or not chemin.startswith("\\\\"):
                continue

            try:
                ip = chemin[2:].split("\\")[0].strip()
            except Exception:
                continue

            if self.plugin._ping_ip(ip):
                readers.append(lecteur)

        if readers:
            result["reseau"] = True
            result["readers"] = readers

        # -------------------------------------------------------
        # 4️⃣ Fin
        # -------------------------------------------------------
        self.status.emit(self.tr("Analyse terminée."))
        self.progress.emit(100)
        self.finished.emit(result)


# ============================================================================
# worker compilation resources.qrc
# ============================================================================
class CompileResourcesWorker(QObject):
    finished = pyqtSignal(bool, str)  # success, message

    def __init__(self, qrc_file: Path, py_file: Path, pyrcc_path: Path):
        super().__init__()
        self.qrc_file = qrc_file
        self.py_file = py_file
        self.pyrcc_path = pyrcc_path

    def run(self):
        """Exécuté dans un QThread → aucune fenêtre CMD et UI fluide."""

        cmd = [
            str(self.pyrcc_path),
            str(self.qrc_file),
            "-o",
            str(self.py_file),
        ]

        try:
            result = subprocess.run(
                cmd,
                stdin=subprocess.DEVNULL,
                stdout=subprocess.PIPE,     # OK car capture_output est désactivé
                stderr=subprocess.PIPE,     # idem
                text=True,
                creationflags=subprocess.CREATE_NO_WINDOW  # ← empêche toute fenêtre CMD
            )

            if result.returncode != 0:
                self.finished.emit(False, result.stderr)
            else:
                self.finished.emit(True, str(self.pyrcc_path))

        except Exception as e:
            self.finished.emit(False, str(e))


# ============================================================================
# Fenêtre throbber (chrono.gif) non bloquante pendant la recherche
# ============================================================================

class ThrobberProgressDialog(QDialog):
    """
    Fenêtre de progression Fluent auto-redimensionnable :
    - s’adapte à la taille réelle du GIF
    - ajuste la largeur/hauteur selon la longueur du texte
    - toujours centrée
    - style Windows 11
    Compatible PyQt5 & PyQt6.
    """

    def __init__(self, parent=None):
        super().__init__(parent)

        # --- Style fenêtre ---
        self.setWindowFlags(
            Qt.FramelessWindowHint |
            Qt.Tool |
            Qt.WindowStaysOnTopHint
        )
        self.setModal(False)
        self.setAttribute(Qt.WA_TranslucentBackground)

        # --- Conteneur principal ---
        self.container = QFrame(self)
        self.container.setObjectName("container")
        self.container.setStyleSheet("""
            #container {
                background-color: rgba(255, 255, 255, 215);
                border-radius: 18px;
                border: 1px solid rgba(255,255,255,150);
            }
            QLabel {
                color: #202020;
                font-size: 11pt;
                font-weight: 500;
            }
            QProgressBar {
                border: none;
                background: rgba(0,0,0,20);
                border-radius: 6px;
                height: 10px;
            }
            QProgressBar::chunk {
                background-color: #0067C0;
                border-radius: 6px;
            }
        """)

        # --- Layout interne ---
        self.layout = QVBoxLayout(self.container)
        self.layout.setContentsMargins(24, 24, 24, 24)
        self.layout.setSpacing(14)

        # GIF
        self.gif_label = QLabel()
        self.gif_label.setAlignment(Qt.AlignCenter)
        self.movie = None

        # Texte
        self.label_txt = QLabel(self.tr("Analyse en cours…"))
        self.label_txt.setAlignment(Qt.AlignCenter)
        self.label_txt.setWordWrap(True)  # ← indispensable pour auto-layout

        # Progress Bar
        self.pbar = QProgressBar()
        self.pbar.setRange(0, 100)
        self.pbar.setValue(0)
        self.pbar.setTextVisible(False)

        # Ajout widgets
        self.layout.addWidget(self.gif_label)
        self.layout.addWidget(self.label_txt)
        self.layout.addWidget(self.pbar)

        # Taille calculée automatiquement
        self.adjustSize()
        self._center_on_screen()

    # ------------------------------------------------------------------
    # CHARGEMENT ADAPTATIF DU GIF
    # ------------------------------------------------------------------

    def _tick_movie(self):
        """
        Force QMovie à avancer d'une frame régulièrement.
        Empêche tout freeze même si le thread principal est très chargé.
        """
        if self.movie:
            self.movie.jumpToFrame(self.movie.currentFrameNumber() + 1)

    def update_gif(self, gif_path: str):
        """Charge un GIF, détecte sa taille réelle, et redimensionne le throbber."""
        if self.movie:
            self.movie.stop()

        self.movie = QMovie(gif_path)
        self.gif_label.setMovie(self.movie)
        self.movie.start()

        # --- Timer interne pour forcer l'animation du GIF ---
        self._timer = QTimer(self)
        self._timer.setInterval(33)  # ≈ 30 FPS
        self._timer.timeout.connect(self._tick_movie)
        self._timer.start()

        self.movie.jumpToFrame(0)  # permet d’obtenir la taille réelle du GIF

        self._recalculate_size()

    # ------------------------------------------------------------------
    # AJUSTEMENT INTELLIGENT EN FONCTION DU GIF + TEXTE
    # ------------------------------------------------------------------
    def _recalculate_size(self):
        """
        Recalcule la taille totale de la fenêtre en fonction :
        - du GIF (largeur/hauteur réelles)
        - du texte (longueur dynamique)
        - des marges internes
        """

        # Taille GIF
        gif_w = self.movie.frameRect().width() if self.movie else 64
        gif_h = self.movie.frameRect().height() if self.movie else 64

        # Appliquer taille mini
        self.gif_label.setFixedSize(gif_w, gif_h)

        # Adapter la largeur du texte au GIF
        max_width = max(gif_w, 220)  # largeur minimale confortable
        self.label_txt.setMaximumWidth(max_width)

        # Demander un recalcul automatique du layout
        self.adjustSize()
        self.container.adjustSize()

        # Taille finale basée sur container
        self.resize(self.container.width(), self.container.height())

        # Centre la fenêtre après resize
        self._center_on_screen()

    # ------------------------------------------------------------------
    # MISE À JOUR DU TEXTE → Resize automatique
    # ------------------------------------------------------------------
    def update_status(self, text: str):
        """
        Met à jour le message texte et recalcule la taille si nécessaire.
        """
        self.label_txt.setText(text)
        self.label_txt.adjustSize()
        self._recalculate_size()
        QApplication.processEvents()

    # ------------------------------------------------------------------
    # MISE À JOUR DE LA PROGRESSION
    # ------------------------------------------------------------------
    def update_progress(self, percent: int):
        self.pbar.setValue(percent)
        QApplication.processEvents()

    # ------------------------------------------------------------------
    # CENTRAGE
    # ------------------------------------------------------------------
    def _center_on_screen(self):
        screen = QApplication.primaryScreen().availableGeometry()
        self.move(
            int(screen.center().x() - self.width() / 2),
            int(screen.center().y() - self.height() / 2)
        )


# ============================================================================
# CLASSE PRINCIPALE DU PLUGIN
# ============================================================================
class Gestion_crise:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Initialisation principale du plugin QGIS."""
        self.iface = iface
        # -------------------------------------------------------------------------
        # 🔥 Charger automatiquement QdrawEVT avant Gestion_crise
        # -------------------------------------------------------------------------
        import qgis.utils
        settings = QSettings()
        self._gc_separator_added = None
        # Observer la suppression/désactivation dans la fenêtre Extensions
        # QgsApplication.pluginRegistry().pluginsRemoved.connect(self._on_plugins_removed)

        def ensure_plugin_loaded(plugin_name: str):
            """Active et charge un plugin QGIS s'il n'est pas encore chargé."""
            # 1. Activer dans qgis.ini
            settings.beginGroup("PythonPlugins")
            settings.setValue(plugin_name, True)
            settings.endGroup()
            # 2. Si déjà chargé → OK
            if plugin_name in qgis.utils.plugins:
                return qgis.utils.plugins[plugin_name]
            # 3. Sinon → tenter reload immédiat
            try:
                qgis.utils.reloadPlugin(plugin_name)
            except Exception:
                pass
            return qgis.utils.plugins.get(plugin_name, None)

        # --- Charger QdrawEVT avant toute initialisation de Gestion_crise ---
        qdraw = ensure_plugin_loaded("qdrawEVT")
        self.toolbar_qdraw = None
        self.gc_anchor = None
        if qdraw:
            if hasattr(qdraw, "get_toolbar"):
                self.toolbar_qdraw = qdraw.get_toolbar()
            if hasattr(qdraw, "get_gc_anchor"):
                self.gc_anchor = qdraw.get_gc_anchor()

        self.plugin_dir = Path(__file__).parent
        self.menu = self.tr('&Gestion de crise')
        self.actions = []
        self.first_start = None
        self.action_debug_toggle = None  # action checkable pour debug
        self.stop_network_check = threading.Event()
        self.deja_monte = False  # ← empêche un second lancement
        self._loading_new_project = False

        # -------------------------------------------------------------------------
        # 🌍 Localisation robuste
        #    - langue source (aucun QM) déterminée par le pays d'exécution de Qgis
        #    - Fallback EN uniquement si :
        #        * langue QGIS != FR
        #        * ET QM de la langue absente
        # -------------------------------------------------------------------------
        settings = QSettings()
        locale_full = settings.value("locale/userLocale", "fr_FR")
        lang = locale_full.split("_")[0].lower()  # fr, en, de, es...

        i18n_dir = self.plugin_dir / "i18n"
        translator = QTranslator()

        def load_qm(code: str) -> bool:
            qm = i18n_dir / f"Gestion_crise_{code}.qm"
            if qm.exists():
                return translator.load(str(qm))
            return False
        loaded = False
        # -----------------------------
        # CAS 1️⃣ : Français → langue source
        # -----------------------------
        if lang == "fr":
            debug("[i18n] Français détecté → langue source (aucun QM chargé)")
        else:
            # -----------------------------
            # CAS 2️⃣ : langue système
            # -----------------------------
            if load_qm(lang):
                loaded = True
                debug(f"[i18n] Traduction chargée : {lang}")
            # -----------------------------
            # CAS 3️⃣ : fallback anglais
            # -----------------------------
            elif load_qm("en"):
                loaded = True
                debug("[i18n] Fallback anglais activé")
            else:
                debug("[i18n] Aucune traduction trouvée → français source")
        # -----------------------------
        # Installation du translator
        # -----------------------------
        if loaded:
            QCoreApplication.installTranslator(translator)
            self.translator = translator

        # --- Variables principales ---
        self.utilisateur = os.environ.get("USERNAME") or Path.home().name or "Inconnu"
        self.datejma = datetime.datetime.today().strftime('%d%m%y')
        self.cheminburo = Path.home() / 'Desktop'
        self.cheminform = self.cheminburo / 'Formation_gestion_Crise'
        self.cheminexo = Path('K:/permanence/Exercices')
        self.chemincrise = Path('K:/permanence/Evenements')

        # self.liste_lecteurs = ['K:\\', 'R:\\', 'S:\\', 'T:\\', 'W:\\']
        self.liste_lecteurs = []

        self.disque_ext = 'PERMANENCE_DDT04'
        self.connexion_reseau = False
        self.connexion_disque = False
        self.montage_HDD = 'Gestion_crise_disque_externe.bat'
        self.demontage_HDD = 'Demontage_disque_externe.bat'
        self.projetBASE_RESOURCE = ":/plugins/gestion_crise/resources/Gestion_crise.qgz"

        # --- Défaut partages réseau ---
        self.defaults_partages = {
            'K:\\': r'\\10.4.8.67\dossiers',
            'R:\\': r'\\10.4.8.41\gb_ref',
            'S:\\': r'\\10.4.8.41\gb_prod',
            'T:\\': r'\\10.4.8.41\gb_cons',
        }
        self.partages = dict(self.defaults_partages)
        self._charger_partages()

        # Assertion sécuritaire
        assert all(l.endswith(":") and len(l) == 2 for l in self.get_configured_drive_letters()), \
            "Erreur de configuration: les lettres doivent être au format 'X:' uniquement"

        # --- Charger l'état du DEBUG depuis QSettings ---
        s = QSettings()
        global DEBUG_MODE
        DEBUG_MODE = bool(int(s.value("gestion_crise/debug", "1")))  # "1" par défaut
        debug("Initialisation complète du plugin Gestion_crise.")

    def _on_plugins_removed(self, plugins):
        """
        Appelé lorsque le plugin est désactivé ou supprimé via la fenêtre Extensions.
        'plugins' est une liste des noms des plugins retirés.
        """
        if "gestion_crise" not in plugins:
            return
        # 🔥 Nettoyage complet des icônes GC dans la toolbar QdrawEVT
        if self.toolbar_qdraw:
            # Actions GC
            for a in self.actions:
                try:
                    self.toolbar_qdraw.removeAction(a)
                except Exception:
                    pass
            # Séparateurs GC
            for act in list(self.toolbar_qdraw.actions()):
                if act.property("gc_separator") is True:
                    self.toolbar_qdraw.removeAction(act)
        # Reset interne
        self.actions.clear()
        self._gc_separator_added = False

    # -------------------------------------------------------------------------
    # Initialisation QGIS
    # -------------------------------------------------------------------------
    def initGui(self):
        """Initialise le plugin dans QGIS (sans aucun accès disque/réseau)."""
        icon_main = ':/plugins/gestion_crise/icon.png'
        icon_out = ':/plugins/gestion_crise/icon_out.png'
        try:
            icon_network = QgsApplication.getThemeIcon("mIconNetwork.svg")
            if icon_network.isNull():
                icon_network = QgsApplication.getThemeIcon("mActionOptions.svg")
        except Exception:
            icon_network = QgsApplication.getThemeIcon("mActionOptions.svg")
        # --- Bouton de lancement du plugin ---
        self.add_action(icon_main, self.tr('Lancement projet gestion de crise'),
                        self.run, self.iface.mainWindow(), add_to_toolbar=True)
        # --- Bouton pour démonter le disque externe ---
        self.add_action(icon_out, self.tr('Démonter le disque externe'),
                        self._action_demontage_clicked, self.iface.mainWindow(), add_to_toolbar=True)

        # --- Action : Configuration des partages réseau ---
        self.action_configurer_partages = QAction(
            icon_network,
            self.tr("Configurer les partages réseau"),
            self.iface.mainWindow(),
        )
        self.action_configurer_partages.setObjectName("gc_configurer_partages")
        self.action_configurer_partages.triggered.connect(self.configurer_partages)

        # Ajout de la ligne de commande Configurer les partages réseau au menu Gestion de crise
        self.iface.addPluginToMenu("&Gestion de crise", self.action_configurer_partages)
        # --- Bouton pour définir les partages réseau dans la barre d'outil (à activer en décommentant si besoin) ---
        # self.add_action(icon_network, self.tr('Configurer les partages réseau'),
        #                 self.configurer_partages, self.iface.mainWindow(), add_to_toolbar=False)

        # --- Icône pour la compilation resources.qrc ---
        icon_rc = QIcon(":/plugins/gestion_crise/resources/rc_compile.svg")
        self.action_compile_resources = QAction(
            icon_rc,
            self.tr("Recompiler les ressources (resources.qrc)"),
            self.iface.mainWindow()
        )
        self.action_compile_resources.triggered.connect(self.compiler_resources)
        # Ajout au menu du plugin
        self.iface.addPluginToMenu(self.menu, self.action_compile_resources)
        # Ajout à la barre d’outils (optionnel)
        # self.iface.addToolBarIcon(self.action_compile_resources)

        # --- Action checkable pour activer/désactiver le mode debug ---
        self.action_debug_toggle = QAction(self.tr("Activer le mode débogage"), self.iface.mainWindow())
        self.action_debug_toggle.setCheckable(True)
        self.action_debug_toggle.setChecked(DEBUG_MODE)
        self.action_debug_toggle.toggled.connect(self._toggle_debug_mode)
        self.iface.addPluginToMenu(self.menu, self.action_debug_toggle)
        self.first_start = True
        debug("Interface QGIS initialisée.")
        proj = QgsProject.instance()

        # --- Action : Aide ---
        # Détermination de l'icône native QGIS pour l'aide
        icon_help = QgsApplication.getThemeIcon("mActionHelpContents.svg")
        self.action_aide = QAction(
            icon_help,
            self.tr("Aide / Manuel d'utilisation"),
            self.iface.mainWindow()
        )
        self.action_aide.setObjectName("gc_action_aide")
        self.action_aide.triggered.connect(self.afficher_aide)
        # On ajoute l’action EN BAS du menu
        self.iface.addPluginToMenu(self.menu, self.action_aide)
        # On garde une trace pour unload()
        self.actions.append(self.action_aide)

        # --- signal "cleared" (existe dans toutes les versions) ---
        proj.cleared.connect(self._on_project_cleared)
        # --- Compatibilité signaux de chargement projet ---
        if hasattr(proj, "projectRead"):
            proj.projectRead.connect(self._on_project_loaded)  # QGIS < 3.40
        else:
            # QGIS >= 3.40
            if hasattr(proj, "readProject"):
                proj.readProject.connect(self._on_project_loaded)
            if hasattr(proj, "projectLoaded"):
                proj.projectLoaded.connect(self._on_project_loaded)
        # --- Fermeture QGIS ---
        try:
            QApplication.instance().aboutToQuit.connect(self._auto_demontage_final)
        except Exception:
            pass
        QTimer.singleShot(500, self._reinserer_icones_gc)
        QTimer.singleShot(1000, self._remove_gc_from_extensions)

    def afficher_aide(self):
        """Affiche le fichier d'aide (HTML / Markdown / TXT) dans une boîte de dialogue."""

        # --- 1) Détermination de la langue ---
        lang_full = QLocale.system().name().lower()  # ex: 'fr_fr', 'en_gb'
        lang = lang_full.split("_")[0]  # 'fr', 'en', 'de', ...

        plugin_dir = Path(__file__).resolve().parent
        help_dir = plugin_dir / "help"

        # --- 2) Ordre de recherche des fichiers d'aide ---
        candidats = [
            help_dir / f"readme_{lang}.html",
            help_dir / f"readme_{lang}.htm",
            help_dir / f"readme_{lang}.md",
            help_dir / f"readme_{lang}.txt",
            help_dir / "readme_fr.html",
            help_dir / "readme_fr.htm",
            help_dir / "readme_fr.md",
            help_dir / "readme_fr.txt",
        ]
        fichier = None
        for cand in candidats:
            if cand.is_file():
                fichier = cand
                break
        # Si aucun fichier trouvé : message simple
        if fichier is None:
            QMessageBox.warning(
                self.iface.mainWindow(),
                self.tr("Aide - Gestion_crise"),
                self.tr("Aucun fichier d'aide n'a été trouvé dans le dossier 'help'."),
            )
            return
        # --- 3) Lecture du contenu ---
        try:
            contenu = fichier.read_text(encoding="utf-8")
        except Exception as e:

            QMessageBox.critical(
                self.iface.mainWindow(),
                self.tr("Aide - Gestion_crise"),
                self.tr(
                    "Impossible de lire le fichier d'aide : %1"
                ).replace(
                    "%1", str(e)
                ),
            )

            return
        suffix = fichier.suffix.lower()
        # --- 4) Préparation HTML / Markdown / Texte brut ---
        html = ""
        if suffix in (".html", ".htm"):
            # On suppose que le contenu est déjà du HTML
            html = contenu
        elif suffix == ".md":
            # Conversion très simple Markdown -> HTML (sans dépendance externe)
            # - # Titre -> <h1>, ## -> <h2>, etc.
            # - lignes normales -> <p>
            lines = contenu.splitlines()
            html_lines = []
            for line in lines:
                stripped = line.strip()
                if stripped.startswith("###### "):
                    html_lines.append("<h6>{}</h6>".format(stripped[7:]))
                elif stripped.startswith("##### "):
                    html_lines.append("<h5>{}</h5>".format(stripped[6:]))
                elif stripped.startswith("#### "):
                    html_lines.append("<h4>{}</h4>".format(stripped[5:]))
                elif stripped.startswith("### "):
                    html_lines.append("<h3>{}</h3>".format(stripped[4:]))
                elif stripped.startswith("## "):
                    html_lines.append("<h2>{}</h2>".format(stripped[3:]))
                elif stripped.startswith("# "):
                    html_lines.append("<h1>{}</h1>".format(stripped[2:]))
                elif stripped == "":
                    html_lines.append("<br>")
                else:
                    # on échappe juste les < et > pour éviter de casser le HTML
                    safe = stripped.replace("&", "&amp;").replace("<", "&lt;").replace(">", "&gt;")
                    html_lines.append("<p>{}</p>".format(safe))
            html = "<html><body>{}</body></html>".format("\n".join(html_lines))
        else:
            # TXT : conversion propre en HTML avec wrapping automatique
            safe = contenu.replace("&", "&amp;").replace("<", "&lt;").replace(">", "&gt;")
            safe = safe.replace("\n", "<br>")
            html = f"""
                <html>
                <head>
                    <style>
                        body {{
                            font-family: 'Segoe UI', Arial, sans-serif;
                            font-size: 11pt;
                            line-height: 1.4;
                            white-space: normal;
                            word-wrap: break-word;
                            margin: 8px;
                        }}
                    </style>
                </head>
                <body>{safe}</body>
                </html>
                """
        # --- 5) Création de la boîte de dialogue ---
        dlg = QDialog(self.iface.mainWindow())
        dlg.setWindowTitle(self.tr("Aide - Gestion_crise"))
        dlg.resize(800, 600)

        layout = QVBoxLayout(dlg)

        browser = QTextBrowser()
        browser.setOpenExternalLinks(True)  # liens cliquables vers le web, si présents
        browser.setHtml(html)
        layout.addWidget(browser)
        # --- 6) Boutons en bas (Fermer) ---
        btn_layout = QHBoxLayout()
        btn_layout.addStretch(1)

        btn_close = QPushButton(self.tr("Fermer"))
        btn_close.clicked.connect(dlg.accept)
        btn_layout.addWidget(btn_close)

        layout.addLayout(btn_layout)

        dlg.exec()

    def _reinserer_icones_gc(self):
        """
        Replace proprement TOUTES les actions Gestion_crise dans la toolbar QdrawEVT :
        - supprime actions et séparateurs GC existants
        - réinsère les actions dans le bon ordre
        - recrée un séparateur unique juste avant les actions QdrawEVT
        Fonction appelée via QTimer.singleShot dans initGui().
        """
        import qgis.utils
        qdraw = qgis.utils.plugins.get("qdrawEVT")
        if not qdraw:
            return
        # Récupérer toolbar & anchor
        self.toolbar_qdraw = getattr(qdraw, "get_toolbar", lambda: None)()
        self.gc_anchor = getattr(qdraw, "get_gc_anchor", lambda: None)()

        if not self.toolbar_qdraw or not self.gc_anchor:
            return
        # ---------------------------------------------------
        # 1️⃣ Nettoyer actions GC déjà présentes
        # ---------------------------------------------------
        for action in self.actions:
            try:
                self.toolbar_qdraw.removeAction(action)
            except:
                pass
        # ---------------------------------------------------
        # 2️⃣ Nettoyer les séparateurs GC existants
        # ---------------------------------------------------
        for act in list(self.toolbar_qdraw.actions()):
            try:
                if act.property("gc_separator") is True:
                    self.toolbar_qdraw.removeAction(act)
            except:
                pass
        # ---------------------------------------------------
        # 3️⃣ Réinsérer les actions GC avant gc_anchor
        # ---------------------------------------------------
        for action in self.actions:
            self.toolbar_qdraw.insertAction(self.gc_anchor, action)
        # ---------------------------------------------------
        # 4️⃣ Ajouter un seul séparateur GC → EVT juste avant gc_anchor
        # ---------------------------------------------------
        sep = QAction(self.toolbar_qdraw)
        sep.setSeparator(True)
        sep.setProperty("gc_separator", True)
        self.toolbar_qdraw.insertAction(self.gc_anchor, sep)
        # Flag interne (facultatif)
        self._gc_separator_added = True

    def _remove_gc_from_extensions(self):
        """Supprime toutes les actions GC que QGIS aurait ajoutées dans Extensions."""
        try:
            tb = self.iface.pluginToolBar()  # barre Extensions
        except Exception:
            return
        if not tb:
            return
        for act in list(tb.actions()):
            if act in self.actions:
                tb.removeAction(act)

    def _auto_demontage_final(self):
        """
        Démontage automatique uniquement si un disque externe est monté,
        et uniquement lorsque :
          ✓ QGIS se ferme
          ✓ l’utilisateur ferme vraiment le dernier projet ouvert
        """

        if not getattr(self, "deja_monte", False):
            return

        debug("[AUTO] Démontage automatique du disque externe…")

        try:
            self.demontage_et_remontage_hdd()
            debug("[AUTO] Démontage auto effectué.")
        except Exception as e:
            debug(f"[AUTO] Erreur démontage automatique : {e}")

    def _evaluate_project_close(self):
        """
        Si cleared() n'est PAS suivi d'un projectRead(),
        cela signifie que l’utilisateur ferme réellement le projet
        → on démonte.
        """
        pass
        # if self._loading_new_project:
        #     # Un nouveau projet arrive : PAS DE DEMONTAGE
        #     self._loading_new_project = False
        #     return
        # # Sinon → vrai "fermer le projet"
        # self._auto_demontage_final()

    def _on_project_loaded(self):
        """
        Appelé lors d'un OUVERTURE DE PROJET.
        Donc on ne démontera PAS.
        """
        self._loading_new_project = True

    def _on_project_cleared(self):
        """
        Appelé quand un projet est vidé.
        On attend quelques ms pour vérifier si clear() est suivi d'un projectRead(),
        ce qui signifie : Nouveau projet / Ouvrir projet.
        """
        QTimer.singleShot(200, self._evaluate_project_close)

    # def _toggle_debug_mode(self, checked: bool):
    #     """Active/désactive le mode DEBUG et persiste la valeur."""
    #     global DEBUG_MODE
    #     DEBUG_MODE = bool(checked)
    #     QSettings().setValue("gestion_crise/debug", "1" if checked else "0")
    #     _debug_sink(f"Mode debug {'activé' if checked else 'désactivé'}.")

    def _toggle_debug_mode(self, checked: bool):
        global DEBUG_MODE, print
        DEBUG_MODE = bool(checked)
        QSettings().setValue("gestion_crise/debug", "1" if checked else "0")
        if checked:
            # Restaurer print normal
            import builtins
            print = builtins.print
        else:
            # Redésactiver print
            def _silent_print(*args, **kwargs):
                pass
            print = _silent_print
        _debug_sink(f"Mode debug {'activé' if checked else 'désactivé'}.")

    def unload(self):
        """
        Désactivation complète du plugin Gestion_crise :
        - supprime les actions du menu
        - supprime les icônes de la toolbar QdrawEVT
        - supprime le séparateur GC→EVT
        - supprime les actions éventuellement tombées dans la barre Extensions
        - nettoie l’état interne
        """
        # -----------------------------------------------------------
        # 1) Supprimer les actions du menu QGIS
        # -----------------------------------------------------------
        for action in self.actions:
            try:
                self.iface.removePluginMenu(self.menu, action)
            except Exception:
                pass

        # Action debug (si présente)
        if getattr(self, "action_debug_toggle", None):
            try:
                self.iface.removePluginMenu(self.menu, self.action_debug_toggle)
            except Exception:
                pass

        # Action compile resources (si présente)
        if getattr(self, "action_compile_resources", None):
            try:
                self.iface.removePluginMenu(self.menu, self.action_compile_resources)
            except Exception:
                pass

        # Action configurer_partages (nouvelle action)
        if getattr(self, "action_configurer_partages", None):
            try:
                self.iface.removePluginMenu(self.menu, self.action_configurer_partages)
            except Exception:
                pass

        # -----------------------------------------------------------
        # 2) Nettoyer la toolbar QdrawEVT si disponible
        # -----------------------------------------------------------
        if getattr(self, "toolbar_qdraw", None):

            # a) Supprimer toutes les actions GC
            for action in self.actions:
                try:
                    self.toolbar_qdraw.removeAction(action)
                except Exception:
                    pass

            # c) Supprimer l’action configurer_partages de la toolbar QdrawEVT
            if getattr(self, "action_configurer_partages", None):
                try:
                    self.toolbar_qdraw.removeAction(self.action_configurer_partages)
                except Exception:
                    pass

            # Action aide (si présente)
            if getattr(self, "action_aide", None):
                try:
                    self.iface.removePluginMenu(self.menu, self.action_aide)
                except Exception:
                    pass

            # b) Supprimer les séparateurs de GC
            try:
                for act in list(self.toolbar_qdraw.actions()):
                    if act.property("gc_separator") is True:
                        self.toolbar_qdraw.removeAction(act)
            except Exception:
                pass

        # -----------------------------------------------------------
        # 3) Fallback : retirer aussi des toolbars QGIS classiques
        #    (barre Extensions, barre Principale, etc.)
        # -----------------------------------------------------------
        for action in self.actions:
            try:
                self.iface.removeToolBarIcon(action)
            except Exception:
                pass

        # Action configurer_partages dans fallback toolbars
        if getattr(self, "action_configurer_partages", None):
            try:
                self.iface.removeToolBarIcon(self.action_configurer_partages)
            except Exception:
                pass

        if getattr(self, "action_aide", None):
            try:
                self.iface.removeToolBarIcon(self.action_aide)
            except Exception:
                pass
        # -----------------------------------------------------------
        # 4) Nettoyer l’état interne
        # -----------------------------------------------------------
        self.actions.clear()
        self._gc_separator_added = False

    # -------------------------------------------------------------------------
    # Fonctions utilitaires
    # -------------------------------------------------------------------------
    def tr(self, message):
        return QCoreApplication.translate('Gestion_crise', message)

    def monter_disque_externe_python_winAPI(self, disque_source: str) -> bool:
        """
        Monte le disque externe en utilisant la lettre détectée par
        _try_mount_external_disk().
        """

        if not disque_source:
            QMessageBox.critical(None, self.tr("Erreur"), self.tr("Aucun disque spécifié."))
            return False

        # Normalisation du lecteur source
        disque_source = self.normalize_drive_letter(disque_source.rstrip("\\"))
        base = Path(disque_source + "\\")

        if not base.exists():
            QMessageBox.critical(
                None,
                self.tr("Erreur"),
                self.tr(
                    "Le disque %1 n'existe pas."
                ).replace(
                    "%1", str(disque_source)
                )
            )

            return False

        # =====================================================================
        # 1️⃣ Suppression des montages existants SUBST et NET USE
        # =====================================================================
        lettres = self.get_configured_drive_letters()
        messages = []

        def supprimer_montage(drive: str):
            """Supprime NET USE ou SUBST sur une lettre donnée."""
            drive = self.normalize_drive_letter(drive)

            # Suppression réseau
            try:
                conns, _, _ = win32net.NetUseEnum(None, 0)
                for c in conns:
                    if c["local"].upper() == drive.upper():
                        win32net.NetUseDel(None, drive)
                        return "réseau supprimé"
            except Exception:
                pass

            # Suppression SUBST
            try:
                target = win32file.QueryDosDevice(drive[0])  # DOIT être une seule lettre
                if target.startswith("\\??\\"):
                    win32file.DefineDosDevice(
                        win32con.DDD_REMOVE_DEFINITION,
                        drive[0],  # une seule lettre !
                        None
                    )
                    return "SUBST supprimé"
            except Exception:
                pass

            return "aucune action"

        for l in lettres:
            lnorm = self.normalize_drive_letter(l)
            messages.append(f"{lnorm} : {supprimer_montage(lnorm)}")

        # =====================================================================
        # 2️⃣ Création des SUBST à partir des partages configurés
        # =====================================================================
        substitutions = {}

        for lecteur, chemin_unc in self.partages.items():
            lecteur_norm = self.normalize_drive_letter(lecteur)
            dossier = self._extract_folder_name_from_unc(chemin_unc)

            if not dossier:
                messages.append(f"{lecteur_norm} → ERREUR : dossier introuvable dans {chemin_unc}")
                continue

            substitutions[lecteur_norm] = base / dossier

        for lecteur, dossier in substitutions.items():
            try:
                if not dossier.exists():
                    messages.append(f"{lecteur} → ERREUR : dossier introuvable : {dossier}")
                    continue

                # lecteur est déjà normalisé → pas besoin de retraiter
                win32file.DefineDosDevice(
                    0,
                    lecteur[0],  # une seule lettre !
                    str(dossier)
                )
                messages.append(f"{lecteur} → {dossier}")

            except Exception as e:
                messages.append(f"{lecteur} → ERREUR : {e}")

        # =====================================================================
        # 3️⃣ Résultat final
        # =====================================================================
        QMessageBox.information(
            None,
            self.tr("Montage terminé"),
            self.tr(
                "[SUCCÈS] Montage terminé depuis %1\n\n%2"
            ).replace(
                "%1", disque_source
            ).replace(
                "%2", "\n".join(messages)
            )        )
        return True

    def add_action(self, icon_path, text, callback, parent=None, add_to_toolbar=True, enabled=True):
        """
        Ajoute proprement une action GC :
          - au menu
          - à la toolbar QdrawEVT si disponible
          - sinon dans la barre Extensions
        """
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(bool(enabled))

        # Menu du plugin
        self.iface.addPluginToMenu(self.menu, action)

        # Placement dans la toolbar
        if add_to_toolbar:
            if self.toolbar_qdraw and self.gc_anchor:
                # Ajouter action GC avant l’ancre
                self.toolbar_qdraw.insertAction(self.gc_anchor, action)

                # Ajouter le séparateur une seule fois
                if not self._gc_separator_added:
                    sep = QAction(self.toolbar_qdraw)
                    sep.setSeparator(True)
                    sep.setProperty("gc_separator", True)
                    self.toolbar_qdraw.insertAction(self.gc_anchor, sep)
                    self._gc_separator_added = True
            # ⚠️ PLUS AUCUN FALLBACK ICI !
            # On ne veut PLUS que QGIS place les actions GC ailleurs
            # => on ne met RIEN dans Extensions.
            # else:
            #     # Fallback : barre Extensions
            #     self.iface.addToolBarIcon(action)

        self.actions.append(action)
        return action

    def _action_demontage_clicked(self):
        """Callback du bouton 'icon_out' dans la barre d’outils."""
        # version avec fenêtre cmd
        # self._demonter_disque_externe()
        # Version python
        self.demontage_et_remontage_hdd()

    def demontage_et_remontage_hdd(self):
        # 🔧 Récupération dynamique des lecteurs configurés
        lettres = self.get_configured_drive_letters()
        messages = []

        # 1️⃣ DEMONTAGE (normalisé)
        for lettre in lettres:
            lnorm = self.normalize_drive_letter(lettre)
            self._supprimer_montage(lnorm)

        # 2️⃣ VÉRIFIER SI TOUT EST BIEN DÉMONTÉ — normaliser AVANT le test
        encore_monte = [
            self.normalize_drive_letter(l)
            for l in lettres
            if self._lecteur_existe(self.normalize_drive_letter(l))
        ]

        if encore_monte:
            messages.append(
                self.tr(
                    "Attention : ces lecteurs existent encore : %1"
                ).replace(
                    "%1", ", ".join(encore_monte)
                )
            )
            messages.append(
                self.tr("Impossible de finaliser le démontage.")
            )

            QMessageBox.warning(
                None,
                self.tr("Démontage incomplet"),
                "\n".join(messages)
            )
            return
        else:
            messages.append(
                self.tr("Tous les montages ont été correctement démontés.")
            )
            self.deja_monte = False  # 🔓 Libérer le flag

        # 3️⃣ Vérifier disponibilité serveurs
        if not self._serveurs_disponibles():
            messages.append(
                self.tr("Serveurs indisponibles.")
            )

            QMessageBox.information(
                None,
                self.tr("Résultat"),
                "\n".join(messages)
            )
            return

        rep = QMessageBox.question(
            None,
            self.tr("Réseau interne"),
            self.tr("Voulez-vous remonter le réseau interne ?"),
            QMessageBox.Yes | QMessageBox.No
        )

        # 4️⃣ REMONTAGE RÉSEAU — 100% dynamique
        if rep == QMessageBox.Yes:
            for lecteur, chemin_unc in self.partages.items():
                lettre_norm = self.normalize_drive_letter(lecteur)

                cmd = [
                    "net", "use",
                    lettre_norm,
                    chemin_unc,
                    "/persistent:no"
                ]

                subprocess.run(
                    cmd,
                    stdout=subprocess.DEVNULL,
                    stderr=subprocess.DEVNULL,
                    creationflags=subprocess.CREATE_NO_WINDOW
                )

        QMessageBox.information(
            None,
            self.tr("Résultat"),
            "\n".join(messages)
        )

    def _demonter_disque_externe(self):
        """Lance le batch dans une fenêtre CMD visible, sans paramètre."""
        bat_path = self.plugin_dir / "resources" / "Demonter_disque_externe.bat"
        if not bat_path.exists():
            QMessageBox.information(
                None,
                self.tr("Information"),
                self.tr(
                    "Script batch introuvable : %1"
                ).replace(
                    "%1", str(bat_path)
                )
            )

        # Ouvre une nouvelle fenêtre CMD et lance directement le batch
        subprocess.Popen(
            ["cmd.exe", "/K", str(bat_path)],
            creationflags=subprocess.CREATE_NEW_CONSOLE)

    def _lecteur_existe(self, lettre: str) -> bool:
        """Retourne True si une lettre est réellement présente dans le système."""
        try:
            return Path(lettre + "\\").exists()
        except Exception:
            return False

    def _supprimer_montage(self, lettre: str):
        """Supprime SUBST ou NET USE pour une lettre donnée."""
        lettre = self.normalize_drive_letter(lettre)
        try:
            subprocess.run(
                ["subst", lettre, "/D"],
                stdout=subprocess.DEVNULL,
                stderr=subprocess.DEVNULL,
                creationflags=subprocess.CREATE_NO_WINDOW
            )
        except Exception:
            pass
        try:
            subprocess.run(
                ["net", "use", lettre, "/delete", "/yes"],
                stdout=subprocess.DEVNULL,
                stderr=subprocess.DEVNULL,
                creationflags=subprocess.CREATE_NO_WINDOW
            )
        except Exception:
            pass

    # -------------------------------------------------------------------------
    # Gestion des partages réseau
    # -------------------------------------------------------------------------
    def _charger_partages(self):
        """Charge la configuration des partages depuis QSettings."""
        s = QSettings()
        keys = [k for k in s.allKeys() if k.startswith("gestion_crise/partage_")]
        if not keys:
            self.partages = dict(self.defaults_partages)
            for lecteur, chemin in self.partages.items():
                s.setValue(f"gestion_crise/partage_{lecteur}", chemin)
        else:
            self.partages.clear()
            for k in keys:
                lecteur = k.replace("gestion_crise/partage_", "")
                self.partages[lecteur] = s.value(k)
        debug(f"Partages réseau chargés : {self.partages}")
        self.liste_lecteurs = self.get_configured_drive_letters()
        debug(f"Lettres configurées : {self.liste_lecteurs}")

    def _ping_ip(self, ip: str) -> bool:
        """Retourne True si l’IP répond au ping Windows (safe depuis un thread)."""
        if not ip:
            return False

        try:
            creationflags = 0
            if hasattr(subprocess, "CREATE_NO_WINDOW"):
                creationflags = subprocess.CREATE_NO_WINDOW

            result = subprocess.run(
                ["ping", "-n", "1", "-w", "250", ip],  # timeout 250 ms
                stdout=subprocess.DEVNULL,
                stderr=subprocess.DEVNULL,
                creationflags=creationflags,
                check=False,
            )
            return result.returncode == 0
        except Exception as e:
            debug(f"[PING] Erreur ping {ip} : {e}")
            return False

    def _extract_folder_name_from_unc(self, chemin: str) -> str:
        """
        Extrait proprement le dernier dossier d’un UNC.
        Exemple : \\10.4.8.41\gb_ref   → gb_ref
                  \\10.4.8.41\dossier\sig → sig
        Jamais d’accès au réseau.
        """
        if not isinstance(chemin, str) or not chemin.startswith("\\\\"):
            return ""

        chemin = chemin.replace("/", "\\").rstrip("\\")
        parts = [p.strip() for p in chemin.split("\\") if p.strip()]

        # UNC minimal = \\srv\share → parts = [srv, share]
        if len(parts) < 2:
            return ""

        return parts[-1].lower()

    def _serveurs_disponibles(self) -> bool:
        """
        Teste uniquement la disponibilité des IP des serveurs.
        NE TOUCHER À AUCUN CHEMIN UNC.
        Renvoie True dès qu'une IP répond (ping rapide).
        """

        # ---- 1️⃣ Extraction des IP (sans lire UNC) ----
        ips = set()
        for chemin in list(self.defaults_partages.values()):  # éviter tout accès réseau inutile
            if isinstance(chemin, str) and chemin.startswith("\\\\"):
                try:
                    ip = chemin[2:].split("\\")[0].strip()
                    if ip:
                        ips.add(ip)
                except Exception:
                    pass

        # ---- 2️⃣ Ping rapide sur chaque IP ----
        for ip in ips:
            debug(f"[IP] Test ping {ip}…")
            if self._ping_ip(ip):
                debug(f"[IP] {ip} répond ✔")
                return True
            debug(f"[IP] {ip} ne répond pas ✖")

        debug("[IP] Aucun serveur n’est disponible.")
        return False

    def _sauver_partages(self):
        """Sauvegarde la configuration des partages dans QSettings."""
        s = QSettings()
        for k in [k for k in s.allKeys() if k.startswith("gestion_crise/partage_")]:
            s.remove(k)
        for lecteur, chemin in self.partages.items():
            s.setValue(f"gestion_crise/partage_{lecteur}", chemin)
        debug("Partages réseau sauvegardés.")

    def configurer_partages(self):
        """Boîte de dialogue de configuration des partages réseau."""

        # 🔥 1 — Affichage immédiat du throbber avant toute action
        th = ThrobberProgressDialog(self.iface.mainWindow())
        gif = ":/plugins/gestion_crise/resources/chrono.gif"
        th.update_gif(gif)
        th.update_status(self.tr("Ouverture de la configuration…"))
        th.pbar.setMaximum(0)
        th.show()
        QApplication.processEvents()  # indispensable pour lancer l’animation !

        dlg = QDialog(self.iface.mainWindow())
        dlg.setWindowTitle(self.tr("Configuration des partages réseau"))
        dlg.setMinimumWidth(750)

        main_layout = QVBoxLayout(dlg)
        grid = QGridLayout()
        main_layout.addLayout(grid)

        edits, labels_status = {}, {}

        # -------------------------------
        # FLAG "dirty" pour modifications
        # -------------------------------
        modifications = {"dirty": False}

        def set_dirty():
            modifications["dirty"] = True

        # ---------------
        # VALIDATIONS
        # ---------------
        def lettre_valide(lecteur: str) -> bool:
            if not isinstance(lecteur, str):
                return False
            l = lecteur.replace("\\", "").upper()
            return len(l) == 2 and l[0].isalpha() and l.endswith(":")

        def unc_valide(chemin: str) -> bool:
            if not isinstance(chemin, str):
                return False
            if not chemin.startswith("\\\\"):
                return False
            parts = [p for p in chemin.split("\\") if p.strip()]
            return len(parts) >= 2  # \\server\share

        # ---------------
        # COLORATION LIVE
        # ---------------
        def update_field_color(lecteur):
            edit = edits.get(lecteur)
            lbl = labels_status.get(lecteur)
            chemin = edit.text().strip()

            if not unc_valide(chemin):
                edit.setStyleSheet("background-color: rgb(255, 200, 200);")  # rouge clair
                lbl.setText(self.tr("🔴"))
                lbl.setStyleSheet("color: red; font-weight: bold;")
                return

            try:
                ip = chemin[2:].split("\\")[0].strip()
                if self._ping_ip(ip):
                    edit.setStyleSheet("background-color: rgb(200, 255, 200);")  # vert clair
                    lbl.setText(self.tr("🟢"))
                    lbl.setStyleSheet("color: green; font-weight: bold;")
                else:
                    edit.setStyleSheet("background-color: rgb(255, 230, 180);")  # orange clair
                    lbl.setText(self.tr("🟠"))
                    lbl.setStyleSheet("color: orange; font-weight: bold;")
            except Exception:
                edit.setStyleSheet("background-color: rgb(255, 200, 200);")
                lbl.setText(self.tr("🔴"))

        # -------------------------------
        # RECONSTRUCTION DE LA GRILLE
        # -------------------------------
        def refresh_grid():
            th = None
            if len(self.partages) > 20:  # seuil ajustable
                th = show_throbber("Actualisation de l’affichage…")

            for i in reversed(range(grid.count())):
                w = grid.itemAt(i).widget()
                if w:
                    w.setParent(None)

            edits.clear()
            labels_status.clear()

            # 🔥 TRI AUTOMATIQUE DES LETTRES
            partages_tries = dict(sorted(self.partages.items(), key=lambda kv: kv[0]))

            for row, (lecteur, chemin) in enumerate(partages_tries.items()):
                lbl = QLabel(f"{lecteur} →")
                edit = QLineEdit(chemin)
                lbl_status = QLabel("")
                btn_remove = QPushButton(self.tr("✖"))
                btn_remove.setFixedWidth(30)

                edits[lecteur] = edit
                labels_status[lecteur] = lbl_status

                # coloration + flag dirty
                def on_change(_, key=lecteur):
                    set_dirty()
                    update_field_color(key)

                edit.textChanged.connect(on_change)

                grid.addWidget(lbl, row, 0)
                grid.addWidget(edit, row, 1)
                grid.addWidget(lbl_status, row, 2)
                grid.addWidget(btn_remove, row, 3)

                btn_remove.clicked.connect(lambda _, key=lecteur: supprimer_partage(key))

                update_field_color(lecteur)

            if th:
                th.close()

        # -------------------------------
        # SUPPRESSION
        # -------------------------------
        def supprimer_partage(lecteur):
            if lecteur in self.partages:
                del self.partages[lecteur]
                set_dirty()
                refresh_grid()

        # -------------------------------
        # AJOUT SÉCURISÉ AVEC ANTI-DOUBLON
        # -------------------------------
        def ajouter_partage():
            # Demande brute à l'opérateur
            lecteur_raw, ok = QInputDialog.getText(
                dlg, "Nouveau partage",
                "Lettre du lecteur (une seule lettre, ex : k ou K) :"
            )
            if not ok or not lecteur_raw.strip():
                return

            # Nettoyage
            lecteur_raw = lecteur_raw.strip()

            # 🔥 Vérifier que c'est EXACTEMENT UNE lettre (minuscule OU majuscule)
            if len(lecteur_raw) != 1 or not lecteur_raw.isalpha():
                QMessageBox.critical(
                    dlg, "Erreur",
                    "Vous devez saisir exactement UNE lettre (minuscule ou majuscule, ex : k ou K)."
                )
                return

            # 🔥 Conversion automatique → MAJ + :\
            lecteur = lecteur_raw.upper() + ":\\"

            # 🔥 ANTI-DOUBLON : la lettre existe déjà
            if lecteur in self.partages:
                rep = QMessageBox.question(
                    dlg,
                    "Doublon détecté",
                    f"Le lecteur {lecteur} existe déjà.\n"
                    f"Chemin actuel : {self.partages[lecteur]}\n\n"
                    "Voulez-vous le remplacer ?",
                    QMessageBox.Yes | QMessageBox.No
                )
                if rep == QMessageBox.No:
                    return

            # Saisie du chemin UNC
            chemin, ok2 = QInputDialog.getText(
                dlg, "Chemin UNC",
                "Chemin UNC (ex : \\\\serveur\\partage) :"
            )
            if not ok2 or not chemin.strip():
                return

            chemin = chemin.strip()

            # Validation UNC
            if not unc_valide(chemin):
                QMessageBox.critical(
                    dlg, "Erreur",
                    f"Le chemin UNC '{chemin}' est invalide."
                )
                return

            # 🔥 Ajout ou remplacement
            self.partages[lecteur] = chemin
            set_dirty()
            refresh_grid()

        # -------------------------------
        # UTILITAIRE : throbber contextuel
        # -------------------------------
        def show_throbber(message="Traitement en cours…"):
            th = ThrobberProgressDialog(self.iface.mainWindow())
            gif = ":/plugins/gestion_crise/resources/chrono.gif"
            th.update_gif(gif)
            th.update_status(message)
            th.pbar.setMaximum(0)  # mode indéterminé
            th.show()
            QApplication.processEvents()
            return th

        # -------------------------------
        # IMPORT AUTOMATIQUE VIA NET USE
        # -------------------------------
        def importer_partages_windows():

            th = show_throbber("Recherche des partages Windows…")

            try:
                output = subprocess.check_output(
                    "net use", shell=True, encoding="cp850"
                )
            except Exception as e:
                th.close()
                QMessageBox.critical(dlg, "Erreur", f"Impossible d’exécuter net use :\n{e}")
                return

            trouves = {}

            for line in output.splitlines():
                parts = line.strip().split()
                if len(parts) >= 2 and parts[0].endswith(":") and parts[1].startswith("\\\\"):
                    lecteur = parts[0].upper() + "\\"
                    unc = parts[1]
                    if lettre_valide(lecteur) and unc_valide(unc):
                        trouves[lecteur] = unc

            th.close()

            if not trouves:
                QMessageBox.information(
                    dlg,
                    self.tr("Aucun partage"),
                    self.tr(
                        "Aucun lecteur réseau valide n'a été détecté via la commande net use."
                    )
                )
                return

            # ------------------------------------------------------------------
            # Message de confirmation (100 % traduisible)
            # ------------------------------------------------------------------
            details = "\n".join(
                f"{k} → {v}" for k, v in trouves.items()
            )

            msg = self.tr(
                "Les partages suivants ont été trouvés :\n\n"
                "%1\n\n"
                "Voulez-vous les importer ?"
            ).replace("%1", details)

            rep = QMessageBox.question(
                dlg,
                self.tr("Import des partages Windows"),
                msg,
                QMessageBox.Yes | QMessageBox.No
            )

            if rep == QMessageBox.No:
                return

            # ------------------------------------------------------------------
            # Throbber (texte traduit)
            # ------------------------------------------------------------------
            th = show_throbber(
                self.tr("Importation des partages Windows en cours…")
            )

            for lecteur, unc in trouves.items():
                self.partages[lecteur] = unc

            set_dirty()
            refresh_grid()

            th.close()

        # -------------------------------
        # TEST DES PARTAGES
        # -------------------------------
        def tester_partages():
            for lecteur in edits:
                update_field_color(lecteur)

        # -------------------------------
        # REINITIALISATION
        # -------------------------------
        def reinitialiser():
            rep = QMessageBox.question(
                dlg,
                self.tr("Réinitialiser"),
                self.tr("Voulez-vous restaurer les partages par défaut ?"),
                QMessageBox.Yes | QMessageBox.No
            )
            if rep != QMessageBox.Yes:
                return

            # 🔥 Affichage du throbber pendant la réinitialisation
            th = show_throbber(
                self.tr("Réinitialisation des partages en cours…")
            )

            # 1) Remise à zéro des partages réseau
            self.partages = dict(self.defaults_partages)

            # 2) Sauvegarde dans QSettings
            self._sauver_partages()

            # 3) Le panneau n’a plus de modification en attente
            modifications["dirty"] = False

            # 4) Reconstruction de la grille
            refresh_grid()

            # 5) Fermeture du throbber AVANT le QMessageBox final
            th.close()

            # 6) Message final à l’utilisateur
            QMessageBox.information(
                dlg,
                self.tr("Réinitialisation terminée"),
                self.tr("Les partages ont été restaurés.")
            )

        # -------------------------------
        # ENREGISTRER
        # -------------------------------
        def enregistrer():
            # 🔥 Affiche le throbber pendant la sauvegarde
            th = show_throbber(
                self.tr("Enregistrement de la configuration en cours…")
            )

            # 1) Reconstruction propre des partages depuis les champs de saisie
            self.partages = {
                lecteur: edits[lecteur].text().strip()
                for lecteur in edits
            }

            # 2) Sauvegarde dans QSettings
            self._sauver_partages()

            # 3) Marquer l'état comme non modifié
            modifications["dirty"] = False

            # 4) Actualisation visuelle si nécessaire
            refresh_grid()

            # 5) Fermer le throbber AVANT d'afficher le message final
            th.close()

            # 6) Confirmation utilisateur
            QMessageBox.information(
                dlg,
                self.tr("Partages enregistrés"),
                self.tr("La configuration a bien été sauvegardée.")
            )

            # 7) Fermeture de la fenêtre de configuration
            dlg.accept()

        # -------------------------------
        # ANNULER
        # -------------------------------
        def annuler():
            if modifications["dirty"]:
                rep = QMessageBox.question(
                    dlg,
                    self.tr("Annuler"),
                    self.tr(
                        "Des modifications n'ont pas été enregistrées.\n"
                        "Voulez-vous vraiment quitter ?"
                    ),
                    QMessageBox.Yes | QMessageBox.No
                )
                if rep == QMessageBox.No:
                    return
            dlg.reject()

        # -------------------------------
        # INIT UI
        # -------------------------------
        refresh_grid()

        # -------------------------------
        # Boutons
        # -------------------------------
        hbox = QHBoxLayout()

        btn_add = QPushButton(self.tr("+ Ajouter"))
        btn_add.clicked.connect(ajouter_partage)
        hbox.addWidget(btn_add)

        btn_import = QPushButton(self.tr("Importer partages Windows"))
        btn_import.clicked.connect(importer_partages_windows)
        hbox.addWidget(btn_import)

        btn_test = QPushButton(self.tr("Tester"))
        btn_test.clicked.connect(tester_partages)
        hbox.addWidget(btn_test)

        btn_reset = QPushButton(self.tr("Réinitialiser"))
        btn_reset.clicked.connect(reinitialiser)
        hbox.addWidget(btn_reset)

        btn_save = QPushButton(self.tr("Enregistrer"))
        btn_save.clicked.connect(enregistrer)
        hbox.addWidget(btn_save)

        btn_cancel = QPushButton(self.tr("Annuler"))
        btn_cancel.clicked.connect(annuler)
        hbox.addWidget(btn_cancel)

        main_layout.addLayout(hbox)

        # Fermer le throbber
        if th:
            th.close()

        dlg.exec()

    # -------------------------------------------------------------------------
    # Détection et montage du disque externe
    # -------------------------------------------------------------------------
    def _try_mount_external_disk(self) -> bool:
        if platform.system().lower() != "windows":
            debug("Non-Windows : montage disque externe ignoré.")
            return False

        # -------------------------------------------------------
        # Liste des lecteurs actuels
        # -------------------------------------------------------
        try:
            lecteurs = [d for d in win32api.GetLogicalDriveStrings().split("\x00") if d]
            debug(f"Lecteurs détectés : {lecteurs}")
        except Exception as e:
            debug(f"Erreur GetLogicalDriveStrings : {e}")
            return False

        # -------------------------------------------------------
        # 1️⃣ Extraction des noms attendus depuis la CONFIG utilisateur
        # -------------------------------------------------------
        noms_attendus = {
            self._extract_folder_name_from_unc(chemin)
            for chemin in self.partages.values()
            if isinstance(chemin, str) and chemin.startswith("\\\\")
        }

        noms_attendus = {n for n in noms_attendus if n}  # élimine les vides

        debug(f"Noms attendus sur disque externe (normalisés) : {noms_attendus}")

        # -------------------------------------------------------
        # 2️⃣ Recherche du disque externe
        # -------------------------------------------------------
        for drive in lecteurs:
            try:
                drive_type = win32file.GetDriveType(drive)
                if drive_type not in (win32con.DRIVE_FIXED, win32con.DRIVE_REMOVABLE):
                    continue

                base = Path(drive)
                volume_label = win32api.GetVolumeInformation(drive)[0]
                debug(f"Test du lecteur {drive} (label={volume_label})")

                # -------------------------------------------------------
                # A. Détection par label exact
                # -------------------------------------------------------
                if volume_label == self.disque_ext:
                    debug(f"Disque externe trouvé par label : {drive}")
                    self.stop_network_check.set()
                    return self.monter_disque_externe_python(drive)

                # -------------------------------------------------------
                # B. Détection par structure : tous les dossiers attendus doivent être présents
                # -------------------------------------------------------
                try:
                    dossiers_presents = {
                        p.name.strip().lower()
                        for p in base.iterdir()
                        if p.is_dir()
                    }
                except Exception as e:
                    debug(f"Impossible de lister {drive} : {e}")
                    continue

                debug(f"Dossiers présents {drive}: {dossiers_presents}")

                if noms_attendus.issubset(dossiers_presents):
                    debug(f"Disque externe trouvé par structure : {drive}")
                    self.stop_network_check.set()
                    return self.monter_disque_externe_python(drive)

            except Exception as e:
                debug(f"Erreur sur {drive}: {e}")
                continue

        debug("Aucun disque externe valide trouvé.")
        return False

    def _monter_disque_externe(self, drive: str) -> bool:
        """Monte le disque externe et met à jour l’interface sans bloquer QGIS."""
        try:
            bat_path = self.plugin_dir / "resources" / "Gestion_crise_disque_externe.bat"
            if not bat_path.exists():
                debug(f"Script batch introuvable : {bat_path}")
                return False

            drive_arg = drive.strip("\\")
            cmd = f'start /MIN cmd.exe /C "{bat_path}" {drive_arg}'
            debug(f"Lancement du batch de montage : {cmd}")

            # --- lancement du processus ---
            p = subprocess.Popen(
                cmd,
                shell=True,
                stdin=subprocess.DEVNULL,
                stdout=subprocess.DEVNULL,
                stderr=subprocess.DEVNULL,
                close_fds=True,
                creationflags=subprocess.DETACHED_PROCESS | subprocess.CREATE_NEW_PROCESS_GROUP
            )

            # --- suppression du warning ResourceWarning ---
            threading.Thread(target=p.wait, daemon=True).start()

            time.sleep(2)
            self.connexion_disque = True

            # ✅ TEXTE UI i18n-compliant
            self.dialogui.label_connexion.setText(
                self.tr("Connexion disque externe (%1)").replace("%1", drive_arg)
            )

            self.dialogui.label_connexion.setStyleSheet(
                'font: 75 12pt "Arial"; background-color: rgb(0, 255, 0);'
            )
            self.dialogui.buttonE.setEnabled(True)
            self.dialogui.buttonC.setEnabled(True)
            self.dialogui.buttonF.setEnabled(True)

            return True

        except Exception as e:
            debug(f"Erreur montage disque externe : {e}")
            return False

    def monter_disque_externe_python(self, drive: str) -> bool:
        """
        Version optimisée du montage SUBST :
        - purge complète
        - SUBST déterministes
        - aucune fenêtre console
        - synchronisation throbber
        """
        try:
            # Normalisation robuste du lecteur source
            drive = self.normalize_drive_letter(drive)
            base = Path(drive + "\\")

            if not base.exists():
                debug(f"[SUBST] ERREUR : le disque {drive} n'existe pas.")
                return False

            debug(f"[SUBST] Montage du disque externe {drive}")

            # Lettres configurées, déjà normalisées
            lettres = [self.normalize_drive_letter(k) for k in self.partages.keys()]

            # -----------------------------
            # 1️⃣ Purge complète
            # -----------------------------
            def purge(lettre_brute):
                lettre = self.normalize_drive_letter(lettre_brute)

                # SUBST /D
                try:
                    subprocess.run(
                        ["subst", lettre, "/D"],
                        stdout=subprocess.DEVNULL,
                        stderr=subprocess.DEVNULL,
                        creationflags=subprocess.CREATE_NO_WINDOW
                    )
                except Exception:
                    pass

                # NET USE /delete
                try:
                    subprocess.run(
                        ["net", "use", lettre, "/delete", "/yes"],
                        stdout=subprocess.DEVNULL,
                        stderr=subprocess.DEVNULL,
                        creationflags=subprocess.CREATE_NO_WINDOW
                    )
                except Exception:
                    pass

            # Exécution purge
            for l in lettres:
                purge(l)

            # -----------------------------
            # 2️⃣ Création SUBST → basé sur dossier final UNC
            # -----------------------------
            for lecteur_brut, chemin_unc in self.partages.items():

                lecteur = self.normalize_drive_letter(lecteur_brut)

                dossier = self._extract_folder_name_from_unc(chemin_unc)
                if not dossier:
                    debug(f"[SUBST] {lecteur} → ERREUR : aucun nom de dossier final trouvé dans {chemin_unc}")
                    continue

                dossier_path = base / dossier

                if not dossier_path.exists():
                    debug(f"[SUBST] {lecteur} → dossier introuvable : {dossier_path}")
                    continue

                debug(f"[SUBST] {lecteur} → {dossier_path}")

                subprocess.run(
                    ["subst", lecteur, str(dossier_path)],
                    stdout=subprocess.DEVNULL,
                    stderr=subprocess.DEVNULL,
                    creationflags=subprocess.CREATE_NO_WINDOW
                )

            # -----------------------------
            # 3️⃣ UI + throbber
            # -----------------------------
            time.sleep(1.0)

            self.connexion_disque = True
            self.deja_monte = True

            self.dialogui.label_connexion.setText(
                self.tr("Connexion disque externe (%1)").replace(
                    "%1", str(drive)
                )
            )

            self.dialogui.label_connexion.setStyleSheet(
                'font: 75 12pt "Arial"; background-color: rgb(0, 255, 0);'
            )
            self.dialogui.buttonE.setEnabled(True)
            self.dialogui.buttonC.setEnabled(True)
            self.dialogui.buttonF.setEnabled(True)

            debug("[SUBST] Montage effectué avec succès.")
            return True

        except Exception as e:
            debug(f"[SUBST] ERREUR montage : {e}")
            return False

    #  Normaliser les lettres de lecteur
    @staticmethod
    def normalize_drive_letter(lettre: str) -> str:
        """
        Normalise toutes les formes possibles de lettres de lecteur Windows :
        - 'K' → 'K:'
        - 'K:' → 'K:'
        - 'K:\' → 'K:'
        - 'K:\\\\' → 'K:'
        - 'k:' → 'K:'

        Retourne une lettre strictement au format Windows SUBST.
        """
        if not isinstance(lettre, str):
            raise ValueError("Lettre de lecteur invalide")

        # retirer tout ce qui est slash/backslash
        lettre = lettre.replace("/", "").replace("\\", "").upper()

        # si fini par :, OK, sinon ajout
        if not lettre.endswith(":"):
            lettre = lettre + ":"

        # forcer 1 seule lettre avant :
        if len(lettre) != 2 or not lettre[0].isalpha():
            raise ValueError(f"Lettre de lecteur invalide : {lettre}")

        return lettre

    # -------------------------------------------------------------------------
    # Détection réseau
    # -------------------------------------------------------------------------
    def detect_lecteurs_reseau(self):
        lecteurs = []
        try:
            output = subprocess.check_output("net use", shell=True, encoding="cp850", stderr=subprocess.DEVNULL)
            for line in output.splitlines():
                parts = line.strip().split()
                if len(parts) >= 2 and parts[0].endswith(":") and parts[1].startswith("\\\\"):
                    lecteurs.append(parts[0].upper() + "\\")
            debug(f"Lecteurs réseau détectés : {lecteurs}")
        except Exception as e:
            debug(f"Erreur net use : {e}")
        return lecteurs

    def detect_partages_unc(self):
        """
        Détecte les lecteurs UNC disponibles SANS JAMAIS accéder au réseau SMB.
        → 100% non bloquant
        → Test IP uniquement
        → Retourne les lettres dont l’IP est joignable.
        """

        accessibles = []

        for lecteur, chemin in self.partages.items():
            try:
                # Vérification syntaxe UNC
                if not isinstance(chemin, str) or not chemin.startswith("\\\\"):
                    debug(f"[UNC] Format invalide : {chemin}")
                    continue

                # Extraction IP
                try:
                    ip = chemin[2:].split("\\")[0].strip()
                except Exception:
                    debug(f"[UNC] Impossible d’extraire IP du chemin : {chemin}")
                    continue

                # --- Test IP uniquement ---
                if self._ping_ip(ip):
                    accessibles.append(lecteur)
                    debug(f"[UNC] {lecteur} → IP {ip} OK (aucun test SMB)")
                else:
                    debug(f"[UNC] {lecteur} → IP {ip} DOWN")

            except Exception as e:
                debug(f"[UNC] Erreur {lecteur} : {e}")
                continue

        return accessibles

    # -------------------------------------------------------------------------
    # Méthode principale (run)
    # -------------------------------------------------------------------------


    def run(self):
        """Point d’entrée déclenché par le bouton."""

        # Interdire deuxième lancement si déjà monté
        if getattr(self, "deja_monte", False):
            QMessageBox.information(
                None,
                self.tr("Montage déjà effectué"),
                self.tr(
                    "Un disque externe est déjà monté.\n"
                    "Veuillez le démonter avant de relancer l’outil."
                ),
            )
            return

        self.stop_network_check.clear()
        self.connexion_reseau = False
        self.connexion_disque = False

        # Préparation interface principale
        self.dialog = QDialog(self.iface.mainWindow())
        self.dialogui = GestionCriseDialog()
        self.dialogui.setupUi(self.dialog)

        self.dialogui.label_connexion.setStyleSheet(
            'font: 75 12pt "Arial"; background-color: rgb(255, 0, 0);'
        )
        self.dialogui.label_connexion.setText('Connexion non disponible')

        self.dialogui.buttonE.clicked.connect(self.exercice)
        self.dialogui.buttonF.clicked.connect(self.formation)
        self.dialogui.buttonC.clicked.connect(self.crise)

        # 1. Throbber immédiat
        self.throbber = ThrobberProgressDialog(self.iface.mainWindow())

        gif_path = ":/plugins/gestion_crise/resources/chrono.gif"

        self.throbber.update_gif(gif_path)

        self.throbber.update_status(self.tr("Recherche du disque externe…"))
        self.throbber.show()

        # 2. Lancer la recherche disque dans un QThread
        self.thread = QThread()
        self.worker = FullProgressSearchWorker(self)

        self.worker.moveToThread(self.thread)
        self.thread.started.connect(self.worker.run)

        self.worker.progress.connect(self.throbber.update_progress)
        self.worker.status.connect(self.throbber.update_status)

        self.worker.finished.connect(self.thread.quit)
        self.worker.finished.connect(self.worker.deleteLater)
        self.thread.finished.connect(self.thread.deleteLater)

        self.worker.finished.connect(self._handle_full_result)

        self.thread.start()

    # -------------------------------------------------------------------------
    # Gestion des résultats de recherche disque
    # -------------------------------------------------------------------------
    def _handle_full_result(self, result: dict):

        # ============================================================
        # CAS 1 : DISQUE EXTERNE DETECTÉ
        # ============================================================
        if result.get("disque"):
            drive = result.get("drive")

            # 🔥 ici on MONTE réellement les lecteurs (SUBST)
            ok = self.monter_disque_externe_python(drive)

            if not ok:
                lettres_msg = ", ".join(self.get_configured_drive_letters())
                msg = QMessageBox(
                    QMessageBox.Critical,
                    self.tr("Montage disque externe impossible"),
                    self.tr(
                        "Le disque externe a été détecté (%1) mais le montage "
                        "des lecteurs (%2) a échoué.\n\n"
                        "Vérifiez la présence des dossiers à la racine du disque, "
                        "puis réessayez."
                    ).replace(
                        "%1", drive
                    ).replace(
                        "%2", lettres_msg
                    ),
                )
                self.bring_window_to_front(msg)
                msg.exec()
                return

            # si tout s’est bien passé, on met simplement à jour l’état
            self.deja_monte = True
            self.connexion_disque = True

            # (facultatif, la fonction monter_disque_externe_python met déjà
            #  à jour ces éléments d’UI, tu peux même enlever ce doublon)

            self.dialogui.label_connexion.setText(
                self.tr("Connexion disque externe (%1)").replace("%1", drive)
            )

            self.dialogui.label_connexion.setStyleSheet(
                'font: 75 12pt "Arial"; background-color: rgb(0,255,0);'
            )
            self.dialogui.buttonE.setEnabled(True)
            self.dialogui.buttonF.setEnabled(True)
            self.dialogui.buttonC.setEnabled(True)

            return self._ouvrir_fenetre_principale()

        # ============================================================
        # CAS 2 : RÉSEAU DISPONIBLE
        # ============================================================
        if result["reseau"]:
            r = result["readers"]
            self.connexion_reseau = True

            self.dialogui.label_connexion.setText(
                self.tr("Connexion réseau (%1)").replace(
                    "%1", ", ".join(r)
                )
            )

            self.dialogui.label_connexion.setStyleSheet(
                'font: 75 12pt "Arial"; background-color: rgb(0,255,0);'
            )
            self.dialogui.buttonE.setEnabled(True)
            self.dialogui.buttonF.setEnabled(True)
            self.dialogui.buttonC.setEnabled(True)

            return self._ouvrir_fenetre_principale()

        # ============================================================
        # CAS 3 : RIEN TROUVÉ
        # ============================================================
        msg = QMessageBox(
            QMessageBox.Critical,
            self.tr("Aucune connexion disponible"),
            self.tr(
                "Aucun disque externe.\n"
                "Aucun serveur ne répond.\n"
                "Aucun lecteur réseau accessible.\n\n"
                "Impossible de continuer."
            )
        )

        self.bring_window_to_front(msg)
        msg.exec()

    def _handle_disk_search_result(self, success: bool):
        # fermer le throbber
        if getattr(self, "throbber", None):
            try:
                self.throbber.close()
            except Exception:
                pass

        # CAS 1 : disque trouvé
        if success:
            self.connexion_disque = True
            self.deja_monte = True
            debug("Connexion disque externe réussie.")
            self.stop_network_check.set()
            self._ouvrir_fenetre_principale()
            return

        # CAS 2 : aucun disque → test réseau
        ip_dispo = self._serveurs_disponibles()
        lecteurs = self.detect_lecteurs_reseau() or self.detect_partages_unc()

        # CAS 3 : rien trouvé
        if not ip_dispo and not lecteurs:
            msg = QMessageBox(
                QMessageBox.Critical,
                self.tr("Aucune connexion disponible"),
                self.tr(
                    "Aucun disque externe.\n"
                    "Aucun serveur ne répond.\n"
                    "Aucun lecteur réseau accessible.\n\n"
                    "Impossible de continuer."
                )
            )

            msg.setWindowModality(Qt.ApplicationModal)
            self.bring_window_to_front(msg)
            msg.exec()
            return

        # CAS 4 : réseau dispo mais pas de disque
        if lecteurs:
            debug(f"Connexion réseau trouvée : {lecteurs}")
            self.connexion_reseau = True
            self.dialogui.label_connexion.setText(f'Connexion réseau ({", ".join(lecteurs)})')
            self.dialogui.label_connexion.setStyleSheet(
                'font: 75 12pt "Arial"; background-color: rgb(0, 255, 0);'
            )
            self.dialogui.buttonE.setEnabled(True)
            self.dialogui.buttonC.setEnabled(True)
            self.dialogui.buttonF.setEnabled(True)

        # Affiche fenêtre principale
        self._ouvrir_fenetre_principale()

        # Vérification réseau asynchrone
        if not self.connexion_disque:
            threading.Thread(
                target=self._verifier_accessibilite_partages_async,
                daemon=True,
            ).start()

    def _handle_full_search_result(self, result: dict):
        # fermer le throbber
        try:
            self.throbber.close()
        except Exception:
            pass

        # disque externe détecté
        if result.get("disque"):
            self.deja_monte = True
            self.connexion_disque = True
            self._ouvrir_fenetre_principale()
            return

        # réseau disponible
        if result.get("reseau"):
            readers = result.get("readers", [])
            self.connexion_reseau = True
            self.dialogui.label_connexion.setText(f"Connexion réseau ({', '.join(readers)})")
            self.dialogui.label_connexion.setStyleSheet(
                'font: 75 12pt "Arial"; background-color: rgb(0, 255, 0);'
            )
            self.dialogui.buttonE.setEnabled(True)
            self.dialogui.buttonC.setEnabled(True)
            self.dialogui.buttonF.setEnabled(True)
            self._ouvrir_fenetre_principale()
            return

        # aucun résultat
        msg = QMessageBox(
            QMessageBox.Critical,
            "Aucune connexion disponible",
            "Aucun disque externe n’a été détecté.\n"
            "Aucun serveu ne répond.\n"
            "Aucun lecteur réseau monté n’est disponible.\n\n"
            "Impossible de continuer.",
        )
        msg.setWindowModality(Qt.ApplicationModal)
        self.bring_window_to_front(msg)
        msg.exec()

    def _ouvrir_fenetre_principale(self):

        # 🔥 fermer le throbber ici
        try:
            if hasattr(self, "throbber") and self.throbber.isVisible():
                self.throbber.close()
        except Exception:
            pass

        self.dialogui.setWindowModality(qt_windowsmodal)
        self.dialogui.setWindowFlags(qt_windowstaysontophint)

        # affichage immédiat juste après fermeture du throbber
        self.dialog.exec()

    # -------------------------------------------------------------------------
    # Vérification asynchrone du réseau
    # -------------------------------------------------------------------------

    def _verifier_accessibilite_partages_async(self):
        """Vérifie les partages réseau sans bloquer QGIS, sauf si disque déjà connecté."""
        if self.connexion_disque or self.stop_network_check.is_set():
            debug("Vérification réseau annulée : disque externe déjà connecté.")
            return
        try:
            inaccessibles = []
            for lecteur, chemin in self.partages.items():
                if self.stop_network_check.is_set():
                    debug("Arrêt anticipé de la vérification réseau.")
                    return
                p = Path(chemin)
                try:
                    if not p.exists():
                        inaccessibles.append(lecteur)
                except Exception:
                    inaccessibles.append(lecteur)
            if inaccessibles:
                lecteurs_str = ", ".join(inaccessibles)

                message = self.tr(
                    "⚠️ Partages réseau inaccessibles : %1"
                ).replace(
                    "%1", lecteurs_str
                )

                QMetaObject.invokeMethod(
                    self.iface.messageBar(), "pushWarning",
                    Qt.QueuedConnection,
                    Q_ARG(str, "Gestion de crise"),
                    Q_ARG(str, message),
                )
                debug(message)
        except Exception as e:
            debug(f"[WARN] Vérification réseau asynchrone échouée : {e}")

    # -------------------------------------------------------------------------
    # OUTILS PROJET QGIS
    # -------------------------------------------------------------------------
    def _extraire_projet_base(self, dossier_destination: Path) -> Path:
        """
        Copie le projet modèle embarqué depuis les ressources Qt
        (:/plugins/gestion_crise/resources/Gestion_crise.qgz)
        vers `dossier_destination`.
        """
        try:
            resource_path = self.projetBASE_RESOURCE  # toujours dans resources.qrc
            file = QFile(resource_path)

            if not file.exists():
                raise FileNotFoundError(
                    f"Le projet QGZ embarqué '{resource_path}' est introuvable dans resources.qrc."
                )

            if not file.open(QFile.ReadOnly):
                raise IOError(
                    f"Impossible d’ouvrir la ressource Qt '{resource_path}'."
                )

            data = file.readAll()
            file.close()

            # Emplacement de sortie
            dest = dossier_destination / "Gestion_crise.qgz"

            # Copie binaire
            with dest.open("wb") as f:
                f.write(bytes(data))

            debug(f"Projet modèle embarqué correctement extrait vers : {dest}")
            return dest

        except Exception as e:
            debug(f"Erreur extraction projet embarqué : {e}")

            QMessageBox.critical(
                None,
                self.tr("Erreur"),
                self.tr(
                    "Impossible de copier le projet modèle embarqué :\n%1"
                ).replace(
                    "%1", str(e)
                )
            )

            return None

    def ouvrir_projet(self, chemproj: Path, nomproj: str, pathToFile: Path):
        """Ouvre le projet modèle et y ajoute les couches évènementielles."""
        projet_copie = self._extraire_projet_base(chemproj)
        if not projet_copie:
            return
        self.project = QgsProject.instance()
        self.project.read(str(projet_copie))
        cible = chemproj / nomproj
        # Ecriture des couches évènement
        self.handler_createLayers(chemproj, nomproj, pathToFile)
        self.project.write(str(cible))
        debug(f"Nouveau projet créé : {cible}")

    # -------------------------------------------------------------------------
    # BOUTONS (exercice, formation, crise)
    # -------------------------------------------------------------------------
    def exercice(self):
        nom, ok = QInputDialog.getText(
                    None,
                    self.tr("Exercice gestion de crise"),
                    self.tr("Nom de l'exercice :")
                    )
        if ok and nom.strip():
            self.dialog.close()
            dossier = self.cheminexo / f"{self.datejma} - {nom}"
            dossier.mkdir(parents=True, exist_ok=True)
            nomproj = f"{self.datejma} - {nom} - {self.utilisateur}.qgz"
            self.ouvrir_projet(dossier, nomproj, dossier)

    def formation(self):
        self.dialog.close()
        dossier = self.cheminform / f"Formation_Qgis_gestion_crise_{self.datejma}"
        dossier.mkdir(parents=True, exist_ok=True)
        nomproj = f"Formation_Qgis_gestion_crise_{self.datejma} - {self.utilisateur}.qgz"
        self.ouvrir_projet(dossier, nomproj, dossier)

    def crise(self):
        nom, ok = QInputDialog.getText(
            None,
            self.tr("Gestion de crise"),
            self.tr("Nom de la crise :")
            )
        if ok and nom.strip():
            self.dialog.close()
            dossier = self.chemincrise / f"{self.datejma} - {nom}"
            dossier.mkdir(parents=True, exist_ok=True)
            nomproj = f"{self.datejma} - {nom} - {self.utilisateur}.qgz"
            self.ouvrir_projet(dossier, nomproj, dossier)

    # -------------------------------------------------------------------------
    # CRÉATION DES COUCHES ÉVÉNEMENT
    # -------------------------------------------------------------------------
    def handler_createLayers(self, chemproj: Path, nomproj: str, pathToFile: Path):
        """Crée les couches événementielles dans le dossier Evenements du projet."""
        self.project = QgsProject.instance()
        root = self.project.layerTreeRoot()
        evt = None
        groupevt = None

        # Recherche du groupe "Evenements"
        for group in root.children():
            test = ''.join(
                x for x in unicodedata.normalize('NFKD', group.name())
                if unicodedata.category(x)[0] == 'L'
            ).upper()
            if test == 'EVENEMENTS':
                evt = True
                groupevt = group
                evenements_dir = chemproj / "Evenements"
                if any((evenements_dir / fn).exists() for fn in [
                    "POLYGONE_EVENEMENT.shp",
                    "LIGNE_EVENEMENT.shp",
                    "POINT_EVENEMENT.shp"
                ]):
                    debug('Couches déjà créées')
                    QMessageBox.warning(
                        self.iface.mainWindow(),
                        self.tr("Commande inutile : "),
                        self.tr("Les couches sont déjà présentes.")
                    )
                    return

        # Création du groupe s’il n’existe pas
        if not evt:
            if nomproj:
                groupevt = root.insertGroup(0, 'Evenements')
            else:
                debug('Projet non enregistré')
                QMessageBox.warning(
                    self.iface.mainWindow(),
                    self.tr("Création impossible"),
                    self.tr("Le projet doit être enregistré avant la création des couches Evenements.")
                )
                return

        # Définition du répertoire de sortie
        evt_dir = chemproj / "Evenements"
        evt_dir.mkdir(parents=True, exist_ok=True)

        # Définition CRS et options
        crs = self.project.crs()
        transform_context = self.project.transformContext()
        save_options = QgsVectorFileWriter.SaveVectorOptions()
        save_options.driverName = "ESRI Shapefile"
        save_options.fileEncoding = "UTF-8"

        # Définition des champs (algorithme inchangé)
        def champs_polygone():
            f = QgsFields()
            for name, t, *args in [
                ('libelle', QVariant.String),
                ('date', QVariant.String),
                ('h_creation', QVariant.String),
                ('source', QVariant.String),
                ('h_constat', QVariant.String),
                ('remarques', QVariant.String),
                ('surface', QVariant.Double, 'double', 10, 0),
                ('utilisatr', QVariant.String)
            ]:
                f.append(QgsField(name, t, *args))
            return f

        def champs_ligne():
            f = QgsFields()
            for name, t, *args in [
                ('libelle', QVariant.String),
                ('date', QVariant.String),
                ('h_creation', QVariant.String),
                ('source', QVariant.String),
                ('h_constat', QVariant.String),
                ('remarques', QVariant.String),
                ('longueur', QVariant.Double, 'double', 10, 0),
                ('utilisatr', QVariant.String)
            ]:
                f.append(QgsField(name, t, *args))
            return f

        def champs_point():
            f = QgsFields()
            for name, t, *args in [
                ('libelle', QVariant.String),
                ('date', QVariant.String),
                ('h_creation', QVariant.String),
                ('source', QVariant.String),
                ('h_constat', QVariant.String),
                ('remarques', QVariant.String),
                ('x_gps', QVariant.Double, 'double', 10, 6),
                ('y_gps', QVariant.Double, 'double', 10, 6),
                ('utilisatr', QVariant.String)
            ]:
                f.append(QgsField(name, t, *args))
            return f

        # Chemin vers les styles QML (dans les ressources embarquées)
        qmlbase = ":/plugins/gestion_crise/resources/"
        couches = [
            (evt_dir / "POLYGONE_EVENEMENT.shp", champs_polygone(), QgsWkbTypes.MultiPolygon,
             qmlbase + "POLYGONE_EVENEMENT.qml"),
            (evt_dir / "LIGNE_EVENEMENT.shp", champs_ligne(), QgsWkbTypes.MultiLineString,
             qmlbase + "LIGNE_EVENEMENT.qml"),
            (evt_dir / "POINT_EVENEMENT.shp", champs_point(), QgsWkbTypes.MultiPoint,
             qmlbase + "POINT_EVENEMENT.qml")
        ]

        # Création des shapefiles + insertion dans le groupe Evenements (algorithme inchangé)
        for path, fields, wkb, qml in couches:
            writer = QgsVectorFileWriter.create(str(path), fields, wkb, crs, transform_context, save_options)
            if writer.hasError() != QgsVectorFileWriter.NoError:
                QMessageBox.critical(
                    self.iface.mainWindow(),
                    self.tr("Erreur création shapefile"),
                    writer.errorMessage()
                )
                del writer
                continue
            del writer

            # Chargement dans QGIS
            layer = self.iface.addVectorLayer(str(path), path.stem, "ogr")
            layer.loadNamedStyle(qml)
            node = root.findLayer(layer.id())
            clone = node.clone()
            parent = node.parent()
            groupevt.insertChildNode(0, clone)
            parent.removeChildNode(node)
            layer.setReadOnly(True)
            layer.triggerRepaint()

        # Déplie le groupe Evenements
        for n in root.children():
            test = ''.join(x for x in unicodedata.normalize('NFKD', n.name())
                           if unicodedata.category(x)[0] == 'L').upper()
            n.setExpanded(test == 'EVENEMENTS')

    def bring_window_to_front(self, widget):
        """
        Force une fenêtre Qt (QMessageBox / QDialog…) au premier plan
        sur Windows. Compatible Qt5 + Qt6 + QGIS 3.x.
        """

        try:
            widget.setWindowFlag(Qt.WindowStaysOnTopHint, True)
            widget.raise_()
            widget.activateWindow()

            # Windows API SetWindowPos → topmost réel (au-dessus des autres applis)
            try:
                hwnd = int(widget.winId())
                windll.user32.SetWindowPos(
                    hwnd,
                    -1,  # HWND_TOPMOST
                    0, 0, 0, 0,
                    0x0003  # SWP_NOMOVE | SWP_NOSIZE
                )
            except Exception:
                pass

            widget.show()
            widget.raise_()
            widget.activateWindow()

        except Exception as e:
            print(f"[WARN] bring_window_to_front: {e}")

    def compiler_resources(self):
        """
        Affiche le throbber chrono.gif et compile resources.qrc
        dans un QThread pour conserver l'animation fluide.
        """

        import sys
        from pathlib import Path
        from qgis.PyQt.QtWidgets import QApplication, QMessageBox
        from qgis.PyQt.QtCore import QThread
        from qgis.PyQt.QtCore import PYQT_VERSION_STR

        # ------------------------------------------------------------
        # 1️⃣ Préparation du throbber animé (non bloquant)
        # ------------------------------------------------------------
        self.throbber_compile = ThrobberProgressDialog(self.iface.mainWindow())
        gif_path = str(Path(__file__).parent / "resources" / "chrono.gif")
        self.throbber_compile.update_gif(gif_path)
        self.throbber_compile.update_status(self.tr("Compilation des ressources…"))
        self.throbber_compile.pbar.setMaximum(0)  # mode indéterminé
        self.throbber_compile.show()

        QApplication.processEvents()

        # ------------------------------------------------------------
        # 2️⃣ Fichiers source / destination
        # ------------------------------------------------------------
        plugin_dir = Path(__file__).parent
        qrc_file = plugin_dir / "resources.qrc"
        py_file = plugin_dir / "resources.py"

        if not qrc_file.exists():
            self.throbber_compile.close()

            QMessageBox.critical(
                None,
                self.tr("Erreur"),
                self.tr(
                    "resources.qrc introuvable :\n%1"
                ).replace(
                    "%1", str(qrc_file)
                )
            )

            return

        # ------------------------------------------------------------
        # 3️⃣ Localisation pyrcc5 / pyrcc6 via sys.executable (fiable)
        # ------------------------------------------------------------
        bin_dir = Path(sys.executable).parent
        qgis_root = bin_dir.parent

        pyrcc_name = "pyrcc6.exe" if PYQT_VERSION_STR.startswith("6") else "pyrcc5.exe"

        pyrcc_path = None
        for p in qgis_root.rglob(pyrcc_name):
            if p.is_file():
                pyrcc_path = p
                break

        if pyrcc_path is None:
            self.throbber_compile.close()
            QMessageBox.critical(None, "Erreur", f"{pyrcc_name} introuvable dans {qgis_root}")
            return

        # ------------------------------------------------------------
        # 4️⃣ Création du thread et lancement
        # ------------------------------------------------------------
        self.compile_thread = QThread()
        self.compile_worker = CompileResourcesWorker(qrc_file, py_file, pyrcc_path)

        self.compile_worker.moveToThread(self.compile_thread)
        self.compile_thread.started.connect(self.compile_worker.run)

        self.compile_worker.finished.connect(self.compile_thread.quit)
        self.compile_worker.finished.connect(self.compile_worker.deleteLater)
        self.compile_thread.finished.connect(self.compile_thread.deleteLater)

        def on_finished(success: bool, msg: str):
            self.throbber_compile.close()
            if success:
                QMessageBox.information(
                    None,
                    self.tr("Compilation réussie"),
                    self.tr("resources.qrc → resources.py généré.")
                )
            else:
                QMessageBox.critical(
                    None,
                    self.tr("Erreur compilation"),
                    msg
                )

        self.compile_worker.finished.connect(on_finished)

        self.compile_thread.start()

    def get_configured_drive_letters(self):
        """
        Retourne la liste normalisée des lettres configurées dans self.partages,
        au format ['K:', 'R:', 'S:', ...].
        """
        lettres = []
        for k in self.partages.keys():
            try:
                lettres.append(self.normalize_drive_letter(k))
            except Exception:
                pass
        return lettres

# ============================================================================
# Worker générique
# ============================================================================
class SimpleWorker(QObject):
    finished = pyqtSignal()
    func = None

    def __init__(self, func):
        super().__init__()
        self.func = func

    def run(self):
        try:
            self.func()
        finally:
            self.finished.emit()


# ============================================================================
# Boîte de message temporaire
# ============================================================================
class CustomMessageBox(QMessageBox):
    def __init__(self, *__args):
        super().__init__()
        self.timeout = 0
        self.autoclose = False
        self.currentTime = 0

    def showEvent(self, QShowEvent):
        self.currentTime = 0
        if self.autoclose:
            self.startTimer(1000)

    def timerEvent(self, *args, **kwargs):
        self.currentTime += 1
        if self.currentTime >= self.timeout:
            self.done(0)

    @staticmethod
    def showWithTimeout(timeoutSeconds, message, title, icon=qmessagebox_information):
        w = CustomMessageBox()
        w.autoclose = True
        w.timeout = timeoutSeconds
        w.setText(message)
        w.setWindowTitle(title)
        w.setIcon(icon)
        w.exec()


