# -*- coding: utf-8 -*-
"""
/***************************************************************************
 gestion_crise
 Plugin QGIS de gestion de crise
 Version finale modernisée (pathlib-only)
 - priorité disque externe (vérifie dossiers finaux des partages à la racine)
 - fallback réseau
 - projet modèle embarqué (2_Gestion_crise.qgz)
 - configuration partages réseau
 - avertissement automatique (asynchrone)
 - compatibilité PyQt5 / PyQt6
 - DEBUG_MODE optionnel + logging fiable (QgsMessageLog + print flush)
 ***************************************************************************/
"""

# --- Librairies système ---
import platform
import threading
import unicodedata
import os
import time
import datetime
import subprocess
from pathlib import Path
from ctypes import windll

# --- Librairies Windows spécifiques ---
import win32api
import win32file
import win32con
import pywintypes

# --- QGIS / PyQt ---
from qgis.PyQt.QtCore import (
    QSettings, QTranslator, QCoreApplication, Qt,
    PYQT_VERSION_STR, QFile, QVariant, QMetaObject, Q_ARG
)
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import (
    QAction, QMessageBox, QInputDialog, QLineEdit, QDialog,
    QLabel, QPushButton, QVBoxLayout, QGridLayout, QHBoxLayout
)
from qgis.core import *
from qgis.core import QgsMessageLog, Qgis  # ← pour log QGIS
from qgis.gui import *

# --- Plugin resources ---
from . import resources
from .Gestion_crise_dialog import Gestion_criseDialog

# --- Compatibilité PyQt5 / PyQt6 ---
if PYQT_VERSION_STR.startswith("5"):
    qmessagebox_information = QMessageBox.Information
    qt_windowsmodal = Qt.WindowModal
    qt_windowstaysontophint = Qt.WindowStaysOnTopHint
elif PYQT_VERSION_STR.startswith("6"):
    qmessagebox_information = QMessageBox.Icon.Information
    qt_windowsmodal = Qt.WindowModality.WindowModal
    qt_windowstaysontophint = Qt.WindowType.WindowStaysOnTopHint

# --- Réglage par défaut (sera surchargé par QSettings à l'init) ---
import sys
import warnings
DEBUG_MODE = False

# --- neutralisation complète des prints et warnings si DEBUG désactivé ---
if not DEBUG_MODE:
    def _silent_print(*args, **kwargs):
        pass
    print = _silent_print  # type: ignore
    warnings.filterwarnings("ignore")

def _debug_sink(msg: str):
    """Écrit dans le journal QGIS + console Python (avec flush)."""
    try:
        QgsMessageLog.logMessage(msg, "Gestion_crise", Qgis.Info)
    except Exception:
        pass
    try:
        print(f"[DEBUG] {msg}", flush=True)
    except Exception:
        pass

def debug(msg: str):
    """Affiche un message de debug si DEBUG_MODE est activé (fiable)."""
    if DEBUG_MODE:
        _debug_sink(msg)


# ============================================================================
# CLASSE PRINCIPALE DU PLUGIN
# ============================================================================
class Gestion_crise:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Initialisation principale du plugin QGIS."""
        self.iface = iface
        self.plugin_dir = Path(__file__).parent
        self.menu = self.tr('&Gestion de crise')
        self.actions = []
        self.first_start = None
        self.action_debug_toggle = None  # action checkable pour debug
        self.stop_network_check = threading.Event()

        # --- Localisation ---
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = self.plugin_dir / 'i18n' / f'Gestion_crise_{locale}.qm'
        if locale_path.exists():
            self.translator = QTranslator()
            self.translator.load(str(locale_path))
            QCoreApplication.installTranslator(self.translator)

        # --- Variables principales ---
        self.utilisateur = os.environ.get("USERNAME") or Path.home().name or "Inconnu"
        self.datejma = datetime.datetime.today().strftime('%d%m%y')
        self.cheminburo = Path.home() / 'Desktop'
        self.cheminform = self.cheminburo / 'Formation_gestion_Crise'
        self.cheminexo = Path('K:/permanence/Exercices')
        self.chemincrise = Path('K:/permanence/Evenements')
        self.liste_lecteurs = ['K:\\', 'R:\\', 'S:\\', 'T:\\', 'W:\\']
        self.disque_ext = 'PERMANENCE_DDT04'
        self.connexion_reseau = False
        self.connexion_disque = False
        self.montage_HDD = 'Gestion_crise_disque_externe.bat'
        self.projetBASE_RESOURCE = ":/plugins/gestion_crise/resources/2_Gestion_crise.qgz"

        # --- Défaut partages réseau ---
        self.defaults_partages = {
            'R:\\': r'\\10.4.8.41\gb_ref',
            'S:\\': r'\\10.4.8.41\gb_prod',
            'T:\\': r'\\10.4.8.41\gb_cons',
            'W:\\': r'\\10.4.8.41\dossiers\fichiers_sig'
        }
        self.partages = dict(self.defaults_partages)
        self._charger_partages()

        # --- Charger l'état du DEBUG depuis QSettings ---
        s = QSettings()
        global DEBUG_MODE
        DEBUG_MODE = bool(int(s.value("gestion_crise/debug", "1")))  # "1" par défaut
        debug("Initialisation complète du plugin Gestion_crise.")

    # -------------------------------------------------------------------------
    # Initialisation QGIS
    # -------------------------------------------------------------------------
    def initGui(self):
        """Initialise le plugin dans QGIS (sans aucun accès disque/réseau)."""
        icon_main = ':/plugins/gestion_crise/icon.png'
        try:
            icon_network = QgsApplication.getThemeIcon("mIconNetwork.svg")
            if icon_network.isNull():
                icon_network = QgsApplication.getThemeIcon("mActionOptions.svg")
        except Exception:
            icon_network = QgsApplication.getThemeIcon("mActionOptions.svg")

        self.add_action(icon_main, self.tr('Lancement projet gestion de crise'),
                        self.run, self.iface.mainWindow(), add_to_toolbar=True)

        self.add_action(icon_network, self.tr('Configurer les partages réseau'),
                        self.configurer_partages, self.iface.mainWindow(), add_to_toolbar=False)

        # --- Action checkable pour activer/désactiver le mode debug ---
        self.action_debug_toggle = QAction(self.tr("Activer le mode débogage"), self.iface.mainWindow())
        self.action_debug_toggle.setCheckable(True)
        self.action_debug_toggle.setChecked(DEBUG_MODE)
        self.action_debug_toggle.toggled.connect(self._toggle_debug_mode)
        self.iface.addPluginToMenu(self.menu, self.action_debug_toggle)

        self.first_start = True
        debug("Interface QGIS initialisée.")

    def _toggle_debug_mode(self, checked: bool):
        """Active/désactive le mode DEBUG et persiste la valeur."""
        global DEBUG_MODE
        DEBUG_MODE = bool(checked)
        QSettings().setValue("gestion_crise/debug", "1" if checked else "0")
        _debug_sink(f"Mode debug {'activé' if checked else 'désactivé'}.")

    def unload(self):
        """Supprime le plugin du menu et de la barre d’outils QGIS."""
        for action in self.actions:
            self.iface.removePluginMenu(self.menu, action)
            self.iface.removeToolBarIcon(action)
        if self.action_debug_toggle:
            self.iface.removePluginMenu(self.menu, self.action_debug_toggle)

    # -------------------------------------------------------------------------
    # Fonctions utilitaires
    # -------------------------------------------------------------------------
    def tr(self, message):
        return QCoreApplication.translate('Gestion_crise', message)

    def add_action(self, icon_path, text, callback, parent=None, add_to_toolbar=True):
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        self.iface.addPluginToMenu(self.menu, action)
        if add_to_toolbar:
            self.iface.addToolBarIcon(action)
        self.actions.append(action)
        return action

    # -------------------------------------------------------------------------
    # Gestion des partages réseau
    # -------------------------------------------------------------------------
    def _charger_partages(self):
        """Charge la configuration des partages depuis QSettings."""
        s = QSettings()
        keys = [k for k in s.allKeys() if k.startswith("gestion_crise/partage_")]
        if not keys:
            self.partages = dict(self.defaults_partages)
            for lecteur, chemin in self.partages.items():
                s.setValue(f"gestion_crise/partage_{lecteur}", chemin)
        else:
            self.partages.clear()
            for k in keys:
                lecteur = k.replace("gestion_crise/partage_", "")
                self.partages[lecteur] = s.value(k)
        debug(f"Partages réseau chargés : {self.partages}")

    def _sauver_partages(self):
        """Sauvegarde la configuration des partages dans QSettings."""
        s = QSettings()
        for k in [k for k in s.allKeys() if k.startswith("gestion_crise/partage_")]:
            s.remove(k)
        for lecteur, chemin in self.partages.items():
            s.setValue(f"gestion_crise/partage_{lecteur}", chemin)
        debug("Partages réseau sauvegardés.")

    def configurer_partages(self):
        """Boîte de dialogue de configuration des partages réseau."""
        dlg = QDialog(self.iface.mainWindow())
        dlg.setWindowTitle("Configuration des partages réseau")
        dlg.setMinimumWidth(650)
        main_layout = QVBoxLayout(dlg)
        grid = QGridLayout()
        main_layout.addLayout(grid)

        edits, labels_status = {}, {}

        def refresh_grid():
            for i in reversed(range(grid.count())):
                w = grid.itemAt(i).widget()
                if w:
                    w.setParent(None)
            for row, (lecteur, chemin) in enumerate(self.partages.items()):
                lbl = QLabel(f"{lecteur} →")
                edit = QLineEdit(chemin)
                lbl_status = QLabel("")
                btn_remove = QPushButton("✖")
                btn_remove.setFixedWidth(30)
                grid.addWidget(lbl, row, 0)
                grid.addWidget(edit, row, 1)
                grid.addWidget(lbl_status, row, 2)
                grid.addWidget(btn_remove, row, 3)
                edits[lecteur] = edit
                labels_status[lecteur] = lbl_status
                btn_remove.clicked.connect(lambda _, key=lecteur: supprimer_partage(key))

        def ajouter_partage():
            lecteur, ok = QInputDialog.getText(dlg, "Nouveau partage", "Lettre du lecteur (ex: U:\\) :")
            if not ok or not lecteur.strip():
                return
            if not lecteur.endswith("\\"):
                lecteur += "\\"
            chemin, ok2 = QInputDialog.getText(dlg, "Nouveau partage", "Chemin UNC (ex: \\\\serveur\\partage) :")
            if not ok2 or not chemin.strip():
                return
            self.partages[lecteur] = chemin.strip()
            refresh_grid()

        def supprimer_partage(lecteur):
            if lecteur in self.partages:
                del self.partages[lecteur]
                refresh_grid()

        def tester_partages():
            for lecteur, edit in edits.items():
                chemin = edit.text().strip()
                lbl = labels_status[lecteur]
                p = Path(chemin)
                try:
                    if p.exists():
                        lbl.setText("✅")
                        lbl.setStyleSheet("color: green; font-weight: bold;")
                    else:
                        lbl.setText("❌")
                        lbl.setStyleSheet("color: red; font-weight: bold;")
                except Exception:
                    lbl.setText("⚠️")
                    lbl.setStyleSheet("color: orange; font-weight: bold;")

        def reinitialiser():
            rep = QMessageBox.question(dlg, "Réinitialiser les partages",
                                       "Voulez-vous restaurer les partages par défaut ?",
                                       QMessageBox.Yes | QMessageBox.No)
            if rep == QMessageBox.Yes:
                self.partages = dict(self.defaults_partages)
                self._sauver_partages()
                refresh_grid()
                QMessageBox.information(dlg, "Réinitialisé", "Les partages ont été restaurés.")

        def enregistrer():
            self.partages = {lecteur: edits[lecteur].text().strip() for lecteur in edits}
            self._sauver_partages()
            QMessageBox.information(dlg, "Partages enregistrés", "Les nouveaux chemins ont été sauvegardés.")
            dlg.accept()

        refresh_grid()

        hbox = QHBoxLayout()
        for text, func in [
            ("+ Ajouter", ajouter_partage),
            ("Tester", tester_partages),
            ("Réinitialiser", reinitialiser),
            ("Enregistrer", enregistrer),
            ("Annuler", dlg.reject)
        ]:
            btn = QPushButton(text)
            btn.clicked.connect(func)
            hbox.addWidget(btn)
        main_layout.addLayout(hbox)
        dlg.exec()

    # -------------------------------------------------------------------------
    # Détection et montage du disque externe
    # -------------------------------------------------------------------------
    def _try_mount_external_disk(self) -> bool:
        if platform.system().lower() != "windows":
            debug("Non-Windows : montage disque externe ignoré.")
            return False

        try:
            lecteurs = [d for d in win32api.GetLogicalDriveStrings().split("\x00") if d]
            debug(f"Lecteurs détectés : {lecteurs}")
        except Exception as e:
            debug(f"Erreur GetLogicalDriveStrings : {e}")
            return False

        # --- corrige ici : nettoyage et insensibilité à la casse ---
        noms_attendus = set()
        for chemin in self.partages.values():
            name = Path(str(chemin).rstrip("\\/")).name.strip().lower()
            if name:
                noms_attendus.add(name)
        debug(f"Noms attendus sur disque externe (normalisés) : {noms_attendus}")

        for drive in lecteurs:
            try:
                drive_type = win32file.GetDriveType(drive)
                if drive_type not in (win32con.DRIVE_FIXED, win32con.DRIVE_REMOVABLE):
                    continue
                volume_label = win32api.GetVolumeInformation(drive)[0]
                base = Path(drive)
                debug(f"Test du lecteur {drive} (label={volume_label})")

                if volume_label == self.disque_ext:
                    debug(f"Disque externe trouvé par label : {drive}")
                    self.stop_network_check.set()  # stoppe la recherche réseau
                    return self._monter_disque_externe(drive)

                dossiers_presents = {p.name.strip().lower() for p in base.iterdir() if p.is_dir()}
                debug(f"Dossiers présents {drive}: {dossiers_presents}")
                if noms_attendus.issubset(dossiers_presents):
                    debug(f"Disque externe trouvé par structure : {drive}")
                    self.stop_network_check.set()  # stoppe la recherche réseau
                    return self._monter_disque_externe(drive)
            except Exception as e:
                debug(f"Erreur sur {drive}: {e}")

        debug("Aucun disque externe valide trouvé.")
        return False

    def _monter_disque_externe(self, drive: str) -> bool:
        """Monte le disque externe et met à jour l’interface sans bloquer QGIS."""
        try:
            bat_path = self.plugin_dir / "resources" / "Gestion_crise_disque_externe.bat"
            if not bat_path.exists():
                debug(f"Script batch introuvable : {bat_path}")
                return False

            drive_arg = drive.strip("\\")
            cmd = f'start /MIN cmd.exe /C "{bat_path}" {drive_arg}'
            debug(f"Lancement du batch de montage : {cmd}")

            # --- lancement du processus ---
            p = subprocess.Popen(
                cmd,
                shell=True,
                stdin=subprocess.DEVNULL,
                stdout=subprocess.DEVNULL,
                stderr=subprocess.DEVNULL,
                close_fds=True,
                creationflags=subprocess.DETACHED_PROCESS | subprocess.CREATE_NEW_PROCESS_GROUP
            )

            # --- suppression du warning ResourceWarning ---
            threading.Thread(target=p.wait, daemon=True).start()

            # --- message utilisateur ---
            CustomMessageBox.showWithTimeout(
                3,
                f"Montage du disque externe ({drive_arg}), veuillez patienter...",
                "Connexion",
                icon=qmessagebox_information
            )

            time.sleep(2)
            self.connexion_disque = True
            self.dialogui.label_connexion.setText(f'Connexion disque externe ({drive_arg})')
            self.dialogui.label_connexion.setStyleSheet(
                'font: 75 12pt "Arial"; background-color: rgb(0, 255, 0);'
            )
            self.dialogui.buttonE.setEnabled(True)
            self.dialogui.buttonC.setEnabled(True)
            self.dialogui.buttonF.setEnabled(True)

            return True

        except Exception as e:
            debug(f"Erreur montage disque externe : {e}")
            return False

    # -------------------------------------------------------------------------
    # Détection réseau
    # -------------------------------------------------------------------------
    def detect_lecteurs_reseau(self):
        lecteurs = []
        try:
            output = subprocess.check_output("net use", shell=True, encoding="cp850", stderr=subprocess.DEVNULL)
            for line in output.splitlines():
                parts = line.strip().split()
                if len(parts) >= 2 and parts[0].endswith(":") and parts[1].startswith("\\\\"):
                    lecteurs.append(parts[0].upper() + "\\")
            debug(f"Lecteurs réseau détectés : {lecteurs}")
        except Exception as e:
            debug(f"Erreur net use : {e}")
        return lecteurs

    def detect_partages_unc(self):
        accessibles = []
        for lecteur, chemin in self.partages.items():
            try:
                if Path(chemin).exists():
                    accessibles.append(lecteur)
            except Exception:
                pass
        debug(f"Partages UNC accessibles : {accessibles}")
        return accessibles

    # -------------------------------------------------------------------------
    # Méthode principale (run)
    # -------------------------------------------------------------------------
    def run(self):
        """Point d’entrée déclenché par le bouton."""
        self.stop_network_check.clear()  # réinitialise le drapeau d'arrêt

        if self.first_start:
            self.first_start = False

        self.connexion_reseau = False
        self.connexion_disque = False

        # Création et configuration du dialogue
        self.dialog = QDialog(self.iface.mainWindow())
        self.dialogui = Gestion_criseDialog()
        self.dialogui.setupUi(self.dialog)
        self.dialogui.label_connexion.setStyleSheet(
            'font: 75 12pt "Arial"; background-color: rgb(255, 0, 0);'
        )
        self.dialogui.label_connexion.setText('Connexion non disponible')

        # Connexion des boutons
        self.dialogui.buttonE.clicked.connect(self.exercice)
        self.dialogui.buttonF.clicked.connect(self.formation)
        self.dialogui.buttonC.clicked.connect(self.crise)

        # 1️⃣ Tentative disque externe
        debug("Recherche disque externe prioritaire...")
        if self._try_mount_external_disk():
            debug("Connexion disque externe réussie, arrêt du thread réseau.")
            self.stop_network_check.set()
            self.dialogui.setWindowModality(qt_windowsmodal)
            self.dialogui.setWindowFlags(qt_windowstaysontophint)
            self.dialog.exec()
            return  # aucune vérification réseau
        else:
            debug("Aucun disque externe trouvé, test réseau...")

        # 2️⃣ Tentative réseau uniquement si disque externe absent
        lecteurs = self.detect_lecteurs_reseau() or self.detect_partages_unc()
        if lecteurs:
            debug(f"Connexion réseau trouvée : {lecteurs}")
            self.connexion_reseau = True
            self.dialogui.label_connexion.setText(f'Connexion réseau ({", ".join(lecteurs)})')
            self.dialogui.label_connexion.setStyleSheet(
                'font: 75 12pt "Arial"; background-color: rgb(0, 255, 0);'
            )
            self.dialogui.buttonE.setEnabled(True)
            self.dialogui.buttonC.setEnabled(True)
            self.dialogui.buttonF.setEnabled(True)
        else:
            debug("Aucune connexion réseau trouvée.")

        # 3️⃣ Affiche la fenêtre si quelque chose est connecté
        if self.connexion_disque or self.connexion_reseau:
            self.dialogui.setWindowModality(qt_windowsmodal)
            self.dialogui.setWindowFlags(qt_windowstaysontophint)
            self.dialog.exec()
        else:
            QMessageBox.warning(
                None, "ATTENTION !",
                "Aucune connexion disponible.\nVérifiez le disque externe ou le réseau."
            )

        # 4️⃣ Vérification réseau asynchrone uniquement si disque non trouvé
        if not self.connexion_disque:
            threading.Thread(target=self._verifier_accessibilite_partages_async, daemon=True).start()

    # -------------------------------------------------------------------------
    # Vérification asynchrone du réseau
    # -------------------------------------------------------------------------
    def _verifier_accessibilite_partages_async(self):
        """Vérifie les partages réseau sans bloquer QGIS, sauf si disque externe déjà connecté."""
        if self.connexion_disque or self.stop_network_check.is_set():
            debug("Vérification réseau annulée : disque externe déjà connecté.")
            return
        try:
            inaccessibles = []
            for lecteur, chemin in self.partages.items():
                if self.stop_network_check.is_set():  # permet d'interrompre proprement
                    debug("Arrêt anticipé de la vérification réseau.")
                    return
                p = Path(chemin)
                try:
                    if not p.exists():
                        inaccessibles.append(lecteur)
                except Exception:
                    inaccessibles.append(lecteur)
            if inaccessibles:
                lecteurs_str = ", ".join(inaccessibles)
                message = f"⚠️ Partages réseau inaccessibles : {lecteurs_str}"
                QMetaObject.invokeMethod(
                    self.iface.messageBar(), "pushWarning",
                    Qt.QueuedConnection,
                    Q_ARG(str, "Gestion de crise"),
                    Q_ARG(str, message)
                )
                debug(message)
        except Exception as e:
            debug(f"[WARN] Vérification réseau asynchrone échouée : {e}")

    # -------------------------------------------------------------------------
    # OUTILS PROJET QGIS
    # -------------------------------------------------------------------------
    def _extraire_projet_base(self, dossier_destination: Path) -> Path:
        """Copie le projet modèle embarqué dans le dossier destination."""
        try:
            file = QFile(self.projetBASE_RESOURCE)
            if not file.exists():
                raise FileNotFoundError("Le projet modèle embarqué est introuvable.")
            if not file.open(QFile.ReadOnly):
                raise IOError("Impossible d’ouvrir la ressource du projet modèle.")
            data = file.readAll()
            file.close()
            dest = dossier_destination / "2_Gestion_crise.qgz"
            with dest.open("wb") as f:
                f.write(bytes(data))
            debug(f"Projet modèle copié vers {dest}")
            return dest
        except Exception as e:
            debug(f"Erreur extraction projet modèle : {e}")
            QMessageBox.critical(None, "Erreur", f"Impossible de copier le projet modèle : {e}")
            return None

    def ouvrir_projet(self, chemproj: Path, nomproj: str, pathToFile: Path):
        """Ouvre le projet modèle et y ajoute les couches évènementielles."""
        projet_copie = self._extraire_projet_base(chemproj)
        if not projet_copie:
            return
        self.project = QgsProject.instance()
        self.project.read(str(projet_copie))
        cible = chemproj / nomproj
        # Ecriture des couches évènement
        self.handler_createLayers(chemproj, nomproj, pathToFile)
        self.project.write(str(cible))
        debug(f"Nouveau projet créé : {cible}")

    # -------------------------------------------------------------------------
    # BOUTONS (exercice, formation, crise)
    # -------------------------------------------------------------------------
    def exercice(self):
        nom, ok = QInputDialog.getText(None, 'Exercice gestion de crise', "Nom de l'exercice :")
        if ok and nom.strip():
            self.dialog.close()
            dossier = self.cheminexo / f"{self.datejma} - {nom}"
            dossier.mkdir(parents=True, exist_ok=True)
            nomproj = f"{self.datejma} - {nom} - {self.utilisateur}.qgz"
            self.ouvrir_projet(dossier, nomproj, dossier)

    def formation(self):
        self.dialog.close()
        dossier = self.cheminform / f"Formation_Qgis_gestion_crise_{self.datejma}"
        dossier.mkdir(parents=True, exist_ok=True)
        nomproj = f"Formation_Qgis_gestion_crise_{self.datejma} - {self.utilisateur}.qgz"
        self.ouvrir_projet(dossier, nomproj, dossier)

    def crise(self):
        nom, ok = QInputDialog.getText(None, 'Gestion de crise', "Nom de la crise :")
        if ok and nom.strip():
            self.dialog.close()
            dossier = self.chemincrise / f"{self.datejma} - {nom}"
            dossier.mkdir(parents=True, exist_ok=True)
            nomproj = f"{self.datejma} - {nom} - {self.utilisateur}.qgz"
            self.ouvrir_projet(dossier, nomproj, dossier)

    # -------------------------------------------------------------------------
    # CRÉATION DES COUCHES ÉVÉNEMENT
    # -------------------------------------------------------------------------
    def handler_createLayers(self, chemproj: Path, nomproj: str, pathToFile: Path):
        """Crée les couches événementielles dans le dossier Evenements du projet."""
        self.project = QgsProject.instance()
        root = self.project.layerTreeRoot()
        evt = None
        groupevt = None

        # Recherche du groupe "Evenements"
        for group in root.children():
            test = ''.join(
                x for x in unicodedata.normalize('NFKD', group.name())
                if unicodedata.category(x)[0] == 'L'
            ).upper()
            if test == 'EVENEMENTS':
                evt = True
                groupevt = group
                evenements_dir = chemproj / "Evenements"
                if any((evenements_dir / fn).exists() for fn in [
                    "POLYGONE_EVENEMENT.shp",
                    "LIGNE_EVENEMENT.shp",
                    "POINT_EVENEMENT.shp"
                ]):
                    debug('Couches déjà créées')
                    QMessageBox.warning(
                        self.iface.mainWindow(),
                        self.tr("Commande inutile : "),
                        self.tr("Les couches sont déjà présentes.")
                    )
                    return

        # Création du groupe s’il n’existe pas
        if not evt:
            if nomproj:
                groupevt = root.insertGroup(0, 'Evenements')
            else:
                debug('Projet non enregistré')
                QMessageBox.warning(
                    self.iface.mainWindow(),
                    self.tr("Création impossible"),
                    self.tr("Le projet doit être enregistré avant la création des couches Evenements.")
                )
                return

        # Définition du répertoire de sortie
        evt_dir = chemproj / "Evenements"
        evt_dir.mkdir(parents=True, exist_ok=True)

        # Définition CRS et options
        crs = self.project.crs()
        transform_context = self.project.transformContext()
        save_options = QgsVectorFileWriter.SaveVectorOptions()
        save_options.driverName = "ESRI Shapefile"
        save_options.fileEncoding = "UTF-8"

        # Définition des champs (algorithme inchangé)
        def champs_polygone():
            f = QgsFields()
            for name, t, *args in [
                ('libelle', QVariant.String),
                ('date', QVariant.String),
                ('h_creation', QVariant.String),
                ('source', QVariant.String),
                ('h_constat', QVariant.String),
                ('remarques', QVariant.String),
                ('surface', QVariant.Double, 'double', 10, 0),
                ('utilisatr', QVariant.String)
            ]:
                f.append(QgsField(name, t, *args))
            return f

        def champs_ligne():
            f = QgsFields()
            for name, t, *args in [
                ('libelle', QVariant.String),
                ('date', QVariant.String),
                ('h_creation', QVariant.String),
                ('source', QVariant.String),
                ('h_constat', QVariant.String),
                ('remarques', QVariant.String),
                ('longueur', QVariant.Double, 'double', 10, 0),
                ('utilisatr', QVariant.String)
            ]:
                f.append(QgsField(name, t, *args))
            return f

        def champs_point():
            f = QgsFields()
            for name, t, *args in [
                ('libelle', QVariant.String),
                ('date', QVariant.String),
                ('h_creation', QVariant.String),
                ('source', QVariant.String),
                ('h_constat', QVariant.String),
                ('remarques', QVariant.String),
                ('x_gps', QVariant.Double, 'double', 10, 6),
                ('y_gps', QVariant.Double, 'double', 10, 6),
                ('utilisatr', QVariant.String)
            ]:
                f.append(QgsField(name, t, *args))
            return f

        # Chemin vers les styles QML (dans les ressources embarquées)
        qmlbase = ":/plugins/gestion_crise/resources/"
        couches = [
            (evt_dir / "POLYGONE_EVENEMENT.shp", champs_polygone(), QgsWkbTypes.MultiPolygon,
             qmlbase + "POLYGONE_EVENEMENT.qml"),
            (evt_dir / "LIGNE_EVENEMENT.shp", champs_ligne(), QgsWkbTypes.MultiLineString,
             qmlbase + "LIGNE_EVENEMENT.qml"),
            (evt_dir / "POINT_EVENEMENT.shp", champs_point(), QgsWkbTypes.MultiPoint,
             qmlbase + "POINT_EVENEMENT.qml")
        ]

        # Création des shapefiles + insertion dans le groupe Evenements (algorithme inchangé)
        for path, fields, wkb, qml in couches:
            writer = QgsVectorFileWriter.create(str(path), fields, wkb, crs, transform_context, save_options)
            if writer.hasError() != QgsVectorFileWriter.NoError:
                QMessageBox.critical(
                    self.iface.mainWindow(),
                    self.tr("Erreur création shapefile"),
                    writer.errorMessage()
                )
                del writer
                continue
            del writer

            # Chargement dans QGIS
            layer = self.iface.addVectorLayer(str(path), path.stem, "ogr")
            layer.loadNamedStyle(qml)
            node = root.findLayer(layer.id())
            clone = node.clone()
            parent = node.parent()
            groupevt.insertChildNode(0, clone)
            parent.removeChildNode(node)
            layer.setReadOnly(True)
            layer.triggerRepaint()

        # Déplie le groupe Evenements
        for n in root.children():
            test = ''.join(x for x in unicodedata.normalize('NFKD', n.name())
                           if unicodedata.category(x)[0] == 'L').upper()
            n.setExpanded(test == 'EVENEMENTS')


# ============================================================================
# Boîte de message temporaire
# ============================================================================
class CustomMessageBox(QMessageBox):
    def __init__(self, *__args):
        super().__init__()
        self.timeout = 0
        self.autoclose = False
        self.currentTime = 0

    def showEvent(self, QShowEvent):
        self.currentTime = 0
        if self.autoclose:
            self.startTimer(1000)

    def timerEvent(self, *args, **kwargs):
        self.currentTime += 1
        if self.currentTime >= self.timeout:
            self.done(0)

    @staticmethod
    def showWithTimeout(timeoutSeconds, message, title, icon=qmessagebox_information):
        w = CustomMessageBox()
        w.autoclose = True
        w.timeout = timeoutSeconds
        w.setText(message)
        w.setWindowTitle(title)
        w.setIcon(icon)
        w.exec()
