# -*- coding: utf-8 -*-

# Qgeric: Graphical queries by drawing simple shapes.
# Author: Jérémy Kalsron
#         jeremy.kalsron@gmail.com
# Contributor : François Thévand
#         francois.thevand@gmail.com
# AttributesTable.py
import os
import unicodedata
from functools import partial

# from qgis.PyQt.uic.properties import QtWidgets
from qgis.PyQt import QtCore, QtGui
from qgis.PyQt.QtCore import (
    Qt, QDate, QTime, QDateTime, QTranslator, QCoreApplication,
    QSettings, QLocale, qVersion, QItemSelectionModel
)
from qgis.PyQt.QtWidgets import (
    QWidget, QTabWidget, QVBoxLayout, QProgressDialog, QStatusBar,
    QAction, QTableWidget, QTableWidgetItem, QFileDialog,
    QToolBar, QMessageBox, QDateEdit, QTimeEdit, QDateTimeEdit,
    QInputDialog, QMenu, QWidgetAction, QComboBox, QLineEdit
)
from qgis.PyQt.QtGui import QIcon, QColor, QGuiApplication
from qgis.core import (
    QgsWkbTypes, QgsVectorLayer, QgsProject, QgsGeometry,
    QgsCoordinateTransform, QgsLayerTreeGroup
)
from qgis.gui import QgsHighlight

from . import odswriter as ods
from . import resources

# Adaptation Qt5/Qt6 pour boutons de QMessageBox
from qgis.PyQt.QtCore import PYQT_VERSION_STR as pyqt_version
if pyqt_version.startswith("5"):
    yes_button = QMessageBox.Yes
    no_button = QMessageBox.No
    cancel_button = QMessageBox.Cancel
    information_icon = QMessageBox.Information
    question_icon = QMessageBox.Question
    critical_icon = QMessageBox.Critical
    warning_icon = QMessageBox.Warning
    qt_windows_modality = Qt.WindowModal
    qt_application_modal = Qt.ApplicationModal
    qt_window_stays_on_top_hint = Qt.WindowStaysOnTopHint
else:
    yes_button = QMessageBox.StandardButton.Yes
    no_button = QMessageBox.StandardButton.No
    cancel_button = QMessageBox.StandardButton.Cancel
    information_icon = QMessageBox.Icon.Information
    question_icon = QMessageBox.Icon.Question
    critical_icon = QMessageBox.Icon.Critical
    warning_icon = QMessageBox.Icon.Warning
    qt_windows_modality = Qt.WindowModality.WindowModal
    qt_application_modal = Qt.WindowModality.ApplicationModal
    qt_window_stays_on_top_hint = Qt.WindowType.WindowStaysOnTopHint

# ─── Compat Qt5 / Qt6 pour DisplayRole et ItemIsEditable ───
try:
    DisplayRole   = Qt.DisplayRole
    ItemIsEditable = Qt.ItemIsEditable
except AttributeError:
    # Qt6 : on puise dans les sous-énums Qt.ItemDataRole et Qt.ItemFlag
    DisplayRole   = Qt.ItemDataRole.DisplayRole
    ItemIsEditable = Qt.ItemFlag.ItemIsEditable


class AttributesTable(QWidget):
    def __init__(self, name, iface):
        super().__init__()
        self.iface = iface
        self.project = QgsProject.instance()
        self.root = self.project.layerTreeRoot()

        # i18n
        override = QSettings().value("locale/overrideFlag", False, type=bool)
        locale = (QLocale.system().name() if not override
                  else QSettings().value("locale/userLocale", ""))
        locale = locale[:2]
        qm = os.path.join(os.path.dirname(__file__),
                          f"i18n/qdrawEVT_{locale}.qm")
        if os.path.exists(qm):
            tr = QTranslator()
            tr.load(qm)
            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(tr)

        self.setWindowTitle(self.tr("Selection made in") + f" {name}")
        self.resize(480, 320)
        self.center()

        # Actions
        self.btn_save_tab = QAction(QIcon(':/plugins/qdrawEVT/resources/icon_save.png'),
                                    self.tr("Save this tab's results"), self)
        self.btn_save_tab.triggered.connect(
            lambda: self.saveAttributes(active=True))

        self.btn_save_all = QAction(QIcon(':/plugins/qdrawEVT/resources/icon_saveAll.png'),
                                    self.tr("Save all results"), self)
        self.btn_save_all.triggered.connect(
            lambda: self.saveAttributes(active=False))

        self.btn_export = QAction(QIcon(':/plugins/qdrawEVT/resources/icon_export.png'),
                                  self.tr("Export selection as memory layer"), self)
        self.btn_export.triggered.connect(self.exportLayer)

        self.btn_zoom = QAction(QIcon(':/plugins/qdrawEVT/resources/icon_Zoom.png'),
                                self.tr("Zoom to selected features"), self)
        self.btn_zoom.triggered.connect(self.zoomToFeature)

        self.btn_toggle = QAction(QIcon(':/plugins/qdrawEVT/resources/icon_HlG.png'),
                                  self.tr("Highlight feature geometry"), self)
        self.btn_toggle.triggered.connect(self.toggleHighlightMode)
        self.selectGeom = False

        # Onglets et barre de progression
        self.tabWidget = QTabWidget()
        self.tabWidget.setTabsClosable(True)
        self.tabWidget.tabCloseRequested.connect(self.closeTab)
        self.tabWidget.currentChanged.connect(self.highlight_features)

        # self.loading = QProgressDialog(self.tr("Loading..."), None, 0, 100, self)
        # self.loading.setWindowModality(Qt.WindowModality.WindowModal)
        # self.loading.setAutoClose(False)

        # 🔧 on nomme l'attribut loadingWindow pour coller à l'existant dans qdrawEVT.py
        self.loadingWindow = QProgressDialog(self.tr("Loading..."), None, 0, 100, self)
        self.loadingWindow.setWindowModality(Qt.WindowModality.WindowModal)
        self.loadingWindow.setAutoClose(False)
        self.loadingWindow.setCancelButton(None)
        # 🔧 alias si jamais vous aviez encore du code qui utilisait `self.loading`
        self.loading = self.loadingWindow

        # Toolbar
        toolbar = QToolBar()
        toolbar.addAction(self.btn_save_tab)
        toolbar.addAction(self.btn_save_all)
        toolbar.addSeparator()
        toolbar.addAction(self.btn_export)
        toolbar.addSeparator()
        toolbar.addAction(self.btn_zoom)
        toolbar.addSeparator()
        toolbar.addAction(self.btn_toggle)

        layout = QVBoxLayout(self)
        layout.setContentsMargins(0,0,0,0)
        layout.addWidget(toolbar)
        layout.addWidget(self.tabWidget)

    def center(self):
        screen = QGuiApplication.primaryScreen().geometry()
        geo = self.geometry()
        self.move((screen.width()-geo.width())//2,
                  (screen.height()-geo.height())//2)

    def toggleHighlightMode(self):
        self.selectGeom = not self.selectGeom
        icon = 'icon_HlC.png' if self.selectGeom else 'icon_HlG.png'
        text = (self.tr("Highlight centroid")
                if self.selectGeom else
                self.tr("Highlight geometry"))
        self.btn_toggle.setIcon(QIcon(f':/plugins/qdrawEVT/resources/{icon}'))
        self.btn_toggle.setText(text)
        self.highlight_features()

    def closeTab(self, idx):
        w = self.tabWidget.widget(idx)
        w.deleteLater()
        self.tabWidget.removeTab(idx)

    def addLayer(self, layer, headers, types, features, visible_cols):
        tab = QWidget()
        tab.layer = layer
        tab.crs = layer.crs()
        tab.sb = QStatusBar()

        table = QTableWidget()
        table.setColumnCount(len(headers))
        table.setRowCount(len(features))
        table.setHorizontalHeaderLabels(headers)
        table.horizontalHeader().setSectionsMovable(True)
        table.itemSelectionChanged.connect(self.highlight_features)
        table.title = layer.name()
        table.filters = [None]*len(headers)
        table.filter_ops = [0]*len(headers)

        for r, feat in enumerate(features):
            for c, idx in enumerate(visible_cols):
                item = QTableWidgetItem()
                item.setData(DisplayRole, feat[idx])
                item.setFlags(item.flags() ^ ItemIsEditable)
                item.feature = feat
                table.setItem(r, c, item)
            # self.loading.setValue(int((r+1)/len(features)*100))
            self.loadingWindow.setValue(int((r+1)/len(features)*100))
            QtGui.QGuiApplication.processEvents()

        lay = QVBoxLayout(tab)
        lay.setContentsMargins(0,0,0,0)
        lay.addWidget(table)
        lay.addWidget(tab.sb)

        title = f"{table.title[:20]}... ({len(features)})"
        self.tabWidget.addTab(tab, title)
        self.tabWidget.setTabToolTip(self.tabWidget.indexOf(tab), table.title)

    def highlight_features(self):
        # retire anciens highlights
        for h in getattr(self, 'highlights', []):
            self.iface.mapCanvas().scene().removeItem(h)
        self.highlights = []

        idx = self.tabWidget.currentIndex()
        if idx < 0: return
        tab = self.tabWidget.widget(idx)
        table = tab.findChild(QTableWidget)
        if not table: return

        items = table.selectedItems()
        if not items:
            tab.sb.clearMessage()
            return

        count = area = length = 0
        for it in items:
            feat = getattr(it, 'feature', None)
            geom = feat.geometry() if feat else None
            if not geom: continue

            shape = geom if self.selectGeom else geom.centroid()
            high = QgsHighlight(self.iface.mapCanvas(), shape, tab.layer)
            high.setColor(QColor(255,0,0))
            self.highlights.append(high)

            g2 = QgsGeometry(geom)
            g2.transform(QgsCoordinateTransform(tab.crs,
                                                QgsProject.instance().crs(),
                                                QgsProject.instance()))
            count += 1
            area += g2.area()
            length += g2.length()

        typ = tab.layer.geometryType()
        if typ == QgsWkbTypes.PolygonGeometry:
            tab.sb.showMessage(f"{self.tr('Selected')} {count}  {self.tr('Area')}: {area:.2f} m²")
        elif typ == QgsWkbTypes.LineGeometry:
            tab.sb.showMessage(f"{self.tr('Selected')} {count}  {self.tr('Length')}: {length:.2f} m")
        else:
            tab.sb.showMessage(f"{self.tr('Selected')} {count}")

    def zoomToFeature(self):
        idx = self.tabWidget.currentIndex()
        if idx < 0: return
        table = self.tabWidget.widget(idx).findChild(QTableWidget)
        ids = [it.feature.id() for it in table.selectedItems() if hasattr(it, 'feature')]
        if not ids: return
        canvas = self.iface.mapCanvas()
        if len(ids) == 1:
            geom = table.selectedItems()[0].feature.geometry()
            canvas.setExtent(geom.buffer(5,0).boundingBox())
        else:
            canvas.zoomToFeatureIds(table.layer, ids)
        canvas.refresh()

    def exportLayer(self):
        idx = self.tabWidget.currentIndex()
        if idx < 0: return
        table = self.tabWidget.widget(idx).findChild(QTableWidget)
        items = table.selectedItems()
        if not items: return

        geom_map = {
            QgsWkbTypes.PointGeometry: "Point",
            QgsWkbTypes.LineGeometry: "LineString",
            QgsWkbTypes.PolygonGeometry: "Polygon"
        }
        geom = items[0].feature.geometry().type()
        geom_type = geom_map.get(geom, "Point")

        name = f"{self.tr('Extract')} {table.title}"
        layer = QgsVectorLayer(f"{geom_type}?crs={table.crs.authid()}", name, "memory")
        layer.startEditing()
        layer.dataProvider().addAttributes(items[0].feature.fields().toList())
        feats = list({it.feature for it in items})
        layer.dataProvider().addFeatures(feats)
        layer.commitChanges()

        grp_name = f"{self.tr('Extracts in')} {self.windowTitle().split()[-1]}"
        grp = self.root.findGroup(grp_name) or \
              self.root.insertChildNode(0, QgsLayerTreeGroup(grp_name))
        self.project.addMapLayer(layer, False)
        grp.insertLayer(0, layer)
        self.expand_node(grp)

        src = QgsProject.instance().mapLayersByName(table.title)
        if src:
            self.iface.setActiveLayer(src[0])
            self.iface.actionCopyLayerStyle().trigger()
            self.iface.setActiveLayer(layer)
            self.iface.actionPasteLayerStyle().trigger()

    def expand_node(self, node):
        node.setExpanded(False)
        node.setExpanded(True)

    def saveAttributes(self, active):
        folder = os.path.join(self.project.homePath(), "Extractions")
        os.makedirs(folder, exist_ok=True)
        path, _ = QFileDialog.getSaveFileName(
            self, self.tr("Save in..."), folder,
            self.tr("OpenDocument Spreadsheet (*.ods)")
        )
        if not path:
            return False
        with ods.writer(path) as w:
            tables = ([self.tabWidget.currentWidget().findChild(QTableWidget)]
                      if active else
                      self.tabWidget.findChildren(QTableWidget))
            for tbl in tables[::-1]:
                sheet = w.new_sheet(tbl.title[:20] + "...")
                sheet.writerow([tbl.title])
                hdr = [tbl.horizontalHeaderItem(i).text()
                       for i in range(tbl.columnCount())]
                sheet.writerow(hdr)
                for r in range(tbl.rowCount()):
                    if tbl.isRowHidden(r):
                        continue
                    row = []
                    for c in range(tbl.columnCount()):
                        text = tbl.item(r, c).text() if tbl.item(r, c) else ""
                        row.append(self.parse_cell(text))
                    sheet.writerow(row)
        msg = QMessageBox()
        msg.setIcon(information_icon)
        msg.setWindowTitle(self.tr("Export terminé"))
        msg.setText(self.tr("Creation completed:\n") + path + "\n\n" + self.tr("Open file?"))
        msg.setStandardButtons(yes_button | no_button)
        msg.setDefaultButton(yes_button)
        msg.setWindowModality(qt_application_modal)
        msg.setWindowFlags(msg.windowFlags() | qt_window_stays_on_top_hint)

        # Affichage bloquant
        btn = msg.exec()

        if btn == yes_button:
            try:
                os.startfile(path)  # Windows uniquement
            except Exception:
                from qgis.PyQt.QtGui import QDesktopServices
                from qgis.PyQt.QtCore import QUrl
                QDesktopServices.openUrl(QUrl.fromLocalFile(path))


    def parse_cell(self, txt):
        try:
            return float(txt.replace(",", "."))
        except:
            return txt

    def clear(self):
        while self.tabWidget.count():
            w = self.tabWidget.widget(0)
            self.tabWidget.removeTab(0)
            w.deleteLater()

    def closeEvent(self, e):
        res = QMessageBox.question(
            self, self.tr("Saving?"), self.tr("Save before exit?"),
            yes_button | no_button | cancel_button
        )
        if res == yes_button:
            if self.saveAttributes(active=False):
                self.clear()
                e.accept()
            else:
                e.ignore()
        elif res == no_button:
            self.clear()
            e.accept()
        else:
            e.ignore()

    def tr(self, msg):
        return QCoreApplication.translate('QdrawEVT', msg)

