# TabbedConsole.py
import os
from qgis.PyQt.QtWidgets import QDialog, QVBoxLayout, QHBoxLayout, QStackedWidget, QWidget, QPushButton, QLabel
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QIcon

# Use your existing dialogs (pure Python)
from .ConfigDialog import ConfigDialog
from .CreateDialog import CreateDialog
from .UploadDialog import UploadDialog
from .PublishDialog import PublishDialog

class QCartaConsole(QDialog):
    """
    Main page with links that navigates to different dialogs:
      - Configure Access
      - Create Store
      - Update Store
      - Publish Map
    """
    def __init__(self, config, parent=None, save_callback=None):
        super().__init__(parent)
        self.config = config
        self.save_callback = save_callback
        self.setWindowTitle("QCarta Console")
        self.resize(900, 650)

        # Main layout - just the stacked widget
        main_layout = QVBoxLayout(self)
        main_layout.setContentsMargins(0, 0, 0, 0)
        
        # Stacked widget for content (main page + dialogs)
        self.content_stack = QStackedWidget()
        main_layout.addWidget(self.content_stack)
        
        # Create dialogs dictionary
        self.dialogs = {}
        
        # Validate and clean up selected server
        selected_server = config.get('_selected_server')
        if selected_server and selected_server not in config:
            # Selected server no longer exists, clear it
            config['_selected_server'] = None
            if self.save_callback:
                self.save_callback(config)
        
        # Create main page first, then dialogs
        self.create_main_page()
        self.setup_dialogs(config)
        
        # Start on main page
        self.content_stack.setCurrentIndex(0)
        
        # Connect to config dialog's server selection change
        if "Configure" in self.dialogs:
            self.dialogs["Configure"].selected_server_dropdown.currentTextChanged.connect(self.update_other_tabs)

    def create_main_page(self):
        """Create the main page with links to different sections"""
        main_page = QWidget()
        main_layout = QVBoxLayout(main_page)
        main_layout.setSpacing(20)
        main_layout.setContentsMargins(40, 40, 40, 40)
        
        # Get plugin directory for logo
        plugin_dir = os.path.dirname(__file__)
        logo_path = os.path.join(plugin_dir, 'logo.png')
        icon_dir = os.path.join(plugin_dir, 'icons')
        
        # Logo
        logo_label = QLabel()
        if os.path.exists(logo_path):
            logo_label.setPixmap(QIcon(logo_path).pixmap(180, 60))
        logo_label.setAlignment(Qt.AlignCenter)
        main_layout.addWidget(logo_label)
        
        # Title
        title_label = QLabel("<b style='font-size:18pt;'>QCarta Console</b>")
        title_label.setAlignment(Qt.AlignCenter)
        main_layout.addWidget(title_label)
        
        # Spacer
        main_layout.addStretch()
        
        # Navigation buttons - map button title to dialog title
        nav_items = [
            ("Configure", "Configure", ConfigDialog, "configure.png", "Add and configure QCarta servers"),
            ("Publish Map", "Publish", PublishDialog, "publish.png", "Publish your map to QCarta"),
            ("Advanced: Create Store", "Create Store", CreateDialog, "create.png", "Create a new QCarta store"),
            ("Advanced: Update Store", "Update Store", UploadDialog, "update.png", "Update an existing QCarta store"),
        ]
        
        for button_title, dialog_title, dialog_cls, icon_name, description in nav_items:
            # Create button container
            button_container = QWidget()
            button_layout = QHBoxLayout(button_container)
            button_layout.setContentsMargins(10, 10, 10, 10)
            
            # Create button with icon
            icon_path = os.path.join(icon_dir, icon_name)
            if os.path.exists(icon_path):
                icon = QIcon(icon_path)
                btn = QPushButton(icon, button_title)
            else:
                btn = QPushButton(button_title)
            
            btn.setMinimumHeight(60)
            btn.setStyleSheet("""
                QPushButton {
                    font-size: 14pt;
                    font-weight: bold;
                    text-align: left;
                    padding: 15px 20px;
                    background-color: #5D5D5D;
                    color: white;
                    border: 2px solid #36454F;
                    border-radius: 5px;
                }
                QPushButton:hover {
                    background-color: #36454F;
                    border-color: #2E8B57;
                }
                QPushButton:pressed {
                    background-color: #2E8B57;
                }
            """)
            
            # Store dialog title for navigation
            btn.dialog_title = dialog_title
            btn.clicked.connect(lambda checked, dt=dialog_title: self.navigate_to_dialog(dt))
            
            button_layout.addWidget(btn)
            main_layout.addWidget(button_container)
        
        # Spacer
        main_layout.addStretch()
        
        # Add main page to stack (index 0)
        self.content_stack.addWidget(main_page)
    
    def setup_dialogs(self, config):
        """Setup the dialogs"""
        # Get the plugin directory for icons
        plugin_dir = os.path.dirname(__file__)
        
        dialog_items = [
            ("Configure", ConfigDialog),
            ("Create Store", CreateDialog),
            ("Publish", PublishDialog),
            ("Update Store", UploadDialog),
        ]
        
        for title, dialog_cls in dialog_items:
            # Create dialog
            if title == "Configure":
                dlg = dialog_cls(config, self.save_callback)
            elif title == "Create Store":
                # Pass selected server and parent console to CreateDialog
                selected_server = config.get('_selected_server')
                dlg = dialog_cls(config, selected_server, self)
            else:
                # Pass selected server to other dialogs
                selected_server = config.get('_selected_server')
                dlg = dialog_cls(config, selected_server)
            dlg.setWindowFlags(Qt.Widget)
            # Neutralize accept/reject so inner dialogs don't close the whole console
            dlg.accept = lambda *a, **k: None
            dlg.reject = lambda *a, **k: None
            
            # Create container widget with back button
            container = QWidget()
            layout = QVBoxLayout(container)
            layout.setContentsMargins(0, 0, 0, 0)
            layout.setSpacing(0)
            
            # Add back button
            back_btn = QPushButton("← Back to Main Page")
            back_btn.setStyleSheet("""
                QPushButton {
                    font-size: 11pt;
                    padding: 8px 15px;
                    background-color: #5D5D5D;
                    color: white;
                    border: none;
                    border-bottom: 1px solid #36454F;
                }
                QPushButton:hover {
                    background-color: #36454F;
                }
            """)
            back_btn.clicked.connect(self.navigate_to_main)
            layout.addWidget(back_btn)
            
            layout.addWidget(dlg)
            layout.addStretch()  # Push content to top
            
            # Add to stacked widget (after main page, so index 1+)
            self.content_stack.addWidget(container)
            self.dialogs[title] = dlg
    
    def navigate_to_dialog(self, dialog_title):
        """Navigate to a specific dialog"""
        # Find the index of the dialog in the stack
        # Main page is at index 0, dialogs start at index 1
        dialog_index = list(self.dialogs.keys()).index(dialog_title) + 1
        self.content_stack.setCurrentIndex(dialog_index)
    
    def navigate_to_main(self):
        """Navigate back to the main page"""
        self.content_stack.setCurrentIndex(0)
    
    def update_other_tabs(self, selected_server):
        """Update other tabs when server selection changes in Configure tab"""
        # Update the config with the new selected server
        self.config['_selected_server'] = selected_server
        
        # Update other dialogs
        for title, dialog in self.dialogs.items():
            if title != "Configure":
                # Update the server label in other dialogs
                if hasattr(dialog, 'server_label'):
                    if selected_server and selected_server in self.config:
                        dialog.server_label.setText(f"Server: {selected_server}")
                        dialog.server_label.setStyleSheet("font-weight: bold; color: #2E8B57;")
                    else:
                        dialog.server_label.setText("Server: No server selected")
                        dialog.server_label.setStyleSheet("font-weight: bold; color: #DC143C;")
                
                # Update the selected server in the dialog
                if hasattr(dialog, 'selected_server'):
                    dialog.selected_server = selected_server
                
                # Trigger server change updates for dialogs that need it
                if hasattr(dialog, 'onServerChanged'):
                    dialog.onServerChanged()
    
    def refresh_store_lists(self):
        """Refresh store lists in all dialogs that have them"""
        for title, dialog in self.dialogs.items():
            if hasattr(dialog, 'updateStores'):
                dialog.updateStores()
