import os
import re
import requests
from qgis.PyQt.QtWidgets import QVBoxLayout, QMessageBox, QLineEdit, QDialog, QVBoxLayout, QLabel, QFormLayout, QComboBox, QComboBox, QHBoxLayout, QProgressBar, QTextEdit, QDialog, QVBoxLayout, QPushButton, QCheckBox, QListWidget, QListWidgetItem, QSizePolicy
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import Qt
from qgis.core import QgsProject, QgsMapLayer
from .util import app_http_login

class PublishDialog(QDialog):
    def __init__(self, config, selected_server=None):
        super().__init__()
    
        self.config = config
        self.selected_server = selected_server
        self.setWindowTitle("Create QCarta Layer")
        self.layout = QVBoxLayout()
    
        logo_label = QLabel()
        logo_path = os.path.join(os.path.dirname(__file__), 'logo.png')
        if os.path.exists(logo_path):
            logo_label.setPixmap(QIcon(logo_path).pixmap(120, 40))
        branding_label = QLabel("<b style='font-size:14pt;'>Publish Project</b><br><span style='font-size:10pt;'>Publish QCarta Project</span>")
        branding_label.setAlignment(Qt.AlignCenter)
    
        self.layout.addWidget(logo_label)
        self.layout.addWidget(branding_label)
    
        form_layout = QFormLayout()

        self.s = None
        self.stores = {}
        self.access_groups = {}
        self.basemaps = {}
        self.is_unsupported_server = False

        # Show selected server as label instead of dropdown
        if selected_server and selected_server in config:
            self.server_label = QLabel(f"Server: {selected_server}")
            self.server_label.setStyleSheet("font-weight: bold; color: #2E8B57;")
        else:
            self.server_label = QLabel("Server: No server selected")
            self.server_label.setStyleSheet("font-weight: bold; color: #DC143C;")
        
        self.store_dropdown = QComboBox()
        self.store_dropdown.currentIndexChanged.connect(self.updateLayers)
    
        self.layer_dropdown = QListWidget()
        self.layer_dropdown.setSelectionMode(QListWidget.ExtendedSelection)
        self.print_layout_dropdown = QComboBox()

        # optional: give inputs some minimum width so the dialog must grow
        self.store_dropdown.setMinimumWidth(250)

        self.layer_name = QLineEdit()
        self.layer_desc = QLineEdit()
        self.basemaps_dropdown = QComboBox()

        self.auto_generate_thumbnail = QCheckBox()

        self.option_public = QCheckBox('Public Access')
        self.option_customized = QCheckBox('Custom Styling')
        
        layer_options = QHBoxLayout()
        layer_options.addWidget(self.option_public)
        layer_options.addStretch()
        
        self.option_qcarta_cache = QCheckBox('Enabled')
        self.option_qcarta_cache.setChecked(True)

        qcarta_cache_box = QHBoxLayout()
        qcarta_cache_box.addWidget(self.option_qcarta_cache)
        qcarta_cache_box.addStretch()

        self.show_charts = QCheckBox('Charts tab')
        self.show_dt = QCheckBox('Data tables')
        self.show_query = QCheckBox('Query tab')
        self.show_fi_edit = QCheckBox('FeatureInfo Edit')

        show_box = QHBoxLayout()
        show_box.addWidget(self.show_charts)
        show_box.addWidget(self.show_query)
        show_box.addWidget(self.show_dt)
        show_box.addWidget(self.show_fi_edit)

        self.access_groups_dropdown = QListWidget()
        self.access_groups_dropdown.setSelectionMode(QListWidget.MultiSelection)
        
        # Checkbox for auto-creating store from project
        self.use_project_store = QCheckBox("Use store from current QGIS project (auto-create if missing)")
        self.use_project_store.setChecked(True)  # Default ON
        self.use_project_store.stateChanged.connect(self.onUseProjectStoreChanged)
        
        # Don't call updateBasemaps() and updateAccessGroups() here - they'll be called in onServerChanged()

        form_layout.addRow(self.server_label)
        form_layout.addRow("Store:", self.store_dropdown)
        form_layout.addRow("", self.use_project_store)
        form_layout.addRow("Layer:", self.layer_dropdown)
        form_layout.addRow("Print Layout:", self.print_layout_dropdown)

        form_layout.addRow("Name:", self.layer_name)
        form_layout.addRow("Description:", self.layer_desc)
        form_layout.addRow("Basemap:", self.basemaps_dropdown)
        
        form_layout.addRow("QCarta Cache:", qcarta_cache_box)
        form_layout.addRow("Layer options:", layer_options)        
        form_layout.addRow("Show options:", show_box)

        form_layout.addRow("Access Groups:", self.access_groups_dropdown)
    
        # Unsupported server message label (initially hidden)
        self.unsupported_message_label = QLabel("Unsupported QCarta server")
        self.unsupported_message_label.setStyleSheet("color: #DC143C; font-weight: bold; padding: 10px;")
        self.unsupported_message_label.setAlignment(Qt.AlignCenter)
        self.unsupported_message_label.setVisible(False)
        self.layout.addWidget(self.unsupported_message_label)
    
        self.layout.addLayout(form_layout)
    
        button_box = QHBoxLayout()
        self.create_btn = QPushButton("Create")
        cancel_btn = QPushButton("Cancel")
        button_box.addWidget(self.create_btn)
        button_box.addWidget(cancel_btn)
        self.layout.addLayout(button_box)
            
        self.setLayout(self.layout)
    
        # make the dialog open bigger and allow growing
        self.setMinimumSize(560, 500)
        self.resize(560, 500)  # initial size
        self.setSizeGripEnabled(True)
    
        # let form fields expand
        form_layout.setFieldGrowthPolicy(QFormLayout.AllNonFixedFieldsGrow)
    
        self.create_btn.clicked.connect(self.create_layer)
        cancel_btn.clicked.connect(self.reject)
        
        # Call onServerChanged after all widgets are created
        self.onServerChanged()
    
    def onServerChanged(self):
        # Reset unsupported state
        self.is_unsupported_server = False
        self.unsupported_message_label.setVisible(False)
        self.create_btn.setEnabled(True)
        
        if not self.selected_server or self.selected_server not in self.config:
            self.store_dropdown.clear()
            self.basemaps_dropdown.clear()
            self.access_groups_dropdown.clear()
            return
            
        server_info = self.config.get(self.selected_server, {})
        if not server_info or not isinstance(server_info, dict):
            self.store_dropdown.clear()
            self.basemaps_dropdown.clear()
            self.access_groups_dropdown.clear()
            return
            
        proto = 'https' if server_info.get('port', 443) == 443 else 'http'
        
        if self.s:
            self.s.close()
        self.s = requests.Session()
        
        try:
            if not app_http_login(self.s, proto, server_info['host'], server_info['username'], server_info['password']):
                QMessageBox.warning(None, "Login error", "Failed to login to with " + server_info['username'] + ' to ' + server_info['host'])
                self.s.close()
                self.s = None
                return
        except Exception as e:
            QMessageBox.warning(None, "HTTP error", "Failed on login: " + str(e))
            return

        # Check for unsupported server by attempting to fetch stores
        stores, is_supported = self.updateStores()
        
        if not is_supported:
            # Server is unsupported (legacy)
            self.is_unsupported_server = True
            self.unsupported_message_label.setVisible(True)
            self.create_btn.setEnabled(False)
            # Clear dropdowns since we can't use this server
            self.store_dropdown.clear()
            self.basemaps_dropdown.clear()
            self.access_groups_dropdown.clear()
            self.layer_dropdown.clear()
            self.print_layout_dropdown.clear()
            return
        
        # Server is supported - proceed with normal updates
        # Update layers based on checkbox state
        if self.use_project_store.isChecked():
            # If checkbox is checked, update layers from project store or project
            self.onUseProjectStoreChanged()
        else:
            # If checkbox is unchecked, update layers from selected store
            self.updateLayers()
        self.updateBasemaps()
        self.updateAccessGroups()
    
    def onUseProjectStoreChanged(self):
        # Enable/disable store dropdown based on checkbox state
        self.store_dropdown.setEnabled(not self.use_project_store.isChecked())
        
        # If checkbox is checked, populate layers from QGIS project
        if self.use_project_store.isChecked():
            try:
                project = QgsProject.instance()
                project_path = project.fileName()
                
                # Always populate from QGIS project first (so user can see layers immediately)
                layers = []
                try:
                    for layer_id, layer in project.mapLayers().items():
                        if layer and (layer.type() == QgsMapLayer.VectorLayer or layer.type() == QgsMapLayer.RasterLayer):
                            layer_name = layer.name()
                            if layer_name:  # Only add if layer has a name
                                layers.append(layer_name)
                except Exception as e:
                    # If we can't get layers, at least clear the dropdown
                    pass
                
                self.layer_dropdown.blockSignals(True)
                self.layer_dropdown.clear()
                if layers:
                    layers.sort()
                    for l in layers:
                        self.layer_dropdown.addItem(QListWidgetItem(l))
                    # Auto-select all layers if we have any
                    self.layer_dropdown.selectAll()
                self.layer_dropdown.blockSignals(False)
                
                # Get print layouts from project
                layouts = []
                try:
                    layout_manager = project.layoutManager()
                    if layout_manager:
                        for layout in layout_manager.layouts():
                            if layout and layout.name():
                                layouts.append(layout.name())
                except:
                    pass  # Layout manager might not be available
                
                self.print_layout_dropdown.blockSignals(True)
                self.print_layout_dropdown.clear()
                if layouts:
                    layouts.sort()
                    self.print_layout_dropdown.addItems(layouts)
                self.print_layout_dropdown.blockSignals(False)
                
                # If session is initialized, also try to get/store info to see if store exists
                # This is informational - we've already populated from QGIS project
                if project_path and self.s and self.selected_server and self.selected_server in self.config:
                    try:
                        store_name = self.sanitize_store_name(project_path)
                        server_info = self.config[self.selected_server]
                        store_info = self.get_store_info(server_info, store_name)
                        # If store exists and has layers, we could optionally update, but for now
                        # we'll keep the QGIS project layers since they're what the user sees
                    except:
                        pass  # Non-critical, we already have layers from QGIS project
            except Exception as e:
                # If anything fails, at least clear the dropdowns
                self.layer_dropdown.clear()
                self.print_layout_dropdown.clear()
        else:
            # Checkbox unchecked, update layers from selected store
            self.updateLayers()
    
    def sanitize_store_name(self, filename):
        """Sanitize a filename to create a valid store name."""
        # Get base filename without extension
        base_name = os.path.splitext(os.path.basename(filename))[0]
        # Replace invalid characters with underscores
        # Allow alphanumeric, underscores, hyphens
        sanitized = re.sub(r'[^a-zA-Z0-9_-]', '_', base_name)
        # Remove multiple consecutive underscores
        sanitized = re.sub(r'_+', '_', sanitized)
        # Remove leading/trailing underscores
        sanitized = sanitized.strip('_')
        # Ensure it's not empty
        if not sanitized:
            sanitized = 'qgis_project'
        return sanitized
    
    def read_in_chunks(self, file_object, chunk_size=65536):
        """Read file in chunks for upload."""
        while True:
            data = file_object.read(chunk_size)
            if not data:
                break
            yield data
    
    def ensure_store(self, store_name, group_ids):
        """
        Ensure a store exists. If not found, auto-create it.
        Returns store_id if successful, None otherwise.
        """
        if not self.selected_server or self.selected_server not in self.config:
            return None
        
        server_info = self.config[self.selected_server]
        proto = 'https' if server_info.get('port', 443) == 443 else 'http'
        
        # Check if store already exists
        try:
            response = self.s.get(proto + '://' + server_info['host'] + '/rest/store/' + store_name, timeout=(10, 30))
            if response.status_code == 200:
                response_json = response.json()
                if response_json.get('success') and 'store' in response_json:
                    # Store exists, return its id
                    return response_json['store']['id']
        except Exception as e:
            # Store doesn't exist or error, will create it
            pass
        
        # Store doesn't exist, create it
        project_path = QgsProject.instance().fileName()
        if not project_path:
            QMessageBox.warning(None, "No Project", "Please save the QGIS project first.")
            return None
        
        project_dir = os.path.dirname(project_path)
        
        qgs_list = []
        file_list = []
        
        try:
            # Walk through project directory to find .qgs/.qgz and other files
            for root, _, files in os.walk(project_dir):
                for file in files:
                    local_path = os.path.join(root, file)
                    relative_path = os.path.relpath(local_path, project_dir)
                    
                    if file.endswith('.qgs') or file.endswith('.qgz'):
                        qgs_list.append(file)
                        # Upload qgs/qgz file in chunks
                        with open(local_path, 'rb') as f:
                            offset = 0
                            post_values = {'action': 'upload_bytes', 'source': os.path.basename(local_path)}
                            
                            for chunk in self.read_in_chunks(f):
                                post_values['start'] = offset
                                post_values['bytes'] = chunk
                                response = self.s.post(proto + '://' + server_info['host'] + '/admin/action/upload.php', 
                                                     data=post_values, timeout=(10, 30))
                                if response.status_code != 200:
                                    raise Exception("Chunk upload failed")
                                offset = offset + len(chunk)
                    else:
                        file_list.append((local_path, relative_path))
        except Exception as e:
            QMessageBox.critical(None, "QGS Upload Failed", f"An error occurred: {e}")
            return None
        
        if not qgs_list:
            QMessageBox.warning(None, "No QGS File", "No .qgs or .qgz file found in project directory.")
            return None
        
        # Create store with qgs/qgz files
        post_values = {'action': 'save', 'name': store_name, 'group_id[]': group_ids, 'source[]': qgs_list}
        
        try:
            response = self.s.post(proto + '://' + server_info['host'] + '/admin/action/qgs.php', 
                                 data=post_values, timeout=(10, 30))
            if response.status_code != 200:
                response_json = response.json()
                QMessageBox.warning(None, "Create error", "Failed to create store: " + response_json.get('message', 'Unknown error'))
                return None
        except Exception as e:
            QMessageBox.critical(None, "Store Creation Failed", f"An error occurred: {e}")
            return None
        
        # Upload remaining files
        try:
            # Get store info to get id and post_max_size
            response = self.s.get(proto + '://' + server_info['host'] + '/rest/store/' + store_name, timeout=(10, 30))
            if response.status_code != 200:
                response_json = response.json()
                QMessageBox.warning(None, "REST error", "Failed to get store info: " + response_json.get('message', 'Unknown error'))
                return None
            
            store_info = response.json()['store']
            store_id = store_info['id']
            
            # Upload remaining files
            for local_path, relative_path in file_list:
                try:
                    with open(local_path, 'rb') as f:
                        offset = 0
                        post_values = {'action': 'upload_bytes', 'source': os.path.basename(local_path)}
                        
                        for chunk in self.read_in_chunks(f):
                            post_values['start'] = offset
                            post_values['bytes'] = chunk
                            response = self.s.post(proto + '://' + server_info['host'] + '/admin/action/upload.php', 
                                                 data=post_values, timeout=(10, 30))
                            if response.status_code != 200:
                                raise Exception("Chunk upload failed")
                            offset = offset + len(chunk)
                    
                    post_values = {'id': store_id, 'action': 'update_file', 'relative_path': relative_path, 
                                 'mtime': os.path.getmtime(local_path)}
                    response = self.s.post(proto + '://' + server_info['host'] + '/admin/action/qgs.php', 
                                         data=post_values, timeout=(10, 30))
                    if response.status_code != 200:
                        # Non-critical error, continue with other files
                        pass
                except Exception as e:
                    # Non-critical error, continue with other files
                    pass
            
            # Update stores dictionary and refresh dropdown
            self.stores[store_name] = {'id': store_id, 'name': store_name}
            self.updateStores()
            
            # Return store id
            return store_id
            
        except Exception as e:
            QMessageBox.warning(None, "File Upload Warning", f"Store created but some files may not have uploaded: {e}")
            # Still return store_id if we have it
            try:
                response = self.s.get(proto + '://' + server_info['host'] + '/rest/store/' + store_name, timeout=(10, 30))
                if response.status_code == 200:
                    response_json = response.json()
                    if response_json.get('success') and 'store' in response_json:
                        return response_json['store']['id']
            except:
                pass
            return None

    def updateStores(self):
        """
        Update stores dropdown. Returns tuple (store_list, is_supported).
        """
        if not self.selected_server or self.selected_server not in self.config:
            self.store_dropdown.clear()
            return [], True
            
        server_info = self.config.get(self.selected_server, {})
        if not server_info or not isinstance(server_info, dict):
            self.store_dropdown.clear()
            return [], True
            
        stores, is_supported = self.get_stores(server_info)
        stores.sort()
        
        self.store_dropdown.blockSignals(True)
        self.store_dropdown.clear()
        self.store_dropdown.addItems(stores)
        self.store_dropdown.blockSignals(False)
        
        return stores, is_supported
    
    def updateLayers(self):
        if not self.selected_server or self.selected_server not in self.config:
            self.layer_dropdown.clear()
            self.print_layout_dropdown.clear()
            return
            
        server_info = self.config.get(self.selected_server, {})
        if not server_info or not isinstance(server_info, dict):
            self.layer_dropdown.clear()
            self.print_layout_dropdown.clear()
            return
            
        store_name = self.store_dropdown.currentText()
        
        store_info = self.get_store_info(server_info,store_name)
        if not store_info or not isinstance(store_info, dict):
            self.layer_dropdown.clear()
            return
        
        # Handle both string and list formats for Layers and Layouts
        layers_data = store_info.get('Layers', [])
        layouts_data = store_info.get('Layouts', [])
        
        if isinstance(layers_data, str):
            layers = layers_data.split(',') if layers_data else []
        else:
            layers = layers_data if isinstance(layers_data, list) else []
            
        if isinstance(layouts_data, str):
            print_layouts = layouts_data.split(',') if layouts_data else []
        else:
            print_layouts = layouts_data if isinstance(layouts_data, list) else []
        
        self.layer_dropdown.blockSignals(True)
        self.layer_dropdown.clear()
        layers.sort()
        for l in layers:
            self.layer_dropdown.addItem(QListWidgetItem(l))
        self.layer_dropdown.blockSignals(False)
        
        self.print_layout_dropdown.blockSignals(True)
        self.print_layout_dropdown.clear()
        print_layouts.sort()
        self.print_layout_dropdown.addItems(print_layouts)
        self.print_layout_dropdown.blockSignals(False)
    
    def updateBasemaps(self):
        if not self.selected_server or self.selected_server not in self.config:
            self.basemaps_dropdown.clear()
            return
            
        server_info = self.config.get(self.selected_server, {})
        if not server_info or not isinstance(server_info, dict):
            self.basemaps_dropdown.clear()
            return

        self.basemaps_dropdown.blockSignals(True)
        self.basemaps_dropdown.clear()
        
        for g in self.get_basemaps(server_info):
            self.basemaps[g['name']] = g['id']
            self.basemaps_dropdown.addItem(g['name'])
        self.basemaps_dropdown.blockSignals(False)
    
    def updateAccessGroups(self):
        if not self.selected_server or self.selected_server not in self.config:
            self.access_groups_dropdown.clear()
            return
            
        server_info = self.config.get(self.selected_server, {})
        if not server_info or not isinstance(server_info, dict):
            self.access_groups_dropdown.clear()
            return

        self.access_groups_dropdown.blockSignals(True)
        self.access_groups_dropdown.clear()
        
        for g in self.get_access_groups(server_info):
            self.access_groups[g['name']] = g['id']
            self.access_groups_dropdown.addItem(QListWidgetItem(g['name']))
        self.access_groups_dropdown.blockSignals(False)
    
    def get_access_groups(self, server_info):
        rv = {}
        
        proto = 'https' if server_info['port'] == 443 else 'http'
        try:
            response = self.s.post(proto + '://' + server_info['host'] + '/admin/action/access_group.php', data={'action':'list'}, timeout=(10, 30))
            if response.status_code == 200:
                response = response.json()
                if response['success']:
                    rv = response['access_groups']
                else:
                    QMessageBox.warning(None, "QCarta Error", 'get_access_groups:' + response['message'])
        except Exception as e:
            QMessageBox.critical(None, "HTTP Error", f"An error occurred: {e}")

        return rv
    
    def get_basemaps(self, server_info):
        rv = {}
        
        proto = 'https' if server_info['port'] == 443 else 'http'
        try:
            response = self.s.post(proto + '://' + server_info['host'] + '/admin/action/basemap.php', data={'action':'list'}, timeout=(10, 30))
            if response.status_code == 200:
                response = response.json()
                if response['success']:
                    rv = response['basemaps']
                else:
                    QMessageBox.warning(None, "QCarta Error", 'get_basemaps:' + response['message'])
        except Exception as e:
            QMessageBox.critical(None, "HTTP Error", f"An error occurred: {e}")

        return rv

    def get_stores(self, server_info):
        """
        Fetch stores from server. Returns tuple (store_list, is_supported).
        is_supported is False if the server response doesn't match modern structure.
        Never throws KeyError - always returns defensively.
        """
        self.stores = {}  # Reset stores
        
        proto = 'https' if server_info.get('port', 443) == 443 else 'http'
        is_supported = True
        
        try:
            response = self.s.get(proto + '://' + server_info['host'] + '/rest/stores', timeout=(10, 30))
            if response.status_code == 200:
                try:
                    response_json = response.json()
                    # Check for modern structure: response must have 'stores' key with 'store' key
                    if not isinstance(response_json, dict):
                        is_supported = False
                    elif 'stores' not in response_json:
                        is_supported = False
                    elif not isinstance(response_json['stores'], dict):
                        is_supported = False
                    elif 'store' not in response_json['stores']:
                        is_supported = False
                    elif not isinstance(response_json['stores']['store'], list):
                        is_supported = False
                    else:
                        # Modern structure detected - parse stores
                        for s in response_json['stores']['store']:
                            if isinstance(s, dict) and 'name' in s:
                                self.stores[s['name']] = s
                except (KeyError, TypeError, ValueError) as e:
                    # Response doesn't match expected structure
                    is_supported = False
        except Exception as e:
            # Network/HTTP errors - treat as unsupported for safety
            is_supported = False

        return list(self.stores.keys()), is_supported
    
    def get_store_info(self, server_info, store_name):
        rv = {}
        
        proto = 'https' if server_info['port'] == 443 else 'http'
        try:
            response = self.s.get(proto + '://' + server_info['host'] + '/rest/store/' + store_name, timeout=(10, 30))
            if response.status_code == 200:
                response = response.json()
                if response['success']:
                    rv = response['store']
                else:
                    QMessageBox.warning(None, "QCarta Error", 'get_store_info:' + response['message'])
        except Exception as e:
            QMessageBox.critical(None, "HTTP Error", f"An error occurred: {e}")

        return rv
            
    def create_layer(self):
        # Prevent creation if server is unsupported
        if self.is_unsupported_server:
            return
        
        # Get server name - try selected_server first, fallback to extracting from label
        server_name = self.selected_server
        if not server_name or not server_name.strip():
            # Try to extract from server label as fallback
            label_text = self.server_label.text()
            if label_text and label_text.startswith("Server: ") and label_text != "Server: No server selected":
                server_name = label_text.replace("Server: ", "").strip()
        
        layer_name = self.layer_name.text().strip() if self.layer_name.text() else ""
        layer_desc = self.layer_desc.text().strip() if self.layer_desc.text() else ""

        # Get basemap ID safely
        basemap_text = self.basemaps_dropdown.currentText()
        if basemap_text and basemap_text in self.basemaps:
            basemap_id = self.basemaps[basemap_text]
        else:
            basemap_id = None

        print_layout = self.print_layout_dropdown.currentText()
        
        # convert access group names to ids
        map_access_groups = []
        for g in self.access_groups_dropdown.selectedItems():
            map_access_groups.append(self.access_groups[g.text()])

        # Validate inputs with more specific error messages
        if not server_name or (isinstance(server_name, str) and not server_name.strip()):
            QMessageBox.warning(self, "Missing Info", "Please select a server in the Configure tab.")
            return
        
        if server_name not in self.config:
            QMessageBox.warning(self, "Invalid Server", f"Selected server '{server_name}' is not configured. Please select a valid server in the Configure tab.")
            return
        
        if not layer_name:
            QMessageBox.warning(self, "Missing Info", "Please enter a layer name.")
            return
        
        if not map_access_groups:
            QMessageBox.warning(self, "Missing Info", "Please select layer access groups.")
            return

        server_info = self.config[server_name]
        proto = 'https' if server_info['port'] == 443 else 'http'
        
        # Get selected layers BEFORE we do anything that might clear the dropdown
        # QListWidget preserves the order of clicks in MultiSelection, so we get selected items through indexes
        selectedIndexes = []
        for idx in self.layer_dropdown.selectedIndexes():
            selectedIndexes.append(idx.row())
        selectedIndexes.sort()

        qgis_layers = []
        for i in selectedIndexes:
            item = self.layer_dropdown.item(i)
            if item:
                qgis_layers.append(item.text())
        
        # Check if we have layers to publish BEFORE ensuring store
        if not qgis_layers:
            QMessageBox.warning(self, "No Layers Selected", "Please select at least one layer to publish.")
            return
        
        # Determine store_id based on checkbox state
        store_id = None
        store_name = None
        if self.use_project_store.isChecked():
            # Derive store_name from project filename
            project_path = QgsProject.instance().fileName()
            if not project_path:
                QMessageBox.warning(None, "No Project", "Please save the QGIS project first.")
                return
            
            store_name = self.sanitize_store_name(project_path)
            store_id = self.ensure_store(store_name, map_access_groups)
            
            if store_id is None:
                QMessageBox.warning(None, "Store Error", "Failed to ensure store exists.")
                return
            
            # Optionally update layers dropdown with layers from the store (for display purposes)
            # But we already have the layers we need from above
            store_info = self.get_store_info(server_info, store_name)
            if store_info and isinstance(store_info, dict):
                # Update layers from this store (for UI consistency)
                layers_data = store_info.get('Layers', [])
                layouts_data = store_info.get('Layouts', [])
                
                if isinstance(layers_data, str):
                    store_layers = layers_data.split(',') if layers_data else []
                else:
                    store_layers = layers_data if isinstance(layers_data, list) else []
                    
                if isinstance(layouts_data, str):
                    print_layouts = layouts_data.split(',') if layouts_data else []
                else:
                    print_layouts = layouts_data if isinstance(layouts_data, list) else []
                
                # Only update dropdown if store has layers (might be empty if just created)
                if store_layers:
                    self.layer_dropdown.blockSignals(True)
                    self.layer_dropdown.clear()
                    store_layers.sort()
                    for l in store_layers:
                        self.layer_dropdown.addItem(QListWidgetItem(l))
                    # Try to restore selection by matching layer names
                    for i in range(self.layer_dropdown.count()):
                        item = self.layer_dropdown.item(i)
                        if item and item.text() in qgis_layers:
                            item.setSelected(True)
                    # If nothing selected, select all
                    if not self.layer_dropdown.selectedIndexes() and store_layers:
                        self.layer_dropdown.selectAll()
                    self.layer_dropdown.blockSignals(False)
                
                self.print_layout_dropdown.blockSignals(True)
                self.print_layout_dropdown.clear()
                if print_layouts:
                    print_layouts.sort()
                    self.print_layout_dropdown.addItems(print_layouts)
                self.print_layout_dropdown.blockSignals(False)
        else:
            # Use store from dropdown
            store_name = self.store_dropdown.currentText()
            if not store_name:
                QMessageBox.warning(self, "Missing Info", "Please select a store or enable 'Use store from current QGIS project'.")
                return
            
            if store_name not in self.stores:
                QMessageBox.warning(self, "Invalid Store", "Selected store not found.")
                return
            
            store_id = self.stores[store_name]['id']
        
        try:
            post_data = {'action':'save', 'id': 0, 'store_id': store_id, 'layers[]': qgis_layers, 'name':layer_name, 'description':layer_desc, 'print_layout': print_layout, 'group_id[]':map_access_groups}
            if basemap_id is not None:
                post_data['basemap_id'] = basemap_id
            if self.option_public.isChecked():
                post_data['public'] = 't'
            
            # Always set proxyfied, exposed, customized, and auto_thumbnail to 'f'
            post_data['proxyfied'] = 'f'
            post_data['exposed'] = 'f'
            post_data['customized'] = 'f'
            post_data['auto_thumbnail'] = 'f'
            
            # QCarta Cache checkbox controls the cached field
            if self.option_qcarta_cache.isChecked():
                post_data['cached'] = 't'
            else:
                post_data['cached'] = 'f'
            
            if self.show_charts.isChecked():
                post_data['show_charts'] = 't'
            if self.show_dt.isChecked():
                post_data['show_dt'] = 't'
            if self.show_query.isChecked():
                post_data['show_query'] = 't'
            if self.show_fi_edit.isChecked():
                post_data['show_fi_edit'] = 't'

            response = self.s.post(proto + '://' + server_info['host'] + '/admin/action/qgs_layer.php', data=post_data, timeout=(10, 30))
            if response.status_code != 200:
                QMessageBox.warning(None, "HTTP error", "HTTP code " + str(response.status_code))
                return

            response = response.json();
            if response['success']:
                layer_url = proto + '://' + server_info['host'] + '/layers/' + response['id'] + '/index.php';
                QMessageBox.information(None, "AcuGIS QCarta", 'Layer published. You can view it at <a href="' + layer_url +'">' + layer_url + '</a>')
            else:
                QMessageBox.warning(None, "QCarta error", response['message'])

        except Exception as e:
            QMessageBox.warning(None, "HTTP error", "HTTP code " + str(response.status_code))
            return

        self.accept()
