import os
from pathlib import Path

from qgis.core import Qgis, QgsApplication, QgsMessageLog, QgsMapLayerProxyModel
from qgis.PyQt.QtGui import QColor
from qgis.PyQt.QtWidgets import QFileDialog, QLabel, QMessageBox

from q4ts.processing.cut_mesh import CutMeshAlgorithm
from q4ts.processing.qgis_utils import get_current_q4ts_gpkg_folder
from q4ts.ui.algorithm_widget import AlgorithmWidget
from q4ts.ui.layer_selection_creation import LayerSelectionCreation
from q4ts.ui.ui_utils import (
    add_mesh_display_style,
    connect_zoom_selection,
    get_filepath_or_temp_output,
    import_mesh_layer,
)


class CutMeshWidget(AlgorithmWidget):
    def __init__(self, parent=None, iface=None):
        super().__init__(
            parent,
            iface,
            os.path.dirname(os.path.realpath(__file__)) + "/cut_mesh_widget.ui",
        )

        self.setWindowIcon(CutMeshAlgorithm().icon())

        self.inputMeshLayerCbx.setFilters(QgsMapLayerProxyModel.Filter.MeshLayer)
        self.inputMeshLayerCbx.layerChanged.connect(self.check_inout_file)
        connect_zoom_selection(self.inputMeshLayerCbx, self.zoomMeshButton, self._iface)

        self.input_layers = CutMeshAlgorithm().input_layers
        self.layer_combo = {}

        self.input_layer_layout.setColumnStretch(1, 1)
        for layer_name, input_layer in self.input_layers.items():
            row = self.input_layer_layout.rowCount()
            widget = LayerSelectionCreation(
                self,
                self._iface,
                layer_name,
                input_layer["style"],
                input_layer["type_wkb"],
                optional=input_layer["optional"],
                directory=input_layer["directory"],
            )
            if "fields" in input_layer:
                widget.setFields(input_layer["fields"])
            if "edit_form" in input_layer:
                widget.setEditForm(input_layer["edit_form"])
            widget.mMapLayerComboBox.layerChanged.connect(self.check_inout_file)
            self.layer_combo[layer_name] = widget
            self.input_layer_layout.addWidget(QLabel(input_layer["display"]), row, 0)
            self.input_layer_layout.addWidget(widget, row, 1)

        self.selectTempMedButton.clicked.connect(self.select_temp_med)
        self.selectOuputTempMedButton.clicked.connect(self.select_output_temp_med)
        self.selectResultSlfButton.clicked.connect(self.select_result_slf)

        self.tempMedEdit.setPlaceholderText(self.tr("[Save to temporary file]"))
        self.tempMedEdit.editingFinished.connect(self.check_inout_file)
        self.resultSlfEdit.setPlaceholderText(self.tr("[Save to temporary file]"))
        self.resultSlfEdit.editingFinished.connect(self.check_inout_file)

        self._set_not_running()
        self.check_inout_file()

    def _set_not_running(self):
        self._set_running(False, self.cut_mesh)

    def select_temp_med(self):
        default_dir = "temp.med"
        if self.resultSlfEdit.text():
            default_dir = str(Path(self.resultSlfEdit.text()).parent / default_dir)
        else:
            default_dir = get_current_q4ts_gpkg_folder() + default_dir
        filename, filter_used = QFileDialog.getSaveFileName(
            self, self.tr("Select file"), default_dir, self.tr("Mesh file (*.med)")
        )

        self.tempMedEdit.setText(filename)
        self.check_inout_file()

    def select_output_temp_med(self):
        default_dir = "output_temp.med"
        if self.resultSlfEdit.text():
            default_dir = str(Path(self.resultSlfEdit.text()).parent / default_dir)
        else:
            default_dir = get_current_q4ts_gpkg_folder() + default_dir
        filename, filter_used = QFileDialog.getSaveFileName(
            self, self.tr("Select file"), default_dir, self.tr("Mesh file (*.med)")
        )

        self.outputTempMedEdit.setText(filename)
        self.check_inout_file()

    def select_result_slf(self):
        default_dir = "cut_mesh.slf"
        if self.tempMedEdit.text():
            default_dir = str(Path(self.tempMedEdit.text()).parent / default_dir)
        else:
            default_dir = get_current_q4ts_gpkg_folder() + default_dir
        filename, filter_used = QFileDialog.getSaveFileName(
            self, self.tr("Select file"), default_dir, self.tr("Mesh file (*.slf)")
        )

        self.resultSlfEdit.setText(filename)
        self.check_inout_file()

    def check_inout_file(self):
        enable, tooltip = self._define_algorithm_status()

        self.createButton.setEnabled(enable)
        self.createButton.setToolTip(tooltip)

    def _define_algorithm_status(self, check_unsaved_layer: bool = False):
        enable = True
        tooltip = ""
        for layer_name, input_layer in self.input_layers.items():
            layer = self.layer_combo[layer_name].currentLayer()
            display_name = input_layer["display"]
            if not input_layer["optional"] and not layer:
                tooltip = self.tr(f"{display_name} must be defined")
                enable = False
            elif layer and check_unsaved_layer:
                enable &= self._check_if_layer_saved(display_name, layer)
        return enable, tooltip

    def cut_mesh(self):
        enable, _ = self._define_algorithm_status(check_unsaved_layer=True)
        if not enable:
            return

        params = {}
        for layer_name in self.input_layers.keys():
            params[layer_name] = self.layer_combo[layer_name].currentLayer()

        params[CutMeshAlgorithm.INPUT_MESH] = self.inputMeshLayerCbx.currentLayer()
        params[CutMeshAlgorithm.TEMP_MED] = get_filepath_or_temp_output(
            self.tempMedEdit
        )
        params[CutMeshAlgorithm.OUTPUT_MESH] = get_filepath_or_temp_output(
            self.resultSlfEdit
        )
        params[CutMeshAlgorithm.OUTPUT_MESH_MED] = get_filepath_or_temp_output(
            self.outputTempMedEdit
        )

        algo_str = "q4ts:cut_mesh"

        alg = QgsApplication.processingRegistry().algorithmById(algo_str)

        if alg:
            self._run_alg(alg, params, self.mesh_created)
        else:
            QMessageBox.critical(
                self,
                self.tr("Processing not available"),
                self.tr(f"{algo_str} processing not available."),
            )

    def mesh_created(self, context, successful, results):
        self._set_not_running()
        self.check_inout_file()

        if not successful:
            QgsMessageLog.logMessage(
                "Cut mesh finished unsucessfully",
                "Cut mesh",
                Qgis.MessageLevel.Warning,
            )
        else:
            mesh_layer = import_mesh_layer(
                self,
                results["OUTPUT_MESH"],
                "cut_mesh",
                self.resultSlfEdit,
                "creation",
            )
            add_mesh_display_style(mesh_layer, QColor("black"))
