# -*- coding: utf-8 -*-
"""
/***************************************************************************
 PygeoapiConfigDialog
                                 A QGIS plugin
 Update pygeoapi configuration file
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-05-16
        git sha              : $Format:%H$
        copyright            : (C) 2025 by ByteRoad
        email                : info@byteroad.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from copy import deepcopy
from datetime import datetime
import os
from wsgiref import headers
import requests
import yaml

from .utils.helper_functions import datetime_to_string
from .utils.data_diff import diff_yaml_dict

from .ui_widgets.utils import get_url_status

from .server_config_dialog import Ui_serverDialog

from .models.top_level.providers.records import ProviderTypes
from .ui_widgets.providers.NewProviderWindow import NewProviderWindow
from .ui_widgets.WarningDialog import ReadOnlyTextDialog
from .ui_widgets import DataSetterFromUi, UiSetter
from .models.ConfigData import ConfigData
from .models.top_level.utils import (
    InlineList,
    get_enum_value_from_string,
)
from .models.top_level.utils import STRING_SEPARATOR

from PyQt5 import QtWidgets, uic
from PyQt5.QtWidgets import (
    QMainWindow,
    QFileDialog,
    QMessageBox,
    QDialogButtonBox,
    QDialog,
    QApplication,
)  # or PyQt6.QtWidgets

from PyQt5.QtCore import (
    Qt,
    QModelIndex,
    QStringListModel,
    QSortFilterProxyModel,
)  # Not strictly needed, can use Python file API instead

# make imports optional for pytests
try:
    from qgis.core import (
        QgsMessageLog,
        QgsRasterLayer,
        QgsVectorLayer,
    )

    from qgis.gui import QgsMapCanvas
except:
    pass

headers = {"accept": "*/*", "Content-Type": "application/json; charset=utf-8"}


class ServerConfigDialog(QDialog, Ui_serverDialog):
    """
    Logic for the Server Configuration Dialog.
    Inherits from QDialog (functionality) and Ui_serverDialog (layout).
    """

    def __init__(self, parent=None):
        super().__init__(parent)
        self.setupUi(self)  # Builds the UI defined in Designer

        # Optional: Set default values based on current config if needed
        # self.ServerHostlineEdit.setText("localhost")

    def get_server_url(self):
        """
        Retrieve the server configuration data entered by the user.
        :return: A dictionary with 'host' and 'port' keys.
        """
        host = self.ServerHostlineEdit.text()
        port = self.ServerSpinBox.value()
        protocol = "http" if self.radioHttp.isChecked() else "https"
        return f"{protocol}://{host}:{port}/admin/config"


# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(
    os.path.join(os.path.dirname(__file__), "pygeoapi_config_dialog_base.ui")
)


class PygeoapiConfigDialog(QtWidgets.QDialog, FORM_CLASS):

    config_data: ConfigData
    yaml_original_data: dict
    ui_setter: UiSetter
    data_from_ui_setter: DataSetterFromUi
    current_res_name = ""

    # these need to be class properties, otherwise, without constant reference, they are not displayed in a widget
    provider_window: QMainWindow
    bbox_map_canvas: "QgsMapCanvas"
    bbox_base_layer: "QgsRasterLayer"
    bbox_extents_layer: "QgsVectorLayer"

    def __init__(self, parent=None):
        """Constructor."""
        super(PygeoapiConfigDialog, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)
        self.config_data = ConfigData()
        self.yaml_original_data = None
        self.ui_setter = UiSetter(self)
        self.data_from_ui_setter = DataSetterFromUi(self)

        class CustomDumper(yaml.SafeDumper):
            pass

        self.dumper = CustomDumper

        # make sure InlineList is represented as a YAML sequence (e.g. for 'bbox')
        self.dumper.add_representer(
            InlineList,
            lambda dumper, data: dumper.represent_sequence(
                "tag:yaml.org,2002:seq", data, flow_style=True
            ),
        )

        # make sure datetime items are not saved as strings with quotes
        def represent_datetime_as_timestamp(dumper, data: datetime):
            value = datetime_to_string(data)

            # emit as YAML timestamp → plain scalar, no quotes
            return dumper.represent_scalar("tag:yaml.org,2002:timestamp", value)

        self.dumper.add_representer(datetime, represent_datetime_as_timestamp)

        # custom assignments
        self.model = QStringListModel()
        self.proxy = QSortFilterProxyModel()

        self.ui_setter.customize_ui_on_launch()
        self.ui_setter.set_ui_from_data()
        self.ui_setter.setup_map_widget()

    def on_button_clicked(self, button):

        role = self.buttonBox.buttonRole(button)
        print(f"Button clicked: {button.text()}, Role: {role}")

        # You can also check the standard button type
        if button == self.buttonBox.button(QDialogButtonBox.Save):
            # proceed only if UI data inputs are valid
            if self._set_validate_ui_data()[0]:

                if self.serverRadio.isChecked():
                    # check #1: show diff with "Procced" and "Cancel" options
                    diff_approved, processed_config_data = (
                        self._diff_original_and_current_data()
                    )
                    if not diff_approved:
                        return

                    self.server_config(data_to_push=processed_config_data)
                else:
                    # check #1: show diff with "Procced" and "Cancel" options
                    diff_approved, processed_config_data = (
                        self._diff_original_and_current_data(get_yaml_output=True)
                    )
                    if not diff_approved:
                        return

                    file_path, _ = QFileDialog.getSaveFileName(
                        self, "Save File", "", "YAML Files (*.yml);;All Files (*)"
                    )
                    # check #2: valid file path
                    if file_path:
                        self.save_to_file(processed_config_data, file_path)

        elif button == self.buttonBox.button(QDialogButtonBox.Open):
            if self.serverRadio.isChecked():
                self.server_config(data_to_push=None)
            else:
                file_name, _ = QFileDialog.getOpenFileName(
                    self, "Open File", "", "YAML Files (*.yml);;All Files (*)"
                )
                self.open_file(file_name)

        elif button == self.buttonBox.button(QDialogButtonBox.Close):
            self.reject()
            return

    def server_config(self, data_to_push: dict | None = None):

        dialog = ServerConfigDialog(self)

        if dialog.exec_():
            url = dialog.get_server_url()
            if data_to_push is not None:
                self.push_to_server(url, data_to_push)
            else:
                self.pull_from_server(url)

    def push_to_server(self, url, data_to_push: dict):

        QMessageBox.information(
            self,
            "Information",
            f"Pushing configuration to: {url}",
        )

        # TODO: support authentication through the QT framework
        try:
            # Send the PUT request to Admin API
            response = requests.put(url, headers=headers, json=data_to_push)
            response.raise_for_status()

            QgsMessageLog.logMessage(f"Success! Status Code: {response.status_code}")

            QMessageBox.information(
                self,
                "Information",
                f"Success! Status Code: {response.status_code}",
            )

        except requests.exceptions.RequestException as e:
            QgsMessageLog.logMessage(f"An error occurred: {e}")
            QMessageBox.critical(
                self,
                "Error",
                f"An error occurred pushing the configuration to the server: {e}",
            )

    def pull_from_server(self, url):

        QMessageBox.information(
            self,
            "Information",
            f"Pulling configuration from: {url}",
        )

        # TODO: support authentication through the QT framework
        try:
            # Send the GET request to Admin API
            response = requests.get(url, headers=headers)
            response.raise_for_status()

            QgsMessageLog.logMessage(f"Success! Status Code: {response.status_code}")

            QMessageBox.information(
                self,
                "Information",
                f"Success! Status Code: {response.status_code}",
            )

            QgsMessageLog.logMessage(f"Response: {response.text}")

            data_dict = response.json()
            self.update_config_data_and_ui(data_dict)

        except requests.exceptions.RequestException as e:
            QgsMessageLog.logMessage(f"An error occurred: {e}")

            QMessageBox.critical(
                self,
                "Error",
                f"An error occurred pulling the configuration from the server: {e}",
            )

    def save_to_file(self, new_config_data: dict, file_path: str):

        if file_path:
            QApplication.setOverrideCursor(Qt.WaitCursor)
            try:
                with open(file_path, "w", encoding="utf-8") as file:
                    yaml.dump(
                        new_config_data,
                        file,
                        Dumper=self.dumper,
                        default_flow_style=False,
                        sort_keys=False,
                        allow_unicode=True,
                        indent=4,
                    )

                # try/except in case of running it from pytests
                try:
                    QgsMessageLog.logMessage(f"File saved to: {file_path}")
                except:
                    pass

            except Exception as e:
                QgsMessageLog.logMessage(f"Error saving file: {e}")
            finally:
                QApplication.restoreOverrideCursor()

    def open_file(self, file_name):

        if not file_name:
            return

        # exit Resource view
        self.exit_resource_edit()

        try:
            # QApplication.setOverrideCursor(Qt.WaitCursor)
            with open(file_name, "r", encoding="utf-8") as file:
                file_content = file.read()
                yaml_original_data_dict = yaml.safe_load(file_content)

                self.update_config_data_and_ui(yaml_original_data_dict)

        except Exception as e:
            QMessageBox.warning(self, "Error", f"Cannot open file:\n{str(e)}")
        # finally:
        #     QApplication.restoreOverrideCursor()

    def update_config_data_and_ui(self, data_dict):
        """Use the data from local file or local server to reset the ConfigData and UI."""

        # reset data
        self.config_data = ConfigData()

        # set data and .all_missing_props:
        self.yaml_original_data = deepcopy(data_dict)
        self.config_data.set_data_from_yaml(data_dict)

        # set UI from data
        self.ui_setter.set_ui_from_data()

        # log messages about missing or mistyped values during deserialization
        # try/except in case of running it from pytests
        try:
            QgsMessageLog.logMessage(
                f"Errors during deserialization: {self.config_data.error_message}"
            )
            QgsMessageLog.logMessage(
                f"Default values used for missing YAML fields: {self.config_data.defaults_message}"
            )

            # summarize all properties missing/overwitten with defaults
            # atm, warning with the full list of properties
            all_missing_props = self.config_data.all_missing_props
            QgsMessageLog.logMessage(
                f"All missing or replaced properties: {all_missing_props}"
            )

            if len(all_missing_props) > 0:
                ReadOnlyTextDialog(
                    self,
                    "Warning",
                    f"All missing or replaced properties (check logs for more details): {all_missing_props}",
                ).exec_()
        except:
            pass  # QgsMessageLog import error in pytests, ignore

    def _set_validate_ui_data(self) -> tuple[bool, list]:
        # Set and validate data from UI
        try:
            self.data_from_ui_setter.set_data_from_ui()
            invalid_props = self.config_data.validate_config_data()
            if len(invalid_props) > 0:

                # in case of running from pytests
                try:
                    QgsMessageLog.logMessage(
                        f"Properties are missing or have invalid values: {invalid_props}"
                    )
                    ReadOnlyTextDialog(
                        self,
                        "Warning",
                        f"Properties are missing or have invalid values: {invalid_props}",
                    ).exec_()
                except:
                    pass

                return False, invalid_props
            return True, []

        except Exception as e:
            QgsMessageLog.logMessage(f"Error deserializing: {e}")
            QMessageBox.warning(f"Error deserializing: {e}")
            return

    def _diff_original_and_current_data(
        self, get_yaml_output=False
    ) -> tuple[bool, dict]:
        """Before saving the file, show the diff and give an option to proceed or cancel."""

        new_config_data = self.config_data.asdict_enum_safe(
            self.config_data, datetime_to_str=True
        )

        # if created from skratch, no original data to compare to
        if not self.yaml_original_data:
            return True, new_config_data

        diff_data = diff_yaml_dict(
            self.yaml_original_data,
            new_config_data,
        )

        # if get_yaml_output, preserve datetime objects without string conversion.
        # This is needed so the yaml dumper is using representer removing quotes from datetime strings
        if get_yaml_output:
            new_config_data = self.config_data.asdict_enum_safe(
                self.config_data, datetime_to_str=False
            )

        # if no diff detected, directly accept the changes
        if (
            len(diff_data["added"])
            + len(diff_data["removed"])
            + len(diff_data["changed"])
            == 0
        ):
            return True, new_config_data

        # if diff detected, show a window with the choice to approve the diff
        QgsMessageLog.logMessage(f"{diff_data}")
        dialog = ReadOnlyTextDialog(self, "Warning", diff_data, True)
        result = dialog.exec_()  # returns QDialog.Accepted (1) or QDialog.Rejected (0)

        if result == QDialog.Accepted:
            return True, new_config_data
        else:
            return False, None

    def open_templates_path_dialog(self):
        """Defining Server.templates.path path, called from .ui file."""

        folder_path = QFileDialog.getExistingDirectory(None, "Select Folder")

        if folder_path:
            self.lineEditTemplatesPath.setText(folder_path)

    def open_templates_static_dialog(self):
        """Defining Server.templates.static path, called from .ui file."""

        folder_path = QFileDialog.getExistingDirectory(None, "Select Folder")

        if folder_path:
            self.lineEditTemplatesStatic.setText(folder_path)

    def open_logfile_dialog(self):
        """Defining Logging.logfile path, called from .ui file."""

        logFile = QFileDialog.getSaveFileName(
            self, "Save Log", "", "log Files (*.log);;All Files (*)"
        )

        if logFile:
            self.lineEditLogfile.setText(logFile[0])

    #################################################################
    ################## methods that are called from .ui file:
    #################################################################

    def add_server_lang(self):
        """Add language to Server Languages list, called from .ui file."""
        self.ui_setter.add_listwidget_element_from_lineedit(
            line_edit_widget=None,
            list_widget=self.listWidgetServerLangs,
            locale_combobox=self.comboBoxServerLangs,
            allow_repeated_locale=False,
            sort=False,
        )

    def add_metadata_id_title(self):
        """Add title to metadata, called from .ui file."""
        self.ui_setter.add_listwidget_element_from_lineedit(
            line_edit_widget=self.addMetadataIdTitleLineEdit,
            list_widget=self.listWidgetMetadataIdTitle,
            locale_combobox=self.comboBoxIdTitleLocale,
            allow_repeated_locale=False,
            sort=True,
        )

    def add_metadata_id_description(self):
        """Add description to metadata, called from .ui file."""
        self.ui_setter.add_listwidget_element_from_lineedit(
            line_edit_widget=self.addMetadataIdDescriptionLineEdit,
            list_widget=self.listWidgetMetadataIdDescription,
            locale_combobox=self.comboBoxIdDescriptionLocale,
            allow_repeated_locale=False,
            sort=True,
        )

    def add_metadata_keyword(self):
        """Add keyword to metadata, called from .ui file."""
        self.ui_setter.add_listwidget_element_from_lineedit(
            line_edit_widget=self.addMetadataKeywordLineEdit,
            list_widget=self.listWidgetMetadataIdKeywords,
            locale_combobox=self.comboBoxKeywordsLocale,
            allow_repeated_locale=True,
            sort=True,
        )

    def add_res_title(self):
        """Called from .ui file."""
        self.ui_setter.add_listwidget_element_from_lineedit(
            line_edit_widget=self.addResTitleLineEdit,
            list_widget=self.listWidgetResTitle,
            locale_combobox=self.comboBoxResTitleLocale,
            allow_repeated_locale=False,
            sort=True,
        )

    def add_res_description(self):
        """Called from .ui file."""
        self.ui_setter.add_listwidget_element_from_lineedit(
            line_edit_widget=self.addResDescriptionLineEdit,
            list_widget=self.listWidgetResDescription,
            locale_combobox=self.comboBoxResDescriptionLocale,
            allow_repeated_locale=False,
            sort=True,
        )

    def add_res_keyword(self):
        """Called from .ui file."""
        self.ui_setter.add_listwidget_element_from_lineedit(
            line_edit_widget=self.addResKeywordsLineEdit,
            list_widget=self.listWidgetResKeywords,
            locale_combobox=self.comboBoxResKeywordsLocale,
            allow_repeated_locale=True,
            sort=True,
        )

    def add_res_link(self):
        """Called from .ui file."""
        self.ui_setter.add_listwidget_element_from_multi_widgets(
            line_widgets_mandatory=[
                self.addResLinksTypeLineEdit,
                self.addResLinksRelLineEdit,
                self.addResLinksHrefLineEdit,
            ],
            line_widgets_optional=[
                self.addResLinksTitleLineEdit,
                self.addResLinkshreflangComboBox,
                self.addResLinksLengthLineEdit,
            ],
            list_widget=self.listWidgetResLinks,
            sort=False,
        )

    def try_add_res_provider(self, provider_index=None, data: list[str] | None = None):
        """Called from .ui file, and from this class."""
        provider_type: ProviderTypes = get_enum_value_from_string(
            ProviderTypes, self.comboBoxResProviderType.currentText().lower()
        )

        if not data:
            self.provider_window = NewProviderWindow(provider_type)
            provider_index = None

        else:
            # if the window is triggered for editing, ignore widget provider type and read it from data instead
            provider_type = get_enum_value_from_string(ProviderTypes, data[0])
            self.provider_window = NewProviderWindow(provider_type, data[1:])

        # add or replace provider data to ConfigData when user clicks 'Add'
        self.provider_window.signal_provider_values.connect(
            lambda provider_window, values: self._validate_and_add_res_provider(
                provider_window, values, provider_type, provider_index
            )
        )

    def _validate_and_add_res_provider(
        self, provider_window, values, provider_type, provider_index: int | None = None
    ):
        """Calls the Provider validation method and displays a warning if data is invalid."""
        invalid_fields = self.config_data.set_validate_new_provider_data(
            values, self.current_res_name, provider_type, provider_index
        )

        self.ui_setter.set_providers_ui_from_data(
            self.config_data.resources[self.current_res_name]
        )
        if len(invalid_fields) > 0:
            QMessageBox.warning(
                provider_window,
                "Warning",
                f"Invalid Provider values: {invalid_fields}",
            )
        else:
            self.provider_window.signal_provider_close.emit()

    def validate_res_extents_crs(self):
        """Called from .ui file."""
        url = self.data_from_ui_setter.get_extents_crs_from_ui(self)
        get_url_status(url, self)

    def delete_server_lang(self):
        """Delete Server language from list, called from .ui file."""
        self.ui_setter.delete_list_widget_selected_item(self.listWidgetServerLangs)

    def delete_metadata_id_title(self):
        """Delete keyword from metadata, called from .ui file."""
        self.ui_setter.delete_list_widget_selected_item(self.listWidgetMetadataIdTitle)

    def delete_metadata_id_description(self):
        """Delete keyword from metadata, called from .ui file."""
        self.ui_setter.delete_list_widget_selected_item(
            self.listWidgetMetadataIdDescription
        )

    def delete_metadata_keyword(self):
        """Delete keyword from metadata, called from .ui file."""
        self.ui_setter.delete_list_widget_selected_item(
            self.listWidgetMetadataIdKeywords
        )

    def delete_res_title(self):
        """Called from .ui file."""
        self.ui_setter.delete_list_widget_selected_item(self.listWidgetResTitle)

    def delete_res_description(self):
        """Called from .ui file."""
        self.ui_setter.delete_list_widget_selected_item(self.listWidgetResDescription)

    def delete_res_keyword(self):
        """Called from .ui file."""
        self.ui_setter.delete_list_widget_selected_item(self.listWidgetResKeywords)

    def delete_res_link(self):
        """Called from .ui file."""
        self.ui_setter.delete_list_widget_selected_item(self.listWidgetResLinks)

    def edit_res_provider(self):
        """Called from .ui file."""
        selected_items = self.listWidgetResProvider.selectedItems()
        if selected_items:
            item = selected_items[0]  # get the first (and only) selected item
            data_list = item.text().split(STRING_SEPARATOR)
            self.try_add_res_provider(self.listWidgetResProvider.row(item), data_list)

    def delete_res_provider(self):
        """Called from .ui file."""

        # first, get selected item text and delete matching provider from Resource providers
        self.data_from_ui_setter.delete_selected_provider_type_and_name(
            self.listWidgetResProvider
        )
        # then, remove the item from the list widget
        self.ui_setter.delete_list_widget_selected_item(self.listWidgetResProvider)

    def filterResources(self, filter):
        """Called from .ui."""
        self.proxy.setDynamicSortFilter(True)
        self.proxy.setFilterFixedString(filter)

    def exit_resource_edit(self):
        """Switch widgets to Preview, reset selected resource. Called from .ui and from this class too."""
        # hide detailed collection UI, show preview
        self.groupBoxCollectionLoaded.hide()
        self.groupBoxCollectionSelect.show()
        self.groupBoxCollectionPreview.show()
        self.ui_setter.refresh_resources_list_ui()

    def save_resource_edit_and_preview(self):
        """Save current changes to the resource data, reset widgets to Preview. Called from .ui."""

        invalid_fields = self.data_from_ui_setter.get_invalid_resource_ui_fields()
        if len(invalid_fields) > 0:
            QMessageBox.warning(
                self,
                "Warning",
                f"Invalid fields' values: {invalid_fields}",
            )
            return

        self.data_from_ui_setter.set_resource_data_from_ui()

        # reset the current resource name, refresh UI list
        self.current_res_name = self.lineEditResAlias.text()
        self.exit_resource_edit()

    def preview_resource(self, model_index: QModelIndex = None):
        """Display basic Resource info, called from .ui."""
        self.ui_setter.preview_resource(model_index)

    def delete_resource(self):
        """Delete selected resource. Called from .ui."""
        # hide detailed collection UI, show preview
        if self.current_res_name == "":
            return

        reply = QMessageBox.question(
            self,
            "Confirm action",
            f"Delete resource '{self.current_res_name}'?",
            QMessageBox.Yes | QMessageBox.No,
        )
        if reply == QMessageBox.Yes:
            self.config_data.delete_resource(self)
            self.ui_setter.preview_resource()
            self.ui_setter.refresh_resources_list_ui()
            self.current_res_name = ""

    def new_resource(self):
        """Called from .ui."""
        # add resource and reload UI
        new_name = self.config_data.add_new_resource()
        self.ui_setter.refresh_resources_list_ui()

        # visually select new resource
        self.ui_setter.select_listcollection_item_by_text(new_name)

        # set new resource as current and load details
        self.current_res_name = new_name
        self.load_resource()

    def load_resource(self):
        """Called from .ui and from this class too."""

        # if no resource selected, do nothing
        if self.current_res_name == "":
            return

        # hide preview collection UI, show detailed UI
        self.groupBoxCollectionPreview.hide()
        self.groupBoxCollectionSelect.hide()
        self.groupBoxCollectionLoaded.show()

        res_data = self.config_data.resources[self.current_res_name]
        # self.ui_setter.setup_resouce_loaded_ui(res_data)

        # first, set ConfigData from UI (e.g. in case language was changed)
        self.data_from_ui_setter.set_data_from_ui()

        # set the values to Resource UI widgets
        self.ui_setter.set_resource_ui_from_data(res_data)
