from __future__ import annotations

from typing import Any

from pyvista import Actor
from qtpy.QtGui import QColor, QIcon
from qtpy.QtWidgets import (
    QButtonGroup,
    QDialog,
    QDoubleSpinBox,
    QFormLayout,
    QGroupBox,
    QHBoxLayout,
    QRadioButton,
    QWidget,
)

from .colors import QICSColorButton


class pvRendererDialog(QDialog):
    def __init__(self, parent: pvRenderer):  # noqa: F821
        super().__init__(parent)
        self.renderer = parent.renderers.active_renderer

        self.setModal(False)  # ignored by self.exec() method !
        self.setWindowIcon(QIcon())
        self.setWindowTitle("3D View settings")
        self.setLayout(QFormLayout())
        # vertical exaggeration
        self.layout().addRow(
            "Vertical exaggeration:", self._init_vertical_exaggeration()
        )
        # background color
        self.layout().addRow(self._init_background())
        # compass
        self.layout().addRow(self._init_orientation_widget())
        # bounding box
        self.layout().addRow(self._init_bbox())

    @property
    def set(self):
        return self.parent().set

    def get(self, name: str, default: Any = None):
        return getattr(self.parent(), name, default)

    def _init_vertical_exaggeration(self) -> QWidget:
        zscale = QDoubleSpinBox()
        zscale.setMinimum(0.01)
        zscale.setMaximum(100)
        zscale.setValue(self.renderer.scale[-1])

        def callback(z):
            self.set("scale", zscale=z)

        zscale.valueChanged.connect(callback)
        return zscale

    def _init_background(self) -> QWidget:
        color = QColor.fromRgbF(*self.renderer.GetBackground())
        color2 = QColor.fromRgbF(*self.renderer.GetBackground2())

        widget = QGroupBox("Background color")
        top = QICSColorButton(color2)
        bot = QICSColorButton(color)
        stack = QFormLayout()
        stack.addRow("top:", top)
        stack.addRow("bottom:", bot)
        widget.setLayout(stack)

        def callback():
            self.set("background", color=bot.color.name(), top=top.color.name())

        top.colorChanged.connect(callback)
        bot.colorChanged.connect(callback)

        return widget

    def _init_orientation_widget(self) -> QWidget:
        ren = self.get("renderer")
        axes = ren.axes_widget if hasattr(ren, "axes_widget") else None
        try:
            color = axes.GetOrientationMarker().prop.color.hex_rgb
        except AttributeError:
            color = "red"
        widget = QGroupBox("Compass")
        widget.setLayout(QHBoxLayout())
        group = QButtonGroup(widget)
        none_opt = QRadioButton("none")
        axes_opt = QRadioButton("axes")
        arrow_opt = QRadioButton("arrow")
        color_btn = QICSColorButton(color)
        widget.layout().addWidget(none_opt)
        widget.layout().addWidget(axes_opt)
        widget.layout().addWidget(arrow_opt)
        widget.layout().addWidget(color_btn)

        group.setExclusive(True)
        group.addButton(none_opt)
        group.addButton(axes_opt)
        group.addButton(arrow_opt)

        if axes:
            if isinstance(axes, Actor):
                arrow_opt.setChecked(True)
            else:
                axes_opt.setChecked(True)
        else:
            none_opt.setChecked(True)

        def callback():
            geometry = group.checkedButton().text()
            if color := color_btn.color:
                color = color.name()
                self.set("orientation_widget", geometry=geometry, color=color)

        group.buttonClicked.connect(callback)
        color_btn.colorChanged.connect(callback)

        return widget

    def _init_bbox(self) -> QWidget:
        # bbox = self.get("bbox", "none")
        widget = QGroupBox("Bounding box")
        widget.setLayout(QFormLayout())

        # w_shape = QWidget()
        # w_shape.setLayout(QHBoxLayout())

        # none = QRadioButton("none")
        # none.setChecked(bbox not in ["box", "ticks", "grid"])
        # none.clicked.connect(lambda: self.set_bbox("none"))
        # box = QRadioButton("box")
        # box.setChecked(bbox == "box")
        # box.clicked.connect(lambda: self.set_bbox("box"))
        # grid = QRadioButton("grid")
        # grid.setChecked(bbox == "grid")
        # grid.clicked.connect(lambda: self.set_bbox("grid"))

        # w_shape.layout().addWidget(none)
        # w_shape.layout().addWidget(box)
        # w_shape.layout().addWidget(grid)
        # widget.layout().addRow(w_shape)

        # picker = QtColorButton(self, color=self.default("bbox_color", "grey"))
        # picker.colorChanged.connect(lambda c: self.set_bbox(color=c.name()))
        # widget.layout().addRow("color", picker)

        # width = QDoubleSpinBox()
        # width.setMinimum(0.01)
        # width.setMaximum(100)
        # width.setValue(float(self.default("bbox_width", 1.0)))
        # width.valueChanged.connect(lambda w: self.set_bbox(width=w))
        # widget.layout().addRow("width", width)

        return widget
